<?php
/**
 * Kalium WordPress Theme
 *
 * Blog posts widget for Elementor.
 *
 * @author Laborator
 * @link   https://kaliumtheme.com
 */
namespace Kalium\Elementor\Widgets;

use Elementor\Plugin;
use Elementor\Widget_Base;
use Elementor\Controls_Manager;
use Elementor\Group_Control_Border;
use Elementor\Group_Control_Box_Shadow;
use Kalium\Elementor\Control_Sets\Query_Posts;

class Blog_Posts extends Widget_Base {

	/**
	 * Register widget.
	 */
	public static function register() {
		Plugin::instance()->widgets_manager->register( new self() );
	}

	/**
	 * Get element name.
	 *
	 * @return string
	 */
	public function get_name() {
		return 'kalium-blog-posts';
	}

	/**
	 * Get element title.
	 *
	 * @return string
	 */
	public function get_title() {
		return 'Blog Posts';
	}

	/**
	 * Get element icon.
	 *
	 * @return string
	 */
	public function get_icon() {
		return 'eicon-post-list';
	}

	/**
	 * Get widget categories.
	 *
	 * @return array
	 */
	public function get_categories() {
		return [ 'kalium' ];
	}

	/**
	 * Register controls.
	 */
	protected function register_controls() {

		/**
		 * General.
		 */

		$this->start_controls_section(
			'section_layout',
			[
				'label' => 'Layout',
			]
		);

		// Layout
		$this->add_control(
			'layout',
			[
				'type'    => Controls_Manager::CHOOSE,
				'label'   => 'Layout',
				'options' => [
					'list' => [
						'title' => 'List',
						'icon'  => 'eicon-post-list',
					],
					'grid' => [
						'title' => 'Grid',
						'icon'  => 'eicon-posts-grid',
					],
				],
				'toggle'  => false,
				'default' => kalium_get_default_theme_option( 'blog_layout' ),
			]
		);

		// Columns
		$grid_columns = kalium_get_default_theme_option( 'blog_grid_columns' );

		$this->add_responsive_control(
			'grid_cols',
			[
				'type'           => Controls_Manager::NUMBER,
				'label'          => 'Columns',
				'default'        => kalium_get_responsive_value( $grid_columns ),
				'tablet_default' => kalium_get_responsive_value( $grid_columns, 'tablet' ),
				'mobile_default' => kalium_get_responsive_value( $grid_columns, 'mobile' ),
				'min'            => 0,
				'max'            => 6,
				'condition'      => [
					'layout' => 'grid',
				],
			]
		);

		// Gap
		$this->add_responsive_control(
			'grid_gap',
			[
				'type'      => Controls_Manager::SLIDER,
				'label'     => 'Gap',
				'default'   => [
					'size' => 30,
				],
				'selectors' => [
					'{{WRAPPER}} .blog-posts' => sprintf( '%s: {{SIZE}}px', kalium_css_var_name( 'grid-gap' ) ),
				],
			]
		);

		// Masonry mode
		$this->add_control(
			'grid_masonry_mode',
			[
				'type'      => Controls_Manager::SWITCHER,
				'label'     => 'Masonry Mode',
				'condition' => [
					'layout' => 'grid',
				],
				'default'   => kalium_get_default_theme_option( 'blog_grid_masonry_mode' ),
			]
		);

		$this->end_controls_section();

		/**
		 * Query section.
		 */

		// Query portfolio section
		Query_Posts::add_controls(
			$this,
			[
				'post_type'  => 'post',
				'term_field' => 'slug',
			]
		);

		/**
		 * Card options.
		 */

		$this->start_controls_section(
			'section_general',
			[
				'label' => 'Card Options',
			]
		);

		// Featured Image
		$this->add_control(
			'post_card_featured_image',
			[
				'type'    => Controls_Manager::SWITCHER,
				'label'   => 'Featured Image',
				'default' => kalium_elementor_switcher_control_value( kalium_get_default_theme_option( 'blog_thumbnails' ) ),
			]
		);

		// Title
		$this->add_control(
			'post_card_title',
			[
				'type'    => Controls_Manager::SWITCHER,
				'label'   => 'Title',
				'default' => kalium_elementor_switcher_control_value( kalium_get_default_theme_option( 'blog_post_title' ) ),
			]
		);

		// Excerpt
		$this->add_control(
			'post_card_excerpt',
			[
				'type'    => Controls_Manager::SWITCHER,
				'label'   => 'Excerpt',
				'default' => kalium_elementor_switcher_control_value( kalium_get_default_theme_option( 'blog_post_excerpt' ) ),
			]
		);

		// Meta
		$this->add_control(
			'post_card_meta',
			[
				'type'    => Controls_Manager::SWITCHER,
				'label'   => 'Post Meta',
				'default' => kalium_elementor_switcher_control_value( kalium_get_default_theme_option( 'blog_post_meta' ) ),
			]
		);

		$this->end_controls_section();

		/**
		 * Featured image.
		 */

		$this->start_controls_section(
			'section_featured_image',
			[
				'label'     => 'Featured Image',
				'condition' => [
					'post_card_featured_image' => 'yes',
				],
			]
		);

		$this->add_control(
			'thumbnail_hover_effect',
			[
				'type'    => Controls_Manager::SWITCHER,
				'label'   => 'Hover Effect',
				'default' => kalium_elementor_switcher_control_value( kalium_get_default_theme_option( 'blog_thumbnail_hover_effect' ) ),
			]
		);

		$this->add_control(
			'thumbnail_placeholder',
			[
				'type'    => Controls_Manager::SWITCHER,
				'label'   => 'Image Placeholder',
				'default' => kalium_elementor_switcher_control_value( kalium_get_default_theme_option( 'blog_thumbnails_placeholder' ) ),
			]
		);

		$this->add_control(
			'heading_image_size',
			[
				'type'      => Controls_Manager::HEADING,
				'label'     => 'Image Size',
				'separator' => 'before',
			]
		);

		$this->add_control(
			'thumbnail_size',
			[
				'type'    => Controls_Manager::SELECT,
				'label'   => 'Image Size',
				'options' => function () {
					return kalium_get_image_size_names( [ 'custom' => true ] );
				},
				'default' => 'large',
			]
		);

		$this->add_control(
			'thumbnail_size_custom',
			[
				'type'      => Controls_Manager::IMAGE_DIMENSIONS,
				'label'     => 'Custom Size',
				'condition' => [
					'thumbnail_size' => 'custom',
				],
			]
		);

		$this->add_control(
			'thumbnail_aspect_ratio',
			[
				'type'    => Controls_Manager::SELECT,
				'label'   => 'Aspect Ratio',
				'default' => 'auto',
				'options' => kalium_get_aspect_ratio_names(
					[
						'custom' => true,
					]
				),
			]
		);

		$this->add_responsive_control(
			'thumbnail_aspect_ratio_custom',
			[
				'type'        => Controls_Manager::TEXT,
				'label'       => 'Custom Aspect Ratio',
				'placeholder' => 'e.g. 16 / 9',
				'ai'          => [
					'active' => false,
				],
				'selectors'   => [
					'{{WRAPPER}} .blog-posts' => sprintf( '%s: {{VALUE}}', kalium_css_var_name( 'post-item-image-ratio' ) ),
				],
				'condition'   => [
					'thumbnail_aspect_ratio' => 'custom',
				],
			]
		);

		$this->add_control(
			'heading_post_formats',
			[
				'type'      => Controls_Manager::HEADING,
				'label'     => 'Post Formats',
				'separator' => 'before',
			]
		);

		$this->add_control(
			'post_formats',
			[
				'type'  => Controls_Manager::SWITCHER,
				'label' => 'Post Format Content',
			]
		);

		$this->add_control(
			'post_format_icon',
			[
				'type'  => Controls_Manager::SWITCHER,
				'label' => 'Post Format Icon',
			]
		);

		$this->end_controls_section();

		/**
		 * Post excerpt.
		 */

		$this->start_controls_section(
			'section_excerpt',
			[
				'label'     => 'Excerpt',
				'condition' => [
					'post_card_excerpt' => 'yes',
				],
			]
		);

		$this->add_control(
			'post_excerpt_length',
			[
				'type'    => Controls_Manager::NUMBER,
				'label'   => 'Max Words',
				'default' => kalium_get_default_theme_option( 'blog_post_excerpt_length' ),
				'min'     => 5,
			]
		);

		$this->end_controls_section();

		/**
		 * Post meta.
		 */

		$this->start_controls_section(
			'section_meta',
			[
				'label'     => 'Post Meta',
				'condition' => [
					'post_card_meta' => 'yes',
				],
			]
		);

		$this->add_control(
			'post_meta_date',
			[
				'type'    => Controls_Manager::SWITCHER,
				'label'   => 'Date',
				'default' => kalium_elementor_switcher_control_value( kalium_get_default_theme_option( 'blog_post_date' ) ),
			]
		);

		$this->add_control(
			'post_meta_category',
			[
				'type'    => Controls_Manager::SWITCHER,
				'label'   => 'Category',
				'default' => kalium_elementor_switcher_control_value( kalium_get_default_theme_option( 'blog_category' ) ),
			]
		);

		$this->add_control(
			'post_meta_icon',
			[
				'type'      => Controls_Manager::SWITCHER,
				'label'     => 'Display Icon',
				'default'   => kalium_elementor_switcher_control_value( kalium_get_default_theme_option( 'blog_post_type_icon' ) ),
				'separator' => 'before',
			]
		);

		$this->end_controls_section();

		/**
		 * Cart style.
		 */

		$this->start_controls_section(
			'section_style_card',
			[
				'label' => 'Card',
				'tab'   => Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'card_bg',
			[
				'type'      => Controls_Manager::COLOR,
				'label'     => 'Background',
				'selectors' => [
					'{{WRAPPER}} .blog-posts' => sprintf( '%s: {{VALUE}}', kalium_css_var_name( 'post-item-bg' ) ),
				],
			]
		);

		$this->add_control(
			'card_title',
			[
				'type'      => Controls_Manager::COLOR,
				'label'     => 'Title',
				'selectors' => [
					'{{WRAPPER}} .blog-posts' => sprintf( '%s: {{VALUE}}', kalium_css_var_name( 'post-item-title' ) ),
				],
			]
		);

		$this->add_control(
			'card_text',
			[
				'type'      => Controls_Manager::COLOR,
				'label'     => 'Text',
				'selectors' => [
					'{{WRAPPER}} .blog-posts' => sprintf( '%s: {{VALUE}}', kalium_css_var_name( 'post-item-text' ) ),
				],
			]
		);

		$this->add_control(
			'card_padding',
			[
				'type'       => Controls_Manager::DIMENSIONS,
				'label'      => 'Padding',
				'size_units' => kalium()->elementor->get_size_units( 'length' ),
				'selectors'  => [
					'{{WRAPPER}} .blog-posts' => sprintf( '%s: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}}', kalium_css_var_name( 'post-item-padding' ) ),
				],
			]
		);

		$this->add_group_control(
			Group_Control_Border::get_type(),
			[
				'name'     => 'card_border',
				'selector' => '{{WRAPPER}} .blog-posts .post-item',
			]
		);

		$this->add_control(
			'card_radius',
			[
				'type'       => Controls_Manager::DIMENSIONS,
				'label'      => 'Border Radius',
				'size_units' => kalium()->elementor->get_size_units( 'length' ),
				'selectors'  => [
					'{{WRAPPER}} .blog-posts' => sprintf( '%s: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}}', kalium_css_var_name( 'post-item-border-radius' ) ),
				],
			]
		);

		$this->add_group_control(
			Group_Control_Box_Shadow::get_type(),
			[
				'name'     => 'card_shadow',
				'selector' => '{{WRAPPER}} .blog-posts .post-item',
			]
		);

		$this->end_controls_section();
	}

	/**
	 * Render.
	 */
	protected function render() {
		$settings = $this->get_settings_for_display();

		// Blog options
		$blog_options = [

			// Layout
			'layout' => $settings['layout'],

			// Loop
			'loop'   => [

				// Thumbnails
				'post_thumbnail'      => [
					'visible'      => kalium_validate_boolean( $settings['post_card_featured_image'] ),

					// Hover layer
					'hover'        => [
						'visible' => kalium_validate_boolean( $settings['thumbnail_hover_effect'] ),
					],

					// Placeholder image
					'placeholder'  => [
						'enabled' => kalium_validate_boolean( $settings['thumbnail_placeholder'] ),
					],

					// Image size
					'size'         => kalium_if_else( 'custom' === $settings['thumbnail_size'], kalium_list_dimensions( $settings['thumbnail_size_custom'] ), $settings['thumbnail_size'] ),

					// Aspect ratio
					'aspect_ratio' => kalium_map_responsive_value(
						kalium_parse_elementor_responsive_value( 'thumbnail_aspect_ratio', $settings ),
						function ( $value, $viewport ) use ( &$settings ) {
							if ( 'custom' === $value ) {
								return kalium_get_responsive_value(
									kalium_parse_elementor_responsive_value( 'thumbnail_aspect_ratio_custom', $settings ),
									$viewport
								);
							}

							return $value;
						}
					),
				],

				// Title
				'post_title'          => kalium_validate_boolean( $settings['post_card_title'] ),

				// Excerpt
				'post_excerpt'        => kalium_validate_boolean( $settings['post_card_excerpt'] ),

				// Post excerpt length
				'post_excerpt_length' => $settings['post_excerpt_length'],

				// Post meta
				'post_meta'           => [
					'visible'       => kalium_validate_boolean( $settings['post_card_meta'] ),
					'icon'          => kalium_validate_boolean( $settings['post_meta_icon'] ),
					'post_date'     => kalium_validate_boolean( $settings['post_meta_date'] ),
					'post_category' => kalium_validate_boolean( $settings['post_meta_category'] ),
				],

				// Post formats
				'post_formats'        => kalium_validate_boolean( $settings['post_formats'] ),

				// Post format icon
				'post_format_icon'    => kalium_validate_boolean( $settings['post_format_icon'] ),

				// Columns
				'columns'             => kalium_parse_elementor_responsive_value( 'grid_cols', $settings ),

				// Masonry mode
				'masonry_mode'        => kalium_validate_boolean( $settings['grid_masonry_mode'] ),
			],
		];

		// Query
		$query = Query_Posts::get_value(
			$this,
			[
				'post_type' => 'post',
			]
		);

		// Query args (vars)
		$query_args = Query_Posts::to_query_args(
			$query,
			[
				'term_field' => 'slug',
			]
		);

		// Init blog options
		kalium_init_blog_options( $this->get_id(), $blog_options );

		// Query
		query_posts( $query_args );

		// Loop before
		kalium_blog_loop_before();

		// Posts loop
		kalium_blog_posts_loop();

		// Loop after
		kalium_blog_loop_after();

		// Reset post data
		wp_reset_postdata();
		wp_reset_query();
	}
}
