<?php
/**
 * Kalium WordPress Theme
 *
 * Theme backups manager.
 *
 * @author Laborator
 * @link   https://kaliumtheme.com
 */
namespace Kalium\Admin;

use Kalium\Core\Settings;
use Theme_Upgrader;
use WP_Error;
use WP_Upgrader;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Direct access not allowed.
}

class Backups {

	/**
	 * Theme backups status option.
	 *
	 * @type string
	 */
	const THEME_BACKUPS_OPTION = 'kalium_theme_backups';

	/**
	 * Theme backup file name.
	 *
	 * @type string
	 */
	const BACKUP_FILE_NAME = 'kalium-{version}-{date}.zip';

	/**
	 * Construct.
	 */
	public function __construct() {

		// Create theme backups before upgrade
		add_filter( 'upgrader_pre_download', [ $this, 'create_theme_backup' ], 1000, 4 );

		// Settings
		add_action( 'kalium_settings_init', [ $this, 'register_setting' ] );
		add_action( 'kalium_settings_section_backups', [ $this, 'settings' ], 100 );
	}

	/**
	 * Get the status for theme backups.
	 *
	 * @return bool
	 */
	public function enabled() {
		return get_option( self::THEME_BACKUPS_OPTION );
	}

	/**
	 * Get saved backups.
	 *
	 * @return array
	 */
	public function get_backups() {
		$backups = [];

		// Load backups
		$upload_dir    = wp_upload_dir();
		$theme_backups = glob( $upload_dir['basedir'] . '/' . preg_replace( '/\{\w+\}/i', '*', self::BACKUP_FILE_NAME ) );

		// Insert backups
		if ( is_array( $theme_backups ) ) {
			foreach ( $theme_backups as $theme_backup ) {
				$relative_backup_file = str_replace( ABSPATH, '', $theme_backup );
				$base_name            = basename( $theme_backup );

				$backups[] = [
					'relative_path' => $relative_backup_file,
					'base_name'     => $base_name,
					'time'          => filemtime( $theme_backup ),
					'size'          => size_format( filesize( $theme_backup ) ),
					'url'           => $upload_dir['baseurl'] . '/' . $base_name,
				];
			}
		}

		// Sort by created date
		uasort(
			$backups,
			function ( $a, $b ) {
				return $a['time'] > $b['time'] ? - 1 : 1;
			}
		);

		return $backups;
	}

	/**
	 * Before updating the theme, create theme backup (if enabled).
	 *
	 * @param bool                       $reply
	 * @param string                     $package
	 * @param Theme_Upgrader|WP_Upgrader $upgrader
	 * @param array                      $hook_extra
	 *
	 * @return string|WP_Error
	 */
	public function create_theme_backup( $reply, $package, $upgrader, $hook_extra ) {
		$theme_name = wp_basename( get_template_directory() );

		if ( $theme_name !== kalium_get_array_key( $hook_extra, 'theme' ) ) {
			return $reply;
		}

		if ( false === kalium()->license->is_active() ) {
			return $reply;
		}

		// Create Theme Backup
		if ( kalium()->backups->enabled() ) {
			$find_replace     = [
				'{version}' => str_replace( '.', '', kalium()->get_version() ),
				'{date}'    => date( 'dmy-Hi' ),
			];
			$upload_dir       = wp_upload_dir();
			$backup_file_name = str_replace( array_keys( $find_replace ), array_values( $find_replace ), self::BACKUP_FILE_NAME );
			$backup_file_path = $upload_dir['basedir'] . '/' . $backup_file_name;

			// Feedback strings
			$upgrader->strings['creating_theme_backup'] = 'Creating theme backup&hellip;';
			$upgrader->strings['theme_backup_created']  = sprintf( 'Theme backup created at <strong>%s</strong>', str_replace( ABSPATH, '', $backup_file_path ) );

			// Creating Theme Backup message
			$upgrader->skin->feedback( 'creating_theme_backup' );

			// Initialize Filesystem API
			kalium()->filesystem->initialize();

			// Zip current version of theme
			$result = kalium()->filesystem->zip_file( kalium()->locate_file(), $backup_file_path );

			// On error
			if ( is_wp_error( $result ) ) {
				return new WP_Error( 'theme_backup_creation_error', '<span>Cannot create theme backup, upgrade process failed.</span>' );
			}

			// Theme backup created
			$upgrader->skin->feedback( 'theme_backup_created' );
		}

		return $reply;
	}

	/**
	 * Register settings.
	 */
	public function register_setting() {
		kalium()->settings->register(
			'theme-backups',
			[
				'setting'           => self::THEME_BACKUPS_OPTION,
				'default'           => true,
				'autoload'          => false,
				'initialize'        => true,
				'input_type'        => 'switch',
				'sanitize_callback' => 'kalium_validate_boolean',
			]
		);
	}

	/**
	 * Backup settings.
	 */
	public function settings() {
		$theme_backups = $this->get_backups();

		?>
		<div class="kalium-dashboard__setting">
			<div class="kalium-dashboard__setting__label">
				<h3>Theme Backups</h3>
				<p>Theme backups are created when you update the theme.</p>
			</div>
			<div class="kalium-dashboard__setting__control">
				<?php Settings::setting( 'theme-backups' ); ?>
			</div>
		</div>

		<?php if ( ! empty( $theme_backups ) ) : ?>
			<ul class="kalium-dashboard__backups">
				<li class="kalium-dashboard__backups__entry kalium-dashboard__backups__entry--heading">
					<div class="kalium-dashboard__backups__entry__name">File name</div>
					<div class="kalium-dashboard__backups__entry__date">Date</div>
					<div class="kalium-dashboard__backups__entry__size">Size</div>
					<div class="kalium-dashboard__backups__entry__actions">Actions</div>
				</li>
				<?php

				foreach ( $theme_backups as $theme_backup ) :
					?>
					<li class="kalium-dashboard__backups__entry">
						<div class="kalium-dashboard__backups__entry__name"><?php echo esc_html( $theme_backup['base_name'] ); ?></div>
						<div class="kalium-dashboard__backups__entry__date"><?php echo esc_html( date_i18n( 'F d, Y', $theme_backup['time'] ) ); ?></div>
						<div class="kalium-dashboard__backups__entry__size"><?php echo esc_html( $theme_backup['size'] ); ?></div>
						<div class="kalium-dashboard__backups__entry__actions">
							<a href="<?php echo esc_url( $theme_backup['url'] ); ?>" target="_blank" rel="noreferrer noopener">Download</a>
						</div>
					</li>
					<?php
				endforeach;
				?>
			</ul>
			<?php
		endif;
	}
}
