<?php
/**
 * Kalium WordPress Theme
 *
 * Assets manager.
 *
 * @author Laborator
 * @link   https://kaliumtheme.com
 */
namespace Kalium\Core;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Direct access not allowed.
}

class Assets {

	/**
	 * Handle prefix.
	 *
	 * @var string
	 */
	public $handle_prefix = 'kalium';

	/**
	 * Waiting list of items to enqueue.
	 *
	 * @var array
	 */
	public $waiting_list = [
		'styles'  => [],
		'scripts' => [],
	];

	/**
	 * Constructor.
	 */
	public function __construct() {
		add_action( 'init', [ $this, 'register_scripts_and_styles' ] );
	}

	/**
	 * Scripts and styles.
	 *
	 * @return array {
	 *  @type string $type
	 *  @type string $handle
	 *  @type string $src
	 *  @type array $deps
	 *  @type string $ver
	 *  @type array $args
	 *  @type string $media
	 * }
	 */
	public function get_all() {
		$scripts_styles = [

			/**
			 * Frontend styles and scripts.
			 */
			[
				'handle' => 'theme-style',
				'src'    => '/style.css',
			],

			// Theme styles
			[
				'handle' => 'theme-bootstrap',
				'src'    => 'css/bootstrap.min.css',
			],
			[
				'handle' => 'theme-base',
				'src'    => 'css/base.min.css',
			],
			[
				'handle' => 'theme-portfolio',
				'src'    => 'css/portfolio.min.css',
			],
			[
				'handle' => 'theme-elementor',
				'src'    => 'css/elementor.min.css',
			],
			[
				'handle' => 'theme-wpbakery',
				'src'    => 'css/wpbakery.min.css',
			],
			[
				'handle' => 'theme-woocommerce',
				'src'    => 'css/woocommerce.min.css',
			],

			// Theme scripts
			[
				'handle' => 'theme-base',
				'src'    => 'js/base.min.js',
				'args'   => [
					'strategy' => 'defer',
				],
			],
			[
				'handle' => 'theme-portfolio',
				'src'    => 'js/portfolio.min.js',
				'args'   => [
					'strategy' => 'defer',
				],
			],
			[
				'handle' => 'theme-woocommerce',
				'src'    => 'js/woocommerce.min.js',
				'args'   => [
					'strategy' => 'defer',
				],
			],
			[
				'handle' => 'theme-sticky-header',
				'src'    => 'js/sticky-header.min.js',
				'args'   => [
					'strategy' => 'defer',
				],
			],

			// Blocks
			[
				'handle' => 'block-content-section',
				'src'    => '/includes/blocks/content-section/index.min.js',
				'deps'   => [
					'wp-block-editor',
				],
			],
			[
				'handle' => 'block-portfolio-items',
				'src'    => '/includes/blocks/portfolio-items/index.min.js',
				'deps'   => [
					'wp-block-editor',
				],
			],
			[
				'handle' => 'block-blog-posts',
				'src'    => '/includes/blocks/blog-posts/index.min.js',
				'deps'   => [
					'wp-block-editor',
				],
			],

			// Theme icons
			[
				'handle' => 'theme-icons-default',
				'src'    => 'icons/main.min.css',
			],
			[
				'handle' => 'theme-icons-social',
				'src'    => 'icons/social.min.css',
			],
			[
				'handle' => 'theme-icons-admin',
				'src'    => 'icons/admin.min.css',
			],

			// CSS loaders
			[
				'handle' => 'css-loaders',
				'src'    => 'css/loaders.min.css',
			],

			/**
			 * Libraries.
			 */

			// GSAP
			[
				'handle' => 'gsap',
				'src'    => 'vendors/gsap/gsap.min.js',
				'args'   => [
					'strategy' => 'defer',
				],
			],
			[
				'handle' => 'gsap-scrolltrigger',
				'src'    => 'vendors/gsap/scrolltrigger.min.js',
				'deps'   => [
					'gsap',
				],
			],

			// VideoJS
			[
				'handle' => 'videojs',
				'src'    => 'vendors/videojs/video.min.css',
			],
			[
				'handle' => 'videojs',
				'src'    => 'vendors/videojs/video.min.js',
				'args'   => [
					'strategy' => 'defer',
				],
			],
			[
				'handle' => 'videojs-youtube',
				'src'    => 'vendors/videojs/video-youtube.min.js',
				'deps'   => [
					$this->prefix_handle( 'videojs' ),
				],
				'args'   => [
					'strategy' => 'defer',
				],
			],

			// YouTube Player API
			[
				'handle' => 'youtube-player-api',
				'src'    => 'https://www.youtube.com/iframe_api',
				'args'   => [
					'strategy' => 'defer',
				],
			],

			// Vimeo Player API
			[
				'handle' => 'vimeo-player-api',
				'src'    => 'vendors/vimeo/player.min.js',
				'args'   => [
					'strategy' => 'defer',
				],
			],

			// Isotope
			[
				'handle' => 'metafizzy-isotope',
				'src'    => 'vendors/metafizzy/isotope.pkgd.min.js',
			],

			// Packery
			[
				'handle' => 'metafizzy-packery',
				'src'    => 'vendors/metafizzy/packery-mode.pkgd.min.js',
			],

			// Infinite Scroll
			[
				'handle' => 'infinite-scroll',
				'src'    => 'vendors/metafizzy/infinite-scroll.pkgd.min.js',
			],

			// Swiper
			[
				'handle' => 'swiper',
				'src'    => 'vendors/swiper/swiper.min.css',
			],
			[
				'handle' => 'swiper',
				'src'    => 'vendors/swiper/swiper.min.js',
				'args'   => [
					'strategy' => 'defer',
				],
			],

			// Fancybox
			[
				'handle' => 'fancybox',
				'src'    => 'vendors/fancyapps/fancybox.min.css',
			],
			[
				'handle' => 'fancybox',
				'src'    => 'vendors/fancyapps/fancybox.min.js',
			],

			// Popper JS
			[
				'handle' => 'popper',
				'src'    => 'vendors/popper/popper.min.js',
			],

			// Tippy
			[
				'handle' => 'tippy',
				'src'    => 'vendors/tippy/tippy-bundle.umd.min.js',
				'deps'   => [
					$this->prefix_handle( 'popper' ),
				],
			],

			// Select2
			[
				'handle' => 'select2',
				'src'    => 'vendors/select2/select2.min.css',
			],
			[
				'handle' => 'select2',
				'src'    => 'vendors/select2/select2.full.min.js',
				'deps'   => [
					'jquery',
				],
			],

			// FontAwesome
			[
				'handle' => 'fontawesome',
				'src'    => 'vendors/font-awesome/css/all.min.css',
			],

			// Legacy Linea font icons
			[
				'handle' => 'legacy-linea',
				'src'    => 'vendors/linea/linea-icons.css',
			],

			// Typed.js
			[
				'handle' => 'typed',
				'src'    => 'vendors/typed/typed.js',
				'deps' => [
					'jquery',
				],
			],

			/**
			 * Admin styles and scripts.
			 */

			// Customizer style
			[
				'handle' => 'theme-customizer',
				'src'    => 'admin/css/customizer.min.css',
				'deps'   => 'wp-components',
			],
			[
				'handle' => 'theme-customizer-dark',
				'src'    => 'admin/css/customizer-dark.min.css',
				'media'  => 'auto' === kalium()->request->cookie( 'kalium_customizer_color_scheme' ) ? '(prefers-color-scheme: dark)' : ( 'dark' === kalium()->request->cookie( 'kalium_customizer_color_scheme' ) ? 'all' : 'none' ),
			],

			// Customizer scripts
			[
				'handle' => 'theme-customizer',
				'src'    => 'admin/js/customizer.min.js',
				'deps'   => [
					'wp-element',
					'wp-components',
					'wp-url',
					'wp-primitives',
				],
			],
			[
				'handle' => 'theme-customizer-preview',
				'src'    => 'admin/js/customizer-preview.min.js',
				'deps'   => [
					'backbone',
					'customize-preview',
				],
			],

			// Template parts
			[
				'handle' => 'template-parts',
				'src'    => 'admin/css/template-parts.min.css',
			],
			[
				'handle' => 'template-parts',
				'src'    => 'admin/js/template-parts.min.js',
			],

			// Template parts edit page
			[
				'handle' => 'template-parts-edit',
				'src'    => 'admin/css/template-parts-edit.min.css',
				'deps'   => [ 'wp-components' ],
			],
			[
				'handle' => 'template-parts-edit',
				'src'    => 'admin/js/template-parts-edit.min.js',
				'deps'   => [ 'wp-blocks', 'wp-plugins', 'wp-api-fetch' ],
			],

			// Template parts frontend
			[
				'handle' => 'template-parts-frontend',
				'src'    => 'admin/js/template-parts-frontend.min.js',
			],

			// Block editor style
			[
				'handle' => 'editor-style',
				'src'    => 'admin/css/editor-style.min.css',
			],
			[
				'handle' => 'block-editor',
				'src'    => 'admin/css/block-editor.min.css',
			],

			// Theme admin
			[
				'handle' => 'theme-admin',
				'src'    => 'admin/css/main.min.css',
			],
			[
				'handle' => 'theme-admin',
				'src'    => 'admin/js/main.min.js',
			],

			// Theme dashboard
			[
				'handle' => 'theme-dashboard',
				'src'    => 'admin/css/dashboard.min.css',
			],
			[
				'handle' => 'theme-dashboard',
				'src'    => 'admin/js/dashboard.min.js',
			],

			// Theme version upgrades
			[
				'handle' => 'theme-version-upgrades',
				'src'    => 'admin/js/version-upgrades.min.js',
				'deps'   => [
					'wp-api-fetch',
					'wp-element',
					'wp-autop',
				],
			],

			// Admin Portfolio
			[
				'handle' => 'admin-portfolio',
				'src'    => 'admin/css/portfolio.min.css',
			],
			[
				'handle' => 'admin-portfolio',
				'src'    => 'admin/js/portfolio.min.js',
				'deps'   => [
					'wp-element',
					'wp-components',
					'wp-api-fetch',
					'wp-url',
				],
			],

			// Grouped metaboxes
			[
				'handle' => 'grouped-metaboxes',
				'src'    => '/includes/libraries/laborator/grouped-metaboxes/assets/grouped-metaboxes.min.css',
			],
			[
				'handle' => 'grouped-metaboxes',
				'src'    => '/includes/libraries/laborator/grouped-metaboxes/assets/grouped-metaboxes.min.js',
			],

			// Starter Sites importer
			[
				'handle' => 'importer',
				'src'    => '/includes/libraries/laborator/importer/assets/importer.min.js',
			],
		];

		return array_map( [ $this, 'prepare_item' ], $scripts_styles );
	}

	/**
	 * Asset groups.
	 *
	 * @return array
	 */
	public function get_groups() {
		return [
			// Theme styles
			'theme-frontend-styles'  => [
				'styles' => [
					'theme-bootstrap',
					'theme-base',
					'theme-portfolio'   => kalium()->is->portfolio_active(),
					'theme-elementor'   => kalium()->is->elementor_active(),
					'theme-wpbakery'    => kalium()->is->wpbakery_active(),
					'theme-woocommerce' => kalium()->is->woocommerce_active(),
					'theme-icons-default',
				],
			],

			// Theme frontend scripts
			'theme-frontend-scripts' => [
				'scripts' => [
					'theme-base',
					'theme-portfolio'   => kalium()->is->portfolio_active(),
					'theme-woocommerce' => kalium()->is->woocommerce_active(),
				],
			],

			// Theme blocks
			'theme-blocks'           => [
				'scripts' => [
					'block-blog-posts',
					'block-portfolio-items',
					'block-content-section',
				],
			],

			// Customizer
			'theme-customizer'       => [
				'styles'  => [
					'theme-customizer',
					'theme-customizer-dark',
				],
				'scripts' => [
					'theme-customizer',
				],
			],

			// Grouped metaboxes
			'grouped-metaboxes'      => [
				'styles'  => [
					'grouped-metaboxes',
				],
				'scripts' => [
					'grouped-metaboxes-store',
					'grouped-metaboxes',
				],
			],

			/**
			 * Libraries.
			 */

			// Isotope
			'isotope'                => [
				'scripts' => [
					'metafizzy-isotope',
					'metafizzy-packery',
				],
			],

			// Tippy
			'tippy'                  => [
				'scripts' => [
					'popper',
					'tippy',
				],
			],
		];
	}

	/**
	 * JS API libraries.
	 *
	 * @return array
	 */
	public function get_js_api_libraries() {
		return [
			'gsap'               => [ 'gsap' ],
			'gsap-scrolltrigger' => [ 'gsap-scrolltrigger' ],
			'swiper'             => [ 'swiper' ],
			'fancybox'           => [ 'fancybox' ],
			'infinite-scroll'    => [ 'infinite-scroll' ],
			'isotope'            => [ 'isotope' ],
			'tippy'              => [ 'tippy' ],
			'videojs'            => [ 'videojs' ],
			'videojs-youtube'    => [ 'videojs-youtube' ],
			'youtube-player-api' => [ 'youtube-player-api' ],
			'vimeo-player-api'   => [ 'vimeo-player-api' ],
		];
	}

	/**
	 * Prefix handle.
	 *
	 * @param string $handle
	 *
	 * @return string
	 */
	public function prefix_handle( $handle ) {
		$handle_prefix = $this->handle_prefix . '-';

		if ( '!' === substr( $handle, 0, 1 ) ) {
			return substr( $handle, 1 );
		} elseif ( 0 === strpos( $handle, $handle_prefix ) ) {
			return $handle;
		}

		return $handle_prefix . $handle;
	}

	/**
	 * Prepare item.
	 *
	 * @param string|array $item
	 *
	 * @return array
	 */
	public function prepare_item( $item ) {
		$item = wp_parse_args(
			$item,
			[
				'handle'    => null,
				'src'       => null,
				'deps'      => null,
				'ver'       => null,
				'args'      => null,
				'in_footer' => null,
				'media'     => null,
			]
		);

		$handle = $item['handle'];
		$src    = $item['src'];
		$deps   = $item['deps'];
		$ver    = $item['ver'];

		// Prefix handle
		$handle = $this->prefix_handle( $handle );

		// Version
		if ( is_null( $ver ) ) {
			$ver = kalium()->get_version() . '.' . kalium()->get_build_num();
		}

		$item_keys = [
			'type',
			'handle',
			'src',
			'deps',
			'ver',
		];

		// Source
		if ( '/' === substr( $src, 0, 1 ) ) {
			$src = substr( $src, 1 );
		} elseif ( 'http' !== substr( $src, 0, 4 ) ) {
			$src = 'assets/' . $src;
		}

		// If its local asset
		if ( 0 !== strpos( $src, 'http' ) ) {
			$src = kalium()->file_url( $src );
		}

		// Style
		if ( false !== strrpos( $src, '.css' ) ) {
			$type  = 'style';
			$media = $item['media'];

			if ( ! empty( $media ) ) {
				$item_keys[] = 'media';
			}
		}
		// Script
		else {
			$type = 'script';
			$args = wp_parse_args(
				$item['args'],
				[
					'in_footer' => is_null( $item['in_footer'] ) || $item['in_footer'],
				]
			);

			$item_keys[] = 'args';
		}

		// Script
		return compact( $item_keys );
	}

	/**
	 * Get items list based on type and handle. It also matches groups.
	 *
	 * @param string $type
	 * @param string $handle
	 *
	 * @return string[]
	 */
	public function get_items_list( $type, $handle ) {
		static $items, $groups;

		if ( is_null( $items ) ) {
			$items = $this->get_all();
		}

		if ( is_null( $groups ) ) {
			$groups = $this->get_groups();
		}

		$list = [];

		// Firstly check groups
		if ( isset( $groups[ $handle ] ) ) {
			$group_items = $groups[ $handle ][ 'script' === $type ? 'scripts' : 'styles' ] ?? null;

			if ( is_array( $group_items ) ) {
				foreach ( $group_items as $item_handle => $available ) {
					if ( is_numeric( $item_handle ) ) {
						$item_handle = $available;
					}

					// Prefix item handle
					$item_handle = $this->prefix_handle( $item_handle );

					if ( $available ) {
						foreach ( $items as $item ) {
							if ( $type === $item['type'] && $item_handle === $item['handle'] ) {
								$list[] = $item['handle'];
							}
						}
					}
				}
			}
		} else {
			$handle = $this->prefix_handle( $handle );

			foreach ( $items as $item ) {
				if ( $type === $item['type'] && $handle === $item['handle'] ) {
					$list[] = $item['handle'];
				}
			}
		}

		return $list;
	}

	/**
	 * Get styles handles list.
	 *
	 * @param string $handle
	 *
	 * @return string[]
	 */
	public function get_styles_list( $handle ) {
		return $this->get_items_list( 'style', $handle );
	}

	/**
	 * Get scripts handles list.
	 *
	 * @param string $handle
	 *
	 * @return string[]
	 */
	public function get_scripts_list( $handle ) {
		return $this->get_items_list( 'script', $handle );
	}

	/**
	 * Get single item by type and handle.
	 *
	 * @param string $type
	 * @param string $handle
	 *
	 * @return array|null
	 */
	public function get_item( $type, $handle ) {
		$handle = $this->prefix_handle( $handle );

		foreach ( $this->get_all() as $item ) {
			if ( $type === $item['type'] && $handle === $item['handle'] ) {
				return $item;
			}
		}

		return null;
	}

	/**
	 * Get single style by handle.
	 *
	 * @param string $handle
	 *
	 * @return array|null
	 */
	public function get_style( $handle ) {
		return $this->get_item( 'style', $handle );
	}

	/**
	 * Get single script by handle.
	 *
	 * @param string $handle
	 *
	 * @return array|null
	 */
	public function get_script( $handle ) {
		return $this->get_item( 'script', $handle );
	}

	/**
	 * Enqueue style.
	 *
	 * @param string $handle
	 */
	public function enqueue_style( $handle ) {
		if ( did_action( 'init' ) ) {
			$styles_list = $this->get_styles_list( $handle );

			// Enqueue styles
			foreach ( $styles_list as $style_handle ) {
				wp_enqueue_style( $style_handle );
			}
		}
		// Called too early
		elseif ( ! in_array( $handle, $this->waiting_list['styles'] ) ) {
			$this->waiting_list['styles'][] = $handle;
		}
	}

	/**
	 * Enqueue script.
	 *
	 * @param string $handle
	 */
	public function enqueue_script( $handle ) {
		if ( did_action( 'init' ) ) {
			$scripts_list = $this->get_scripts_list( $handle );

			// Enqueue scripts
			foreach ( $scripts_list as $script_handle ) {
				wp_enqueue_script( $script_handle );
			}
		}
		// Called too early
		elseif ( ! in_array( $handle, $this->waiting_list['scripts'] ) ) {
			$this->waiting_list['scripts'][] = $handle;
		}
	}

	/**
	 * Dequeue style.
	 *
	 * @param string $handle
	 */
	public function dequeue_style( $handle ) {
		if ( did_action( 'init' ) ) {
			$styles_list = $this->get_styles_list( $handle );

			// Dequeue styles
			foreach ( $styles_list as $style_handle ) {
				wp_dequeue_style( $style_handle );
			}
		}
		// Called too early
		elseif ( in_array( $handle, $this->waiting_list['styles'] ) ) {
			$this->waiting_list['styles'] = array_diff( $this->waiting_list['styles'], [ $handle ] );
		}
	}


	/**
	 * Dequeue script.
	 *
	 * @param string $handle
	 */
	public function dequeue_script( $handle ) {
		if ( did_action( 'init' ) ) {
			$scripts_list = $this->get_scripts_list( $handle );

			// Dequeue scripts
			foreach ( $scripts_list as $script_handle ) {
				wp_dequeue_script( $script_handle );
			}
		}
		// Called too early
		elseif ( in_array( $handle, $this->waiting_list['scripts'] ) ) {
			$this->waiting_list['scripts'] = array_diff( $this->waiting_list['scripts'], [ $handle ] );
		}
	}

	/**
	 * Print styles.
	 *
	 * @param string|array $handle
	 *
	 * @since 4.3
	 */
	public function print_styles( $handle ) {
		if ( did_action( 'init' ) ) {
			$styles_list = $this->get_styles_list( $handle );

			// Print styles
			wp_print_styles( $styles_list );
		}
	}

	/**
	 * Print scripts.
	 *
	 * @param string|array $handle
	 *
	 * @since 4.3
	 */
	public function print_scripts( $handle ) {
		if ( did_action( 'init' ) ) {
			$scripts_list = $this->get_scripts_list( $handle );

			// Print scripts
			wp_print_scripts( $scripts_list );
		}
	}

	/**
	 * Enqueue item or group.
	 *
	 * @param string $handle
	 */
	public function enqueue( $handle ) {
		$this->enqueue_style( $handle );
		$this->enqueue_script( $handle );
	}

	/**
	 * Dequeue item or group.
	 *
	 * @param string $handle
	 */
	public function dequeue( $handle ) {
		$this->dequeue_style( $handle );
		$this->dequeue_script( $handle );
	}

	/**
	 * Add inline script.
	 *
	 * @param string $handle
	 * @param string $data
	 * @param string $position
	 *
	 * @return bool
	 */
	public function add_inline_script( $handle, $data, $position = 'after' ) {
		return wp_add_inline_script( $this->prefix_handle( $handle ), $data, $position );
	}

	/**
	 * Add inline style.
	 *
	 * @param string $handle
	 * @param string $data
	 *
	 * @return bool
	 */
	public function add_inline_style( $handle, $data ) {
		return wp_add_inline_style( $this->prefix_handle( $handle ), $data );
	}

	/**
	 * Register assets.
	 */
	public function register_scripts_and_styles() {
		foreach ( $this->get_all() as $item ) {
			$handle = $item['handle'];
			$src    = $item['src'];
			$deps   = $item['deps'];
			$ver    = $item['ver'];

			// Register style
			if ( 'style' === $item['type'] ) {
				wp_register_style( $handle, $src, $deps, $ver, $item['media'] ?? null );
			} if ( 'script' === $item['type'] ) {
				wp_register_script( $handle, $src, $deps, $ver, $item['args'] );
			}
		}

		// Register asset groups as WP dependencies
		foreach ( $this->get_groups() as $group_handle => $group_items ) {
			$group_handle  = $this->prefix_handle( $group_handle );
			$group_styles  = $this->get_styles_list( $group_handle );
			$group_scripts = $this->get_scripts_list( $group_handle );

			if ( ! empty( $group_styles ) && ! wp_style_is( $group_handle, 'registered' ) ) {
				wp_register_style( $group_handle, false, $group_styles, null );
			}

			if ( ! empty( $group_scripts ) && ! wp_script_is( $group_handle, 'registered' ) ) {
				wp_register_script( $group_handle, false, $group_styles, null );
			}
		}

		$this->_enqueue_early();
	}

	/**
	 * Enqueued items earlier than "init" action is fired, are queued for later enqueue.
	 */
	public function _enqueue_early() {
		foreach ( $this->waiting_list['styles'] as $item ) {
			$this->enqueue_style( $item );
		}

		foreach ( $this->waiting_list['scripts'] as $item ) {
			$this->enqueue_script( $item );
		}
	}
}
