<?php
/**
 * Kalium WordPress Theme
 *
 * Checker methods for any purpose.
 *
 * @link https://kaliumtheme.com
 */
namespace Kalium\Core;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Direct access not allowed.
}

class Is {

	/**
	 * Check if plugin is installed.
	 *
	 * @param string $plugin
	 *
	 * @return bool
	 */
	public function plugin_installed( $plugin ) {
		if ( ! function_exists( 'get_plugins' ) ) {
			include_once ABSPATH . 'wp-admin/includes/plugin.php';
		}

		// Plugins
		$plugins = get_plugins();

		return isset( $plugins[ $plugin ] );
	}

	/**
	 * Check if plugin is active.
	 *
	 * @param string|array $plugin
	 *
	 * @return bool
	 */
	public function plugin_active( $plugin ) {
		if ( ! function_exists( 'is_plugin_active' ) ) {
			include_once ABSPATH . 'wp-admin/includes/plugin.php';
		}

		return is_plugin_active( $plugin );
	}

	/**
	 * Check if any of the given plugins is active.
	 *
	 * @param array|string $plugins
	 *
	 * @return bool
	 */
	public function any_plugin_active( $plugins ) {
		if ( ! is_array( $plugins ) ) {
			$plugins = [ $plugins ];
		}

		foreach ( $plugins as $plugin ) {
			if ( $this->plugin_active( $plugin ) ) {
				return true;
			}
		}

		return false;
	}

	/**
	 * Check if all the given plugins are active.
	 *
	 * @param array|string $plugins
	 *
	 * @return bool
	 */
	public function plugins_are_active( $plugins ) {
		if ( ! is_array( $plugins ) ) {
			$plugins = [ $plugins ];
		}

		foreach ( $plugins as $plugin ) {
			if ( ! $this->plugin_active( $plugin ) ) {
				return false;
			}
		}

		return true;
	}

	/**
	 * Check if template parts are enabled.
	 *
	 * @return bool
	 * @since 4.2
	 */
	public function template_parts_active() {
		return \Kalium\Template_Parts\Template_Parts::get_setting( 'enabled' );
	}

	/**
	 * Check if Portfolio Post Type is active.
	 *
	 * @return bool
	 */
	public function portfolio_active() {
		return class_exists( 'Portfolio_Post_Type' );
	}

	/**
	 * Check if WPBakery Page Builder plugin is active.
	 *
	 * @return bool
	 */
	public function wpbakery_active() {
		return class_exists( 'Vc_Manager' );
	}

	/**
	 * Check if WooCommerce plugin is active.
	 *
	 * @return bool
	 */
	public function woocommerce_active() {
		return class_exists( 'WooCommerce' );
	}

	/**
	 * Check if ACF plugin is active.
	 *
	 * @return bool
	 */
	public function acf_active() {
		return function_exists( 'get_field' );
	}

	/**
	 * Check if Elementor plugin is active.
	 *
	 * @return bool
	 */
	public function elementor_active() {
		return defined( 'ELEMENTOR_PATH' );
	}

	/**
	 * Check if GreenShift plugin is active.
	 * 
	 * @return bool
	 * @since 4.3
	 */
	public function greenshift_active() {
		return defined( 'GREENSHIFT_DIR_PATH' );
	}

	/**
	 * Check if Elementor Pro plugin is active.
	 *
	 * @return bool
	 */
	public function elementor_pro_active() {
		return defined( 'ELEMENTOR_PRO_PATH' );
	}

	/**
	 * Check if Slider Revolution is active.
	 *
	 * @since 4.0
	 */
	public function revslider_active() {
		return defined( 'RS_PLUGIN_PATH' );
	}

	/**
	 * Check if WPML plugin is activated.
	 *
	 * @return bool
	 */
	public function wpml_active() {
		return function_exists( 'icl_object_id' );
	}

	/**
	 * Check if Polylang plugin is activated.
	 *
	 * @return bool
	 * @since 4.2.2
	 */
	public function polylang_active() {
		return defined( 'POLYLANG_VERSION' );
	}

	/**
	 * Check if current site is using multilingual plugins.
	 *
	 * @return bool
	 * @since 4.2.2
	 */
	public function multilingual() {
		return $this->wpml_active() || $this->polylang_active();
	}

	/**
	 * Check if Breadcrumb NavXT plugin is active.
	 */
	public function breadcrumb_navxt_active() {
		return function_exists( 'bcn_display' );
	}

	/**
	 * Check if Rank Math plugin is active.
	 *
	 * @return bool
	 * @since 3.4
	 */
	public function rank_math_active() {
		return function_exists( 'rank_math' );
	}

	/**
	 * Check if Gutenberg plugin is active.
	 *
	 * @return bool
	 * @since 4.0
	 */
	public function gutenberg_active() {
		return defined( 'GUTENBERG_VERSION' );
	}

	/**
	 * Checks if current screen is block editor.
	 *
	 * @return bool
	 */
	public function block_editor() {
		if ( ! is_admin() || ! function_exists( 'get_current_screen' ) ) {
			return false;
		}

		if ( $screen = get_current_screen() ) {
			if ( 'post' === $screen->base ) {
				return use_block_editor_for_post_type( $screen->post_type );
			} elseif ( $screen->is_block_editor() ) {
				return true;
			}
		}

		return false;
	}

	/**
	 * Check if given file is SVG.
	 *
	 * @param string $file
	 *
	 * @return bool
	 */
	public function svg( $file ) {
		$file_info = pathinfo( $file );

		return 'svg' == strtolower( kalium_get_array_key( $file_info, 'extension' ) );
	}

	/**
	 * Check if blog page is being displayed.
	 * It includes category, tag and author pages.
	 *
	 * @return bool
	 * @since 4.0
	 */
	public function blog_page() {
		return is_home() || is_tag() || is_category() || is_author() || is_year() || is_month();
	}

	/**
	 * Debug mode.
	 *
	 * @return bool
	 * @since 4.1.2
	 */
	public function debugging() {
		return defined( 'KALIUM_DEBUG' ) && KALIUM_DEBUG;
	}

	/**
	 * Check if provided string is JSON.
	 *
	 * @param string $string
	 *
	 * @return bool
	 * @since 4.0
	 */
	public function json( $string ) {
		return is_string( $string ) && is_array( json_decode( $string, true ) ) && JSON_ERROR_NONE === json_last_error();
	}

	/**
	 * Checks if given string is URL.
	 *
	 * @param string $string
	 *
	 * @return bool
	 * @since 4.0
	 */
	public function url( $string ) {
		return filter_var( $string, FILTER_VALIDATE_URL );
	}

	/**
	 * Check if the specified AJAX action is currently executing.
	 *
	 * @return bool
	 * @since 4.1.1
	 */
	public function ajax_action( $name ) {
		return defined( 'DOING_AJAX' ) && DOING_AJAX && $name === kalium()->request->request( 'action' );
	}

	/**
	 * Check if the current request is a REST API request.
	 *
	 * @return bool
	 * @since 4.3
	 */
	public function rest_request() {
		return defined( 'REST_REQUEST' ) && REST_REQUEST;
	}

	/**
	 * Check if the current request is a WP CLI request.
	 *
	 * @return bool
	 * @since 4.3.1
	 */
	public function wp_cli() {
		return defined( 'WP_CLI' ) && WP_CLI;
	}
}
