<?php
/**
 * Kalium WordPress Theme
 *
 * Theme license.
 *
 * @author Laborator
 * @link   https://kaliumtheme.com
 */
namespace Kalium\Core;

use FS_Plugin_License;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Direct access not allowed.
}

class License {

	/**
	 * Get license object.
	 *
	 * @return FS_Plugin_License|bool
	 */
	public function get_license() {
		return kalium()->freemius->_get_license();
	}

	/**
	 * Get license ID.
	 *
	 * @return string|null
	 */
	public function get_license_id() {
		if ( kalium()->freemius->is_registered() ) {
			$site_id = kalium()->freemius->get_site()->id;
			$license = $this->get_license();

			return $license && ! empty( $license->id ) ? ( $site_id . '_' . $license->id ) : null;
		}

		return null;
	}

	/**
	 * Get license key.
	 *
	 * @return string|null
	 */
	public function get_license_key() {
		if ( $license_data = $this->get_data() ) {
			return $license_data['secret_key'];
		}

		return null;
	}

	/**
	 * Check if theme is registered.
	 *
	 * @return boolean
	 */
	public function is_theme_registered() {
		return kalium()->freemius->is_registered();
	}

	/**
	 * Check if license is expired.
	 *
	 * @return bool
	 */
	public function is_expired() {
		$license = $this->get_license();

		return ! $license || $license->is_expired();
	}

	/**
	 * Checks if license is lifetime deal.
	 *
	 * @return bool
	 */
	public function is_lifetime() {
		$license = $this->get_license();

		if ( $license ) {
			return $license->is_lifetime();
		}

		return false;
	}

	/**
	 * Checks if its trial license.
	 *
	 * @return bool
	 * @since 4.1.1
	 */
	public function is_trial() {
		return kalium()->freemius->is_trial();
	}

	/**
	 * Checks if theme has active subscription.
	 *
	 * @return bool
	 */
	public function is_active() {
		return $this->is_lifetime() || ! $this->is_expired();
	}

	/**
	 * Checks if subscription auto-renews.
	 *
	 * @return bool
	 */
	public function is_auto_renew() {
		$license = $this->get_license();

		if ( $license && ! $this->is_lifetime() ) {
			$subscription = kalium()->freemius->_get_subscription( $license->id );

			if ( $subscription && $subscription->is_active() ) {
				return ! $subscription->is_first_payment_pending();
			}
		}

		return false;
	}

	/**
	 * Checks is license white labeled.
	 *
	 * @return bool
	 */
	public function is_white_labeled() {
		if ( $license = $this->get_license() ) {
			return $license->is_whitelabeled;
		}

		return false;
	}

	/**
	 * Wrap URL with license key.
	 *
	 * @param string $url
	 *
	 * @return string
	 */
	public function wrap_license_key( $url ) {
		if ( $url && ( $license_key = $this->get_license_key() ) ) {
			return add_query_arg( 'license_key', urlencode( $license_key ), $url );
		}

		return $url;
	}

	/**
	 * Get account page URL.
	 *
	 * @return string
	 */
	public function get_account_url() {
		return kalium()->freemius->get_account_url();
	}

	/**
	 * Get plan.
	 *
	 * @return string
	 */
	public function get_plan_name() {
		if ( $plan = kalium()->freemius->get_plan() ) {
			return $plan->name;
		}

		return null;
	}

	/**
	 * Get current plan title.
	 *
	 * @return string
	 */
	public function get_plan_title() {
		if ( $plan = kalium()->freemius->get_plan() ) {
			return $plan->title;
		}

		return 'Unknown';
	}

	/**
	 * Get time left.
	 *
	 * @return int
	 */
	public function get_time_left() {
		if ( $license = $this->get_license() ) {
			if ( $this->is_lifetime() ) {
				return 0;
			}

			if ( ! $license->is_expired() ) {
				return strtotime( $license->expiration ) - time();
			}
		}

		return -1;
	}

	/**
	 * Get license status code.
	 *
	 * 0 - expired
	 * 1 - expiring
	 * 2 - active
	 * 3 - cancelled
	 *
	 * @return int
	 */
	public function get_status_code() {
		$license   = $this->get_license();
		$time_left = $this->get_time_left();

		if ( $license && $license->is_cancelled ) {
			return 3;
		}

		if ( -1 < $time_left ) {
			if ( 30 * DAY_IN_SECONDS < $time_left || 0 === $time_left ) {
				return 2;
			}

			return 1;
		}

		return 0;
	}

	/**
	 * Get status title.
	 *
	 * @return string
	 */
	public function get_status_title() {
		$code = $this->get_status_code();

		if ( 1 === $code ) {
			return 'Expiring';
		} elseif ( 3 === $code ) {
			return 'Canceled';
		}

		return 2 === $code ? 'Active' : 'Expired';
	}

	/**
	 * Get license data.
	 *
	 * @return array|null
	 */
	public function get_data() {
		if ( $license = $this->get_license() ) {
			$user = kalium()->freemius->get_user();
			$site = kalium()->freemius->get_site();

			return [
				'plan'              => $this->get_plan_title(),
				'subscription_term' => $this->is_lifetime() ? 'Lifetime' : 'Annual',
				'expiration'        => $this->is_lifetime() ? false : strtotime( $license->expiration ),
				'secret_key'        => $license->secret_key,
				'secret_key_masked' => $license->get_html_escaped_masked_secret_key(),
				'name'              => $user ? $user->get_name() : null,
				'email'             => $user ? $user->email : null,
				'site_id'           => $site ? $site->id : null,
			];
		}

		return null;
	}

	/**
	 * Badge class of the subscription status.
	 *
	 * @return string
	 */
	public function get_badge_class() {
		$bade_classes = [
			1 => 'warning',
			2 => 'success',
		];

		return $bade_classes[ $this->get_status_code() ] ?? 'danger';
	}

	/**
	 * Check if current plan has certain feature.
	 *
	 * @return bool
	 */
	public function has_feature( $feature_name ) {
		static $features;

		if ( is_null( $features ) ) {
			$features = $this->get_features();
		}

		return ! empty( $features[ $feature_name ] ) && in_array( $this->get_plan_name(), $features[ $feature_name ] );
	}

	/**
	 * List of features supported by plans.
	 *
	 * @return array
	 */
	private function get_features() {
		return [
			'white-label' => [
				'agency',
			],
		];
	}
}
