<?php
/**
 * Kalium WordPress Theme
 *
 * Settings API.
 *
 * @author Laborator
 * @link   https://kaliumtheme.com
 */
namespace Kalium\Core;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Direct access not allowed.
}

class Settings {

	/**
	 * Registered settings.
	 *
	 * @var array
	 */
	private $settings = [];

	/**
	 * Instance.
	 *
	 * @var self
	 */
	private static $instance;

	/**
	 * Alias method for setting rendering.
	 *
	 * @param string $id
	 */
	public static function setting( $id ) {
		self::$instance->render( $id );
	}

	/**
	 * Construct.
	 */
	public function __construct() {
		if ( ! is_null( self::$instance ) ) {
			return;
		}

		// Initialize
		add_action( 'init', [ $this, 'init' ], 0 );

		// Store instance
		self::$instance = $this;
	}

	/**
	 * Initialization.
	 */
	public function init() {

		// Update settings
		add_action( 'wp_ajax_kalium_update_setting', [ $this, 'update_settings' ] );

		/**
		 * Settings init hook.
		 */
		do_action( 'kalium_settings_init' );
	}

	/**
	 * Register setting or array of settings.
	 *
	 * @param string|array $id
	 * @param array        $args
	 */
	public function register( $id, $args = [] ) {
		if ( is_array( $id ) ) {
			foreach ( $id as $setting_key => $setting_args ) {
				$this->register( $setting_key, $setting_args );
			}
			return;
		}

		$args = wp_parse_args(
			$args,
			[
				// Core settings
				'setting_type'      => 'option',
				'setting'           => null,
				'autoload'          => true,
				'initialize'        => false,
				'default'           => null,

				// Edit (admin)
				'auto_update'       => true,
				'input_type'        => null,
				'input_attributes'  => [],
				'choices'           => null,
				'sanitize_callback' => null,
				'validate_callback' => null,
				'render_callback'   => null,
				'capability'        => 'activate_plugins',
			]
		);

		if ( isset( $this->settings[ $id ] ) || empty( $args['setting'] ) ) {
			return;
		}

		$this->settings[ $id ] = $args;

		// Setting ID
		$setting_id = kalium_get_array_first( is_array( $args['setting'] ) ? $args['setting'] : explode( '.', $args['setting'] ) );

		// Initialize setting
		if ( true === $args['initialize'] ) {
			add_option( $setting_id, $args['default'], '', $args['autoload'] );
		}
	}

	/**
	 * Get setting entry.
	 *
	 * @param string $id
	 *
	 * @return array|null
	 */
	public function get( $id ) {
		return $this->settings[ $id ] ?? null;
	}

	/**
	 * Get all settings.
	 *
	 * @param bool $js_export
	 *
	 * @return array
	 */
	public function get_all( $js_export = false ) {
		if ( $js_export ) {
			return array_map(
				function ( $setting ) {
					return [
						'input_type'  => $setting['input_type'],
						'auto_update' => $setting['auto_update'],
					];
				},
				$this->settings
			);
		}
		return $this->settings;
	}

	/**
	 * Settings nonce.
	 *
	 * @return string
	 */
	public function get_nonce() {
		return wp_create_nonce( 'kalium_settings_nonce' );
	}

	/**
	 * Renders a setting.
	 *
	 * @param string $id
	 */
	public function render( $id ) {
		$setting = $this->get( $id );

		if ( ! $setting ) {
			return;
		}

		// Setting vars
		$input_type       = $setting['input_type'];
		$input_attributes = $setting['input_attributes'];
		$choices          = $setting['choices'];
		$default_value    = $setting['default'];
		$setting_path     = is_array( $setting['setting'] ) ? $setting['setting'] : explode( '.', $setting['setting'] );
		$setting_id       = array_shift( $setting_path );
		$setting_type     = $setting['setting_type'];
		$setting_value    = 'theme_mod' === $setting_type ? get_theme_mod( $setting_id, null ) : get_option( $setting_id, null );
		$render_callback  = $setting['render_callback'];

		// Retrieve setting value
		if ( is_null( $setting_value ) ) {
			$setting_value = $default_value;
		} else {
			while ( $key = array_shift( $setting_path ) ) {
				$setting_value = $setting_value[ $key ] ?? null;

				if ( is_null( $setting_value ) ) {
					$setting_value = $setting['default'];
					break;
				}
			}
		}

		// Input attributes
		if ( ! is_array( $input_attributes ) ) {
			$input_attributes = [];
		}

		if ( empty( $input_attributes['class'] ) ) {
			$input_attributes['class'] = [];
		} elseif ( is_string( $input_attributes['class'] ) ) {
			$input_attributes['class'] = explode( ' ', $input_attributes['class'] );
		}

		// ID
		$input_attributes['id'] = $id;

		// JS Setting attribute
		$input_attributes['data-setting-id'] = $id;

		// Render setting
		if ( ! is_callable( $render_callback ) ) {
			$input_attributes['class'][] = 'setting-control';

			// Input type
			switch ( $input_type ) {
				case 'switch':
					$checked_state = $setting_value ? 'true' : 'false';

					array_unshift( $input_attributes['class'], 'input-switch' );

					kalium_render_element(
						[
							'tag_name'   => 'button',
							'attributes' => array_replace_recursive(
								$input_attributes,
								[
									'class'        => $input_attributes['class'],
									'role'         => 'switch',
									'aria-checked' => $checked_state,
									'data-checked' => $checked_state,
								]
							),
							'content'    => '<span></span>',
						]
					);
					break;

				case 'text':
					array_unshift( $input_attributes['class'], 'input-text' );

					kalium_render_element(
						[
							'tag_name'   => 'input',
							'attributes' => array_replace_recursive(
								$input_attributes,
								[
									'type'  => 'text',
									'close' => false,
									'value' => $setting_value,
								]
							),
						]
					);
					break;

				case 'textarea':
					array_unshift( $input_attributes['class'], 'input-text' );

					kalium_render_element(
						[
							'tag_name'   => 'textarea',
							'attributes' => $input_attributes,
							'content'    => $setting_value,
						]
					);
					break;

				case 'select':
					array_unshift( $input_attributes['class'], 'input-select' );

					// Choices (options)
					$select_options = [];

					if ( is_callable( $choices ) ) {
						$choices = call_user_func( $choices, $setting_value, $setting );
					}

					if ( is_array( $choices ) ) {
						foreach ( $choices as $choice_value => $choice_label ) {
							$select_options[] = kalium_render_element(
								[
									'tag_name'   => 'option',
									'attributes' => [
										'value'    => $choice_value,
										'selected' => $choice_value == $setting_value, // phpcs:ignore Universal.Operators.StrictComparisons.LooseEqual
									],
									'content'    => $choice_label,
								],
								false
							);
						}
					}

					kalium_render_element(
						[
							'tag_name'   => 'select',
							'attributes' => $input_attributes,
							'content'    => implode( '', $select_options ),
						]
					);
					break;
			}
		}
	}

	/**
	 * Update batch settings.
	 */
	public function update_settings() {
		$response = [
			'errors'  => [],
			'updated' => 0,
		];

		// Check referer
		check_ajax_referer( 'kalium_settings_nonce', 'settings_nonce' );

		// Settings array
		$settings = kalium_decode_json( wp_unslash( kalium()->request->input( 'settings' ) ), true, 512, JSON_OBJECT_AS_ARRAY );

		if ( ! is_array( $settings ) ) {
			$response['errors'][] = 'No settings provided.';
		} else {
			foreach ( $settings as $setting_entry ) {
				$setting_id    = $setting_entry['id'];
				$setting_value = $setting_entry['value'];

				if ( $setting = $this->get( $setting_id ) ) {

					// Check permissions
					if ( ! empty( $setting['capability'] ) && false === current_user_can( $setting['capability'] ) ) {
						$response['errors'][] = sprintf( 'No permissions to update "%s" setting!', $setting_id );
						break;
					}

					// Sanitize value
					if ( is_callable( $setting['sanitize_callback'] ) ) {
						$setting_value = call_user_func( $setting['sanitize_callback'], $setting_value );
					}

					// Validate value
					$validate_result = true;

					if ( is_callable( $setting['validate_callback'] ) ) {
						$validate_result = call_user_func( $setting['validate_callback'], $setting_value );

						if ( is_wp_error( $validate_result ) ) {
							$response['errors'] = array_merge( $response['errors'], $validate_result->get_error_message() );
							break;
						}
					}

					if ( $validate_result ) {
						$this->update_setting( $setting_id, $setting_value );
						$response['updated'] += $validate_result;
					}
				}
			}
		}

		wp_send_json( $response );
	}

	/**
	 * Update single setting.
	 *
	 * @param string $id
	 * @param mixed  $value
	 */
	public function update_setting( $id, $value ) {
		if ( $setting = $this->get( $id ) ) {
			$setting_path  = is_array( $setting['setting'] ) ? $setting['setting'] : explode( '.', $setting['setting'] );
			$setting_id    = array_shift( $setting_path );
			$setting_type  = $setting['setting_type'];
			$setting_value = 'theme_mod' === $setting_type ? get_theme_mod( $setting_id, null ) : get_option( $setting_id, null );

			$current = & $setting_value;

			while ( $key = array_shift( $setting_path ) ) {
				if ( ! isset( $current[ $key ] ) || ! is_array( $current[ $key ] ) ) {
					$current[ $key ] = [];
				}

				$current = & $current[ $key ];
			}

			$current = $value;

			if ( 'theme_mod' === $setting_type ) {
				set_theme_mod( $setting_id, $setting_value );
			} else {
				update_option( $setting_id, $setting_value );
			}
		}
	}
}
