<?php
/**
 * Kalium Styles class
 *
 * Laborator.co
 * www.laborator.co
 */
namespace Kalium\Core;

use Exception;
use Spatie\Color\Factory;
use Spatie\Color\Color;
use Spatie\Color\Hex;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Direct access not allowed.
}

class Style {

	/**
	 * CSS vars.
	 *
	 * @var array
	 */
	public $css_vars = [];

	/**
	 * Construct.
	 */
	public function __construct() {
		add_action( 'wp_print_scripts', [ $this, 'print_css_vars' ], 1000 );
	}

	/**
	 * Set CSS var.
	 *
	 * @param string $name
	 * @param string $value
	 */
	public function set_css_var( $name, $value ) {
		$this->css_vars[ $name ] = $value;
	}

	/**
	 * Prefixed CSS var name.
	 *
	 * @param ...string $args
	 *
	 * @return string
	 */
	public function var_name() {
		$var = [
			apply_filters( 'kalium_css_var_name_prefix', 'k' ),
		];

		foreach ( func_get_args() as $arg ) {
			if ( empty( $arg ) ) {
				continue;
			}

			if ( is_string( $arg ) || is_numeric( $arg ) ) {
				$var[] = str_replace( [ '_', ' ' ], '-', $arg );
			}
		}

		return '--' . strtolower( implode( '-', $var ) );
	}

	/**
	 * CSS var reference.
	 */
	public function var_reference() {
		if ( empty( func_get_arg( 0 ) ) ) {
			return null;
		}

		return sprintf( 'var(%s)', call_user_func_array( [ $this, 'var_name' ], func_get_args() ) );
	}

	/**
	 * Parses color and returns respective instance if valid color is provided.
	 *
	 * @param string $color
	 *
	 * @return Color|null
	 */
	public function color( $color ) {
		if ( $color instanceof Color ) {
			return $color;
		}

		if ( $color ) {
			try {
				return Factory::fromString( $color );
			} catch ( Exception $e ) {
				// Do not handle exceptions, just return null
			}
		}

		return null;
	}

	/**
	 * Mix two colors.
	 *
	 * @param string $color_a
	 * @param string $color_b
	 * @param int    $weight
	 *
	 * @return string
	 */
	public function color_mix( $color_a, $color_b, $weight = 0 ) {
		$color_a = $this->color( $color_a );
		$color_b = $this->color( $color_b );

		if ( $color_a && $color_b ) {
			$rgb_a  = $color_a->toRgb();
			$rgb_b  = $color_b->toRgb();
			$weight = $weight / 100;

			$rmix = round( $rgb_b->red() + ( $rgb_a->red() - $rgb_b->red() ) * $weight );
			$gmix = round( $rgb_b->green() + ( $rgb_a->green() - $rgb_b->green() ) * $weight );
			$bmix = round( $rgb_b->blue() + ( $rgb_a->blue() - $rgb_b->blue() ) * $weight );

			return (string) $this->color( sprintf( 'rgb(%d, %d, %d)', $rmix, $gmix, $bmix ) )->toHex();
		}

		return null;
	}

	/**
	 * Shade color.
	 *
	 * @param string $color
	 * @param int    $weight
	 *
	 * @return string
	 */
	public function color_shade( $color, $weight ) {
		return $this->color_mix( '#000000', $color, $weight );
	}

	/**
	 * Tint color.
	 *
	 * @param string $color
	 * @param int    $weight
	 *
	 * @return string
	 */
	public function color_tint( $color, $weight ) {
		return $this->color_mix( '#ffffff', $color, $weight );
	}

	/**
	 * Color shift.
	 *
	 * @param string $color
	 * @param int    $weight
	 *
	 * @return string
	 */
	public function color_shift( $color, $weight ) {
		return $weight > 0 ? $this->color_shade( $color, $weight ) : $this->color_tint( $color, -$weight );
	}

	/**
	 * Get color alpha.
	 *
	 * @param string $color
	 *
	 * @return float|null
	 */
	public function color_alpha( $color ) {
		$color = $this->color( $color );

		if ( ! $color ) {
			return null;
		}

		if ( method_exists( $color, 'alpha' ) ) {
			$alpha = $color->alpha();

			if ( $color instanceof Hex ) {
				$alpha = hexdec( $alpha ) / 255;
			}

			return $alpha;
		}

		return 1;
	}

	/**
	 * Checks if given color is dark color based on WCAG standard.
	 *
	 * @param string $color
	 *
	 * @return bool
	 * @see https://www.w3.org/TR/AERT/#color-contrast
	 */
	public function is_dark_color( $color ) {
		if ( $color = $this->color( $color ) ) {
			$rgb = $color->toRgb();

			$brightness = ( $rgb->red() * 299 + $rgb->green() * 587 + $rgb->blue() * 114 ) / 1000 / 255;

			return $brightness < 0.5;
		}

		return false;
	}

	/**
	 * Print CSS vars (private function).
	 */
	public function print_css_vars() {
		if ( ! empty( $this->css_vars ) ) {
			kalium_print_inline_style(
				[
					'id'       => 'theme-vars',
					'selector' => kalium_get_root_selector(),
					'vars'     => $this->css_vars,
				]
			);
		}
	}
}
