<?php
/**
 * Kalium WordPress Theme
 *
 * Template loader class.
 *
 * @author Laborator
 * @link   https://kaliumtheme.com
 */
namespace Kalium\Core;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Direct access not allowed.
}

class Template_Loader {

	/**
	 * Get template.
	 *
	 * @param string $file
	 * @param array  $args
	 */
	public function get_template( $file, $args = [] ) {
		$located = apply_filters(
			'kalium_get_template',
			locate_template( "templates/{$file}" ),
			$file,
			$args
		);

		if ( ! file_exists( $located ) ) {
			kalium_doing_it_wrong( __FUNCTION__, sprintf( '%s does not exist.', '<code>' . $file . '</code>' ), '2.1' );
			return;
		}

		$args = apply_filters( 'kalium_get_template_args', $args, $file );

		if ( ! empty( $args ) && is_array( $args ) ) {
			extract( $args );
		}

		do_action( 'kalium_template_before-' . $file, $located, $args );

		include $located;

		do_action( 'kalium_template_after-' . $file, $located, $args );
	}

	/**
	 * Replace WooCommerce template.
	 *
	 * @param string|array    $template_fragment
	 * @param string|callable $new_template
	 * @param bool|callable   $do_replace
	 *
	 * @return callable
	 * @since 4.0
	 */
	public function replace_woocommerce_template( $template_fragment, $new_template = null, $do_replace = true ) {

		/**
		 * Group of templates to replace.
		 */
		if ( is_array( $template_fragment ) ) {
			$replaced_templates = [];

			foreach ( $template_fragment as $template_fragment_entry ) {
				$replaced_templates[] = $this->replace_woocommerce_template( $template_fragment_entry, $new_template, $do_replace );
			}

			return function () use ( $replaced_templates ) {
				foreach ( $replaced_templates as $unset_hook ) {
					$unset_hook();
				}
			};
		}

		/**
		 * Replace template content.
		 *
		 * @param string $template
		 *
		 * @return string
		 */
		$replace_template_content = function ( $template, $template_name = '', $args = [] ) use ( $template_fragment, $new_template, $do_replace ) {
			return $this->replace_template( $template, $template_fragment, $new_template, $args, $do_replace );
		};

		// Replace template content
		add_filter( 'wc_get_template', $replace_template_content, 1000, 3 );
		add_filter( 'wc_get_template_part', $replace_template_content, 1000 );

		return function () use ( $replace_template_content ) {
			remove_filter( 'wc_get_template', $replace_template_content, 1000 );
			remove_filter( 'wc_get_template_part', $replace_template_content, 1000 );
		};
	}

	/**
	 * Blank template file.
	 *
	 * @return string
	 * @since 4.0
	 */
	public function get_blank_template() {
		return kalium()->locate_file( 'includes/index.php' );
	}

	/**
	 * Match template path.
	 *
	 * @param string $template
	 * @param string $template_fragment
	 *
	 * @return bool
	 * @since 4.0
	 */
	private function match_template_fragment( $template, $template_fragment ) {
		return preg_match( '#' . str_replace( '\*', '[\w\-]+', preg_quote( $template_fragment ) ) . '$#', $template );
	}

	/**
	 * Validate $do_replace condition.
	 *
	 * @param bool|callable $do_replace
	 * @param string        $template_fragment
	 *
	 * @return bool
	 * @since 4.0
	 */
	private function do_replace( $do_replace, $template_fragment ) {
		$result = false;

		if ( is_bool( $do_replace ) ) {
			$result = $do_replace;
		} elseif ( is_callable( $do_replace ) ) {
			$result = (bool) call_user_func( $do_replace );
		}

		return apply_filters( 'kalium_template_do_replace', $result, $template_fragment );
	}

	/**
	 * Parse template replace argument.
	 *
	 * @param string|callable $template
	 * @param array           $args
	 *
	 * @return string|false
	 * @since 4.0
	 */
	private function parse_template_content( $template, $args = [] ) {
		$replace_content = null;

		// Callable function
		if ( is_callable( $template ) ) {
			ob_start();

			$callable_return  = call_user_func( $template, $args );
			$callable_content = ob_get_clean();

			// Replace content
			if ( ! is_null( $callable_return ) ) {
				if ( kalium()->locate_file( 'templates/' . $callable_return ) ) {
					$replace_content = $this->parse_template_content( $callable_return, $args );
				} else {
					$replace_content = $callable_return;
				}
			} else {
				$replace_content = $callable_content;
			}
		} elseif ( ! is_null( $template ) ) {
			ob_start();

			// Load template from file
			kalium_get_template( $template, $args );

			$replace_content = ob_get_clean();
		}

		return $replace_content;
	}

	/**
	 * Replace template.
	 *
	 * @param string          $template
	 * @param string          $template_fragment
	 * @param string|callable $new_template
	 * @param array           $args
	 * @param bool|callable   $do_replace
	 *
	 * @return string
	 * @since 4.0
	 */
	private function replace_template( $template, $template_fragment, $new_template, $args = [], $do_replace = true ) {
		if ( is_admin() && ! wp_doing_ajax() && ! kalium()->elementor->is_edit_mode() ) {
			return $template;
		}

		if ( $this->match_template_fragment( $template, $template_fragment ) && $this->do_replace( $do_replace, $template_fragment ) ) {
			$template_content = $this->parse_template_content( $new_template, $args );

			if ( false !== $template_content ) {

				// Print template
				echo $template_content;

				return $this->get_blank_template();
			}
		}

		return $template;
	}
}
