<?php
/**
 * Version upgrade task class.
 *
 * @author Laborator
 * @link   https://kaliumtheme.com
 */
namespace Kalium\Core;

use JsonSerializable;

class Version_Upgrade_Task implements JsonSerializable {

	/**
	 * Upgrade tasks option.
	 *
	 * @const string
	 */
	const UPGRADE_TASKS_OPTION = 'kalium_upgrade_tasks';

	/**
	 * Task entry ID.
	 *
	 * @var string
	 */
	public $id;

	/**
	 * The version of the theme that is required to run this task.
	 *
	 * @var string
	 */
	public $version;

	/**
	 * Auto-executing task.
	 *
	 * @var bool
	 */
	public $auto;

	/**
	 * String labels when task type is not background.
	 *
	 * @var array
	 */
	public $labels = [

		/**
		 * Title of the admin notice.
		 *
		 * @type string
		 */
		'title'               => 'Database Upgrade Required',

		/**
		 * Description that will be shown on admin notice.
		 *
		 * @type string
		 */
		'description'         => 'This update of the theme requires a database upgrade to maintain full compatibility and optimal performance. Please run the update now to keep your site running smoothly.',

		/**
		 * Description displayed when the database update is auto run task.
		 *
		 * @type string
		 */
		'description_auto'    => 'Theme database upgrade in progress...',

		/**
		 * Button text.
		 *
		 * @type string
		 */
		'upgrade_start'       => 'Run Upgrade',

		/**
		 * Busy text.
		 *
		 * @type string
		 */
		'upgrade_in_progress' => 'Upgrade in process...',

		/**
		 * Completed text.
		 *
		 * @type string
		 */
		'upgrade_complete'    => 'Upgrade completed!',
	];

	/**
	 * Construct.
	 *
	 * @param string $id
	 * @param string $version
	 * @param array  $args
	 */
	public function __construct( $id, $version, $args = [] ) {
		$args = wp_parse_args(
			$args,
			[
				'auto'   => false,
				'labels' => null,
			]
		);

		// Fill data
		$this->id      = $id;
		$this->version = $version;
		$this->auto    = $args['auto'];
		$this->labels  = array_merge( $this->labels, wp_parse_args( $args['labels'] ) );
	}

	/**
	 * Check if is eligible.
	 *
	 * @return bool
	 */
	public function is_eligible() {
		if ( ! empty( $this->version ) ) {
			return kalium()->version_upgrades->is_older_than( $this->version ) && version_compare( $this->version, kalium()->get_version(), '<=' );
		}

		return true;
	}

	/**
	 * Get sub-tasks.
	 *
	 * @return array<callable>
	 */
	public function get_sub_tasks() {
		$file = kalium()->locate_file( "includes/upgrade/{$this->id}.php" );

		return $file ? require $file : [];
	}

	/**
	 * Check if task is completed.
	 *
	 * @return bool
	 */
	public function is_completed() {
		$upgrade_tasks = get_option( self::UPGRADE_TASKS_OPTION, [] );

		foreach ( $upgrade_tasks as $upgrade_task ) {
			if ( $this->id === $upgrade_task['id'] ) {
				return ! empty( $upgrade_task['completed'] );
			}
		}

		return false;
	}

	/**
	 * Get task index.
	 *
	 * @return int|bool
	 */
	public function get_index() {
		$upgrade_tasks = get_option( self::UPGRADE_TASKS_OPTION, [] );

		return array_search( $this->id, array_column( $upgrade_tasks, 'id' ), true );
	}

	/**
	 * Get task data, if available.
	 *
	 * @return array|false
	 */
	public function get_data() {
		$upgrade_tasks = get_option( self::UPGRADE_TASKS_OPTION, [] );
		$task_index    = $this->get_index();

		if ( false !== $task_index ) {
			return $upgrade_tasks[ $task_index ];
		}

		return null;
	}

	/**
	 * Set subtask result (stored in database).
	 *
	 * @param int  $subtask_id
	 * @param bool $result
	 */
	public function set_subtask_result( $subtask_id, $result ) {
		$upgrade_tasks = get_option( self::UPGRADE_TASKS_OPTION, [] );
		$task_index    = $this->get_index();
		$sub_tasks     = $this->get_sub_tasks();

		if ( false !== $task_index ) {
			$upgrade_tasks[ $task_index ]['results'][ $subtask_id ] = $result;

			// Mark task as completed when all tasks are run
			if ( count( $upgrade_tasks[ $task_index ]['results'] ) === count( $sub_tasks ) ) {
				$upgrade_tasks[ $task_index ] = array_merge(
					$upgrade_tasks[ $task_index ],
					[
						'completed' => true,
						'timestamp' => time(),
					]
				);
			}
		} else {
			$upgrade_tasks[] = [
				'id'        => $this->id,
				'completed' => 1 === count( $sub_tasks ),
				'results'   => [
					$subtask_id => $result,
				],
				'timestamp' => time(),
			];
		}

		update_option( self::UPGRADE_TASKS_OPTION, $upgrade_tasks, false );
	}

	/**
	 * Execute task.
	 *
	 * @param int $sub_task_id
	 *
	 * @return array<bool>|bool
	 */
	public function execute( $sub_task_id = null ) {
		$results = kalium_get_array_key( $this->get_data(), 'results', [] );

		if ( $sub_tasks = $this->get_sub_tasks() ) {
			kalium()->require_file( 'includes/upgrade/functions.php' );

			// Run single sub-task
			if ( is_numeric( $sub_task_id ) && isset( $sub_tasks[ $sub_task_id ] ) ) {
				$sub_tasks = [
					$sub_task_id => $sub_tasks[ $sub_task_id ],
				];
			}

			// Execute sub-tasks
			foreach ( $sub_tasks as $subtask_id => $task_entry ) {
				$result = false;

				// Skip already completed tasks
				if ( array_key_exists( $subtask_id, $results ) && $results[ $subtask_id ] ) {
					continue;
				}

				if ( is_callable( $task_entry ) ) {
					try {
						$result = @call_user_func( $task_entry, $this );
					} catch ( \Exception $e ) {
					}
				}

				// Set subtask result
				$results[ $subtask_id ] = $result;
				$this->set_subtask_result( $subtask_id, $result );
			}
		}

		if ( is_numeric( $sub_task_id ) ) {
			return current( $results );
		}

		return $results;
	}

	/**
	 * Serialize for JSON.
	 *
	 * @return array
	 */
	public function jsonSerialize(): array {
		return [
			'id'            => $this->id,
			'version'       => $this->version,
			'labels'        => $this->labels,
			'auto'          => $this->auto,
			'eligible'      => $this->is_eligible(),
			'completed'     => $this->is_completed(),
			'totalSubtasks' => count( $this->get_sub_tasks() ),
		];
	}
}
