<?php
/**
 * Kalium WordPress Theme
 *
 * Customizer Control Type: Conditions.
 *
 * @author Laborator
 * @link   https://kaliumtheme.com
 */
namespace Kalium\Customize\Control_Types;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Direct access not allowed.
}

class Conditions extends Base_Control {

	/**
	 * Control type.
	 *
	 * @var string
	 */
	public $type = 'kalium-conditions';

	/**
	 * Comparators options.
	 *
	 * Available comparators to check against.
	 *
	 * @var array
	 */
	public $comparators = [];

	/**
	 * Operators options.
	 *
	 * Available comparison operators.
	 *
	 * @var array
	 */
	public $operators = [];

	/**
	 * Value options.
	 *
	 * Available values for each comparator type.
	 *
	 * @var array
	 */
	public $value_options = [];

	/**
	 * Empty state message.
	 *
	 * @var string
	 */
	public $empty_state_message = '';

	/**
	 * Add condition button text.
	 *
	 * @var string
	 */
	public $add_condition_text = '';

	/**
	 * Inline label.
	 *
	 * @var bool
	 */
	public $inline_label = false;

	/**
	 * Constructor.
	 *
	 * @param \WP_Customize_Manager $manager Customizer manager.
	 * @param string                $id      Control ID.
	 * @param array                 $args    Control arguments.
	 */
	public function __construct( $manager, $id, $args = [] ) {
		parent::__construct( $manager, $id, $args );

		// Set default comparators if not provided
		if ( empty( $this->comparators ) ) {
			$this->comparators = $this->get_default_comparators();
		}

		// Set default operators if not provided
		if ( empty( $this->operators ) ) {
			$this->operators = $this->get_default_operators();
		}

		// Set default value options if not provided
		if ( empty( $this->value_options ) ) {
			$this->value_options = $this->get_default_value_options();
		}

		// Set default empty state message
		if ( empty( $this->empty_state_message ) ) {
			$this->empty_state_message = 'No conditions set. Click below to add your first condition.';
		}

		// Set default add condition button text
		if ( empty( $this->add_condition_text ) ) {
			$this->add_condition_text = 'Add Condition';
		}
	}

	/**
	 * Get default comparators.
	 *
	 * @return array
	 */
	protected function get_default_comparators() {
		return [
			'general_page'     => 'General Page',
			'pages'            => 'Page',
			'posts'            => 'Post',
			'archives'         => 'Archive',
			'taxonomy_archive' => 'Taxonomy Archive',
			'user'             => 'User',
			'woocommerce'      => 'WooCommerce Page',
			'day_of_month'     => 'Day of Month',
			'month'            => 'Month',
			'year'             => 'Year',
			'time'             => 'Time',
		];
	}

	/**
	 * Get default operators.
	 *
	 * @return array
	 */
	protected function get_default_operators() {
		return [
			'equal_to'         => 'Is',
			'not_equal_to'     => 'Is Not',
		];
	}

	/**
	 * Get default value options.
	 *
	 * @return array
	 */
	protected function get_default_value_options() {
		return [
			// General Page
			'general_page' => [
				'sitewide'       => 'Entire Site',
				'front-page'     => 'Front Page',
				'posts-page'     => 'Blog Page',
				'not-found-page' => '404 Error Page',
				'search-page'    => 'Search Page',
			],
			
			// Pages (Singular pages)
			'pages' => $this->get_pages_options(),
			
			// Posts (Singular posts)
			'posts' => $this->get_posts_options(),
			
			// Archives
			'archives' => [
				'blog'             => 'Blog Archive',
				'date'             => 'Date Archive',
				'author'           => 'Author Archive',
				'custom-post-type' => 'Custom Post Type Archive',
			],
			
			// Taxonomy Archives
			'taxonomy_archive' => [
				'category'        => 'Category Archive',
				'tag'             => 'Tag Archive',
				'custom-taxonomy' => 'Custom Taxonomy Archive',
			],
			
			// User
			'user' => [
				'logged-in'     => 'Logged In',
				'logged-out'    => 'Logged Out',
				'administrator' => 'Administrator',
				'editor'        => 'Editor',
				'author'        => 'Author',
				'contributor'   => 'Contributor',
				'subscriber'    => 'Subscriber',
			],
			
			// WooCommerce
			'woocommerce' => [
				'product-page'     => 'Product Page',
				'products-page'    => 'Shop Archive',
				'product-category' => 'Product Category Archive',
				'product-tag'      => 'Product Tag Archive',
				'cart-page'        => 'Cart Page',
				'checkout-page'    => 'Checkout Page',
				'account-page'     => 'My Account Page',
			],
			
			// Day of Month (1-31)
			'day_of_month' => $this->get_days_of_month_options(),
			
			// Month
			'month' => [
				'1'  => 'January',
				'2'  => 'February',
				'3'  => 'March',
				'4'  => 'April',
				'5'  => 'May',
				'6'  => 'June',
				'7'  => 'July',
				'8'  => 'August',
				'9'  => 'September',
				'10' => 'October',
				'11' => 'November',
				'12' => 'December',
			],
			
			// Year (current year and next 5 years)
			'year' => $this->get_years_options(),
			
			// Time (24 hours + 24 half hours = 48 entries)
			'time' => $this->get_time_options(),
		];
	}
	
	/**
	 * Get pages options.
	 *
	 * @return array
	 */
	protected function get_pages_options() {
		$pages = get_pages( [
			'number'      => 100,
			'post_status' => 'publish',
			'sort_column' => 'post_title',
		] );
		
		$options = [ 'any' => 'Any Page' ];
		
		foreach ( $pages as $page ) {
			$options[ $page->ID ] = $page->post_title;
		}
		
		return $options;
	}
	
	/**
	 * Get posts options.
	 *
	 * @return array
	 */
	protected function get_posts_options() {
		$posts = get_posts( [
			'numberposts' => 100,
			'post_status' => 'publish',
			'orderby'     => 'title',
			'order'       => 'ASC',
		] );
		
		$options = [ 'any' => 'Any Post' ];
		
		foreach ( $posts as $post ) {
			$options[ $post->ID ] = $post->post_title;
		}
		
		return $options;
	}
	
	/**
	 * Get days of month options.
	 *
	 * @return array
	 */
	protected function get_days_of_month_options() {
		$options = [];
		
		for ( $day = 1; $day <= 31; $day++ ) {
			$options[ $day ] = $day;
		}
		
		return $options;
	}
	
	/**
	 * Get years options.
	 *
	 * @return array
	 */
	protected function get_years_options() {
		$current_year = (int) date( 'Y' );
		$options      = [];
		
		// Current year and next 5 years
		for ( $year = $current_year; $year <= $current_year + 5; $year++ ) {
			$options[ $year ] = $year;
		}
		
		return $options;
	}
	
	/**
	 * Get time options (24 hours + 24 half hours).
	 *
	 * @return array
	 */
	protected function get_time_options() {
		$options = [];
		
		// Generate 48 time entries (every 30 minutes)
		for ( $hour = 0; $hour < 24; $hour++ ) {
			// On the hour
			$time_value = sprintf( '%02d:00', $hour );
			$time_label = sprintf( '%02d:00', $hour );
			$options[ $time_value ] = $time_label;
			
			// Half hour
			$time_value = sprintf( '%02d:30', $hour );
			$time_label = sprintf( '%02d:30', $hour );
			$options[ $time_value ] = $time_label;
		}
		
		return $options;
	}

	/**
	 * Add JSON data.
	 */
	public function to_json() {
		parent::to_json();

		$this->json['comparators']         = $this->comparators;
		$this->json['operators']           = $this->operators;
		$this->json['valueOptions']        = $this->value_options;
		$this->json['emptyStateMessage']   = $this->empty_state_message;
		$this->json['addConditionText']    = $this->add_condition_text;
	}

}
