<?php
/**
 * Kalium WordPress Theme
 *
 * Customizer class for Kalium.
 *
 * @author Laborator
 * @link   https://kaliumtheme.com
 *
 * @var \WP_Customize_Manager $wp_customize
 */
namespace Kalium\Customize;

use WP_Customize_Control;
use WP_Customize_Manager;
use WP_Customize_Panel;
use WP_Customize_Section;
use WP_Customize_Setting;
use TypoLab;
use TypoLab_Font_Export_Import;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Direct access not allowed.
}

class Customize {

	/**
	 * Settings groups.
	 */
	use Settings_Groups\Style;
	use Settings_Groups\Pagination;
	use Settings_Groups\Image_Size;
	use Settings_Groups\Element_Visibility;
	use Settings_Groups\Post_Navigation;

	/**
	 * Preview update callbacks.
	 *
	 * @var array
	 */
	public $preview_update_callbacks = [];

	/**
	 * Priority counter.
	 *
	 * @var array
	 */
	public $priority_index = [];

	/**
	 * Current priority index group.
	 *
	 * @var string
	 */
	public $priority_index_group = '';

	/**
	 * Current section stack.
	 *
	 * @var array
	 */
	public $_current_section_stack = [];

	/**
	 * Current tab stack.
	 *
	 * @var array
	 */
	public $_current_tab_stack = [];

	/**
	 * Constructor.
	 */
	public function __construct() {
		add_action( 'after_setup_theme', [ $this, 'add_supports' ] );
		add_action( 'customize_register', [ $this, 'customize_core' ], 0 );
		add_action( 'customize_register', [ $this, 'customize_js_vars' ] );
		add_action( 'customize_register', [ $this, 'customize_register_options' ], 10000 );
		add_action( 'customize_preview_init', [ $this, 'customize_preview_init' ] );
		add_action( 'customize_controls_print_footer_scripts', [ $this, 'customize_footer_scripts' ] );
		add_action( 'customize_update_typolab', [ $this, 'update_typolab_setting' ], 10, 2 );
		add_action( 'wp_ajax_kalium_export_options', [ $this, 'export_options' ] );
		add_action( 'wp_ajax_kalium_import_options', [ $this, 'import_options' ] );
		add_action( 'wp_ajax_kalium_transfer_options', [ $this, 'transfer_options' ] );
		add_action( 'wp_ajax_kalium_reset_options', [ $this, 'reset_options' ] );
	}

	/**
	 * Theme supports and other stuff.
	 */
	public function add_supports() {

		// Add theme support for selective refresh for widgets
		add_theme_support( 'customize-selective-refresh-widgets' );
	}

	/**
	 * Customize core.
	 *
	 * @param WP_Customize_Manager $wp_customize
	 */
	public function customize_core( $wp_customize ) {

		// Register section types
		foreach ( $this->get_supported_section_types() as $section_type ) {
			$wp_customize->register_section_type( $section_type );
		}

		// Register control types (for JS rendering)
		foreach ( $this->get_supported_control_types() as $control_type ) {
			$wp_customize->register_control_type( $control_type );
		}
	}

	/**
	 * Add panel wrapper.
	 *
	 * @param string $id
	 * @param array  $args
	 */
	public function add_panel( $id, $args = [] ) {
		global $wp_customize;

		if ( ! isset( $wp_customize ) ) {
			return;
		}

		// Apply args
		$args = wp_parse_args(
			$args,
			[
				'title'           => '',
				'description'     => '',
				'active_callback' => '',
			]
		);

		// Register panel
		$wp_customize->add_panel( $id, $args );
	}

	/**
	 * Add section wrapper.
	 *
	 * @param string $id
	 * @param array  $args
	 */
	public function add_section( $id, $args = [] ) {
		global $wp_customize;

		if ( ! isset( $wp_customize ) ) {
			return;
		}

		$args = wp_parse_args(
			$args,
			[
				'type'            => 'kalium-section',
				'panel'           => '',
				'title'           => '',
				'description'     => '',
				'active_callback' => '',
				'parent_section'  => $this->get_last_section_id(),
				'start_section'   => true,
			]
		);

		// Revert to default section type
		if ( empty( $args['type'] ) ) {
			$args['type'] = 'kalium-section';
		}

		// Generate ID prefix
		$section_stack_ids = end( $this->_current_section_stack );

		if ( $id_prefix = kalium_get_array_key( $section_stack_ids, 'id' ) ) {
			$id = $id_prefix . '_' . $id;
		}

		// Add section from registered section types
		$supported_section_types = $this->get_supported_section_types();

		// Register section
		if ( $section_type = $supported_section_types[ $args['type'] ] ?? null ) {
			$section_type_instance = new $section_type( $wp_customize, $id, $args );
			$wp_customize->add_section( $section_type_instance );
		} else {
			$wp_customize->add_section( $id, $args );
		}

		// Start section
		if ( $args['start_section'] ) {
			$this->start_section( $id, $args );
		}
	}

	/**
	 * Add control and setting wrapper.
	 *
	 * @param string $id
	 * @param array  $args
	 */
	public function add_control( $id, $args = [] ) {
		global $wp_customize;

		$args = wp_parse_args(
			$args,
			[
				'type'       => null,
				'default'    => null,
				'section'    => $this->get_last_section_id(),
				'dependency' => [],
				'setting'    => [],
			]
		);

		// Register theme option
		kalium()->theme_options->add_option( $id, $args );

		// Register setting and control on Customizer
		if ( isset( $wp_customize ) ) {
			if ( is_array( $args['setting'] ) ) {
				$setting = wp_parse_args(
					$args['setting'],
					[
						'default' => $args['default'],
					]
				);

				$this->add_setting( $id, $setting );
			}

			// Tab dependency
			if ( $tab_dependency = $this->get_current_tab_dependency() ) {
				if ( is_array( $args['dependency'] ) ) {
					$args['dependency'] = array_merge( $tab_dependency, $args['dependency'] );
				} else {
					$args['dependency'] = $tab_dependency;
				}
				if ( defined( 'QCC' ) ) {
					print_r( $args['dependency'] );
					exit;
				}
			}

			// Add control from registered control types
			$supported_control_types = $this->get_supported_control_types();

			if ( $control_type = $supported_control_types[ $args['type'] ] ?? null ) {
				$control_type_instance = new $control_type( $wp_customize, $id, $args );
				$wp_customize->add_control( $control_type_instance );
			} else {
				$wp_customize->add_control( $id, $args );
			}
		}
	}

	/**
	 * Add setting.
	 *
	 * @param string $id
	 * @param array  $args
	 */
	public function add_setting( $id, $args = [] ) {
		global $wp_customize;

		$wp_customize->add_setting( $id, $args );
	}

	/**
	 * Add preview update callback.
	 *
	 * @param array $args
	 */
	public function add_preview_update( $args ) {
		global $wp_customize;

		if ( ! isset( $wp_customize ) || empty( $args['vars'] ) ) {
			return;
		}

		$args = wp_parse_args(
			$args,
			[
				'type'     => 'template', // [template,js,css]
				'vars'     => null,
				'content'  => null,
				'selector' => null, // (type:template)
				'after'    => null, // (type:template)
			]
		);

		$this->preview_update_callbacks[] = $args;
	}

	/**
	 * Start section (does not register on Customizer).
	 *
	 * @param string $id
	 * @param array  $args
	 */
	public function start_section( $id, $args = [] ) {
		global $wp_customize, $_current_tab_control_id;

		if ( ! isset( $wp_customize ) ) {
			return;
		}

		$this->_current_section_stack[] = array_merge(
			[
				'id' => $id,
			],
			$args
		);

		// Each section generates tab control id
		$_current_tab_control_id = $id . '_tab';
	}

	/**
	 * Reset or end current section.
	 *
	 * @param bool $reset
	 */
	public function end_section( $reset = false ) {
		global $wp_customize;

		if ( ! isset( $wp_customize ) ) {
			return;
		}

		if ( $reset ) {
			$this->_current_section_stack = [];
		} else {
			array_pop( $this->_current_section_stack );
		}
	}

	/**
	 * Get last/current section ID.
	 *
	 * @return string
	 */
	public function get_last_section_id() {
		if ( ! empty( $this->_current_section_stack ) ) {
			return end( $this->_current_section_stack )['id'];
		}

		return null;
	}

	/**
	 * Prefixed current section ID.
	 *
	 * @param string $id
	 * @param bool   $section_stack_prefix
	 *
	 * @return string
	 */
	public function section_id( $id, $section_stack_prefix = true ) {
		if ( $section_stack_prefix ) {
			$section_stack_ids = end( $this->_current_section_stack );

			if ( $id_prefix = kalium_get_array_key( $section_stack_ids, 'id' ) ) {
				$id = $id_prefix . '_' . $id;
			}
		}

		return $id;
	}

	/**
	 * Start tab.
	 *
	 * @param string $title
	 */
	public function start_tab( $title ) {
		global $wp_customize, $_current_tab_control_id;

		if ( ! isset( $wp_customize ) ) {
			return;
		}

		$tab_entry_id = sanitize_title( $title );

		// Initialize current tab and register control
		if ( ! isset( $this->_current_tab_stack[ $_current_tab_control_id ] ) ) {
			$this->_current_tab_stack[ $_current_tab_control_id ] = [
				'current_tab' => null,
				'choices'     => [],
			];

			kalium_customize_register_control(
				$_current_tab_control_id,
				[
					'type'    => 'kalium-tabs',
					'choices' => function () use ( $_current_tab_control_id ) {
						return $this->_current_tab_stack[ $_current_tab_control_id ]['choices'];
					},
					'default' => $tab_entry_id,
				]
			);
		}

		// Insert tab
		$this->_current_tab_stack[ $_current_tab_control_id ]['choices'][] = [
			'value' => $tab_entry_id,
			'label' => $title,
		];

		// Set current tab
		$this->_current_tab_stack[ $_current_tab_control_id ]['current_tab'] = $tab_entry_id;
	}

	/**
	 * End current tab.
	 */
	public function end_tab() {
		global $wp_customize, $_current_tab_control_id;

		if ( ! isset( $wp_customize ) ) {
			return;
		}

		if ( ! empty( $this->_current_tab_stack[ $_current_tab_control_id ]['current_tab'] ) ) {
			$this->_current_tab_stack[ $_current_tab_control_id ]['current_tab'] = null;
		}
	}

	/**
	 * Get current tab.
	 *
	 * @return array|false
	 */
	public function get_current_tab_dependency() {
		global $wp_customize, $_current_tab_control_id;

		if ( isset( $wp_customize ) && $_current_tab_control_id && ! empty( $this->_current_tab_stack[ $_current_tab_control_id ]['current_tab'] ) ) {
			return [
				[
					'name'  => $_current_tab_control_id,
					'value' => $this->_current_tab_stack[ $_current_tab_control_id ]['current_tab'],
				],
			];
		}

		return [];
	}

	/**
	 * Prefixed settings array.
	 *
	 * @param string $namespace
	 * @param array  $names
	 *
	 * @return array
	 */
	public function get_prefixed_settings( $namespace, $names ) {
		return array_combine(
			$names,
			array_map(
				function ( $name ) use ( $namespace ) {
					return $namespace . '_' . $name;
				},
				$names
			)
		);
	}

	/**
	 * Supported section types.
	 *
	 * @return array
	 */
	public function get_supported_section_types() {
		return apply_filters(
			'kalium_customize_section_types',
			[
				'kalium-section'       => Section_Types\Base_Section::class,
				'kalium-section-group' => Section_Types\Section_Group::class,
				'kalium-search'        => Section_Types\Search_Section::class,
			]
		);
	}

	/**
	 * Supported control types.
	 *
	 * @return array
	 */
	public function get_supported_control_types() {
		return apply_filters(
			'kalium_customize_control_types',
			[
				'kalium-text'            => Control_Types\Text::class,
				'kalium-textarea'        => Control_Types\Textarea::class,
				'kalium-select'          => Control_Types\Select::class,
				'kalium-custom-select'   => Control_Types\Custom_Select::class,
				'kalium-checkbox'        => Control_Types\Checkbox::class,
				'kalium-checkbox-button' => Control_Types\Checkbox_Button::class,
				'kalium-dimensions'      => Control_Types\Dimensions::class,
				'kalium-radio'           => Control_Types\Radio::class,
				'kalium-radio-button'    => Control_Types\Radio_Button::class,
				'kalium-radio-image'     => Control_Types\Radio_Image::class,
				'kalium-toggle'          => Control_Types\Toggle::class,
				'kalium-range'           => Control_Types\Range::class,
				'kalium-unit'            => Control_Types\Unit::class,
				'kalium-color'           => Control_Types\Palette::class,
				'kalium-media'           => Control_Types\Media::class,
				'kalium-quantity'        => Control_Types\Quantity::class,
				'kalium-multi-numeric'   => Control_Types\Multi_Numeric::class,
				'kalium-align-matrix'    => Control_Types\Align_Matrix::class,
				'kalium-gradient-picker' => Control_Types\Gradient_Picker::class,
				'kalium-box-shadow'      => Control_Types\Box_Shadow::class,
				'kalium-box'             => Control_Types\Box::class,
				'kalium-border'          => Control_Types\Border::class,
				'kalium-border-box'      => Control_Types\Border_Box::class,
				'kalium-conditions'      => Control_Types\Conditions::class,
				'kalium-action-button'   => Control_Types\Action_Button::class,
				'kalium-section-link'    => Control_Types\Section_Link::class,
				'kalium-tabs'            => Control_Types\Tabs::class,
				'kalium-options-export'  => Control_Types\Options_Export::class,
				'kalium-options-import'  => Control_Types\Options_Import::class,
				'kalium-options-reset'   => Control_Types\Options_Reset::class,
				'kalium-html'            => Control_Types\HTML::class,
			]
		);
	}

	/**
	 * Get section groups.
	 *
	 * @return array
	 */
	public function get_section_groups() {
		return apply_filters(
			'kalium_customize_section_groups',
			[
				// General
				[
					'id'       => 'kalium_group_general',
					'title'    => 'General Options',
					'priority' => 1,
					'sections' => [
						[
							'id'      => 'kalium_general',
							'title'   => 'General',
							'icon'    => 'cog',
							'options' => [
								'general/social-icons.php',
								'general/integrations.php',
								'general/image-loading-placeholder.php',
								'general/media.php',
								'general/lightbox.php',
								'general/custom-javascript.php',
								'general/breadcrumb.php',
								'general/scroll-to-top.php',
								'general/maintenance-mode.php',
							],
						],
						[
							'id'      => 'kalium_header',
							'title'   => 'Header',
							'icon'    => 'header',
							'options' => [
								'header/header.php',
							],
						],
						[
							'id'      => 'kalium_footer',
							'title'   => 'Footer',
							'icon'    => 'footer',
							'options' => [
								'footer/footer.php',
							],
						],
						[
							'id'      => 'kalium_sidebars',
							'title'   => 'Sidebars',
							'icon'    => 'sidebar',
							'options' => [
								'general/sidebars.php',
							],
						],
						[
							'id'      => 'kalium_styling',
							'title'   => 'Styling',
							'icon'    => 'color',
							'options' => [
								'styling/brand.php',
								'styling/layout.php',
								'styling/colors.php',
								'styling/typography.php',
								'styling/links.php',
								'styling/buttons.php',
								'styling/forms.php',
								'styling/site-frame.php',
							],
						],
						[
							'id'      => 'kalium_performance',
							'title'   => 'Performance',
							'icon'    => 'trendingUp',
							'options' => [
								'performance.php',
							],
						],
					],
				],

				// Post Types
				[
					'id'       => 'kalium_group_post_types',
					'title'    => 'Post Types',
					'priority' => 100,
					'sections' => [
						[
							'id'      => 'kalium_blog',
							'title'   => 'Blog',
							'icon'    => 'postFeaturedImage',
							'options' => [
								'blog/blog-page.php',
								'blog/post-page.php',
							],
						],
						[
							'type'       => 'kalium-section',
							'id'         => 'kalium_portfolio',
							'title'      => 'Portfolio',
							'icon'       => 'category',
							'enabled'    => kalium()->is->portfolio_active(),
							'options'    => [
								'portfolio/portfolio-page.php',
								'portfolio/project-page.php',
							],
							'setting_id' => 'kalium_portfolio_post_type',
						],
						[
							'id'      => 'WooCommerce',
							'title'   => 'WooCommerce',
							'icon'    => 'store',
							'enabled' => kalium()->is->woocommerce_active(),
							'options' => [
								'woocommerce/general.php',
								'woocommerce/product-catalog.php',
								'woocommerce/product-page.php',
								'woocommerce/cart-and-checkout.php',
								'woocommerce/my-account.php',
							],
						],
						// TODO: Work in progress
						// ...$this->get_custom_post_types_sections(),
						[
							'id'      => 'kalium_search_page',
							'title'   => 'Search Results',
							'icon'    => 'search',
							'options' => [
								'general/search-results.php',
							],
						],
					],
				],

				// Other
				[
					'id'       => 'kalium_group_other',
					'title'    => 'Other',
					'priority' => 900,
					'sections' => [
						[
							'id'      => 'kalium_options',
							'title'   => 'Manage Options',
							'icon'    => 'backup',
							'options' => [
								'other/manage-options.php',
							],
						],
						[
							'id'    => 'kalium_documentation',
							'title' => 'Documentation',
							'icon'  => 'lifesaver',
							'href'  => admin_url( 'admin.php?page=kalium&tab=help' ),
						],
					],
				],
			]
		);
	}

	/**
	 * Customize JS vars.
	 */
	public function customize_js_vars() {
		kalium_add_inline_script(
			'theme-customizer',
			kalium_admin_js_define_vars(
				[
					'themeBaseColorsArgs'      => kalium_get_theme_base_colors_args(),
					'themeColorVars'           => kalium_get_theme_color_vars(),
					'responsiveViewports'      => kalium_get_responsive_viewports(),
					'otherPanelsSectionsIcons' => function () {
						return [
							'nav_menus'         => 'menu',
							'widgets'           => 'widget',
							'static_front_page' => 'home',
							'custom_css'        => 'code',
						];
					},
				],
				[
					'sub_prop'    => 'customize',
					'echo'        => false,
					'script_wrap' => false,
				]
			),
			'before'
		);
	}

	/**
	 * Customize register options.
	 */
	public function customize_register_options() {

		// Set higher priority number for core panels/sections
		$this->move_core_panels();

		// Move WooCommerce sections and controls to Kalium_WooCommerce
		$this->move_woocommerce_sections_and_controls();

		// Search
		$this->add_section(
			'kalium_search',
			[
				'type'            => 'kalium-search',
				'title'           => 'Search',
				'priority'        => 1,
				'no_results_text' => 'No results found!',
				'max_results'     => 15,
				'parent_section'  => null,
				'start_section'   => false,
			]
		);

		// Register theme options
		$this->register_theme_options();
	}

	/**
	 * Customize preview on frontend.
	 */
	public function customize_preview_init() {
		kalium_enqueue( 'theme-customizer-preview' );

		kalium_add_inline_script(
			'theme-customizer-preview',
			kalium_admin_js_define_var(
				'previewUpdateCallbacks',
				$this->preview_update_callbacks,
				[
					'sub_prop'    => 'customizePreview',
					'echo'        => false,
					'script_wrap' => false,
				]
			)
		);
	}

	/**
	 * Customize footer scripts.
	 */
	public function customize_footer_scripts() {

		// Admin color
		$color_scheme = get_user_option( 'admin_color' );

		if ( ! empty( $color_scheme ) ) {
			wp_print_inline_script_tag( sprintf( 'document.body.className += " admin-color-%s"', esc_js( $color_scheme ) ) );
		}
	}

	/**
	 * Typolab setting.
	 *
	 * @param bool                 $value
	 * @param WP_Customize_Setting $setting
	 */
	public function update_typolab_setting( $value, $setting ) {
		$font_settings = TypoLab::get_font_settings();

		switch ( $setting->id ) {

			// Font Preload
			case 'typolab_font_preload':
				$font_settings['font_preload'] = $value;
				break;

			// Pull Google Fonts
			case 'typolab_pull_google_fonts':
				$font_settings['pull_google_fonts'] = TypoLab::$pull_google_fonts = $value;

				if ( $value ) {
					TypoLab_Font_Export_Import::install_fonts();
				}
				break;

			// Pull Google Fonts
			case 'typolab_font_display':
				$font_settings['font_display'] = $value;
				break;
		}

		TypoLab::set_option( 'font_settings', $font_settings );
	}

	/**
	 * Reset options.
	 */
	public function reset_options() {
		if ( current_user_can( 'manage_options' ) ) {
			foreach ( kalium()->theme_options->get_all() as $id => $args ) {
				remove_theme_mod( $id );
			}

			wp_send_json_success();
		}
	}

	/**
	 * Export options.
	 */
	public function export_options() {
		if ( current_user_can( 'manage_options' ) ) {
			wp_send_json(
				[
					'type'       => 'kalium-customize-export',
					'theme'      => wp_get_theme()->get_stylesheet(),
					'date'       => date( 'r' ),
					'options'    => kalium()->theme_options->get_modified_options(),
					'custom_css' => wp_get_custom_css(),
				]
			);
		}
	}

	/**
	 * Import options.
	 */
	public function import_options() {
		if ( current_user_can( 'manage_options' ) ) {
			$options = kalium_decode_json(
				wp_unslash(
					kalium()->request->request( 'options' )
				),
				true
			);

			if ( isset( $options['type'] ) && 'kalium-customize-export' === $options['type'] ) {

				// Reset options first
				foreach ( kalium()->theme_options->get_all() as $id => $args ) {
					remove_theme_mod( $id );
				}

				// Insert new options
				$modified_options = [];

				foreach ( $options['options'] as $id => $value ) {
					set_theme_mod( $id, $value );

					$modified_options[] = $id;
				}

				// Custom CSS
				if ( ! empty( $options['custom_css'] ) ) {
					wp_update_custom_css_post( $options['custom_css'] );
				}

				// Success
				wp_send_json_success(
					[
						'modified' => $modified_options,
					]
				);
			}

			// Fail
			wp_send_json_error();
		}
	}

	/**
	 * Transfer options.
	 */
	public function transfer_options() {
		if ( current_user_can( 'manage_options' ) ) {
			$parent_slug = kalium()->get_slug();

			switch ( kalium()->request->input( 'transfer_mode' ) ) {
				case 'child':
					foreach ( wp_get_themes() as $theme ) {
						if ( $theme->parent() && $parent_slug === $theme->parent()->get_stylesheet() ) {
							$src_stylesheet = $theme->get_stylesheet();
							break;
						}
					}
					break;

				case 'parent':
					$src_stylesheet = $parent_slug;
					break;
			}

			if ( empty( $src_stylesheet ) ) {
				return;
			}

			// Transfer theme mods
			$src_mods = get_option( 'theme_mods_' . $src_stylesheet );

			foreach ( kalium()->theme_options->get_all() as $id => $args ) {
				if ( 'theme_mod' === kalium_get_array_key( $args, 'setting_type' ) && array_key_exists( $id, $src_mods ) ) {
					set_theme_mod( $id, $src_mods[ $id ] );
				}
			}

			// Custom CSS
			wp_update_custom_css_post( wp_get_custom_css( $src_stylesheet ) );

			// Options transferred
			wp_send_json_success();
		}
	}

	/**
	 * Get custom post types sections.
	 *
	 * @return array
	 */
	public function get_custom_post_types_sections() {
		return array_map(
			function ( $post_type ) {
				if ( ! post_type_exists( $post_type ) ) {
					return false;
				}

				$section_id     = 'kalium_cpt_' . $post_type;
				$post_type_obj  = get_post_type_object( $post_type );
				$post_type_icon = 'pin';

				return [
					'id'        => $section_id,
					'title'     => $post_type_obj->label,
					'post_type' => $post_type,
					'icon'      => $post_type_icon,
					'options'   => [
						'custom-post-type/archive.php',
						'custom-post-type/singular.php',
					],
				];
			},
			kalium()->custom_post_types->get_supported_post_types()
		);
	}

	/**
	 * Get next priority number.
	 *
	 * @return int
	 */
	public function get_next_priority( $incrementor = 0 ) {
		if ( ! isset( $this->priority_index[ $this->priority_index_group ] ) ) {
			$this->priority_index[ $this->priority_index_group ] = 0;
		}

		$current_index  = & $this->priority_index[ $this->priority_index_group ];
		$current_index += 10 + $incrementor;

		return $current_index;
	}

	/**
	 * Set priority group.
	 *
	 * @param string $name
	 * @param int    $starting_index
	 */
	public function set_priority_group( $name, $starting_index = 0 ) {
		$this->priority_index_group = $name;

		// Set starting index for current group
		$this->priority_index[ $this->priority_index_group ] = $starting_index;
	}

	/**
	 * Register everything (including section groups, sections and theme options).
	 */
	public function register_theme_options() {
		foreach ( $this->get_section_groups() as $section_group ) {

			// Section group
			$this->add_section(
				$section_group['id'],
				[
					'type'          => 'kalium-section-group',
					'title'         => $section_group['title'],
					'priority'      => $section_group['priority'],
					'separator'     => $section_group['separator'] ?? true,
					'start_section' => false,
				]
			);

			// Skip if there are no sections to register
			if ( empty( $section_group['sections'] ) ) {
				continue;
			}

			$section_index = 0;

			foreach ( $section_group['sections'] as $section ) {
				$section = wp_parse_args(
					$section,
					[
						'id'      => null,
						'title'   => null,
						'enabled' => true,
					]
				);

				// Disabled section
				if ( ! $section['enabled'] ) {
					continue;
				}

				// Set priority group
				$this->set_priority_group( $section_group['id'], $section_group['priority'] );

				// Section
				$this->add_section(
					$section['id'],
					[
						'type'       => $section['type'] ?? null,
						'title'      => $section['title'],
						'priority'   => $this->get_next_priority( $section_index ),
						'href'       => $section['href'] ?? null,
						'icon'       => $section['icon'] ?? null,
						'setting_id' => $section['setting_id'] ?? null,
					]
				);

				// Skip if there are no options to load
				if ( empty( $section['options'] ) ) {
					continue;
				}

				foreach ( $section['options'] as $file ) {
					kalium()->require_file(
						'includes/theme-options/' . $file,
						[
							'section' => $section,
						]
					);
				}

				// End sections
				$this->end_section( true );

				++$section_index;
			}
		}
	}

	/**
	 * Autofocus section link.
	 *
	 * @param string $section_id
	 *
	 * @return string
	 */
	public function section_link( $section_id ) {
		return add_query_arg(
			[
				'autofocus' => [
					'section' => $section_id,
				],
			],
			admin_url( 'customize.php' )
		);
	}

	/**
	 * Set order for core panels and sections.
	 */
	private function move_core_panels() {
		/** @var WP_Customize_Manager $wp_customize */
		global $wp_customize;

		$core_starting_priority  = 2000;
		$other_starting_priority = 1000;

		$core_panels = [
			'widgets',
			'woocommerce',
			'themes',
			'nav_menus',
		];

		$core_sections = [
			'static_front_page',
			'custom_css',
			'title_tagline',
			'colors',
			'header_image',
			'background_image',
		];

		/** @var WP_Customize_Panel $panel */
		foreach ( $wp_customize->panels() as $panel ) {
			if ( in_array( $panel->id, $core_panels ) ) {
				$panel->priority += $core_starting_priority;
			} else {
				$panel->priority += $other_starting_priority;

				$add_plugins_section_group = true;
			}
		}

		/** @var WP_Customize_Section $section */
		foreach ( $wp_customize->sections() as $section ) {
			if ( empty( $section->panel ) ) {
				if ( in_array( $section->id, $core_sections ) ) {
					$section->priority += $core_starting_priority;
				} elseif ( ! ( $section instanceof Section_Types\Base_Section ) ) {
					$section->priority += $other_starting_priority;

					$add_plugins_section_group = true;
				}
			}
		}

		// Create Section groups
		$this->add_section(
			'kalium_group_core',
			[
				'type'          => 'kalium-section-group',
				'title'         => 'WordPress Core',
				'priority'      => $core_starting_priority,
				'start_section' => false,
			]
		);

		if ( isset( $add_plugins_section_group ) ) {
			$this->add_section(
				'kalium_group_other_plugins',
				[
					'type'          => 'kalium-section-group',
					'title'         => 'Plugins',
					'priority'      => $other_starting_priority,
					'separator'     => true,
					'start_section' => false,
				]
			);
		}
	}

	/**
	 * Move WooCommerce sections and controls to Kalium_WooCommerce.
	 */
	private function move_woocommerce_sections_and_controls() {
		/** @var WP_Customize_Manager $wp_customize */
		global $wp_customize;

		if ( kalium()->is->woocommerce_active() ) {

			// Move WooCommerce sections
			foreach ( $wp_customize->sections() as $section ) {
				/** @var WP_Customize_Section $section */
				if ( 'woocommerce' === $section->panel ) {

					// Deregister current section as default section type
					$wp_customize->remove_section( $section->id );

					// Move current options from this WooCommerce section
					foreach ( $wp_customize->controls() as $control ) {
						/** @var WP_Customize_Control $control */
						if ( $section->id === $control->section ) {
							$control->priority += 1000;
						}
					}

					// Add section link
					kalium_customize_register_control(
						$section->id,
						[
							'type'         => 'kalium-section-link',
							'label'        => $section->title,
							'section_link' => $section->id,
							'priority'     => $section->priority,
							'section'      => 'WooCommerce',
						]
					);

					// Wrap with Kalium_Section
					kalium_customize_register_section(
						$section->id,
						[
							'title'          => $section->title,
							'visible'        => false,
							'start_section'  => false,
							'parent_section' => 'WooCommerce',
						]
					);
				}
			}

			// Move store notice as last control
			$store_notice_control = $wp_customize->get_control( 'woocommerce_store_notice' );

			if ( $store_notice_control ) {
				$store_notice_control->priority = 900;
			}
		}
	}
}
