<?php
/**
 * Product navigation settings groups.
 *
 * @author Laborator
 * @link   https://kaliumtheme.com
 */
namespace Kalium\Customize\Settings_Groups;

use WP_Term;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Direct access not allowed.
}

trait Post_Navigation {

	/**
	 * Post navigation props.
	 *
	 * @var array
	 */
	private $post_navigation_props = [
		'layout',
		'align_horizontal',
		'show_info',
		'loop',
		'in_same_term',
		'reverse',
		'arrows',
		'subtitle',
		'archive',
		'taxonomy',
		'arrow_type',
		'title_content_type',
		'title_custom_prev',
		'title_custom_next',
		'subtitle_content_type',
		'subtitle_taxonomy',
		'subtitle_custom_prev',
		'subtitle_custom_next',
		'image',
		'image_align',
		'subtitle_position',
		'subtitle_show_on_hover',
		'archive_url',
		'archive_taxonomy_link',
		'archive_taxonomy',
		'visibility',
		'label',
		'label_visibility',
		'image_visibility',

		// Style
		'style_color',
		'style_archive_color',
		'style_subtitle_color',
		'style_container_bg',
		'style_container_border',
		'style_container_border_radius',
		'style_container_height',
		'style_container_shadow',
		'style_alt_container_bg',
		'style_alt_separator_color',
		'style_alt_color',
		'style_image_width',
		'style_image_border_radius',
	];

	/**
	 * Register post navigation settings.
	 *
	 * @param string $namespace
	 * @param array  $args
	 *
	 * @since 4.1.2
	 */
	public function register_post_navigation_settings( $namespace, $args = [] ) {
		$settings = $this->get_prefixed_settings( $namespace, $this->post_navigation_props );

		$args = wp_parse_args(
			$args,
			[
				'post_type'    => 'post',
				'selector'     => '.post-navigation',
				'layout_types' => null,
				'defaults'     => null,
				'choices'      => null,
				'placeholders' => null,
			]
		);

		// Defaults
		$defaults = wp_parse_args(
			$args['defaults'],
			[
				'layout'                 => null,
				'align_horizontal'       => null,
				'show_info'              => null,

				// Options
				'loop'                   => true,
				'in_same_term'           => false,
				'reverse'                => false,
				'arrows'                 => true,
				'image'                  => false,
				'subtitle'               => true,
				'archive'                => false,

				// Same term taxonomy
				'taxonomy'               => null,

				// Arrows
				'arrow_type'             => null,

				// Image
				'image_align'            => null,

				// Title
				'title_content_type'     => null,
				'title_custom_prev'      => null,
				'title_custom_next'      => null,

				// Subtitle
				'subtitle_content_type'  => null,
				'subtitle_taxonomy'      => null,
				'subtitle_custom_prev'   => null,
				'subtitle_custom_next'   => null,
				'subtitle_position'      => null,
				'subtitle_show_on_hover' => null,

				// Back to archive
				'archive_url'            => null,
				'archive_taxonomy_link'  => null,
				'archive_taxonomy'       => null,
			]
		);

		// Choice
		$choices = wp_parse_args( $args['choices'] );

		// Placeholders
		$placeholders = wp_parse_args(
			$args['placeholders'],
			[
				// Title custom labels
				'title_custom_prev'    => esc_attr__( 'Previous', 'kalium' ),
				'title_custom_next'    => esc_attr__( 'Next', 'kalium' ),

				// Subtitle custom labels
				'subtitle_custom_prev' => esc_attr__( 'Older Post', 'kalium' ),
				'subtitle_custom_next' => esc_attr__( 'Newer Post', 'kalium' ),
			]
		);

		$is_preview = is_customize_preview();

		$button_radius = kalium_get_array_first(
			kalium_get_responsive_value(
				kalium_get_theme_option(
					'button_radius'
				)
			)
		);

		// Archive URL placeholder
		if ( $is_preview && ! isset( $placeholders['placeholders'] ) ) {
			$placeholders['archive_url'] = get_post_type_archive_link( $args['post_type'] );
		}

		// Layouts
		$layouts = [
			'type-1' => [
				'label' => 'Type 1',
				'image' => kalium()->assets_url( 'admin/images/theme-options/post-navigation-1.svg' ),
			],
			'type-2' => [
				'label' => 'Type 2',
				'image' => kalium()->assets_url( 'admin/images/theme-options/post-navigation-2.svg' ),
			],
			'type-3' => [
				'label' => 'Type 3',
				'image' => kalium()->assets_url( 'admin/images/theme-options/post-navigation-3.svg' ),
			],
			'type-4' => [
				'label' => 'Type 4',
				'image' => kalium()->assets_url( 'admin/images/theme-options/post-navigation-4.svg' ),
			],
		];

		if ( is_array( $args['layout_types'] ) ) {
			$layouts = array_intersect_key( $layouts, array_flip( $args['layout_types'] ) );
		}

		// Taxonomies
		$taxonomies = static function () use ( &$args ) {
			return array_map(
				function ( $taxonomy ) {
					/** @var \WP_Taxonomy $taxomomy */
					return $taxonomy->labels->name;
				},
				get_object_taxonomies( $args['post_type'], 'objects' )
			);
		};

		// Subtitle content types
		$subtitle_content_types = array_merge(
			$choices['subtitle_content_type'] ?? [],
			[
				'taxonomy' => 'Taxonomy',
				'custom'   => 'Custom Labels',
			]
		);

		asort( $subtitle_content_types );

		// Dependencies
		$dependencies = [
			'except_layout_type_2' => [
				'name'    => $settings['layout'],
				'value'   => [
					'type-2',
				],
				'compare' => 'NOT IN',
			],
			'except_layout_type_3' => [
				'name'    => $settings['layout'],
				'value'   => [
					'type-3',
				],
				'compare' => 'NOT IN',
			],
			'with_container_props' => [
				'name'  => $settings['layout'],
				'value' => [
					'type-2',
					'type-4',
				],
			],
		];

		// General tab
		kalium_customize_general_tab();

		// Layout
		$this->add_control(
			$settings['layout'],
			[
				'type'          => 'kalium-radio-image',
				'heading_title' => 'Layout',
				'choices'       => $layouts,
				'default'       => $defaults['layout'],
				'show_labels'   => false,
				'tooltips'      => true,
				'columns'       => 2,
			]
		);

		// Align horizontal
		$this->add_control(
			$settings['align_horizontal'],
			[
				'type'        => 'kalium-radio-button',
				'label'       => 'Alignment',
				'input_style' => 'width:120px',
				'choices'     => [
					'start'  => [
						'label' => 'Left',
						'icon'  => 'alignLeft',
						'size'  => 16,
					],
					'center' => [
						'label' => 'Center',
						'icon'  => 'alignCenter',
						'size'  => 16,
					],
					'end'    => [
						'label' => 'Right',
						'icon'  => 'alignRight',
						'size'  => 16,
					],
				],
				'default'     => $defaults['align_horizontal'],
				'setting'     => [
					'transport' => 'postMessage',
				],
				'dependency'  => [
					$settings['layout'] => [
						'type-3' ,
					],
				],
			]
		);

		// Show info
		$this->add_control(
			$settings['show_info'],
			[
				'type'       => 'kalium-toggle',
				'label'      => 'Show Info on Hover',
				'default'    => $defaults['show_info'],
				'dependency' => [
					$settings['layout'] => 'type-4',
				],
			]
		);

		// Loop
		$this->add_control(
			$settings['loop'],
			[
				'type'          => 'kalium-toggle',
				'heading_title' => 'Options',
				'label'         => 'Loop',
				'help'          => 'When reaching the first or last item, navigation continues from the next available post.',
				'separator'     => true,
				'default'       => $defaults['loop'],
			]
		);

		// Reverse (swap order)
		$this->add_control(
			$settings['reverse'],
			[
				'type'    => 'kalium-toggle',
				'label'   => 'Reverse Order',
				'help'    => 'Swaps the positions of previous and next links.',
				'default' => $defaults['reverse'],
				'setting' => [
					'transport' => 'postMessage',
				],
			]
		);

		// Back to archive
		$this->add_control(
			$settings['in_same_term'],
			[
				'type'    => 'kalium-toggle',
				'label'   => 'Same Taxonomy Navigation',
				'help'    => 'Keeps navigation to posts within the same category or taxonomy.',
				'default' => $defaults['in_same_term'],
			]
		);

		// Arrows
		$this->add_control(
			$settings['arrows'],
			[
				'type'       => 'kalium-toggle',
				'label'      => 'Arrows',
				'help'       => 'Toggles the display of navigation arrows for next and previous links.',
				'default'    => $defaults['arrows'],
				'dependency' => [
					$settings['layout'] => [
						'type-1',
					],
				],
			]
		);

		// Image
		$this->add_control(
			$settings['image'],
			[
				'type'       => 'kalium-toggle',
				'label'      => 'Image',
				'help'       => 'Displays the featured image of the linked post.',
				'default'    => $defaults['image'],
				'dependency' => [
					$dependencies['except_layout_type_3'],
				],
			]
		);

		// Subtitle
		$this->add_control(
			$settings['subtitle'],
			[
				'type'       => 'kalium-toggle',
				'label'      => 'Subtitle',
				'help'       => 'Shows additional text below or above the title.',
				'default'    => $defaults['subtitle'],
				'dependency' => [
					$dependencies['except_layout_type_3'],
				],
			]
		);

		// Back to archive
		$this->add_control(
			$settings['archive'],
			[
				'type'       => 'kalium-toggle',
				'label'      => 'Back to Archive',
				'help'       => 'The navigation link that returns to the archive page.',
				'default'    => $defaults['archive'],
				'dependency' => [
					$dependencies['except_layout_type_2'],
				],
			]
		);

		// Taxonomy
		$this->add_control(
			$settings['taxonomy'],
			[
				'type'          => 'kalium-select',
				'heading_title' => 'Same Taxonomy Navigation',
				'label'         => 'Navigation Taxonomy',
				'separator'     => true,
				'input_style'   => 'width:120px',
				'choices'       => $taxonomies,
				'default'       => $defaults['taxonomy'],
				'dependency'    => [
					$settings['in_same_term'] => true,
				],
			]
		);

		// Arrow type
		$this->add_control(
			$settings['arrow_type'],
			[
				'type'          => 'kalium-radio-image',
				'heading_title' => 'Arrows',
				'label'         => 'Arrow Type',
				'inline_label'  => true,
				'input_style'   => 'width:150px;',
				'choices'       => [
					'type-1' => [
						'label' => 'Type 1',
						'image' => kalium()->assets_url( 'admin/images/theme-options/woocommerce/product-navigation-arrow-type-1.svg' ),
					],
					'type-2' => [
						'label' => 'Type 2',
						'image' => kalium()->assets_url( 'admin/images/theme-options/woocommerce/product-navigation-arrow-type-2.svg' ),
					],
				],
				'default'       => $defaults['arrow_type'],
				'setting'       => [
					'transport' => 'postMessage',
				],
				'separator'     => true,
				'show_labels'   => false,
				'tooltips'      => true,
				'columns'       => 2,
				'dependency'    => array_merge(
					[
						[
							'name'     => $settings['layout'],
							'value'    => 'type-1',
							'compare'  => '!=',
							'relation' => 'OR',
						],
						[
							'name'  => $settings['layout'],
							'value' => 'type-1',
						],
						[
							'name'  => $settings['arrows'],
							'value' => true,
						],
					],
					kalium()->customize->get_current_tab_dependency()
				),
			]
		);

		// Image alignment
		$this->add_control(
			$settings['image_align'],
			[
				'type'          => 'kalium-radio-button',
				'heading_title' => 'Image',
				'label'         => 'Alignment',
				'separator'     => true,
				'choices'       => [
					'outside' => 'Outside',
					'inside'  => 'Inside',
				],
				'default'       => $defaults['image_align'],
				'setting'       => [
					'transport' => 'postMessage',
				],
				'dependency'    => [
					$settings['image'] => true,
					$dependencies['except_layout_type_3'],
				],
			]
		);

		// Image size
		$this->register_image_size_settings(
			$settings['image'],
			[
				'heading_title' => null,
				'separator'     => false,
				'dependency'    => [
					$settings['image'] => true,
					$dependencies['except_layout_type_3'],
				],
			]
		);

		// Image aspect ratio
		$this->register_aspect_ratio_settings(
			$settings['image'],
			[
				'dependency' => [
					$settings['image'] => true,
					$dependencies['except_layout_type_3'],
				],
			]
		);

		// Title content type
		$this->add_control(
			$settings['title_content_type'],
			[
				'type'          => 'kalium-select',
				'heading_title' => 'Title',
				'label'         => 'Content to Show',
				'separator'     => true,
				'choices'       => array_merge(
					$choices['title_content_type'] ?? [],
					[
						'post_title' => 'Post Title',
						'custom'     => 'Custom Titles',
					]
				),
				'default'       => $defaults['title_content_type'],
				'dependency'    => [
					$dependencies['except_layout_type_3'],
				],
			]
		);

		// Title custom previous label
		$this->add_control(
			$settings['title_custom_prev'],
			[
				'type'        => 'kalium-text',
				'label'       => 'Previous Title',
				'input_style' => 'width:100px',
				'default'     => $defaults['title_custom_prev'],
				'placeholder' => $placeholders['title_custom_prev'] ?? null,
				'dependency'  => [
					$settings['title_content_type'] => 'custom',
					$dependencies['except_layout_type_3'],
				],
			]
		);

		// Title custom next label
		$this->add_control(
			$settings['title_custom_next'],
			[
				'type'        => 'kalium-text',
				'label'       => 'Next Title',
				'input_style' => 'width:100px',
				'default'     => $defaults['title_custom_next'],
				'placeholder' => $placeholders['title_custom_next'] ?? null,
				'dependency'  => [
					$settings['title_content_type'] => 'custom',
					$dependencies['except_layout_type_3'],
				],
			]
		);

		// Subtitle content
		$this->add_control(
			$settings['subtitle_content_type'],
			[
				'type'          => 'kalium-select',
				'heading_title' => 'Subtitle',
				'label'         => 'Content to Show',
				'separator'     => true,
				'choices'       => $subtitle_content_types,
				'default'       => $defaults['subtitle_content_type'],
				'dependency'    => [
					$settings['subtitle'] => true,
					$dependencies['except_layout_type_3'],
				],
			]
		);

		// Subtitle Taxonomy
		$this->add_control(
			$settings['subtitle_taxonomy'],
			[
				'type'        => 'kalium-select',
				'label'       => 'Taxonomy',
				'input_style' => 'width:120px',
				'choices'     => $taxonomies,
				'default'     => $defaults['subtitle_taxonomy'],
				'dependency'  => [
					$settings['subtitle']              => true,
					$settings['subtitle_content_type'] => 'taxonomy',
					$dependencies['except_layout_type_3'],
				],
			]
		);

		// Subtitle custom previous label
		$this->add_control(
			$settings['subtitle_custom_prev'],
			[
				'type'        => 'kalium-text',
				'label'       => 'Previous Label',
				'input_style' => 'width:100px',
				'default'     => $defaults['subtitle_custom_prev'],
				'placeholder' => $placeholders['subtitle_custom_prev'] ?? null,
				'dependency'  => [
					$settings['subtitle']              => true,
					$settings['subtitle_content_type'] => 'custom',
					$dependencies['except_layout_type_3'],
				],
			]
		);

		// Subtitle custom next label
		$this->add_control(
			$settings['subtitle_custom_next'],
			[
				'type'        => 'kalium-text',
				'label'       => 'Next Label',
				'input_style' => 'width:100px',
				'default'     => $defaults['subtitle_custom_next'],
				'placeholder' => $placeholders['subtitle_custom_next'] ?? null,
				'dependency'  => [
					$settings['subtitle']              => true,
					$settings['subtitle_content_type'] => 'custom',
					$dependencies['except_layout_type_3'],
				],
			]
		);

		// Subtitle position
		$this->add_control(
			$settings['subtitle_position'],
			[
				'type'       => 'kalium-select',
				'label'      => 'Position',
				'choices'    => [
					'above' => 'Above',
					'below' => 'Below',
				],
				'default'    => $defaults['subtitle_position'],
				'setting'    => [
					'transport' => 'postMessage',
				],
				'dependency' => [
					$settings['subtitle'] => true,
					$dependencies['except_layout_type_3'],
				],
			]
		);

		// Subtitle show on hover
		$this->add_control(
			$settings['subtitle_show_on_hover'],
			[
				'type'       => 'kalium-toggle',
				'label'      => 'Show on Hover',
				'default'    => $defaults['subtitle_show_on_hover'],
				'setting'    => [
					'transport' => 'postMessage',
				],
				'dependency' => [
					$settings['subtitle'] => true,
					$dependencies['except_layout_type_3'],
				],
			]
		);

		// Archive URL
		$this->add_control(
			$settings['archive_url'],
			[
				'type'          => 'kalium-text',
				'heading_title' => 'Back to Archive',
				'label'         => 'Archive URL',
				'separator'     => true,
				'inline_label'  => false,
				'default'       => $defaults['archive_url'],
				'placeholder'   => $placeholders['archive_url'] ?? null,
				'dependency'    => [
					$settings['archive'] => true,
					$dependencies['except_layout_type_2'],
				],
			]
		);

		// Use taxonomy archive as archive link
		$this->add_control(
			$settings['archive_taxonomy_link'],
			[
				'type'       => 'kalium-toggle',
				'label'      => 'Taxonomy as Archive Link',
				'help'       => 'Links to the post/item\'s taxonomy archive if a taxonomy is assigned.',
				'default'    => $defaults['archive_taxonomy_link'],
				'dependency' => [
					$settings['archive'] => true,
					$dependencies['except_layout_type_2'],
				],
			]
		);

		// Archive Taxonomy
		$this->add_control(
			$settings['archive_taxonomy'],
			[
				'type'        => 'kalium-select',
				'label'       => 'Taxonomy',
				'input_style' => 'width:120px',
				'choices'     => $taxonomies,
				'default'     => $defaults['archive_taxonomy'],
				'dependency'  => [
					$settings['archive']               => true,
					$settings['archive_taxonomy_link'] => true,
					$dependencies['except_layout_type_2'],
				],
			]
		);

		// Visibility
		$this->register_visibility_setting(
			$namespace,
			[
				'label'        => 'Navigation Visible On',
				'inline_label' => true,
			]
		);

		// Label visibility
		$this->register_visibility_setting(
			$settings['label'],
			[
				'label'         => 'Label Visible On',
				'heading_title' => null,
				'separator'     => false,
				'inline_label'  => true,
				'default'       => [
					'desktop',
					'tablet',
				],
				'dependency'    => [
					$dependencies['except_layout_type_3'],
				],
			]
		);

		// Image visibility
		$this->register_visibility_setting(
			$settings['image'],
			[
				'label'         => 'Image Visible On',
				'heading_title' => null,
				'separator'     => false,
				'inline_label'  => true,
				'default'       => [
					'desktop',
					'tablet',
				],
				'dependency'    => [
					$settings['image'] => true,
					$dependencies['except_layout_type_3'],
				],
			]
		);

		// End of General tab
		kalium_customize_end_tab();

		// Style tab
		kalium_customize_style_tab();

		// Colors / title and icon
		kalium_customize_register_control(
			$settings['style_color'],
			[
				'type'          => 'kalium-color',
				'heading_title' => 'Colors',
				'label'         => 'Title & Arrows',
				'reset'         => true,
				'choices'       => [
					'normal' => 'Normal',
					'hover'  => 'Hover',
				],
				'default'       => [
					'normal' => kalium_color_reference( 'theme_colors_headings' ),
					'hover'  => kalium_color_reference( 'theme_colors_link.hover' ),
				],
				'setting'       => [
					'transport' => 'postMessage',
				],
				'dependency'    => [
					$dependencies['except_layout_type_3'],
				],
			]
		);

		// Back to archive color
		kalium_customize_register_control(
			$settings['style_archive_color'],
			[
				'type'       => 'kalium-color',
				'label'      => 'Back to Archive',
				'reset'      => true,
				'choices'    => [
					'normal' => 'Normal',
					'hover'  => 'Hover',
				],
				'default'    => [
					'normal' => kalium_color_reference( $settings['style_color'] . '.normal' ),
					'hover'  => kalium_color_reference( $settings['style_color'] . '.hover' ),
				],
				'setting'    => [
					'transport' => 'postMessage',
				],
				'dependency' => [
					$dependencies['except_layout_type_3'],
				],
			]
		);

		// Subtitle color
		kalium_customize_register_control(
			$settings['style_subtitle_color'],
			[
				'type'       => 'kalium-color',
				'label'      => 'Subtitle',
				'reset'      => true,
				'choices'    => [
					'normal' => 'Normal',
					'hover'  => 'Hover',
				],
				'default'    => [
					'normal' => kalium_color_reference( 'theme_colors_text.normal' ),
					'hover'  => kalium_color_reference( 'theme_colors_text.normal' ),
				],
				'setting'    => [
					'transport' => 'postMessage',
				],
				'dependency' => [
					$dependencies['except_layout_type_3'],
				],
			]
		);

		// Container background
		kalium_customize_register_control(
			$settings['style_container_bg'],
			[
				'type'          => 'kalium-color',
				'heading_title' => 'Container',
				'label'         => 'Background',
				'reset'         => true,
				'separator'     => true,
				'default'       => kalium_color_reference( 'theme_colors_body' ),
				'setting'       => [
					'transport' => 'postMessage',
				],
				'dependency'    => [
					$dependencies['with_container_props'],
				],
			]
		);

		// Container border
		kalium_customize_register_control(
			$settings['style_container_border'],
			[
				'type'         => 'kalium-border',
				'label'        => 'Border',
				'reset'        => true,
				'responsive'   => true,
				'inline_label' => true,
				'input_style'  => 'max-width:100px',
				'border_style' => true,
				'default'      => [
					'color' => kalium_color_reference( 'theme_colors_border' ),
					'width' => '1px',
					'style' => 'solid',
				],
				'setting'      => [
					'transport' => 'postMessage',
				],
				'dependency'   => [
					$dependencies['with_container_props'],
				],
			]
		);

		// Container border radius
		kalium_customize_register_control(
			$settings['style_container_border_radius'],
			[
				'type'             => 'kalium-range',
				'label'            => 'Border Radius',
				'reset'            => true,
				'responsive'       => true,
				'suffix'           => 'PX',
				'initial_position' => 3,
				'placeholder'      => $button_radius,
				'setting'          => [
					'transport' => 'postMessage',
				],
				'dependency'       => [
					$dependencies['with_container_props'],
				],
			]
		);

		// Container height
		kalium_customize_register_control(
			$settings['style_container_height'],
			[
				'type'       => 'kalium-range',
				'label'      => 'Min. Height',
				'reset'      => true,
				'responsive' => true,
				'default'    => 100,
				'min'        => 30,
				'max'        => 300,
				'suffix'     => 'PX',
				'setting'    => [
					'transport' => 'postMessage',
				],
				'dependency' => [
					[
						'name'  => $settings['layout'],
						'value' => 'type-2',
					],
				],
			]
		);

		// Container shadow
		kalium_customize_register_control(
			$settings['style_container_shadow'],
			[
				'type'               => 'kalium-box-shadow',
				'label'              => 'Box Shadow',
				'reset'              => true,
				'responsive'         => true,
				'placeholder'        => [
					'offset_x'      => 0,
					'offset_y'      => 0,
					'blur_radius'   => 0,
					'spread_radius' => 0,
				],
				'default'            => [
					'color' => kalium_color_reference( 'theme_colors.color-1' ),
				],
				'units'              => kalium_get_default_units(),
				'enable_color_clear' => true,
				'setting'            => [
					'transport' => 'postMessage',
				],
				'dependency'         => [
					$dependencies['with_container_props'],
				],
			]
		);

		// Container alt background
		kalium_customize_register_control(
			$settings['style_alt_container_bg'],
			[
				'type'          => 'kalium-color',
				'heading_title' => 'Colors',
				'label'         => 'Background',
				'reset'         => true,
				'default'       => kalium_color_reference( 'theme_colors.color-4' ),
				'setting'       => [
					'transport' => 'postMessage',
				],
				'dependency'    => [
					$settings['layout'] => 'type-3',
				],
			]
		);

		// Separator color
		kalium_customize_register_control(
			$settings['style_alt_separator_color'],
			[
				'type'       => 'kalium-color',
				'label'      => 'Separator',
				'reset'      => true,
				'default'    => kalium_color_reference( 'theme_colors_body', 0.1 ),
				'setting'    => [
					'transport' => 'postMessage',
				],
				'dependency' => [
					$settings['layout'] => 'type-3',
				],
			]
		);

		// Arrows color
		kalium_customize_register_control(
			$settings['style_alt_color'],
			[
				'type'       => 'kalium-color',
				'label'      => 'Arrows',
				'reset'      => true,
				'choices'    => [
					'normal' => 'Normal',
					'hover'  => 'Hover',
				],
				'default'    => [
					'normal' => kalium_color_reference( 'theme_colors_body' ),
					'hover'  => kalium_color_reference( 'theme_colors_body' ),
				],
				'setting'    => [
					'transport' => 'postMessage',
				],
				'dependency' => [
					$settings['layout'] => 'type-3',
				],
			]
		);

		// Image width
		kalium_customize_register_control(
			$settings['style_image_width'],
			[
				'type'          => 'kalium-range',
				'heading_title' => 'Image',
				'label'         => 'Width',
				'separator'     => true,
				'reset'         => true,
				'responsive'    => true,
				'default'       => 50,
				'min'           => 30,
				'max'           => 250,
				'suffix'        => 'PX',
				'setting'       => [
					'transport' => 'postMessage',
				],
				'dependency'    => [
					$settings['image'] => true,
					$dependencies['except_layout_type_3'],
				],
			]
		);

		// Image border radius
		kalium_customize_register_control(
			$settings['style_image_border_radius'],
			[
				'type'       => 'kalium-multi-numeric',
				'label'      => 'Border Radius',
				'reset'      => true,
				'responsive' => true,
				'units'      => kalium_get_default_units( 'box-size' ),
				'default'    => [
					'link' => true,
				],
				'setting'    => [
					'transport' => 'postMessage',
				],
				'dependency' => [
					$settings['image'] => true,
					$dependencies['except_layout_type_3'],
				],
			]
		);

		// End of style tab
		kalium_customize_end_tab();

		// Preview update
		$selector = $args['selector'];

		kalium_customize_register_preview_update(
			[
				'type'    => 'css',
				'vars'    => [
					$settings['align_horizontal'],
					$settings['reverse'],
					$settings['arrow_type'],
					$settings['image_align'],
					$settings['image'] . '_aspect_ratio',
					$settings['image'] . '_aspect_ratio_custom',
					$settings['subtitle_position'],
					$settings['subtitle_show_on_hover'],
					$namespace . '_visibility',
					$settings['label'] . '_visibility',
					$settings['image'] . '_visibility',
					$settings['style_color'],
					$settings['style_archive_color'],
					$settings['style_subtitle_color'],
					$settings['style_container_bg'],
					$settings['style_container_border'],
					$settings['style_container_border_radius'],
					$settings['style_container_height'],
					$settings['style_container_shadow'],
					$settings['style_alt_container_bg'],
					$settings['style_alt_color'],
					$settings['style_alt_separator_color'],
					$settings['style_image_width'],
					$settings['style_image_border_radius'],
				],
				'content' => <<<EOD
<# 
var className = 'post-navigation';
var linkClassName = className + '__link';
var container = data.api.getElement( '{$selector}' );
var selector = '#' + container.id;
var layoutType = data.api.getSetting( '{$settings['layout']}' );

if ( container ) {
	var prev = container.querySelector( '.' + linkClassName + '--prev' ),
		next = container.querySelector( '.' + linkClassName + '--next' ),
		image = container.querySelector( '.' + linkClassName + '-image' ),
		label = container.querySelector( '.' + linkClassName + '-label' );
	
	// Alignment
	data.api.toggleClassPlaceholder( container, className + '--align-*' );
	container.classList.add( className + '--align-' + data.{$settings['align_horizontal']} );
	
	// Reverse
	container.classList[ data.{$settings['reverse']} ? 'add' : 'remove' ]( className + '--reverse' );
	
	// Arrow type
	var iconType = data.{$settings['arrow_type']};
	
	data.api.toggleClassPlaceholder( prev, linkClassName + '--icon-type-*' );
	data.api.toggleClassPlaceholder( next, linkClassName + '--icon-type-*' );
	
	prev.classList.add( linkClassName + '--icon-' + iconType );
	next.classList.add( linkClassName + '--icon-' + iconType );
	
	// Image alignment
	var imageAlignment = data.{$settings['image_align']};
	
	prev.classList[ 'inside' === imageAlignment ? 'add' : 'remove' ]( linkClassName + '--image-end' );
	next.classList[ 'inside' === imageAlignment ? 'add' : 'remove' ]( linkClassName + '--image-end' );
	
	// Image aspect ratio
	var imageAspectRatio = data.{$settings['image']}_aspect_ratio;
	var imageAspectRatioCustom = data.{$settings['image']}_aspect_ratio_custom;
	
	// Subtitle
	var subtitlePosition = data.{$settings['subtitle_position']};
	var isShowOnHover = data.{$settings['subtitle_show_on_hover']};
	var dataConfig = container.dataset.config ? container.dataset.config.split( ' ' ) : [];
	
	if ( isShowOnHover && ! dataConfig.includes( 'animate-subtitle' ) ) {
		dataConfig.push( 'animate-subtitle' );
	} else if ( ! isShowOnHover && dataConfig.includes( 'animate-subtitle' ) ) {
		dataConfig = dataConfig.filter( ( item ) => item !== "animate-subtitle" );
	}
	
	container.dataset.config = dataConfig.join( ' ' );
	
	data.api.toggleClassPlaceholder( prev, linkClassName + '--subtitle-*' );
	data.api.toggleClassPlaceholder( next, linkClassName + '--subtitle-*' );
	
	prev.classList.add( linkClassName + '--subtitle-' + subtitlePosition );
	next.classList.add( linkClassName + '--subtitle-' + subtitlePosition );
	
	// Visibility
	data.api.applyDeviceVisibilityClasses( container, data.{$namespace}_visibility );
	data.api.applyDeviceVisibilityClasses( label, data.{$settings['label']}_visibility );
	data.api.applyDeviceVisibilityClasses( image, data.{$settings['image']}_visibility );
}
#>
{{{ data.api.printInlineStyle( {
	selector: selector,
	varName: 'custom-aspect-ratio',
	value: imageAspectRatio,
	parser: function( value ) {
		if ( 'custom' === value ) {
			return data.api.currentResponsiveValue( imageAspectRatioCustom );
		}
		
		return 'auto' === value ? null : value;
	},
} ) }}}

{{{ data.api.printInlineStyle( {
	selector: selector,
	varName: 'pn-image-width',
	value: data.{$settings['style_image_width']},
	addUnit: 'px',
} ) }}}

{{{ data.api.printInlineStyle( {
	selector: selector,
	varName: 'pn-image-border-radius',
	value: data.{$settings['style_image_border_radius']},
	parser: 'length',
} ) }}}

{{{ data.api.printStyleVars( selector, { normal: 'link-color', hover: 'link-hover-color' }, data.{$settings['style_color']} ) }}}
{{{ data.api.printStyleVars( selector + ' .back-to-archive', { normal: 'link-color', hover: 'link-hover-color' }, data.{$settings['style_archive_color']} ) }}}
{{{ data.api.printStyleVars( selector, { normal: 'pn-subtitle-color', hover: 'pn-subtitle-hover-color' }, data.{$settings['style_subtitle_color']} ) }}}
{{{ data.api.printStyleVars( selector, 'pn-bg', data.{$settings['style_container_bg']} ) }}}

{{{ data.api.printInlineStyle( {
	selector: selector,
	varName: 'pn-height',
	value: data.{$settings['style_container_height']},
	addUnit: 'px',
} ) }}}

{{{ data.api.printInlineStyle( {
	selector: selector,
	varName: 'pn-border',
	parser: function ( value ) {
		value.color = data.api.parseColor( value.color );
		
		return value;
	},
	value: data.{$settings['style_container_border']},
} ) }}}

{{{ data.api.printInlineStyle( {
	selector: selector,
	varName: 'pn-border-radius',
	value: data.{$settings['style_container_border_radius']},
	addUnit: 'px',
} ) }}}

<# if ( 'type-3' === layoutType ) { #>
	{{{ data.api.printStyleVars( selector, 'pn-bg', data.{$settings['style_alt_container_bg']} ) }}}
	{{{ data.api.printStyleVars( selector, 'pn-border-color', data.{$settings['style_alt_separator_color']} ) }}}
	{{{ data.api.printStyleVars( selector + ', ' + selector + ' .back-to-archive', { normal: 'link-color', hover: 'link-hover-color' }, data.{$settings['style_alt_color']} ) }}}
<# } #>

<# if ( [ 'type-2', 'type-4' ].includes( layoutType ) ) { #>
	<# var boxShadow = data.{$settings['style_container_shadow']}; #>
	<# label && prev.classList.add( 'hover' ); #>
	
	{{{ data.api.printInlineStyle( {
		selector: selector,
		varName: 'pn-box-shadow',
		parser: 'box-shadow',
		value: boxShadow,
	} ) }}}
	
	<# if ( 'type-2' === layoutType ) { #>
		{{{ data.api.printInlineStyle( {
			selector: selector,
			varName: 'pn-next-box-shadow',
			parser: function( value ) {
				value.offset_x *= -1;
				
				return data.api.toBoxShadow( value );
			},
			value: boxShadow,
		} ) }}}
	<# } #>
<# } #>
EOD,
			]
		);
	}

	/**
	 * Get post navigation settings.
	 *
	 * @param string $namespace
	 * @param array  $args
	 *
	 * @return array
	 * @since 4.1.2
	 */
	public function get_post_navigation_settings( $namespace, $args = [] ) {
		$settings = $this->get_prefixed_settings( $namespace, $this->post_navigation_props );

		$args = wp_parse_args(
			$args,
			[
				'post_type' => null,
			]
		);

		$settings_data = array_map( 'kalium_get_theme_option', $settings );
		$settings_data = array_merge(
			$settings_data,
			[
				'image_size'         => $this->get_image_size_value( $settings['image'] ),
				'image_aspect_ratio' => $this->get_aspect_ratio_value( $settings['image'] ),
			]
		);

		// Visibility
		$settings_data = array_merge(
			$settings_data,
			[
				'visibility'       => kalium_get_theme_option( $settings['visibility'] ),
				'label_visibility' => kalium_get_theme_option( $settings['label_visibility'] ),
				'image_visibility' => kalium_get_theme_option( $settings['image_visibility'] ),
			]
		);

		// Style
		$settings_data['style'] = array_filter(
			array_map(
				function ( $option ) {
					return kalium_get_theme_option( $option, true );
				},
				[
					'color'                   => $settings['style_color'],
					'archive_color'           => $settings['style_archive_color'],
					'subtitle_color'          => $settings['style_subtitle_color'],
					'container_bg'            => $settings['style_container_bg'],
					'container_border'        => $settings['style_container_border'],
					'container_border_radius' => $settings['style_container_border_radius'],
					'container_height'        => $settings['style_container_height'],
					'container_shadow'        => $settings['style_container_shadow'],
					'alt_container_bg'        => $settings['style_alt_container_bg'],
					'alt_separator_color'     => $settings['style_alt_separator_color'],
					'alt_color'               => $settings['style_alt_color'],
					'image_width'             => $settings['style_image_width'],
					'image_border_radius'     => $settings['style_image_border_radius'],
				]
			)
		);

		return $this->parse_post_navigation_settings( $settings_data, $args );
	}

	/**
	 * Parse post navigation settings.
	 *
	 * @param array $settings
	 * @param array $args
	 *
	 * @return array
	 * @since 4.1.2
	 */
	public function parse_post_navigation_settings( $settings, $args = [] ) {
		$parsed_args = [
			'layout'  => $settings['layout'],
			'loop'    => $settings['loop'],
			'reverse' => $settings['reverse'],
		];

		// Horizontal alignment
		if ( 'type-3' === $parsed_args['layout'] ) {
			$parsed_args['align_horizontal'] = $settings['align_horizontal'];
		}

		// Show info
		if ( 'type-4' === $parsed_args['layout'] ) {
			$parsed_args['show_info'] = $settings['show_info'];
		}

		// In same term
		if ( $settings['in_same_term'] ) {
			$parsed_args['taxonomy'] = $settings['taxonomy'] ?: kalium_get_array_first(
				get_object_taxonomies(
					$args['post_type']
				)
			);
		}

		// Title
		switch ( $settings['title_content_type'] ) {
			case 'post_title':
				$parsed_args['title'] = '{{ post_title }}';
				break;

			default:
				$parsed_args['title'] = [
					$settings['title_custom_prev'],
					$settings['title_custom_next'],
				];
		}

		// Subtitle
		if ( $settings['subtitle'] ) {
			switch ( $settings['subtitle_content_type'] ) {
				case 'product_price':
					$parsed_args['subtitle'] = '{{ ' . $settings['subtitle_content_type'] . ' }}';
					break;

				case 'taxonomy':
					$parsed_args['subtitle'] = '{{ taxonomy:' . $settings['subtitle_taxonomy'] . ' }}';
					break;

				default:
					$parsed_args['subtitle'] = [
						$settings['subtitle_custom_prev'] ?: __( 'Older Post', 'kalium' ),
						$settings['subtitle_custom_next'] ?: __( 'Newer Post', 'kalium' ),
					];
			}
		}

		$parsed_args['subtitle_position']      = $settings['subtitle_position'];
		$parsed_args['subtitle_show_on_hover'] = $settings['subtitle_show_on_hover'];

		// Arrows
		$parsed_args['icon_type'] = $settings['arrow_type'];

		if ( 'type-1' === $parsed_args['layout'] ) {
			$parsed_args['icon'] = $settings['arrows'];
		}

		// Image
		$parsed_args['image'] = 'type-3' !== $parsed_args['layout'] && $settings['image'];

		if ( $parsed_args['image'] ) {
			// Align
			$parsed_args['image_align'] = kalium_enum_value(
				$settings['image_align'],
				[
					'outside' => 'start',
					'inside'  => 'end',
				]
			);

			// Size
			$parsed_args['image_size'] = $settings['image_size'];

			// Aspect ratio
			$parsed_args['image_aspect_ratio'] = $settings['image_aspect_ratio'];
		}

		// Back to archive
		if ( $settings['archive'] ) {
			$archive_url = get_post_type_archive_link( $args['post_type'] );

			if ( ! empty( $settings['archive_url'] ) ) {
				$archive_url = $settings['archive_url'];
			}

			if ( $settings['archive_taxonomy_link'] && $settings['archive_taxonomy'] && is_singular() ) {
				$first_term = kalium_get_array_first(
					wp_get_post_terms(
						get_queried_object_id(),
						$settings['archive_taxonomy']
					)
				);

				if ( $first_term instanceof WP_Term ) {
					$archive_url = get_term_link( $first_term );
				}
			}

			$parsed_args['archive_url'] = $archive_url;
		}

		// Disable animation on type-4 layout
		if ( 'type-4' === $parsed_args['layout'] ) {
			$parsed_args['animate'] = false;
		}

		// Visibility
		$parsed_args['visibility']       = $settings['visibility'];
		$parsed_args['label_visibility'] = $settings['label_visibility'];
		$parsed_args['image_visibility'] = $settings['image_visibility'];

		// Style settings
		$parsed_args['style'] = $settings['style'];

		return $parsed_args;
	}
}
