<?php
/**
 * Kalium WordPress Theme
 *
 * Portfolio core class.
 *
 * @author Laborator
 * @link   https://kaliumtheme.com
 */
namespace Kalium\Features;

use Kalium\Core\Settings;
use WP_Post;
use WP_Query;
use WP_REST_Request;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Direct access not allowed.
}

class Portfolio {

	/**
	 * Single use values in portfolio loop.
	 *
	 * @var array
	 */
	public $_loop_once_values = [];

	/**
	 * Constructor.
	 */
	public function __construct() {

		// Portfolio type based options
		foreach ( $this->get_post_types() as $post_type ) {

			// Columns
			add_filter( "manage_edit-{$post_type}_columns", [ $this, 'post_type_columns' ], 1000 );
			add_action( "manage_{$post_type}_posts_custom_column", [ $this, 'post_type_column_content' ], 1000, 2 );
		}

		// Filter by item type
		add_action( 'pre_get_posts', [ $this, 'portfolio_filter_by_item_type' ] );

		// Item type dropdown filter
		add_action( 'restrict_manage_posts', [ $this, 'portfolio_filter_by_item_type_dropdown' ] );

		// Reset likes
		add_filter( 'post_row_actions', [ $this, 'row_action_reset_likes' ], 10, 2 );

		// Reset likes for portfolio item
		add_action( 'wp_ajax_kalium_portfolio_reset_likes', [ $this, 'reset_likes_for_item' ] );

		// Edit portfolio item JS
		add_action( 'admin_enqueue_scripts', [ $this, 'edit_portfolio_item_js' ] );

		// Rest API
		add_action( 'rest_api_init', [ $this, 'rest_api_init' ] );

		// Add custom copy-item-data entries
		add_filter( 'kalium_portfolio_copy_item_data_entries', [ $this, 'add_custom_cid_entries' ], 10, 2 );

		// Replaces portfolio rewrite tags
		add_filter( 'post_type_link', [ $this, 'replace_taxonomy_rewrite_tags' ], 10, 2 );

		// Default portfolio archive page
		add_filter( 'post_type_archive_link', [ $this, 'replace_default_archive_link' ], 10, 2 );
		add_filter( 'term_link', [ $this, 'replace_default_taxonomy_link' ], 10, 3 );

		// Settings
		add_action( 'kalium_settings_init', [ $this, 'register_settings' ] );
		add_action( 'kalium_dashboard_settings_sections', [ $this, 'register_settings_section' ] );
		add_action( 'kalium_settings_section_portfolio', [ $this, 'settings' ] );

		// Block editor data
		add_filter( 'kalium_block_editor_data', [ $this, 'add_block_editor_data' ] );
		add_filter( 'block_type_metadata', [ $this, 'setup_portfolio_items_block_metadata' ] );
	}

	/**
	 * Portfolio post types.
	 *
	 * @return array
	 */
	public function get_post_types() {
		return apply_filters(
			'kalium_portfolio_post_types',
			[
				'portfolio',
			]
		);
	}

	/**
	 * Portfolio taxonomies.
	 *
	 * @return array
	 * @since 4.0.9
	 */
	public function get_taxonomies() {
		return apply_filters(
			'kalium_portfolio_taxonomies',
			[
				'portfolio_category',
				'portfolio_tag',
			]
		);
	}

	/**
	 * Checks if portfolio is enabled.
	 *
	 * @return bool
	 */
	public function is_enabled() {
		return kalium()->is->portfolio_active();
	}

	/**
	 * Is edit page.
	 *
	 * @return bool
	 */
	public function is_edit_page() {
		global $pagenow, $wp_query;

		return 'edit.php' === $pagenow && $wp_query->is_main_query() && in_array( $wp_query->get( 'post_type' ), $this->get_post_types() );
	}

	/**
	 * Checks if provided WP_Query instance is pointed to Portfolio items.
	 *
	 * @param WP_Query $query
	 *
	 * @return bool
	 */
	public function is_portfolio_query( $query ) {
		static $portfolio_post_types, $portfolio_taxonomies;

		$current_post_type = (array) $query->get( 'post_type' );

		if ( is_null( $portfolio_post_types ) ) {
			$portfolio_post_types = $this->get_post_types();
			$portfolio_taxonomies = get_object_taxonomies( $portfolio_post_types );
		}

		$result = false;

		// Portfolio archive
		if ( $query->is_main_query() && ( $query->is_post_type_archive( $portfolio_post_types ) || $query->is_tax( $portfolio_taxonomies ) ) ) {
			$result = true;
		}

		// Portfolio query
		if ( ! $result && ( ! empty( $current_post_type ) && 0 === count( array_diff( $current_post_type, $portfolio_post_types ) ) ) ) {
			$result = true;
		}

		/**
		 * Determine portfolio query hook.
		 *
		 * @param bool $result
		 * @param WP_Query $query
		 */
		return apply_filters( 'kalium_is_portfolio_query', $result, $query );
	}

	/**
	 * Post type columns.
	 *
	 * @param array $columns
	 *
	 * @return array
	 */
	public function post_type_columns( $columns ) {

		// Remove tags
		unset( $columns['taxonomy-portfolio_tag'] );

		// Likes column
		if ( kalium_get_theme_option( 'portfolio_likes' ) ) {
			$columns = kalium()->helpers->insert_at(
				$columns,
				'before',
				'date',
				[
					'likes' => 'Likes',
				]
			);
		}

		// Item Type
		$columns = array_merge(
			$columns,
			[
				'item_type' => 'Item Type',
			]
		);

		return $columns;
	}

	/**
	 * Item types.
	 *
	 * @return array
	 */
	public function get_item_types() {
		return [
			'type-1' => 'Side Portfolio',
			'type-2' => 'Columned',
			'type-3' => 'Carousel',
			'type-4' => 'Zig Zag',
			'type-5' => 'Fullscreen',
			'type-6' => 'Lightbox',
			'type-7' => 'Design Your Own',
		];
	}

	/**
	 * Default portfolio item type.
	 *
	 * @return string|false
	 */
	public function get_default_item_type() {
		return get_option( 'portfolio_default_item_type' );
	}

	/**
	 * Column content.
	 *
	 * @param  string $column
	 */
	public function post_type_column_content( $column, $post_id ) {

		switch ( $column ) {

			// Likes
			case 'likes':
				printf(
					'<span class="likes-num">%s</span>',
					number_format_i18n(
						kalium_get_array_key(
							kalium_get_post_likes(),
							'count'
						)
					)
				);
				break;

			// Item Type
			case 'item_type':
				$item_type  = kalium_get_field( 'item_type', $post_id );
				$item_types = $this->get_item_types();

				if ( isset( $item_types[ $item_type ] ) ) {
					printf(
						'<img src="%1$s" alt="%3$s" /> <a href="%2$s" class="portfolio-item-type-column">%3$s</a>',
						esc_attr( kalium()->assets_url( 'admin/images/portfolio/portfolio-item-' . $item_type . '.svg' ) ),
						esc_url(
							add_query_arg(
								[
									'portfolio_item_type' => $item_type,
								]
							)
						),
						esc_html( $item_types[ $item_type ] )
					);
				}
				break;
		}
	}

	/**
	 * Filter by item type.
	 *
	 * @param WP_Query $query
	 */
	public function portfolio_filter_by_item_type( $query ) {
		if ( $this->is_edit_page() && ( $item_type = kalium()->request->query( 'portfolio_item_type' ) ) ) {
			$query->set( 'meta_key', 'item_type' );
			$query->set( 'meta_value', $item_type );
		}
	}

	/**
	 * Filter by item type dropdown.
	 */
	public function portfolio_filter_by_item_type_dropdown() {
		if ( ! in_array( get_current_screen()->post_type, $this->get_post_types() ) ) {
			return;
		}

		$current_item_type = kalium()->request->query( 'portfolio_item_type' );

		?>
		<select name="portfolio_item_type" class="postform">
			<option value="">All Item Types</option>
			<?php
			foreach ( $this->get_item_types() as $item_type => $name ) {
				?>
				<option value="<?php echo esc_attr( $item_type ); ?>"<?php echo selected( $current_item_type, $item_type ); ?>><?php echo esc_html( $name ); ?></option>
				<?php
			}
			?>
		</select>
		<?php
	}

	/**
	 * Reset likes.
	 */
	public function reset_likes_for_item() {
		if ( ! current_user_can( 'edit_posts' ) ) {
			return;
		}

		if ( $post_id = kalium()->request->request( 'post_id' ) ) {
			$post_likes = get_post_meta( $post_id, 'post_likes', true );

			if ( is_array( $post_likes ) ) {
				update_post_meta( $post_id, 'post_likes', [] );

				echo 'success';
				exit;
			}
		}
	}

	/**
	 * Reset likes.
	 *
	 * @param array   $actions
	 * @param WP_Post $post
	 *
	 * @return array
	 */
	public function row_action_reset_likes( $actions, $post ) {
		if ( in_array( $post->post_type, $this->get_post_types() ) && kalium_get_theme_option( 'portfolio_likes' ) ) {
			$actions = kalium()->helpers->insert_at(
				$actions,
				'after',
				'view',
				[
					'resetlikes' => sprintf( '<a href="#" data-id="%s" class="portfolio-likes-reset">Reset Likes</a>', $post->ID ),
				]
			);
		}

		return $actions;
	}

	/**
	 * Register setting.
	 */
	public function register_settings() {
		kalium()->settings->register(
			'portfolio-post-type',
			[
				'setting'           => 'kalium_portfolio_post_type',
				'input_type'        => 'switch',
				'sanitize_callback' => 'kalium_validate_boolean',
			]
		);

		kalium()->settings->register(
			'portfolio-default-item-type',
			[
				'setting'    => 'portfolio_default_item_type',
				'input_type' => 'select',
				'choices'    => [
					'default' => 'Default',
					'type-1'  => 'Side Portfolio',
					'type-2'  => 'Columned',
					'type-3'  => 'Carousel',
					'type-4'  => 'Zig Zag',
					'type-5'  => 'Fullscreen',
					'type-6'  => 'Lightbox',
					'type-7'  => 'Design Your Owm',
				],
				'default'    => 'default',
			]
		);

		kalium()->settings->register(
			'portfolio-default-archive-page',
			[
				'setting'          => 'portfolio_default_archive_page',
				'input_type'       => 'select',
				'input_attributes' => [
					'style' => 'width:150px',
				],
				'choices'          => function () {
					return [ 'default' => 'Default' ] + wp_list_pluck( get_pages(), 'post_title', 'ID' );
				},
			]
		);
	}

	/**
	 * Register settings section.
	 *
	 * @param array $settings_sections
	 *
	 * @return array
	 * @since 4.4
	 */
	public function register_settings_section( $settings_sections ) {
		$settings_sections[] = [
			'id'       => 'portfolio',
			'title'    => 'Portfolio',
			'icon'     => 'kalium-admin-icon-po-portfolio-item',
			'priority' => 20,
		];

		return $settings_sections;
	}

	/**
	 * Portfolio settings.
	 */
	public function settings() {
		?>
		<div class="kalium-dashboard__setting">
			<div class="kalium-dashboard__setting__label">
				<h3>Portfolio Extension</h3>
				<p>This will enable the portfolio functionality and its associated single item templates.</p>
			</div>
			<div class="kalium-dashboard__setting__control">
				<?php Settings::setting( 'portfolio-post-type' ); ?>
			</div>
		</div>
		<div class="kalium-dashboard__setting kalium-dashboard__setting--separator">
			<div class="kalium-dashboard__setting__label">
				<h3>Preselected Item Type</h3>
				<p>Set preselected project type when creating a new project.</p>
			</div>
			<div class="kalium-dashboard__setting__control">
				<?php Settings::setting( 'portfolio-default-item-type' ); ?>
			</div>
		</div>
		<div class="kalium-dashboard__setting kalium-dashboard__setting--separator">
			<div class="kalium-dashboard__setting__label">
				<h3>Default Archive Page</h3>
				<p>Choose the portfolio archive page that will be used as the default display for your projects.</p>
			</div>
			<div class="kalium-dashboard__setting__control">
				<?php Settings::setting( 'portfolio-default-archive-page' ); ?>
			</div>
		</div>
		<?php
	}

	/**
	 * Edit portfolio item script.
	 *
	 * @since 4.0.9
	 */
	public function edit_portfolio_item_js() {
		$screen = function_exists( 'get_current_screen' ) ? get_current_screen() : null;

		if ( ! $screen || empty( $screen->post_type ) || $screen->post_type !== $screen->id || ! in_array( $screen->post_type, $this->get_post_types() ) ) {
			return;
		}

		kalium_enqueue( 'admin-portfolio' );
	}

	/**
	 * Copy-item-data entries.
	 *
	 * @param int $id
	 *
	 * @return array
	 * @since 4.0.9
	 */
	public function get_copy_item_data_entries( $id ) {
		return apply_filters(
			'kalium_portfolio_copy_item_data_entries',
			[
				// Post title
				[
					'type'     => 'post',
					'value'    => 'post_title',
					'title'    => 'Post Title',
					'selected' => true,
					'filter'   => function ( $value ) {
						return sprintf( '%s (Copy)', $value );
					},
				],

				// Post content
				[
					'type'     => 'post',
					'value'    => 'post_content',
					'title'    => 'Post Content',
					'selected' => true,
				],

				// Featured image
				[
					'type'     => 'post_meta',
					'value'    => '_thumbnail_id',
					'title'    => 'Featured Image',
					'selected' => false,
				],
			],
			get_post( $id )
		);
	}

	/**
	 * Copy item data entry.
	 *
	 * @param array $entry
	 * @param int   $source_id
	 * @param int   $dest_id
	 *
	 * @since 4.0.9
	 */
	public function copy_item_data( $entry, $source_id, $dest_id ) {
		$entry = wp_parse_args(
			$entry,
			[
				'type'   => null,
				'value'  => null,
				'filter' => null,
			]
		);

		$source = get_post( $source_id );
		$dest   = get_post( $dest_id );

		// If any
		if ( ! $source || ! $dest || $source_id === $dest_id ) {
			return false;
		}

		// Filter value
		$filter_value = is_callable( $entry['filter'] ) ? $entry['filter'] : function ( $value ) {
			return $value;
		};

		switch ( $entry['type'] ) {

			// Post data
			case 'post':
				$post_update = [];

				switch ( $entry['value'] ) {

					// Post title
					case 'post_title':
						$post_update['post_title'] = $filter_value( $source->post_title );
						$post_update['post_name']  = sanitize_title( $post_update['post_title'] );
						break;

					// Post content
					case 'post_content':
						$post_update['post_content'] = $filter_value( $source->post_content );
						break;
				}

				// Update post
				if ( ! empty( $post_update ) ) {
					wp_update_post(
						array_merge(
							$post_update,
							[
								'ID' => $dest_id,
							]
						)
					);

					// Copy elementor content
					if ( isset( $post_update['post_content'] ) && kalium()->elementor->is_built_with_elementor( $source_id ) ) {
						kalium()->elementor->duplicate_elementor_page( $source_id, $dest_id );
					}

					return true;
				}
				break;

			// Post meta
			case 'post_meta':
				$meta_value = get_post_meta( $source_id, $entry['value'] );

				// Existing post meta only
				if ( ! empty( $meta_value ) ) {
					update_post_meta(
						$dest_id,
						$entry['value'],
						$filter_value(
							reset( $meta_value )
						)
					);

					return true;
				}
				break;

			// Taxonomy
			case 'taxonomy':
				$terms = wp_get_object_terms( $source_id, $entry['value'] );

				if ( ! is_wp_error( $terms ) ) {
					wp_set_object_terms( $dest_id, $filter_value( wp_list_pluck( $terms, 'term_id' ) ), $entry['value'] );
					return true;
				}
				break;

			// ACF Field Group
			case 'acf_field_group':
				$fields = acf_get_fields( $entry['value'] );

				if ( ! empty( $fields ) ) {
					foreach ( $fields as $field ) {
						$field_value = get_field( $field['key'], $source_id );

						update_field( $field['key'], $filter_value( $field_value ), $dest_id );
					}

					return true;
				}
				break;
		}

		return false;
	}

	/**
	 * Add custom copy-item-data entries.
	 *
	 * @param array   $copy_item_data_entries
	 * @param WP_Post $post
	 *
	 * @return array
	 * @since 4.0.9
	 */
	public function add_custom_cid_entries( $copy_item_data_entries, $post ) {
		$item_type  = kalium_get_field( 'item_type', $post->ID );
		$taxonomies = get_post_taxonomies( $post->ID );

		// Taxonomies
		$selected_taxonomies = [
			'portfolio_category',
		];

		foreach ( $taxonomies as $tax ) {
			$taxonomy_object = get_taxonomy( $tax );

			$copy_item_data_entries[] = [
				'type'     => 'taxonomy',
				'value'    => $tax,
				'title'    => $taxonomy_object->label,
				'selected' => in_array( $tax, $selected_taxonomies, true ),
			];
		}

		// ACF fields
		$acf_fields = [
			'type-1' => [
				'group_5ba0c48768d0b', // Side Portfolio
				'group_5ba0c487b1831', // General Details
				'group_5ba0c487ca369', // Project Link
				'group_5ba0c487d4320', // Checklists
				'group_5ba0c487e4256', // Project Gallery
				'group_5fc75ed114e2b', // Featured Video
				'group_5ba0c488cf1ee', // Other Settings
			],
			'type-2' => [
				'group_5ba0c48780e3d', // Columned
				'group_5ba0c487b1831', // General Details
				'group_5ba0c487ca369', // Project Link
				'group_5ba0c487d4320', // Checklists
				'group_5ba0c487e4256', // Project Gallery
				'group_5fc75ed114e2b', // Featured Video
				'group_5ba0c488cf1ee', // Other Settings
			],
			'type-3' => [
				'group_5ba0c48790af7', // Carousel
				'group_5ba0c487b1831', // General Details
				'group_5ba0c487ca369', // Project Link
				'group_5ba0c487d4320', // Checklists
				'group_5ba0c48846cf6', // Project Gallery
				'group_5fc75ed114e2b', // Featured Video
				'group_5ba0c488cf1ee', // Other Settings
			],
			'type-4' => [
				'group_5ba0c48794e83', // Zig Zag
				'group_5ba0c487b1831', // General Details
				'group_5ba0c487ca369', // Project Link
				'group_5ba0c487d4320', // Checklists
				'group_5ba0c48866e5d', // Project Gallery
				'group_5fc75ed114e2b', // Featured Video
				'group_5ba0c488cf1ee', // Other Settings
			],
			'type-5' => [
				'group_5ba0c4879adbe', // Fullscreen
				'group_5ba0c487b1831', // General Details
				'group_5ba0c487ca369', // Project Link
				'group_5ba0c487d4320', // Checklists
				'group_5ba0c488c86e4', // Project Gallery
				'group_5fc75ed114e2b', // Featured Video
				'group_5ba0c488cf1ee', // Other Settings
			],
			'type-6' => [
				'group_5ba0c487a27c2', // Lightbox
				'group_5ba0c487b1831', // General Details
				'group_5fc75ed114e2b', // Featured Video
				'group_5ba0c488cf1ee', // Other Settings
			],
			'type-7' => [
				'group_5ba0c487b1831', // General Details
				'group_5fc75ed114e2b', // Featured Video
				'group_5ba0c488cf1ee', // Other Settings
			],
		];

		if ( ! empty( $acf_fields[ $item_type ] ) ) {
			foreach ( $acf_fields[ $item_type ] as $field_group_id ) {
				if ( $field_group = acf_get_field_group( $field_group_id ) ) {
					$title = preg_replace( '/\(Portfolio Type \d+\)/', 'Options', $field_group['title'] );

					$copy_item_data_entries[] = [
						'type'     => 'acf_field_group',
						'value'    => $field_group_id,
						'title'    => $title,
						'selected' => true,
					];
				}
			}
		}

		return $copy_item_data_entries;
	}

	/**
	 * REST API init.
	 *
	 * @since 4.0.9
	 */
	public function rest_api_init() {

		// Portfolio items
		register_rest_route(
			kalium_rest_namespace(),
			'/portfolio/items',
			[
				'methods'             => 'GET',
				'permission_callback' => kalium_hook_return_value( 'manage_options' ),
				'callback'            => function () {
					$post_type  = kalium()->request->query( 'post_type', $this->get_post_types() );
					$exclude_id = kalium()->request->query( 'exclude_id' );

					$args = [
						'post_type'      => $post_type,
						'posts_per_page' => -1,
					];

					if ( is_numeric( $exclude_id ) ) {
						$args['post__not_in'][] = $exclude_id;
					}

					$query = new WP_Query( $args );
					$items = [];

					$item_type_names = $this->get_item_types();

					if ( $query->have_posts() ) {
						while ( $query->have_posts() ) {
							$query->the_post();

							$item_id   = get_the_ID();
							$item_type = kalium_get_field( 'item_type', $item_id );

							$items[] = [
								'id'         => $item_id,
								'post_title' => get_the_title(),
								'post_meta'  => [
									'item_type'      => $item_type,
									'item_type_name' => $item_type_names[ $item_type ] ?? '',
								],
							];
						}

						wp_reset_postdata();
					}

					return rest_ensure_response( $items );
				},
			]
		);

		// Portfolio - copy item data
		register_rest_route(
			kalium_rest_namespace(),
			'/portfolio/copy-item-data/(?P<source_id>\d+)',
			[
				'methods'             => [ 'GET', 'POST' ],
				'permission_callback' => kalium_hook_return_value( 'manage_options' ),
				'args'                => [
					'source_id' => [
						'required' => true,
					],
				],
				'callback'            => function ( $request ) {
					/** @var WP_REST_Request $request */
					$source_id   = $request->get_param( 'source_id' );
					$cid_entries = $this->get_copy_item_data_entries( $source_id );

					if ( 'POST' === $request->get_method() ) {
						$copied_entries = 0;
						$dest_id        = kalium()->request->query( 'dest_id' );
						$cid_selection  = $request->get_json_params();

						// Map with cid entry id
						$cid_entries = array_combine(
							array_map(
								function ( $entry ) {
									return $entry['type'] . '|' . $entry['value'];
								},
								$cid_entries
							),
							$cid_entries
						);

						if ( is_array( $cid_selection ) && ! empty( $cid_selection ) ) {
							foreach ( $cid_selection as $cid_entry_id => $include ) {
								if ( ! $include || ! isset( $cid_entries[ $cid_entry_id ] ) ) {
									continue;
								}

								$this->copy_item_data( $cid_entries[ $cid_entry_id ], $source_id, $dest_id );

								$copied_entries++;
							}

							// Update item type
							update_field(
								'item_type',
								get_field( 'item_type', $source_id ),
								$dest_id
							);
						}

						return rest_ensure_response( $copied_entries );
					} else {
						return rest_ensure_response( $cid_entries );
					}
				},
			]
		);
	}

	/**
	 * Replaces rewrite tags (e.g. %portfolio_category%) in portfolio permalinks with actual taxonomy term slugs, supporting nested term structures.
	 *
	 * @param string  $post_link
	 * @param WP_Post $post
	 *
	 * @return string
	 * @since 4.2
	 */
	public function replace_taxonomy_rewrite_tags( $post_link, $post ) {
		if ( in_array( $post->post_type, kalium_get_portfolio_post_types() ) ) {
			$taxonomies = get_object_taxonomies( $post->post_type );

			foreach ( $taxonomies as $taxonomy ) {
				$rewrite_tag = '%' . $taxonomy . '%';

				if ( false !== strpos( $post_link, $rewrite_tag ) ) {
					$terms = get_the_terms( $post->ID, $taxonomy );

					if ( empty( $terms ) ) {
						$term_slug = 'uncategorized';
					} else {
						$terms = wp_list_sort( $terms, [ 'term_id' => 'ASC' ] );

						/** @var WP_Term $term_object */
						$term_object = $terms[0];
						$term_slug   = $term_object->slug;

						if ( $term_object->parent ) {
							$term_slug = untrailingslashit(
								get_term_parents_list(
									$term_object->term_id,
									$taxonomy,
									[
										'link'      => false,
										'separator' => '/',
										'format'    => 'slug',
									]
								)
							);
						}
					}

					$post_link = str_replace( $rewrite_tag, $term_slug, $post_link );
				}
			}
		}

		return $post_link;
	}

	/**
	 * Get default archive page ID.
	 *
	 * @return int|false
	 * @since 4.1.2
	 */
	public function get_default_archive_page_id() {
		static $default_archive_page;

		if ( is_null( $default_archive_page ) ) {
			$default_archive_page = get_option( 'portfolio_default_archive_page' );

			if ( ! is_numeric( $default_archive_page ) ) {
				$default_archive_page = false;
			}
		}

		return $default_archive_page;
	}

	/**
	 * Setup default portfolio archive page URL.
	 *
	 * @param string $link
	 * @param string $post_type
	 *
	 * @return string
	 * @since 4.1.2
	 */
	public function replace_default_archive_link( $link, $post_type ) {
		if ( in_array( $post_type, $this->get_post_types(), true ) ) {
			if ( $default_archive_page = $this->get_default_archive_page_id() ) {
				return get_permalink( $default_archive_page );
			}

			$permalinks = kalium_get_portfolio_permalinks();

			if ( isset( $permalinks[ $post_type ] ) && empty( $permalinks[ $post_type ] ) ) {
				return home_url();
			}
		}

		return $link;
	}

	/**
	 * Setup default portfolio archive page.
	 *
	 * @param string $termlink
	 * @param string $term
	 * @param string $taxonomy
	 *
	 * @return string
	 * @since 4.1.2
	 */
	public function replace_default_taxonomy_link( $termlink, $term, $taxonomy ) {
		global $wp_rewrite;

		if ( $default_archive_page = $this->get_default_archive_page_id() ) {
			$taxonomy_obj = get_taxonomy( $taxonomy );

			if ( ! empty( array_intersect( $taxonomy_obj->object_type, $this->get_post_types() ) ) ) {
				$archive_permalink = get_permalink( $default_archive_page );
				$permastruct       = $wp_rewrite->get_extra_permastruct( $taxonomy );
				$term_slug         = $term->slug;

				if ( $permastruct ) {
					$termlink = str_replace( "%$taxonomy%", $term_slug, $permastruct );
					$termlink = ltrim( $termlink, '/' );
					$termlink = trailingslashit( $archive_permalink ) . trailingslashit( $termlink );
				} else {
					$termlink = add_query_arg( 'page_id', $default_archive_page, $termlink );
				}
			}
		}

		return $termlink;
	}

	/**
	 * Add block editor data.
	 *
	 * @param array $data
	 *
	 * @return array
	 * @since 4.3
	 */
	public function add_block_editor_data( $data ) {
		$data['portfolioSettings'] = [
			'postTypes' => array_filter(
				array_combine(
					$this->get_post_types(),
					array_map(
						static function ( $post_type ) {
							$post_type_obj = get_post_type_object( $post_type );

							return $post_type_obj ? [
								'name'       => $post_type_obj->name,
								'label'      => $post_type_obj->label,
								'singular'   => $post_type_obj->labels->singular_name,
								'taxonomies' => array_map(
									static function ( $taxonomy_obj ) {
										return [
											'name'     => $taxonomy_obj->name,
											'label'    => $taxonomy_obj->label,
											'singular' => $taxonomy_obj->labels->singular_name,
										];
									},
									get_object_taxonomies( $post_type, 'objects' )
								),
							] : null;
						},
						$this->get_post_types()
					)
				)
			),
		];

		return $data;
	}

	/**
	 * Setup portfolio items block metadata.
	 *
	 * @param array $metadata
	 *
	 * @return array
	 * @since 4.3
	 */
	public function setup_portfolio_items_block_metadata( $metadata ) {
		if ( 'kalium/portfolio-items' === $metadata['name'] ) {
			$metadata['attributes']['title']['default']       = kalium_get_theme_option( 'portfolio_title' );
			$metadata['attributes']['description']['default'] = kalium_get_theme_option( 'portfolio_description' );

			// Query
			$metadata['attributes']['query']['default']['postsPerPage'] = kalium_get_theme_option( 'portfolio_items_per_page' );

			// Grid columns
			$columns = kalium_get_theme_option( 'portfolio_grid_columns' );

			$metadata['attributes']['columnsTablet']['default'] = kalium_get_responsive_value( $columns, 'tablet' );
			$metadata['attributes']['columnsMobile']['default'] = kalium_get_responsive_value( $columns, 'mobile' );

			// Grid gap
			$gap = kalium_get_theme_option( 'portfolio_grid_columns_gap' );

			$metadata['attributes']['columnsGapTablet']['default'] = kalium_get_responsive_value( $gap, 'tablet' );
			$metadata['attributes']['columnsGapMobile']['default'] = kalium_get_responsive_value( $gap, 'mobile' );

			// Pagination
			$metadata['attributes']['pagination']['default']['buttonText']     = esc_html__( 'Show more', 'kalium' );
			$metadata['attributes']['pagination']['default']['reachedEndText'] = esc_html__( 'No more portfolio items to show', 'kalium' );
		}

		return $metadata;
	}
}
