<?php
/**
 * Block Editor class.
 *
 * @author Laborator
 * @link   https://kaliumtheme.com
 */
namespace Kalium\Integrations;

use TypoLab_Font_Loader;

class Block_Editor {

	/**
	 * Construct.
	 */
	public function __construct() {
		add_action( 'init', [ $this, 'register_blocks' ] );
		add_action( 'enqueue_block_editor_assets', [ $this, 'enqueue_assets' ] );
		add_filter( 'block_categories_all', [ $this, 'register_block_category' ], 100 );
		add_action( 'block_editor_settings_all', [ $this, 'extend_block_editor_settings' ], 1000 );
		add_action( 'enqueue_block_assets', [ $this, 'enqueue_block_styles' ] );
		add_action( 'enqueue_block_editor_assets', [ $this, 'enqueue_block_editor_styles' ] );
	}

	/**
	 * Get theme blocks list.
	 *
	 * @return array
	 * @since 4.3
	 */
	public function get_blocks_list() {
		return [
			[
				'name'    => 'content-section',
				'include' => kalium()->is->template_parts_active(),
				'ssr'     => true,
			],
			[
				'name'    => 'blog-posts',
				'include' => true,
				'ssr'     => true,
			],
			[
				'name'    => 'portfolio-items',
				'include' => kalium()->portfolio->is_enabled(),
				'ssr'     => true,
			],
		];
	}


	/**
	 * Get block editor additional styles.
	 *
	 * @return string
	 * @since 4.3
	 */
	public function get_block_editor_additional_styles() {
		ob_start();

		// Block Editor CSS vars
		$block_editor_vars = [];

		// Set container width
		kalium_set_container_max_width();

		// Set block spacing
		kalium_set_block_spacing();

		// Theme color vars
		kalium_print_theme_color_vars();

		// Button styles
		kalium_print_button_style();

		// Link styles
		kalium_print_link_styles();

		// TypoLab Fonts
		TypoLab_Font_Loader::load_fonts( 'styles' );

		// Widgets spacing
		if ( $sidebar_widgets_spacing = kalium_get_theme_option( 'sidebar_widgets_spacing' ) ) {
			$block_editor_vars['sb-widgets-spacing'] = $sidebar_widgets_spacing . 'px';
		}

		// When sidebar is enabled
		if ( kalium_has_sidebar_on_singular() ) {
			$container_max_width = kalium_get_container_max_width( true );
			$sidebar_width       = kalium_get_sidebar_width( true );
			$sidebar_gap         = kalium_get_theme_option( 'sidebar_gap' ) ?: kalium_css_var_reference( 'sidebar-gap' );

			// Convert container width to percentage value
			if ( false === strpos( $container_max_width, '%' ) && is_numeric( $sidebar_width ) ) {
				$sidebar_width = ( $sidebar_width / 100 );
				$sidebar_width = "$container_max_width * $sidebar_width";
			} else {
				$sidebar_width .= '%';
			}

			// Adjust container width
			kalium_set_css_var(
				'content-width',
				sprintf(
					'calc(%s - (%s + %s))',
					$container_max_width,
					$sidebar_width,
					is_numeric( $sidebar_gap ) ? "{$sidebar_gap}px" : $sidebar_gap,
				)
			);
		}

		// Block editor vars
		kalium_print_inline_style(
			[
				'selector' => kalium_get_root_selector(),
				'vars'     => $block_editor_vars,
			]
		);

		return strip_tags( ob_get_clean() );
	}

	/**
	 * Get theme data for block editor store.
	 *
	 * @return array
	 * @since 4.3
	 */
	public function get_kalium_block_editor_data() {
		$kalium_data = [
			'test' => time(),
		];

		/**
		 * Filter theme data passed to block editor store.
		 *
		 * @param array $kalium_data
		 *
		 * @since 4.3
		 */
		return apply_filters( 'kalium_block_editor_data', $kalium_data );
	}

	/**
	 * Register Gutenberg blocks.
	 */
	public function register_blocks() {
		foreach ( $this->get_blocks_list() as $block_type ) {
			if ( ! isset( $block_type['include'] ) || ! empty( $block_type['include'] ) ) {
				$block_path = kalium()->locate_file( 'includes/blocks/' . $block_type['name'] );

				if ( file_exists( $block_path ) ) {
					register_block_type( $block_path );

					// Include frontend assets for SSR blocks
					if ( ! empty( $block_type['ssr'] ) ) {
						add_filter( 'render_block_kalium/' . $block_type['name'], [ $this, 'block_inject_frontend_assets' ], 10 );
					}
				}
			}
		}
	}

	/**
	 * Enqueue block editor assets.
	 *
	 * @since 4.3
	 */
	public function enqueue_assets() {
		kalium_enqueue( 'theme-blocks' );
	}

	/**
	 * Register Kalium block category.
	 *
	 * @param array $categories
	 *
	 * @return array
	 * @since 4.3
	 */
	public function register_block_category( $categories ) {
		return array_merge(
			[
				[
					'slug'  => 'kalium',
					'title' => kalium_get_name(),
				],
			],
			$categories
		);
	}

	/**
	 * Inject frontend assets for SSR blocks.
	 *
	 * @param string $parsed_block
	 *
	 * @return array
	 * @since 4.3
	 */
	public function block_inject_frontend_assets( $parsed_block ) {
		if ( ! kalium_is_rest_request() ) {
			return $parsed_block;
		}

		$head_assets = kalium()->helpers->capture_output(
			static function () {

				// jQuery dependency
				wp_print_scripts( 'jquery' );

				// Print custom CSS
				wp_custom_css_cb();

				// Theme DataLayer API
				kalium_print_js_api( true );
			}
		);

		$footer_assets = kalium()->helpers->capture_output(
			static function () {

				// Kalium modules (libraries)
				kalium_print_js_libraries();

				// Theme frontend scripts
				kalium_print_scripts( 'theme-frontend-scripts' );
			}
		);

		return $head_assets . $parsed_block . $footer_assets;
	}

	/**
	 * Configure block editor settings.
	 *
	 * @param array $settings
	 *
	 * @return array
	 * @since 4.3
	 */
	public function extend_block_editor_settings( $settings ) {

		// Add theme data to block editor store
		$settings['kaliumData'] = $this->get_kalium_block_editor_data();

		// Add dynamic block styles
		$settings['styles'][] = [
			'css' => kalium()->block_editor->get_block_editor_additional_styles(),
		];

		return $settings;
	}

	/**
	 * Enqueue block assets.
	 *
	 * @since 4.3
	 */
	public function enqueue_block_styles() {
		// Enqueue TypoLab fonts
		TypoLab_Font_Loader::load_fonts( 'enqueues' );
	}

	/**
	 * Block editor styles.
	 *
	 * @since 4.3
	 */
	public function enqueue_block_editor_styles() {
		kalium_enqueue( 'block-editor' );
	}
}
