<?php
/**
 * Kalium WordPress Theme
 *
 * Elementor compatibility class.
 *
 * @author Laborator
 * @link   https://kaliumtheme.com
 */
namespace Kalium\Integrations;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Direct access not allowed.
}

use Elementor\Plugin;
use TypoLab;
use TypoLab_Font;
use TypoLab_Font_Loader;
use WP_REST_Request;
use WP_REST_Response;

class Elementor {

	/**
	 * Construct.
	 */
	public function __construct() {
		if ( ! kalium()->is->elementor_active() ) {
			return;
		}

		// Elementor namespace autoloader
		spl_autoload_register(
			kalium_class_autoloader(
				[
					'namespace' => 'Kalium\Elementor',
					'base_dir'  => kalium()->locate_file( 'includes/elementor' ),
				]
			)
		);

		// Includes
		kalium()->require_file( 'includes/elementor/helpers.php' );
		kalium()->require_file( 'includes/elementor/control-sets/query-posts.php' );

		// Elementor init
		add_action( 'elementor/init', [ $this, 'init' ] );

		// Register element categories
		add_action( 'elementor/elements/categories_registered', [ $this, 'register_categories' ] );
		add_action( 'elementor/elements/categories_registered', [ $this, 'reorder_categories' ] );

		// Register widgets
		add_action( 'elementor/widgets/register', [ $this, 'register_widgets' ] );

		// TypoLab integration
		add_action( 'elementor/fonts/groups', [ $this, 'typolab_font_group' ] );
		add_action( 'elementor/fonts/additional_fonts', [ $this, 'typolab_fonts_list' ] );
		add_action( 'elementor/fonts/print_font_links/typolab', [ $this, 'typolab_enqueue_fonts' ] );

		// Add theme colors in Global colors
		add_action( 'rest_request_after_callbacks', [ $this, 'add_global_colors' ], 10000, 3 );
		add_action( 'rest_request_after_callbacks', [ $this, 'display_global_colors' ], 10000, 3 );

		// Custom base colors
		add_action( 'wp_head', [ $this, 'print_additional_base_colors' ] );

		// Theme lightbox integration
		add_action( 'elementor/widgets/register', [ $this, 'theme_lightbox_integration' ], 100 );

		// Portfolio resize container
		add_action( 'elementor/editor/after_enqueue_scripts', [ $this, 'portfolio_items_relayout_isotope' ] );

		// Missing Header on Elementor Pro issue workaround
		if ( kalium()->is->elementor_pro_active() ) {
			add_action( 'template_redirect', [ $this, 'missing_header_in_elementor_pro_issue' ], 10000 );
		}
	}

	/**
	 * Check if post ID is built with Elementor.
	 *
	 * @param int $post_id
	 *
	 * @return bool
	 */
	public function is_built_with_elementor( $post_id ) {
		if ( ! kalium()->is->elementor_active() ) {
			return false;
		}

		return Plugin::instance()->documents->get( $post_id )->is_built_with_elementor();
	}

	/**
	 * Output elementor content.
	 *
	 * @param int  $post_id
	 * @param bool $with_css
	 *
	 * @return string
	 */
	public function get_builder_content( $post_id, $with_css = false ) {
		if ( ! kalium()->is->elementor_active() ) {
			return false;
		}

		return Plugin::$instance->frontend->get_builder_content_for_display( $post_id, $with_css );
	}

	/**
	 * Whether current request is the elementor preview iframe.
	 *
	 * @param int $post_id
	 *
	 * @return bool
	 */
	public function is_preview( $post_id = 0 ) {
		if ( ! kalium()->is->elementor_active() ) {
			return false;
		}

		if ( $post_id > 0 ) {
			return Plugin::instance()->preview->is_preview_mode( $post_id );
		}

		return Plugin::instance()->preview->is_preview();
	}

	/**
	 * Checks if the Elementor edit mode is active.
	 *
	 * @return bool
	 * @since 4.0
	 */
	public function is_edit_mode() {
		if ( ! kalium()->is->elementor_active() ) {
			return false;
		}

		return Plugin::instance()->editor->is_edit_mode();
	}

	/**
	 * Checks if current request is made on Elementor AJAX mode.
	 *
	 * @return bool
	 * @since 4.0
	 */
	public function is_editor_ajax() {
		return is_admin() && isset( $_POST['action'] ) && 'elementor_ajax' === $_POST['action'];
	}

	/**
	 * Get elementor data for post.
	 *
	 * @param int $post_id
	 *
	 * @return array|null
	 */
	public function get_elements_data( $post_id ) {
		$document = Plugin::instance()->documents->get( $post_id );

		if ( $document ) {
			return $document->get_elements_data();
		}

		return null;
	}

	/**
	 * Elementor init.
	 */
	public function init() {

		// Disable default schemes
		foreach ( [ 'elementor_disable_color_schemes', 'elementor_disable_typography_schemes' ] as $disable_scheme_option ) {
			if ( false === get_option( $disable_scheme_option ) ) {
				update_option( $disable_scheme_option, 'yes' );
			}
		}

		// Fix the excerpt length in the Elementor Preview that is overridden by the post-excerpt block
		if ( $this->is_editor_ajax() ) {
			global $wp_filter;
			unset( $wp_filter['excerpt_length'][ PHP_INT_MAX ] );
		}
	}

	/**
	 * Register element categories.
	 *
	 * @param \Elementor\Elements_Manager $elements_manager
	 */
	public function register_categories( $elements_manager ) {
		$elements_manager->add_category(
			'kalium',
			[
				'title' => kalium_get_name(),
				'icon'  => 'eicon-theme-style',
			]
		);
	}

	/**
	 * Reorder categories.
	 *
	 * @param \Elementor\Elements_Manager $elements_manager
	 */
	public function reorder_categories( $elements_manager ) {
		$positions = [
			'kalium' => 'basic',
		];

		/**
		 * Insert item after specified array key.
		 */
		$insert_before_key = function ( $arr, $move_key, $after_key = null ) {
			if ( ! array_key_exists( $move_key, $arr ) ) {
				return $arr;
			}

			$result   = [];
			$inserted = false;

			$move_value = $arr[ $move_key ];
			unset( $arr[ $move_key ] );

			foreach ( $arr as $current_key => $current_value ) {
				$result[ $current_key ] = $current_value;

				if ( $current_key === $after_key && ! $inserted ) {
					$result[ $move_key ] = $move_value;

					$inserted = true;
				}
			}

			if ( ! $inserted ) {
				$result[ $move_key ] = $move_value;
			}

			return $result;
		};

		/**
		 * Bindable closure to replace the $categories in $elements_manager.
		 */
		$set_categories = function ( $categories ) {
			$this->categories = $categories;
		};

		// Reorder categories
		$new_categories = $elements_manager->get_categories();

		foreach ( $positions as $move_item => $after_item ) {
			$new_categories = $insert_before_key( $new_categories, $move_item, $after_item );
		}

		$set_categories->call( $elements_manager, $new_categories );
	}

	/**
	 * Register widgets.
	 */
	public function register_widgets() {
		$register_widgets = [
			[
				'class'   => '\Kalium\Elementor\Widgets\Portfolio_Items',
				'include' => kalium()->is->portfolio_active(),
			],
			[
				'class'   => '\Kalium\Elementor\Widgets\Blog_Posts',
				'include' => true,
			],
			[
				'class'   => '\Kalium\Elementor\Widgets\Content_Section',
				'include' => kalium()->is->template_parts_active(),
			],
		];

		foreach ( $register_widgets as $widget ) {
			if ( empty( $widget['include'] ) ) {
				continue;
			}

			call_user_func( [ $widget['class'], 'register' ] );
		}
	}

	/**
	 * TypoLab font group.
	 *
	 * @param array $font_groups
	 *
	 * @return array
	 * @since 4.0
	 */
	public function typolab_font_group( $font_groups ) {
		if ( TypoLab::is_enabled() ) {
			$typolab_group = [ 'typolab' => 'Theme Fonts' ];
			$font_groups   = $typolab_group + $font_groups;
		}

		return $font_groups;
	}

	/**
	 * TypoLab fonts list.
	 *
	 * @param array $fonts
	 *
	 * @return array
	 * @since 4.0
	 */
	public function typolab_fonts_list( $fonts ) {
		if ( TypoLab::is_enabled() ) {
			$typolab_fonts = TypoLab::get_fonts( [ 'status' => 'active' ] );

			foreach ( $typolab_fonts as $typolab_font ) {
				/** @var TypoLab_Font $typolab_font $id */
				$family_name = $typolab_font->get_family_name();

				// Add font to the list
				$fonts[ $family_name ] = 'typolab';

				// Enqueue the font
				add_action(
					'admin_footer',
					function () use ( $family_name ) {
						TypoLab_Font_Loader::load_font_family( $family_name );
					}
				);
			}
		}

		return $fonts;
	}

	/**
	 * Enqueue TypoLab font family.
	 *
	 * @param string $font_family
	 *
	 * @since 4.0
	 */
	public function typolab_enqueue_fonts( $font_family ) {
		TypoLab_Font_Loader::load_font_family( $font_family );
	}

	/**
	 * Modify Elementor REST response to add theme colors in Global Colors.
	 *
	 * Kinda hacky, but it is the only way for now!
	 *
	 * @param WP_REST_Response $response
	 * @param array            $handler
	 * @param WP_REST_Request  $request
	 *
	 * @return WP_REST_Response
	 * @since 4.0
	 */
	public function add_global_colors( $response, $handler, $request ) {
		$req_route = $request->get_route();

		// Provide global colors to Elementor
		if ( '/elementor/v1/globals' === $req_route ) {
			$data = $response->get_data();

			$data['colors'] = array_merge( $data['colors'], $this->get_global_colors() );

			$response->set_data( $data );
		}

		return $response;
	}

	/**
	 * Display global colors on frontend.
	 *
	 * @param WP_REST_Response $response
	 * @param array            $handler
	 * @param WP_REST_Request  $request
	 *
	 * @return WP_REST_Response
	 * @since 4.0
	 */
	public function display_global_colors( $response, $handler, $request ) {
		$req_route = $request->get_route();

		if ( 0 !== strpos( $req_route, '/elementor/v1/globals' ) ) {
			return $response;
		}

		$rest_id = substr( $req_route, strrpos( $req_route, '/' ) + 1 );
		$colors  = $this->get_global_colors();

		if ( isset( $colors[ $rest_id ] ) ) {
			return rest_ensure_response( $colors[ $rest_id ] );
		}

		return $response;
	}

	/**
	 * Print additional base colors.
	 *
	 * @since 4.0.9
	 */
	public function print_additional_base_colors() {
		$vars          = [];
		$custom_colors = array_slice( kalium_get_theme_base_colors(), 8 );

		foreach ( $custom_colors as $color_entry ) {
			$id    = $color_entry['id'];
			$prop  = '--e-global-color-' . substr( str_replace( '-', '_', kalium_css_var_name( $id ) ), 2 );
			$value = kalium_css_var_reference( $id );

			$vars[ $prop ] = $value;
		}

		kalium_print_inline_style(
			[
				'selector' => kalium_get_root_selector(),
				'props'    => $vars,
			]
		);
	}

	/**
	 * Theme default lightbox integration with elementor.
	 */
	public function theme_lightbox_integration() {
		$lightbox_widgets = [
			'image-gallery' => 'open_lightbox',
			'image'         => 'open_lightbox',
			'gallery'       => 'open_lightbox',
		];

		$elementor = Plugin::instance();

		foreach ( $lightbox_widgets as $widget_id => $lightbox_attr_name ) {
			$widget_instance = $elementor->widgets_manager->get_widget_types( $widget_id );

			if ( $widget_instance ) {
				$lightbox_control = $widget_instance->get_controls( $lightbox_attr_name );

				if ( ! isset( $lightbox_control['options'] ) ) {
					continue;
				}

				$options_theme_lightbox = [
					'theme-default' => kalium_get_name() . ' Default',
				];

				$options_default = array_slice( $lightbox_control['options'], 0, 1 );
				$options_other   = array_slice( $lightbox_control['options'], 1 );

				$lightbox_control['options'] = array_merge( $options_default, $options_theme_lightbox, $options_other );
				$widget_instance->update_control( $lightbox_attr_name, $lightbox_control );
			}
		}

		add_action(
			'elementor/widget/before_render_content',
			function ( $widget ) use ( $lightbox_widgets ) {
				/** @var \Elementor\Widget_Base $widget */
				if ( isset( $lightbox_widgets[ $widget->get_name() ] ) ) {
					$open_lightbox = $widget->get_settings_for_display( $lightbox_widgets[ $widget->get_name() ] );

					if ( 'theme-default' === $open_lightbox ) {
						$widget->add_render_attribute( 'link', 'data-theme-lightbox', $widget->get_id() );
					}
				}
			}
		);
	}

	/**
	 * Get global colors.
	 *
	 * @return array
	 * @since 4.0
	 */
	private function get_global_colors() {
		$theme_global_colors = kalium_get_theme_global_colors(
			[
				'prefix'    => true,
				'separator' => '_',
			]
		);

		$elementor_global_colors = [];

		foreach ( $theme_global_colors as $color_entry ) {
			$elementor_global_colors[ $color_entry['slug'] ] = [
				'id'    => $color_entry['slug'],
				'title' => $color_entry['name'],
				'value' => $color_entry['color'],
			];
		}

		return $elementor_global_colors;
	}

	/**
	 * Get units on elementor
	 *
	 * @param string|array $type
	 *
	 * @return array
	 */
	public function get_size_units( $type = 'length-percentage', $custom = true ) {
		$size_units = array_keys( kalium_get_default_units( $type ) );

		if ( $custom ) {
			$size_units[] = 'custom';
		}

		return $size_units;
	}

	/**
	 * Update portfolio container when portfolio gap setting is updated.
	 * This is a workaround solution that will resize the iframe container
	 * to allow the Isotope re-layout itself.
	 *
	 * @since 4.0.8
	 */
	public function portfolio_items_relayout_isotope() {
		?>
		<script>
			window.addEventListener( 'elementor/commands/run/before', function ( ev ) {
				if (
					'document/elements/set-settings' === ev.detail.command &&
					'kalium-portfolio-items-legacy' === ev.detail.args.container.model.get( 'widgetType' )
				) {
					var isGap = Object.keys( ev.detail.args.settings ).some(
						function ( key ) {
							return key.startsWith( 'portfolio_gap' );
						}
					);

					if ( isGap ) {
						elementor.$preview.height(
							'100.0' + Math.floor( Math.random() * 999 ) + '%'
						);

						setTimeout( function () {
							elementor.$preview.style.removeProperty( 'height' );
						} );
					}
				}
			} );
		</script>
		<?php
	}

	/**
	 * Duplicate Elementor-specific meta data from one post to another.
	 *
	 * @param int $post_id
	 * @param int $new_post_id
	 *
	 * @return bool
	 * @since 4.0.9
	 */
	public function duplicate_elementor_page( $post_id, $new_post_id ) {
		if ( ! kalium()->is->elementor_active() ) {
			return false;
		}

		$elementor_meta_keys = [
			'_elementor_edit_mode',
			'_elementor_template_type',
			'_elementor_version',
			'_elementor_data',
			'_elementor_page_assets',
			'_elementor_css',
		];

		foreach ( $elementor_meta_keys as $meta_key ) {
			$meta_value = get_post_meta( $post_id, $meta_key, true );

			if ( $meta_value ) {
				update_post_meta( $new_post_id, $meta_key, wp_slash( $meta_value ) );
			}
		}

		// Clear Elementor cache
		Plugin::$instance->files_manager->clear_cache();

		return true;
	}

	/**
	 * An issue related to the theme header removed after Elementor Pro assigns a footer but no header exists.
	 *
	 * @see https://support.laborator.co/ticket/header-missing-after-custom-footer/
	 * @since 4.2.2
	 */
	public function missing_header_in_elementor_pro_issue() {
		$module = \ElementorPro\Modules\ThemeBuilder\Module::instance();

		$conditions_manager = $module->get_conditions_manager();

		$headers = $conditions_manager->get_documents_for_location( 'header' );
		$footers = $conditions_manager->get_documents_for_location( 'footer' );

		if ( empty( $headers ) && ! empty( $footers ) ) {
			$theme_support = $module->get_component( 'theme_support' );
			remove_action( 'get_header', [ $theme_support, 'get_header' ] );
		}
	}
}
