<?php
/**
 * Kalium WordPress Theme
 *
 * GreenShift compatibility class.
 *
 * @author Laborator
 * @link   https://kaliumtheme.com
 */
namespace Kalium\Integrations;

use TypoLab;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Direct access not allowed.
}

class GreenShift {

	/**
	 * Construct.
	 */
	public function __construct() {
		if ( ! kalium()->is->greenshift_active() ) {
			return;
		}

		add_filter( 'greenshift_hide_landscape_breakpoint', '__return_true' );
		add_filter( 'render_block', [ $this, 'theme_lightbox_compatibility' ], 10, 2 );
		add_filter( 'gspb_local_font_array', [ $this, 'merge_typolab_fonts' ] );
		add_action( 'admin_menu', [ $this, 'remove_demo_importer_page' ], 999 );
	}

	/**
	 * Modify GreenShift Image Advanced block to add lightbox class.
	 *
	 * @param string $block_content
	 * @param array  $block
	 *
	 * @return string
	 */
	public function theme_lightbox_compatibility( $block_content, $block ) {
		if ( false === apply_filters( 'kalium_greenshift_use_theme_lightbox', true, $block ) || 'greenshift-blocks/image' !== $block['blockName'] ) {
			return $block_content;
		}

		// Add theme lightbox
		if ( ! empty( $block['attrs']['lightbox'] ) ) {
			$html = new \WP_HTML_Tag_Processor( $block_content );

			if ( $html->next_tag() ) {
				// Remove the gssimglightbox class
				$html->remove_class( 'gssimglightbox' );

				$block_content = preg_replace_callback(
					'/<img[^>]+>/',
					function ( $matches ) {
						if ( preg_match( '/src=(\'|")(?<src>[^\'"]+)(\'|")/', $matches[0], $attribute_matches ) ) {
							return kalium_element(
								[
									'tag_name'   => 'a',
									'attributes' => [
										'href' => $attribute_matches['src'],
										'data-theme-lightbox' => 'true',
									],
									'content'    => $matches[0],
								]
							);
						}
					},
					$html->get_updated_html()
				);
			}
		}

		return $block_content;
	}

	/**
	 * Merge TypoLab fonts with GreenShift local font array.
	 *
	 * @param array|string $localfont
	 *
	 * @return array|string
	 */
	public function merge_typolab_fonts( $localfont ) {
		$fonts = TypoLab::is_enabled() ? TypoLab::get_fonts( [ 'status' => 'active' ] ) : null;

		if ( ! empty( $fonts ) ) {
			$localfont = kalium_decode_json( $localfont, true );

			foreach ( $fonts as $font ) {
				$localfont[ $font->get_family_name() ] = [
					'woff2'     => '',
					'woff'      => '',
					'tiff'      => '',
					'ttf'       => '',
					'preloaded' => '',
				];
			}

			return json_encode( $localfont );
		}

		return $localfont;
	}

	/**
	 * Remove GreenShift admin page to prevent conflicts with theme's demo importer.
	 */
	public function remove_demo_importer_page() {
		remove_submenu_page( 'greenshift_dashboard', 'greenshift_demo' );
	}
}
