<?php
/**
 * Kalium WordPress Theme
 *
 * WPBakery Page Builder compatibility class.
 *
 * @author Laborator
 * @link   https://kaliumtheme.com
 */
namespace Kalium\Integrations;

use Exception;
use Kalium\Template_Parts\Template_Parts;
use WPBMap;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Direct access not allowed.
}

class WPBakery {

	/**
	 * Theme lightbox data.
	 *
	 * @var array
	 */
	public $theme_lightbox_data;

	/**
	 * General attributes (reused).
	 *
	 * @var array
	 */
	public static $laborator_vc_general_params = [

		// Reveal Effect (Extended)
		'reveal_effect' => [
			'type'        => 'dropdown',
			'heading'     => 'Reveal Effect',
			'param_name'  => 'reveal_effect',
			'std'         => 'fadeInLab',
			'value'       => [
				'None'                        => 'none',
				'Fade In'                     => 'fadeIn',
				'Slide and Fade'              => 'fadeInLab',
				'Fade In (one by one)'        => 'fadeIn-one',
				'Slide and Fade (one by one)' => 'fadeInLab-one',
			],
			'description' => 'Set reveal effect for this element.',
		],
	];

	/**
	 * Construct.
	 */
	public function __construct() {
		if ( ! kalium()->is->wpbakery_active() ) {
			return;
		}

		// Init
		add_action( 'vc_before_init', [ $this, 'init' ] );

		// Template redirect
		add_action( 'template_redirect', [ $this, 'vc_deregister_isotope' ], 100 );

		// Row wrapper
		add_filter( 'vc_shortcode_output', [ $this, 'vc_row' ], 100, 3 );

		// Lightbox support for single and gallery images elements
		add_action( 'vc_after_init', [ $this, 'vc_lightbox_option_for_image_elements' ] );
		add_filter( 'vc_shortcode_prepare_atts', [ $this, 'vc_lightbox_elements' ], 100, 2 );
		add_filter( VC_SHORTCODE_CUSTOM_CSS_FILTER_TAG, [ $this, 'vc_lightbox_class' ], 10, 3 );
		add_filter( 'vc_gitem_post_data_get_link_real_link', [ $this, 'vc_gitem_lightbox_link' ], 10, 3 );

		// Lightbox support for grid items
		add_filter( 'vc_gitem_add_link_param', [ $this, 'vc_grid_item_lightbox_option' ] );
		add_filter( 'vc_gitem_zone_image_block_link', [ $this, 'vc_grid_item_lightbox_link' ], 100, 3 );

		// Disable full width row JS
		add_action( 'wp_print_scripts', [ $this, 'wpb_disable_full_width_row_js' ] );

		// Frontend Edit modify admin bar link title
		add_action( 'admin_bar_menu', [ $this, 'admin_bar_button_frontend_edit' ], 1001 );

		// Add retina param for Single Image element
		add_action( 'vc_after_init', [ $this, 'vc_single_image_retina_param' ] );
		add_filter( 'vc_shortcode_output', [ $this, 'vc_single_image_retina_width' ], 100, 3 );

		// TTA Tabs
		add_action( 'vc_after_mapping', [ $this, 'vc_tta_tabs_setup' ] );

		// Widget skin class for VC elements
		add_filter( VC_SHORTCODE_CUSTOM_CSS_FILTER_TAG, [ $this, 'vc_widget_sidebar_classes' ], 10, 2 );
		add_filter( 'vc_shortcode_output', [ $this, 'vc_widget_sidebar_wrap' ], 10, 4 );

		// Theme plugin mode
		add_action( 'admin_init', [ $this, 'vc_theme_plugin_mode_option' ], 0 );

		if ( $this->is_theme_plugin_mode() ) {
			add_action( 'vc_before_init', 'vc_set_as_theme' );
			add_action( 'vc_before_init', [ $this, 'disable_updater' ] );
			add_action( 'vc_after_init', [ $this, 'vc_allow_automatic_update' ] );
			add_filter( 'vc_single_param_edit_holder_output', [ $this, 'vc_hide_ai_icon' ] );
			add_filter( 'transient_wpb_notice_list', [ $this, 'vc_hide_notices' ] );
			add_filter( 'get_user_metadata', [ $this, 'vc_hide_user_notices' ], 10, 3 );
		}
	}

	/**
	 * WPB init.
	 */
	public function init() {
		// Supported post types
		vc_set_default_editor_post_types(
			array_merge(
				[
					'page',
					Template_Parts::POST_TYPE,
				],
				kalium_get_portfolio_post_types()
			)
		);

		// Register widgets list
		$register_widgets_list = [
			'lab_team_members',
			'lab_service_box',
			'lab_heading',
			'lab_scroll_box',
			'lab_clients',
			'lab_vc_social_networks',
			'lab_message',
			'lab_button',
			'lab_contact_form',
			'lab_google_map',
			'lab_text_autotype',
			'lab_blog_posts',
			'lab_divider',
			'lab_pricing_table',
			'lab_dribbble_gallery',
		];

		// Portfolio widgets
		if ( kalium()->portfolio->is_enabled() ) {
			$register_widgets_list = array_merge(
				[
					'lab_portfolio_items',
					'lab_masonry_portfolio',
					'lab_portfolio_share_like',
				],
				$register_widgets_list
			);
		}

		// Breadcrumb widget
		if ( kalium()->is->breadcrumb_navxt_active() ) {
			$register_widgets_list[] = 'kalium_breadcrumb';
		}

		// WooCommerce widgets
		if ( kalium()->is->woocommerce_active() ) {
			$register_widgets_list[] = 'lab_products_carousel';
		}

		// Template parts widgets
		if ( kalium()->is->template_parts_active() ) {
			$register_widgets_list[] = 'kalium_section';
		}

		// Register WPB widgets
		foreach ( $register_widgets_list as $shortcode_template ) {
			kalium()->require_file( 'includes/wpbakery/widgets/' . $shortcode_template . '.php' );
		}

		// Other
		kalium()->require_file( 'includes/wpbakery/custom-font-icons.php' );
	}

	/**
	 * Frontend Edit modify admin bar link title
	 *
	 * @param \WP_Admin_Bar $wp_admin_bar
	 */
	public function admin_bar_button_frontend_edit( $wp_admin_bar ) {
		if ( $node = $wp_admin_bar->get_node( 'vc_inline-admin-bar-link' ) ) {
			$node->title = 'Edit Frontend';
			$wp_admin_bar->add_node( get_object_vars( $node ) );
		}
	}

	/**
	 * Retina param for Single Image element.
	 *
	 * @since 3.1
	 */
	public function vc_single_image_retina_param() {
		vc_add_param(
			'vc_single_image',
			[
				'type'        => 'checkbox',
				'heading'     => 'Retina image',
				'param_name'  => 'retina_image',
				'description' => 'Enabling this option will reduce the size of image for 50%, example if image is 500x500 it will be 250x250.',
				'value'       => [
					'Yes' => 'yes',
				],
			]
		);
	}

	/**
	 * Disable updater, our theme provides updates for this plugin.
	 *
	 * @since 3.4
	 */
	public function disable_updater() {
		vc_manager()->disableUpdater();
	}

	/**
	 * Allow automatic update of the plugin.
	 */
	public function vc_allow_automatic_update() {
		remove_filter( 'upgrader_pre_download', [ vc_updater(), 'preUpgradeFilter' ], 10, 4 );
	}

	/**
	 * Theme-plugin mode option.
	 */
	public function vc_theme_plugin_mode_option() {
		$theme_plugin_mode_field_name = 'kalium_wpb_theme_plugin_mode';

		$field_callback = function () use ( $theme_plugin_mode_field_name ) {
			$checked  = vc_license()->isActivated() ? false : (bool) get_option( $theme_plugin_mode_field_name, true );
			$disabled = vc_license()->isActivated();

			?>
			<label>
				<?php
				// Checkbox
				kalium_render_element(
					[
						'tag_name'   => 'input',
						'attributes' => [
							'type'     => 'checkbox',
							'id'       => $theme_plugin_mode_field_name,
							'name'     => $theme_plugin_mode_field_name,
							'value'    => 1,
							'checked'  => $checked,
							'disabled' => $disabled ?? false,
						],
					]
				);

				// Label
				kalium_xtranslate( 'esc_html_e', 'Enable', 'js_composer' );
				?>
			</label>
			<br/>
			<p class="description indicator-hint">The theme automatically provides WPBakery plugin updates. If you hold a WPBakery license and intend to apply it to this site, please turn off this feature.</p>
			<?php
		};

		register_setting( 'wpb_js_composer_settings_general', $theme_plugin_mode_field_name, [ 'sanitize_callback' => 'kalium_validate_boolean' ] );
		add_settings_field( $theme_plugin_mode_field_name, 'Theme plugin mode', $field_callback, 'vc_settings_general', 'wpb_js_composer_settings_general' );

		// Create option with default value
		add_option( 'kalium_wpb_theme_plugin_mode', true );
	}

	/**
	 * Hide AI icon from param type.
	 *
	 * @param string $output
	 *
	 * @return string
	 */
	public function vc_hide_ai_icon( $output ) {
		return str_replace( 'class="vc_ui-icon-ai"', 'class="vc_ui-icon-ai hidden" style="display:none"', $output );
	}

	/**
	 * Hide VC notices.
	 *
	 * @return array
	 */
	public function vc_hide_notices() {
		return [ 'empty_api_response' => true ];
	}

	/**
	 * Hide VC user notices.
	 * 
	 * @param mixed $value
	 * @param int $object_id
	 * @param string $meta_key
	 *
	 * @return array
	 */
	public function vc_hide_user_notices( $value, $object_id, $meta_key ) {
		if ( 'wpb_expire_close_list' !== $meta_key ) {
			return $value;
		}

		// Get current user metadata
		remove_filter( 'get_user_metadata', [ $this, 'vc_hide_user_notices' ], 10, 3 );
		$metadata = get_user_meta( $object_id, $meta_key, true );
		add_filter( 'get_user_metadata', [ $this, 'vc_hide_user_notices' ], 10, 3 );

		return json_encode(
			array_merge(
				(array) kalium_decode_json( $metadata, true ),
				[
					'not_active_license' => [
						defined( 'WPB_VC_VERSION' ) ? WPB_VC_VERSION : false,
					],
				]
			)
		);
	}

	/**
	 * Row wrapper.
	 *
	 * @param $output
	 * @param $object
	 * @param $atts
	 *
	 * @return string
	 */
	public function vc_row( $output, $object, $atts ) {

		// VC Section and Row
		if ( in_array( $object->settings( 'base' ), [ 'vc_section', 'vc_row' ] ) ) {
			$row_container_classes = [
				'vc-row-container',
			];

			// Row width
			if ( empty( $atts['full_width'] ) ) {
				$row_container_classes[] = 'container';
			}
			// Stretch row
			elseif ( 'stretch_row' == $atts['full_width'] ) {
				$row_container_classes[] = 'row-stretch';
			}
			// Stretch row and content
			elseif ( 'stretch_row_content' == $atts['full_width'] ) {
				$row_container_classes[] = 'vc-row-container--stretch-content';
			}
			// Stretch row and content (no spaces)
			elseif ( 'stretch_row_content_no_spaces' == $atts['full_width'] ) {
				$row_container_classes[] = 'vc-row-container--stretch-content-no-spaces';
			}

			// Custom classes
			if ( ! empty( $atts['el_class'] ) ) {
				$classes = explode( ' ', $atts['el_class'] );

				foreach ( $classes as $class ) {
					$row_container_classes[] = "parent--{$class}";
				}
			}

			// Remove clearfix
			$output = str_replace( [ '<div class="vc_row-full-width vc_clearfix"></div>', 'data-vc-full-width-temp="true"' ], '', $output );

			// Wrap the row
			$output = sprintf( '<div class="%2$s">%1$s</div>', $output, kalium_tokenize_list( $row_container_classes ) );
		}

		return $output;
	}

	/**
	 * Add retina image size in frontend.
	 *
	 * @param $output
	 * @param $object
	 * @param $atts
	 *
	 * @return string
	 *
	 * @since 3.1
	 */
	public function vc_single_image_retina_width( $output, $object, $atts ) {

		// Single Image element with retina image checked option
		if ( 'vc_single_image' === $object->settings( 'base' ) && 'yes' === kalium_get_array_key( $atts, 'retina_image' ) && preg_match( '/<img .*?width="(?<width>[0-9]{2,})"/', $output, $matches ) ) {
			return str_replace( '<img ', '<img style="max-width:' . round( intval( $matches['width'] ) / 2 ) . 'px" ', $output );
		}

		return $output;
	}

	/**
	 * Lightbox option for Single Image, Gallery Images and Images carousel elements.
	 */
	public function vc_lightbox_option_for_image_elements() {

		// Elements that support this "action" attribute
		foreach (
			[
				'vc_single_image'    => 'onclick',
				'vc_gallery'         => 'onclick',
				'vc_images_carousel' => 'onclick',
				'vc_gitem_image'     => 'link',
			] as $element_id => $attribute_id
		) {
			$param = WPBMap::getParam( $element_id, $attribute_id );

			// Add to select list
			if ( ! empty( $param ) && is_array( $param ) ) {
				$param['value']['Open in theme default lightbox'] = 'kalium_lightbox';
				vc_update_shortcode_param( $element_id, $param );
			}
		}
	}

	/**
	 * Elements that use theme lightbox.
	 *
	 * @param array  $atts
	 * @param string $shortcode
	 *
	 * @return array
	 */
	public function vc_lightbox_elements( $atts, $shortcode ) {
		if ( 'vc_gallery' === $shortcode || 'vc_single_image' === $shortcode ) {
			if ( 'kalium_lightbox' === kalium_get_array_key( $atts, 'onclick' ) ) {
				$atts['onclick'] = 'img_link_large';

				$this->theme_lightbox_data = [
					'kalium_lightbox' => true,
				];
			}
		}

		return $atts;
	}

	/**
	 * Theme lightbox class.
	 *
	 * @param string $classes
	 * @param string $base
	 * @param array  $atts
	 *
	 * @return string
	 */
	public function vc_lightbox_class( $classes, $base, $atts ) {
		if ( isset( $this->theme_lightbox_data['kalium_lightbox'] ) ) {
			$classes .= ' ' . kalium_get_lightbox_class();
		}

		$this->theme_lightbox_data = null;

		return $classes;
	}

	/**
	 * Grid item lightbox link.
	 *
	 * @param string   $link
	 * @param array    $atts
	 * @param \WP_Post $post
	 *
	 * @return string
	 */
	public function vc_gitem_lightbox_link( $link, $atts, $post ) {
		if ( 'kalium_lightbox' === kalium_get_array_key( $atts, 'link' ) ) {
			$href_link = vc_gitem_template_attribute_post_full_image_url(
				'',
				[
					'post' => $post,
					'data' => '',
				]
			);

			return 'a href="' . $href_link . '" data-theme-lightbox="true"';
		}

		return $link;
	}

	/**
	 * Add lightbox param as link for grid items.
	 *
	 * @param array $param
	 *
	 * @return array
	 */
	public function vc_grid_item_lightbox_option( $param ) {
		$param['value']['Open in theme default lightbox (use theme\'s built-in lightbox for videos and images)'] = 'kalium_lightbox';

		return $param;
	}

	/**
	 * Grid item lightbox link.
	 *
	 * @param $image_block
	 * @param $link
	 * @param $css_class
	 *
	 * @return string
	 */
	public function vc_grid_item_lightbox_link( $image_block, $link, $css_class ) {
		if ( 'kalium_lightbox' === $link ) {
			return '<a {{ post_full_image_url_href }} class="' . esc_attr( $css_class ) . '" data-theme-lightbox="' . kalium_generate_unique_string( 10 ) . '" title="{{ post_title }}"></a>';
		}

		return $image_block;
	}

	/**
	 * Dequeue isotope, theme includes it.
	 */
	public function vc_deregister_isotope() {
		wp_dequeue_style( 'isotope' );
	}

	/**
	 * Disable full width row JS.
	 */
	public function wpb_disable_full_width_row_js() {
		echo '<script>window.wpb_disable_full_width_row_js = true</script>';
	}

	/**
	 * Checks if WPBakery is used as theme-plugin mode.
	 *
	 * @return bool
	 */
	public function is_theme_plugin_mode() {
		if ( function_exists( 'vc_license' ) && vc_license()->isActivated() ) {
			return false;
		}

		return get_option( 'kalium_wpb_theme_plugin_mode', true );
	}

	/**
	 * VC Tabs.
	 */
	public function vc_tta_tabs_setup() {

		$new_param         = [ 'Theme Styled (if selected, other style settings will be ignored)' => 'theme-styled' ];
		$new_param_minimal = [ 'Theme Styled Minimal (if selected, other style settings will be ignored)' => 'theme-styled-minimal' ];

		$tabs_param      = WPBMap::getParam( 'vc_tta_tabs', 'style' );
		$accordion_param = WPBMap::getParam( 'vc_tta_accordion', 'style' );

		if ( ! is_array( $tabs_param ) || ! is_array( $accordion_param ) ) {
			return;
		}

		$tabs_param['value']      = array_merge( $new_param, $new_param_minimal, $tabs_param['value'] );
		$accordion_param['value'] = array_merge( $new_param, $accordion_param['value'] );

		try {
			vc_update_shortcode_param( 'vc_tta_tabs', $tabs_param );
			vc_update_shortcode_param( 'vc_tta_accordion', $accordion_param );
		} catch ( Exception $e ) {
		}
	}

	/**
	 * Widget skin class for VC elements.
	 *
	 * @param string $classes
	 * @param string $base
	 *
	 * @return string
	 */
	public function vc_widget_sidebar_classes( $classes, $base ) {
		if ( 'vc_widget_sidebar' === $base ) {
			$sidebar_options = kalium_get_sidebar_options();
			$sidebar_classes = kalium_get_sidebar_classes();

			// Styled sidebar
			if ( 'styled' === $sidebar_options['type'] ) {
				$sidebar_classes[] = 'sidebar-styled';
			}

			// Separated widgets
			if ( $sidebar_options['widgets_separate'] ) {
				$sidebar_classes[] = 'sidebar-widgets-separated';
			}

			// Is sticky sidebar
			if ( $sidebar_options['sticky'] ) {
				$sidebar_classes[] = 'has-sticky-' . $sidebar_options['sticky_behavior'];
			}

			$classes .= ' ' . kalium_tokenize_list( $sidebar_classes );
		}

		return $classes;
	}

	/**
	 * Wrap shortcode container with widget-area class.
	 *
	 * @param string             $output
	 * @param \WPBakeryShortCode $shortcode_class
	 * @param array              $atts
	 * @param string             $base
	 *
	 * @return string
	 * @since 4.0
	 */
	public function vc_widget_sidebar_wrap( $output, $shortcode_class, $atts, $base ) {
		if ( 'vc_widget_sidebar' === $base ) {
			$wrapper = '<div class="wpb_wrapper">';
			$pos     = strpos( $output, $wrapper );

			if ( false !== $pos ) {
				$output = substr_replace( $output, '<div class="wpb_wrapper widget-area">', $pos, strlen( $wrapper ) );
			}

			// Sidebar style
			$output .= kalium_print_inline_style(
				[
					'selector' => '.wpb_widgetised_column',
					'vars'     => kalium_get_sidebar_css_vars(),
				],
				false
			);
		}

		return $output;
	}
}
