<?php
/**
 * Kalium WordPress Theme
 *
 * WPML compatibility class.
 *
 * @author Laborator
 * @link   https://kaliumtheme.com
 */
namespace Kalium\Integrations;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Direct access not allowed.
}
class WPML {

	/**
	 * Construct.
	 */
	public function __construct() {
		if ( ! kalium()->is->wpml_active() ) {
			return;
		}

		// Update translation preferences
		add_action( 'wp_ajax_kalium_update_wpml_cf_preferences', [ $this, 'update_wpml_cf_preferences' ] );

		// Register strings
		add_action( 'customize_save_after', [ $this, 'register_translatable_strings' ] );

		// Translate Laborator Builder strings
		add_filter( 'laborator_builder_attribute_value', [ $this, 'translate_lb_attribute_value' ], 10, 3 );
	}

	/**
	 * Update translation preferences for local fields.
	 *
	 * @since 4.1.3
	 */
	public function update_wpml_cf_preferences() {
		if ( ! current_user_can( 'manage_options' ) || ! class_exists( 'WPML_ACF' ) ) {
			return;
		}

		$cft_preferences = & $GLOBALS['iclTranslationManagement']->settings['custom_fields_translation'];

		foreach ( acf_get_local_fields() as $field ) {
			$field_name = $field['name'] ?? null;

			if ( empty( $field_name ) ) {
				continue;
			}

			// Set translation preference for field key reference
			$cft_preferences[ '_' . $field_name ] = 1;

			// Trigger update for the field
			acf_update_field( $field );
		}

		wp_send_json_success();
	}

	/**
	 * Register translatable strings.
	 *
	 * @param \WP_Customize_Manager $wp_customize
	 *
	 * @since 4.2.1
	 */
	public function register_translatable_strings( $wp_customize ) {
		foreach ( $wp_customize->controls() as $control ) {
			if ( 'laborator-builder' === $control->type ) {
				$setting = $control->setting;
				$value   = kalium_get_theme_option( $setting->id );

				$this->register_lb_translatable_attributes( $control->content_type_id, $value );
			}
		}
	}

	/**
	 * Register translatable attributes of Laborator Builder elements.
	 *
	 * @param string $content_type_id
	 * @param array  $value
	 *
	 * @since 4.2.1
	 */
	public function register_lb_translatable_attributes( $content_type_id, $value ) {
		$content_type = laborator_builder_get_content_type( $content_type_id );

		if ( ! $content_type || empty( $value['elements'] ) ) {
			return;
		}

		$elements = \Laborator_Builder\Parser::parse( $content_type, $value['elements'] );

		/**
		 * @param \Laborator_Builder\Elements $elements
		 */
		$traverser = function ( $elements ) use ( &$traverser ) {
			foreach ( $elements->get_elements() as $element ) {
				// Register translatable attributes of Laborator Builder elements
				foreach ( $element->get_attributes() as $attribute ) {
					if ( $attribute->is_translatable() ) {
						kalium_wpml_register_single_string( $attribute->get_value(), $attribute->get_translation_key_name() );
					}
				}

				if ( ! empty( $element->get_children() ) ) {
					$traverser( $element->get_children() );
				}
			}
		};

		// Register strings for translation
		$traverser( $elements );
	}

	/**
	 * Translate Laborator Builder attribute values.
	 *
	 * @param string                       $value
	 * @param \Laborator_Builder\Attribute $attribute
	 *
	 * @return string
	 *
	 * @since 4.2.1
	 */
	public function translate_lb_attribute_value( $value, $attribute ) {
		if ( $attribute->is_translatable() ) {
			return kalium_wpml_translate_single_string( $value, $attribute->get_translation_key_name() );
		}

		return $value;
	}
}
