<?php
/**
 * Kalium WordPress Theme
 *
 * Media handler class.
 *
 * @author Laborator
 * @link   https://kaliumtheme.com
 */
namespace Kalium\Media;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Direct access not allowed.
}

class Media_Handler {

	/**
	 * Handler name.
	 *
	 * @var string
	 */
	public $handler_name;

	/**
	 * Handler args.
	 *
	 * @var array
	 */
	public $handler_options = [];

	/**
	 * Get handler name.
	 *
	 * @return string
	 */
	public function get_handler_name() {
		return $this->handler_name;
	}

	/**
	 * Get handler options.
	 *
	 * @return array
	 */
	public function get_handler_options() {
		return $this->handler_options;
	}

	/**
	 * Initialize.
	 */
	public function init() {
		$this->handler_options['autoplay'] = kalium_get_theme_option( 'media_autoplay' );
		$this->handler_options['loop']     = kalium_get_theme_option( 'media_loop' );
	}

	/**
	 * Get extra attributes.
	 *
	 * @param array  $atts
	 * @param string $media_type
	 *
	 * @return array
	 */
	public function get_extra_attributes( $atts, $media_type ) {
		$extra_atts = [];

		// Controls
		$extra_atts['controls'] = kalium_validate_boolean( $atts['controls'] ?? true );

		// Playsinline
		$extra_atts['playsinline'] = kalium_validate_boolean( $atts['playsinline'] ?? false );

		// Global loop
		if ( ! isset( $atts['loop'] ) && ! empty( $this->handler_options['loop'] ) ) {
			$extra_atts['loop'] = true;
		}

		// Global autoplay
		if ( ! isset( $atts['autoplay'] ) && ! empty( $this->handler_options['autoplay'] ) ) {
			if ( 'yes' === $this->handler_options['autoplay'] ) {
				$extra_atts['autoplay'] = true;
			} elseif ( 'on-viewport' === $this->handler_options['autoplay'] ) {
				$extra_atts['autoplay-on-viewport'] = true;
			}
		} elseif ( isset( $atts['autoplay'] ) && 'on-viewport' === $atts['autoplay'] ) {
			$extra_atts['autoplay'] = false;

			$extra_atts['autoplay-on-viewport'] = true;
		}

		// Auto mute autoplay media
		if ( apply_filters( 'kalium_media_mute_autoplay', true ) && ! isset( $atts['muted'] ) ) {
			if ( ! empty( $atts['autoplay'] ) || ! empty( $extra_atts['autoplay'] ) ) {
				$extra_atts['muted'] = true;
			}
		}

		// Support for Attachment ID as poster and Audio poster
		if ( ! empty( $atts['poster'] ) ) {
			if ( is_numeric( $atts['poster'] ) && ( $image_src = wp_get_attachment_image_src( $atts['poster'], 'large' ) ) ) {
				$extra_atts['poster'] = $image_src[0];
			} elseif ( 'audio' === $media_type && is_string( $atts['poster'] ) ) {
				$extra_atts['poster'] = $atts['poster'];
			}
		} else {
			// Workaround to prevent deprecated notice in PHP 8.1+
			$extra_atts['poster'] = '';
		}

		// Set width and height attributes for audio with poster
		if ( 'audio' === $media_type && ! empty( $extra_atts['poster'] ) ) {
			$extra_atts['width']  = $atts['width'] ?? 16;
			$extra_atts['height'] = $atts['height'] ?? 9;
		}

		// Object fit
		if ( ! empty( $atts['object_fit'] ) ) {
			$extra_atts['object_fit'] = kalium_enum_value( $atts['object_fit'], [ 'contain', 'cover' ] );
		}

		return $extra_atts;
	}

	/**
	 * Prepare video attributes.
	 *
	 * @param array $out
	 * @param array $pairs
	 * @param array $atts
	 *
	 * @return array
	 */
	public function prepare_atts_video( $out, $pairs, $atts ) {
		return array_merge( $out, $this->get_extra_attributes( $atts, 'video' ) );
	}

	/**
	 * Prepare audio attributes.
	 *
	 * @param array $out
	 * @param array $pairs
	 * @param array $atts
	 *
	 * @return array
	 */
	public function prepare_atts_audio( $out, $pairs, $atts ) {
		$atts = wp_parse_args(
			$atts,
			[
				'width'  => 170,
				'height' => 20,
				'poster' => null,
			]
		);

		// Add dimensions based on poster size
		if ( $atts['poster'] && ! isset( $atts['width'], $atts['height'] ) ) {
			if ( $dimensions = kalium()->media->get_image_dimensions( $atts['poster'] ) ) {
				$atts = array_merge( $atts, $dimensions );
			}
		}

		return array_merge( $out, $this->get_extra_attributes( $atts, 'audio' ) );
	}

	/**
	 * Video parser.
	 *
	 * @param string $output
	 * @param array  $atts
	 * @param string $video
	 * @param int    $post_id
	 * @param string $library
	 *
	 * @return string
	 */
	public function parse_video( $output, $atts, $video, $post_id, $library ) {
		return $output;
	}

	/**
	 * Audio parser.
	 *
	 * @param string $output
	 * @param array  $atts
	 * @param string $audio
	 * @param int    $post_id
	 * @param string $library
	 *
	 * @return string
	 */
	public function parse_audio( $output, $atts, $audio, $post_id, $library ) {
		return $output;
	}

	/**
	 * Process output.
	 *
	 * @param string $output
	 * @param array  $atts
	 *
	 * @return string
	 */
	public function process_output( $output, $atts ) {
		// Hide controls
		if ( isset( $atts['controls'] ) && ! $atts['controls'] ) {
			$output = preg_replace( '/\s+controls(=(?:"|\')[^"\']+(?:"|\'))?/', '', $output );
		}

		// Media options attribute
		if ( preg_match( '/<(audio|video|iframe)\b/', $output, $matches ) ) {
			$media_type    = $matches[1];
			$media_options = [];
			$forward_atts  = [
				'autoplay',
				'controls',
				'playsinline',
				'muted',
				'loop',
			];

			foreach ( $forward_atts as $attr_name ) {
				if ( ! empty( $atts[ $attr_name ] ) ) {
					$media_options[ $attr_name ] = $atts[ $attr_name ];
				}
			}

			// Autoplay on viewport
			if ( ! empty( $atts['autoplay-on-viewport'] ) ) {
				$media_options['autoplayOnViewport'] = true;
			}

			// Playsinline attribute
			if ( kalium_validate_boolean( $atts['playsinline'] ) && in_array( $media_type, [ 'video', 'audio' ], true ) ) {
				$output = kalium()->helpers->add_element_attribute( $output, $media_type, 'playsinline', true );
			}

			if ( ! empty( $media_options ) ) {
				$output = kalium()->helpers->add_element_attribute( $output, $media_type, 'data-media-options', $media_options );
			}

			// Object fit
			if ( ! empty( $atts['object_fit'] ) ) {
				$output = kalium()->helpers->add_element_attribute( $output, $media_type, 'data-object-fit', $atts['object_fit'] );
			}
		}

		return $output;
	}

	/**
	 * Wrap video element.
	 *
	 * @param string $output
	 * @param array  $atts
	 *
	 * @return string
	 */
	public function wrap_video( $output, $atts ) {
		$width  = kalium_get_array_key( $atts, 'width' );
		$height = kalium_get_array_key( $atts, 'height' );

		$args = [
			'element' => 'div',
			'width'   => is_numeric( $width ) ? $width : 640,
			'height'  => is_numeric( $height ) ? $height : 360,
		];

		return kalium()->images->aspect_ratio_wrap(
			$output,
			$args,
			[
				'class' => $this->wrapper_classes( 'video', $atts ),
			]
		);
	}

	/**
	 * Wrap audio element.
	 *
	 * @param string $output
	 * @param array  $atts
	 *
	 * @return string
	 */
	public function wrap_audio( $output, $atts ) {
		$args = [
			'element' => 'div',
		];

		// Audio dimensions
		if ( isset( $atts['width'], $atts['height'] ) ) {
			$args['width']  = $atts['width'];
			$args['height'] = $atts['height'];
		}

		return kalium()->images->aspect_ratio_wrap(
			$output,
			$args,
			[
				'class' => $this->wrapper_classes( 'audio', $atts ),
			]
		);
	}

	/**
	 * Wrapper classes.
	 *
	 * @param string $type
	 * @param array  $atts
	 *
	 * @return array
	 */
	private function wrapper_classes( $type, $atts ) {
		$classes = [
			$type,
		];

		if ( ! empty( $atts['wrapper_class'] ) ) {
			$classes[] = kalium_tokenize_list( $atts['wrapper_class'] );
		}

		return $classes;
	}
}
