<?php
/**
 * Kalium WordPress Theme
 *
 * Media handler.
 *
 * @author Laborator
 * @link   https://kaliumtheme.com
 */
namespace Kalium\Media;

use Kalium\Media\Handlers\VideoJS as VideoJS_Handler;
use Kalium\Media\Handlers\Native as Native_Handler;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Direct access not allowed.
}

class Media {

	/**
	 * Media handler.
	 *
	 * @var Media_Handler
	 */
	public $media_handler;

	/**
	 * Constructor.
	 */
	public function __construct() {
		add_action( 'wp', [ $this, 'init' ], 0 );
	}

	/**
	 * Initialize media handler.
	 */
	public function init() {
		// Skip on TinyMCE editor preview
		if ( wp_doing_ajax() && in_array( kalium()->request->request( 'action' ), [ 'parse-media-shortcode', 'parse-embed' ], true ) ) {
			return;
		}

		switch ( apply_filters( 'kalium_media_library', kalium_get_theme_option( 'media_player_library' ) ) ) {
			// VideoJS
			case 'videojs':
				$this->media_handler = new VideoJS_Handler();
				break;

			// Native
			default:
				$this->media_handler = new Native_Handler();
		}

		if ( isset( $this->media_handler ) ) {
			add_action( 'wp', [ $this->media_handler, 'init' ] );
			add_action( 'wp_print_scripts', [ $this, 'define_js_vars' ] );

			foreach ( [ 'video', 'audio' ] as $media_type ) {
				add_filter( "shortcode_atts_{$media_type}", [ $this->media_handler, 'prepare_atts_' . $media_type ], 10, 3 );
				add_filter( "wp_{$media_type}_shortcode_library", [ $this->media_handler, 'get_handler_name' ] );
				add_filter( "wp_{$media_type}_shortcode", [ $this->media_handler, 'parse_' . $media_type ], 100, 5 );
				add_filter( "wp_{$media_type}_shortcode", [ $this->media_handler, 'process_output' ], 110, 5 );
				add_filter( "wp_{$media_type}_shortcode", [ $this->media_handler, 'wrap_' . $media_type ], 120, 2 );
			}
		}
	}

	/**
	 * Define JS vars.
	 */
	public function define_js_vars() {
		if ( ! did_action( 'wp' ) || did_action( 'admin_print_scripts' ) ) {
			return;
		}

		kalium_js_define_vars(
			[
				'mediaPlayer' => [
					'handler' => $this->media_handler->get_handler_name(),
					'options' => (object) $this->media_handler->get_handler_options(),
				],
			]
		);
	}

	/**
	 * Video element.
	 *
	 * @param array $args
	 *
	 * @return string
	 */
	public function video( $args = [] ) {
		return wp_video_shortcode( $args );
	}

	/**
	 * Audio element.
	 *
	 * @param array $args
	 *
	 * @return string
	 */
	public function audio( $args = [] ) {
		return wp_audio_shortcode( $args );
	}

	/**
	 * Get dimensions from provided attachment ID or URL.
	 *
	 * @param int|string $image_url
	 *
	 * @return array|null
	 */
	public function get_image_dimensions( $image_url ) {
		$attachment_id = is_numeric( $image_url ) ? $image_url : attachment_url_to_postid( $image_url );

		if ( $attachment_id && ( $file = get_attached_file( $attachment_id ) ) ) {
			if ( 0 === strpos( wp_check_filetype( $file )['type'], 'image/' ) ) {
				$metadata = wp_get_attachment_metadata( $attachment_id );

				if ( isset( $metadata['width'], $metadata['height'] ) ) {
					return [
						'width'  => $metadata['width'],
						'height' => $metadata['height'],
					];
				}
			}
		}

		return null;
	}

	/**
	 * Check if the url is a YouTube video.
	 *
	 * @param string $url
	 *
	 * @return bool
	 */
	public function is_youtube( $url ) {
		return preg_match( '#^https?://(?:www\.)?(?:youtube\.com/(watch|embed)|youtu\.be/)#', $url );
	}

	/**
	 * Check if the url is a Vimeo video.
	 *
	 * @param string $url
	 *
	 * @return bool
	 */
	public function is_vimeo( $url ) {
		return preg_match( '#^https?://(.+\.)?vimeo\.com/.*#', $url );
	}

	/**
	 * Check if provided URL is video type.
	 *
	 * @param string $url
	 *
	 * @return bool
	 */
	public function is_video( $url ) {
		return in_array( wp_check_filetype( $url )['ext'], wp_get_video_extensions(), true ) || $this->is_youtube( $url ) || $this->is_vimeo( $url );
	}

	/**
	 * Check if provided URL is audio type.
	 *
	 * @param string $url
	 *
	 * @return bool
	 */
	public function is_audio( $url ) {
		return in_array( wp_check_filetype( $url )['ext'], wp_get_audio_extensions(), true );
	}

	/**
	 * Extract media source.
	 *
	 * @param string $str
	 *
	 * @return string
	 */
	public function extract_media_source( $str ) {
		$sources = wp_extract_urls( $str );

		while ( $source = array_shift( $sources ) ) {
			// First non-image URL
			if ( ! preg_match( '/\.(jpe?g|png|gif)(\?.*)?$/i', $source ) ) {
				break;
			}
		}

		return $source;
	}

	/**
	 * Adjust embed URL to given attributes.
	 *
	 * @param string $provider
	 * @param string $url
	 * @param array  $atts
	 *
	 * @return string
	 */
	public function adjust_embed_url( $provider, $url, $atts ) {
		$atts = wp_parse_args(
			$atts,
			[
				'controls'    => null,
				'playsinline' => null,
				'autoplay'    => null,
				'muted'       => null,
				'loop'        => null,
			]
		);

		switch ( $provider ) {
			// YouTube
			case 'youtube':
				$url = add_query_arg(
					[
						'enablejsapi'    => 1, // Enable JS API
						'iv_load_policy' => 3, // Hide video annotations
					],
					$url
				);

				// Muted
				if ( kalium_validate_boolean( $atts['muted'] ) ) {
					$url = add_query_arg( 'mute', 1, $url );
				}

				// Loop
				if ( kalium_validate_boolean( $atts['loop'] ) ) {
					$url = add_query_arg(
						[
							'loop'     => 1,
							'playlist' => kalium_get_youtube_video_id_from_url( $url ),
						],
						$url
					);
				}
				break;

			// Vimeo
			case 'vimeo':
				// Muted
				if ( kalium_validate_boolean( $atts['muted'] ) ) {
					$url = add_query_arg( 'muted', 1, $url );
				}

				// Loop
				if ( kalium_validate_boolean( $atts['loop'] ) ) {
					$url = add_query_arg( 'loop', 1, $url );
				}
				break;
		}

		/**
		 * Common attributes for all providers.
		 */

		// Controls
		if ( ! is_null( $atts['controls'] ) && ! kalium_validate_boolean( $atts['controls'] ) ) {
			$url = add_query_arg( 'controls', 0, $url );
		}

		// Playsinline
		if ( kalium_validate_boolean( $atts['playsinline'] ) ) {
			$url = add_query_arg( 'playsinline', 1, $url );
		}

		// Autoplay
		if ( kalium_validate_boolean( $atts['autoplay'] ) ) {
			$url = add_query_arg( 'autoplay', 1, $url );
		}

		return $url;
	}

	/**
	 * Embed YouTube URL.
	 *
	 * @param string $url
	 * @param array  $atts
	 *
	 * @return string
	 */
	public function embed_youtube( $url, $atts ) {
		$iframe_embed = wp_oembed_get( $url );
		$embed_urls   = wp_extract_urls( $iframe_embed );
		$first_url    = $embed_urls[0] ?? null;
		$embed_url    = $this->adjust_embed_url( 'youtube', $first_url, $atts );

		// Replace source with adjusted URL
		$iframe_embed = str_replace( htmlentities( $first_url ), $embed_url, $iframe_embed );

		return $iframe_embed;
	}

	/**
	 * Embed Vimeo URL.
	 *
	 * @param string $url
	 * @param array  $atts
	 *
	 * @return string
	 */
	public function embed_vimeo( $url, $atts ) {
		$iframe_embed = wp_oembed_get( $url );

		if ( false === $iframe_embed ) {
			return $url;
		}

		$embed_urls = wp_extract_urls( $iframe_embed );
		$first_url  = $embed_urls[0] ?? null;
		$embed_url  = $this->adjust_embed_url( 'vimeo', $first_url, $atts );

		// Replace source with adjusted URL
		$iframe_embed = str_replace( htmlentities( $first_url ), $embed_url, $iframe_embed );

		return $iframe_embed;
	}
}
