<?php
/**
 * Display conditions.
 *
 * @author Laborator
 * @link   https://kaliumtheme.com
 * @since 4.2
 */
namespace Kalium\Template_Parts;

use WP_Post;

class Display_Conditions {

	/**
	 * Construct.
	 */
	public function __construct() {
		add_action( 'rest_api_init', [ $this, 'rest_endpoints' ] );
	}

	/**
	 * Get available types.
	 *
	 * @return array
	 */
	public function get_types() {
		return apply_filters(
			'kalium_display_conditions_types',
			[
				[
					'name'    => 'general-page',
					'label'   => 'General Page',
					'choices' => [
						'sitewide'       => [
							'label' => 'Entire Site',
						],
						'front-page'     => [
							'label' => 'Front Page',
						],
						'posts-page'     => [
							'label' => 'Blog Page',
						],
						'not-found-page' => [
							'label' => '404 Error Page',
						],
						'search-page'    => [
							'label' => 'Search Page',
						],
					],
				],
				[
					'name'    => 'archive',
					'label'   => 'Archive',
					'choices' => [
						'posts'            => [
							'label' => 'Blog Archive',
						],
						'date'             => [
							'label' => 'Date Archive',
						],
						'author'           => [
							'label' => 'Author Archive',
						],
						'custom-post-type' => [
							'label'     => 'Custom Post Type Archive',
							'sub_value' => [
								'type'   => 'select',
								'name'   => 'post_type',
								'source' => [
									'type'    => 'array',
									'choices' => $this->get_post_types_choices(),
								],
							],
						],
					],
				],
				[
					'name'    => 'taxonomy',
					'label'   => 'Taxonomy',
					'choices' => [
						'category'        => [
							'label'     => 'Category Archive',
							'sub_value' => [
								'type'        => 'select',
								'name'        => 'terms',
								'multiple'    => true,
								'placeholder' => get_taxonomy( 'category' )->labels->all_items,
								'source'      => [
									'type'        => 'dynamic',
									'object_type' => 'taxonomy',
									'taxonomy'    => 'category',
								],
							],
						],
						'tag'             => [
							'label'     => 'Tag Archive',
							'sub_value' => [
								'type'        => 'select',
								'name'        => 'terms',
								'multiple'    => true,
								'placeholder' => get_taxonomy( 'post_tag' )->labels->all_items,
								'source'      => [
									'type'        => 'dynamic',
									'object_type' => 'taxonomy',
									'taxonomy'    => 'post_tag',
								],
							],
						],
						'custom-taxonomy' => [
							'label'     => 'Custom Taxonomy Archive',
							'sub_value' => [
								'type'   => 'select',
								'name'   => 'taxonomy',
								'source' => [
									'type'    => 'array',
									'choices' => array_map(
										function ( $choice ) {
											$taxonomy    = $choice['value'];
											$placeholder = get_taxonomy( $taxonomy )->labels->all_items;

											$choice['sub_value'] = [
												'type'     => 'select',
												'name'     => 'terms',
												'multiple' => true,
												'placeholder' => $placeholder,
												'source'   => [
													'type' => 'dynamic',
													'object_type' => 'taxonomy',
													'taxonomy' => $taxonomy,
												],
											];

											return $choice;
										},
										$this->get_taxonomy_choices()
									),
								],
							],
						],
					],
				],
				[
					'name'    => 'singular',
					'label'   => 'Singular Content',
					'choices' => [
						'post'             => [
							'label'     => 'Single Post',
							'sub_value' => [
								'type'        => 'select',
								'name'        => 'post_ids',
								'placeholder' => get_post_type_object( 'post' )->labels->all_items,
								'multiple'    => true,
								'source'      => [
									'type'        => 'dynamic',
									'object_type' => 'post_type',
									'post_type'   => 'post',
									'args'        => [
										'status' => 'publish',
										'limit'  => 100,
									],
								],
							],
						],
						'page'             => [
							'label'     => 'Single Page',
							'sub_value' => [
								'type'        => 'select',
								'name'        => 'post_ids',
								'placeholder' => get_post_type_object( 'page' )->labels->all_items,
								'multiple'    => true,
								'source'      => [
									'type'        => 'dynamic',
									'object_type' => 'post_type',
									'post_type'   => 'page',
									'args'        => [
										'status' => 'publish',
										'limit'  => 100,
									],
								],
							],
						],
						'custom-post-type' => [
							'label'     => 'Custom Post Type Single',
							'sub_value' => [
								'type'        => 'select',
								'name'        => 'post_type',
								'placeholder' => 'Any Post Type',
								'source'      => [
									'type'    => 'array',
									'choices' => array_map(
										function ( $choice ) {
											$post_type   = $choice['value'];
											$placeholder = get_post_type_object( $post_type )->labels->all_items;

											// Sub value
											$choice['sub_value'] = [
												'type'     => 'select',
												'name'     => 'post_ids',
												'multiple' => true,
												'placeholder' => $placeholder,
												'source'   => [
													'type' => 'dynamic',
													'object_type' => 'post_type',
													'post_type' => $post_type,
													'args' => [
														'status' => 'publish',
														'limit'  => 100,
													],
												],
											];

											return $choice;
										},
										$this->get_post_types_choices()
									),
								],
							],
						],
						'attachment'       => [
							'label' => 'Attachment',
						],
					],
				],
				[
					'name'    => 'current-post',
					'label'   => 'Current Post',
					'choices' => [
						'post-featured-image' => [
							'label'                => 'Featured Image',
							'additional_operators' => $this->get_additional_operators( 'has' ),
						],
						'post-type'           => [
							'label'                => 'Type',
							'additional_operators' => $this->get_additional_operators( 'has' ),
							'sub_value'            => [
								'type'     => 'select',
								'name'     => 'post_type',
								'multiple' => true,
								'source'   => [
									'type'    => 'array',
									'choices' => $this->get_post_types_choices( true, true ),
								],
							],
						],
						'post-meta'           => [
							'label'                => 'Meta',
							'additional_operators' => $this->get_additional_operators( 'string' ),
							'sub_value'            => [
								[
									'type'        => 'text',
									'name'        => 'meta_key',
									'placeholder' => 'Meta Key',
								],
								[
									'type'        => 'text',
									'name'        => 'meta_value',
									'placeholder' => 'Meta Value',
								],
							],
						],
						'post-id'             => [
							'label'     => 'ID',
							'sub_value' => [
								'type'        => 'text',
								'name'        => 'post_id',
								'placeholder' => 'Post ID',
							],
						],
						'post-format'         => [
							'label'                => 'Format',
							'additional_operators' => $this->get_additional_operators( 'has' ),
							'sub_value'            => [
								'type'        => 'select',
								'name'        => 'post_format',
								'multiple'    => true,
								'placeholder' => 'Select Format',
								'source'      => [
									'type'    => 'array',
									'choices' => array_map(
										function ( $label, $value ) {
											return [
												'value' => $value,
												'label' => $label,
											];
										},
										get_post_format_strings(),
										array_keys( get_post_format_strings() )
									),
								],
							],
						],
						'post-status'         => [
							'label'                => 'Status',
							'additional_operators' => $this->get_additional_operators( 'has' ),
							'sub_value'            => [
								'type'   => 'select',
								'name'   => 'post_status',
								'source' => [
									'type'    => 'array',
									'choices' => array_map(
										function ( $label, $value ) {
											return [
												'value' => $value,
												'label' => $label,
											];
										},
										get_post_statuses(),
										array_keys( get_post_statuses() )
									),
								],
							],
						],
						'post-author'         => [
							'label'                => 'Author',
							'additional_operators' => $this->get_additional_operators( 'has' ),
							'sub_value'            => [
								'type'        => 'select',
								'name'        => 'post_author',
								'multiple'    => true,
								'placeholder' => 'Select Author',
								'source'      => [
									'type'        => 'dynamic',
									'object_type' => 'user',
								],
							],
						],
						'post-term'           => [
							'label'                => 'Term',
							'additional_operators' => $this->get_additional_operators( 'has' ),
							'sub_value'            => [
								'type'   => 'select',
								'name'   => 'taxonomy',
								'source' => [
									'type'    => 'array',
									'choices' => array_map(
										function ( $choice ) {
											$taxonomy    = $choice['value'];
											$placeholder = ucwords( get_taxonomy( $taxonomy )->labels->name );

											$choice['sub_value'] = [
												'type'     => 'select',
												'name'     => 'terms',
												'multiple' => true,
												'placeholder' => sprintf( 'Select %s', $placeholder ),
												'source'   => [
													'type' => 'dynamic',
													'object_type' => 'taxonomy',
													'taxonomy' => $taxonomy,
												],
											];

											return $choice;
										},
										$this->get_taxonomy_choices( true )
									),
								],
							],
						],
						'post-sticky'         => [
							'label' => 'Sticky',
						],
					],
				],
				[
					'name'    => 'user',
					'label'   => 'User',
					'choices' => [
						'logged-in'  => [
							'label' => 'Logged In',
						],
						'logged-out' => [
							'label' => 'Logged Out',
						],
						'user-role'  => [
							'label'     => 'User Role',
							'sub_value' => [
								'type'        => 'select',
								'name'        => 'role',
								'multiple'    => false,
								'placeholder' => 'All Roles',
								'source'      => [
									'type'    => 'array',
									'choices' => array_map(
										function ( $label, $key ) {
											return [
												'value' => $key,
												'label' => $label,
											];
										},
										( $role_names = wp_roles()->get_names() ),
										array_keys( $role_names )
									),
								],
							],
						],
					],
				],
				[
					'name'    => 'woocommerce',
					'label'   => 'WooCommerce',
					'choices' => [
						'product-page'     => [
							'label' => 'Product Page',
						],
						'products-page'    => [
							'label' => 'Shop Archive',
						],
						'product-category' => [
							'label' => 'Product Category Archive',
						],
						'product-tag'      => [
							'label' => 'Product Tag Archive',
						],
						'cart-page'        => [
							'label' => 'Cart Page',
						],
						'checkout-page'    => [
							'label' => 'Checkout Page',
						],
						'account-page'     => [
							'label' => 'My Account Page',
						],
					],
				],
				[
					'name'    => 'date-time',
					'label'   => 'Date & Time',
					'choices' => [
						'date' => [
							'label'                => 'Specific Date',
							'additional_operators' => $this->get_additional_operators( 'date_time' ),
							'sub_value'            => [
								'type'          => 'date',
								'name'          => 'date',
								'start_of_week' => get_option( 'start_of_week' ),
							],
						],
						'time' => [
							'label'                => 'Specific Time',
							'additional_operators' => $this->get_additional_operators( 'date_time' ),
							'sub_value'            => [
								'type' => 'time',
								'name' => 'time',
							],
						],
						'days' => [
							'label'     => 'Specific Days',
							'sub_value' => [
								'type'        => 'select',
								'name'        => 'days',
								'multiple'    => true,
								'placeholder' => 'Any Day',
								'default'     => [
									'monday',
									'tuesday',
									'wednesday',
									'thursday',
									'friday',
									'saturday',
									'sunday',
								],
								'source'      => [
									'type'    => 'array',
									'choices' => [
										[
											'value' => 'monday',
											'label' => 'Monday',
										],
										[
											'value' => 'tuesday',
											'label' => 'Tuesday',
										],
										[
											'value' => 'wednesday',
											'label' => 'Wednesday',
										],
										[
											'value' => 'thursday',
											'label' => 'Thursday',
										],
										[
											'value' => 'friday',
											'label' => 'Friday',
										],
										[
											'value' => 'saturday',
											'label' => 'Saturday',
										],
										[
											'value' => 'sunday',
											'label' => 'Sunday',
										],
									],
								],
							],
						],
					],
				],
				[
					'name'    => 'custom-condition',
					'label'   => 'Custom Condition',
					'choices' => [
						'php-callback'     => [
							'label'     => 'PHP Callback',
							'sub_value' => [
								'type'        => 'text',
								'name'        => 'callback',
								'placeholder' => 'Enter Function Name',
							],
						],
						'url-query-string' => [
							'label'                => 'URL Query String',
							'additional_operators' => $this->get_additional_operators( 'string' ),
							'sub_value'            => [
								[
									'type'        => 'text',
									'name'        => 'key',
									'placeholder' => 'Query Key',
								],
								[
									'type'        => 'text',
									'name'        => 'value',
									'placeholder' => 'Query Value',
								],
							],
						],
						'referrer'         => [
							'label'                => 'Referrer URL',
							'additional_operators' => $this->get_additional_operators( 'string' ),
							'sub_value'            => [
								'type'        => 'text',
								'name'        => 'url',
								'placeholder' => 'Referrer URL',
							],
						],
					],
				],
			]
		);
	}

	/**
	 * Get available operators.
	 *
	 * @return array
	 */
	public function get_operators() {
		return apply_filters(
			'kalium_display_conditions_operators',
			[
				'==' => 'Is',
				'!=' => 'Is Not',
			]
		);
	}

	/**
	 * Get additional operators for one or more groups.
	 *
	 * @return array
	 */
	public function get_additional_operators() {
		$groups = [
			'string'    => [
				'starts'    => 'Starts With',
				'ends'      => 'Ends With',
				'regex'     => 'Matches Regex',
				'not-regex' => 'Not Matches Regex',
			],
			'date_time' => [
				'<'  => 'Before',
				'>'  => 'After',
				'<=' => 'On or Before',
				'>=' => 'On or After',
			],
			'has'       => [
				'==' => 'Has',
				'!=' => 'Has Not',
			],
		];

		$result = [];

		$get_groups = 'all' === func_get_arg( 0 ) ? array_keys( $groups ) : func_get_args();

		foreach ( $get_groups as $group ) {
			if ( isset( $groups[ $group ] ) ) {
				$result = array_merge( $result, $groups[ $group ] );
			}
		}

		return $result;
	}

	/**
	 * Compare value.
	 *
	 * @param mixed  $actual
	 * @param string $operator
	 * @param mixed  $expected
	 *
	 * @return bool
	 */
	public function compare( $actual, $operator, $expected ) {
		switch ( $operator ) {
			case '==':
				// phpcs:ignore WordPress.PHP.StrictComparisons.LooseComparison
				return $actual == $expected;

			case '!=':
				// phpcs:ignore WordPress.PHP.StrictComparisons.LooseComparison
				return $actual != $expected;

			case 'starts':
				return is_string( $actual ) && is_string( $expected ) && str_starts_with( $actual, $expected );

			case 'ends':
				return is_string( $actual ) && is_string( $expected ) && str_ends_with( $actual, $expected );

			case 'regex':
				$pattern = '/' . str_replace( '/', '\/', $expected ) . '/';
				return @preg_match( $pattern, $actual ) === 1;

			case 'not-regex':
				$pattern = '/' . str_replace( '/', '\/', $expected ) . '/';
				return @preg_match( $pattern, $actual ) !== 1;

			case '<':
				return $actual < $expected;

			case '>':
				return $actual > $expected;

			case '<=':
				return $actual <= $expected;

			case '>=':
				return $actual >= $expected;

			default:
				return false;
		}
	}

	/**
	 * Check condition.
	 *
	 * @param array $condition
	 *
	 * @return bool
	 */
	public function check( $condition ) {
		global $wp_query, $post;

		$type       = $condition['type'];
		$operator   = $condition['operator'];
		$value      = $condition['value'];
		$sub_values = $condition['sub_values'] ?? [];

		switch ( $type ) {

			// General/system page
			case 'general-page':
				if ( 'sitewide' === $value ) {
					return $this->compare( true, $operator, true );
				} elseif ( 'front-page' === $value ) {
					return $this->compare( $wp_query->is_front_page(), $operator, true );
				} elseif ( 'posts-page' === $value ) {
					return $this->compare( $wp_query->is_home(), $operator, true );
				} elseif ( 'not-found-page' === $value ) {
					return $this->compare( $wp_query->is_404(), $operator, true );
				} elseif ( 'search-page' === $value ) {
					return $this->compare( $wp_query->is_search(), $operator, true );
				}

				return false;

			// Singular
			case 'singular':
				if ( is_singular() ) {
					$post      = get_queried_object();
					$post_type = $value;
					$post_ids  = $sub_values['post_ids'] ?? null;

					if ( ! empty( $post_ids ) ) {
						$post_ids = array_map( $this->translate_object_ids(), $post_ids );
					}

					if ( 'custom-post-type' === $post_type ) {
						$post_type = $sub_values['post_type'] ?? null;
					}

					if ( ! empty( $post_ids ) && is_array( $post_ids ) ) {
						foreach ( $post_ids as $post_id ) {
							if ( $this->compare( $post_id, $operator, $post->ID ) ) {
								return true;
							}
						}
					} else {
						return $this->compare( get_post_type( $post ), $operator, $post_type );
					}
				}

				return false;

			// Archive
			case 'archive':
				if ( 'posts' === $value ) {
					return $this->compare( $wp_query->is_home() || $wp_query->is_category() || $wp_query->is_tag() || $wp_query->is_date() || $wp_query->is_author(), $operator, true );
				} elseif ( 'date' === $value ) {
					return $this->compare( $wp_query->is_date(), $operator, true );
				} elseif ( 'author' === $value ) {
					return $this->compare( $wp_query->is_author(), $operator, true );
				} elseif ( 'custom-post-type' === $value ) {
					$post_type = $sub_values['post_type'] ?? null;

					return $this->compare( is_post_type_archive( $post_type ), $operator, true );
				}

				return false;

			// Taxonomy
			case 'taxonomy':
				$taxonomy = $sub_values['taxonomy'] ?? ( 'tag' === $value ? 'post_tag' : 'category' );
				$terms    = $sub_values['terms'] ?? null;

				if ( ! empty( $terms ) ) {
					$terms = array_map( $this->translate_object_ids( 'TERM' ), $terms );
				}

				if ( 'category' === $value ) {
					if ( ! empty( $terms ) && is_array( $terms ) ) {
						return $this->compare( $wp_query->is_category( $terms ), $operator, true );
					}

					return $this->compare( $wp_query->is_category(), $operator, true );
				} elseif ( 'tag' === $value ) {
					if ( ! empty( $terms ) && is_array( $terms ) ) {
						return $this->compare( $wp_query->is_tag( $terms ), $operator, true );
					}

					return $this->compare( $wp_query->is_tag(), $operator, true );
				}

				if ( ! empty( $terms ) && is_array( $terms ) ) {
					return $this->compare( $wp_query->is_tax( $taxonomy, $terms ), $operator, true );
				}

				return $this->compare( $wp_query->is_tax( $taxonomy ), $operator, true );

			// Current post
			case 'current-post':
				if ( $post instanceof WP_Post ) {
					switch ( $value ) {
						case 'post-featured-image':
							$has_thumbnail = has_post_thumbnail( $post );

							return $this->compare( $has_thumbnail, $operator, true );

						case 'post-type':
							$post_type     = get_post_type( $post );
							$allowed_types = $sub_values['post_type'] ?? [];

							return $this->compare( in_array( $post_type, $allowed_types, true ), $operator, true );

						case 'post-meta':
							if ( isset( $sub_values['meta_key'], $sub_values['meta_value'] ) ) {
								$meta_value = get_post_meta( $post->ID, $sub_values['meta_key'], true );

								return $this->compare( $meta_value, $operator, $sub_values['meta_value'] );
							}

							return false;

						case 'post-id':
							$post_id = $sub_values['post_id'] ?? null;

							if ( is_numeric( $post_id ) ) {
								$post_id = kalium_get_translated_object_id( $post_id, get_post( $post_id ) );
							}

							return $this->compare( $post_id, $operator, $post->ID );

						case 'post-format':
							$post_formats = $sub_values['post_format'] ?? [];

							return $this->compare( in_array( get_post_format( $post ), $post_formats ), $operator, true );

						case 'post-status':
							$status = $sub_values['post_status'] ?? '';

							return $this->compare( $status, $operator, get_post_status( $post ) );

						case 'post-author':
							$author_id = (int) $post->post_author;
							$authors   = $sub_values['post_author'] ?? [];

							return $this->compare( in_array( $author_id, $authors ), $operator, true );

						case 'post-term':
							$taxonomy = $sub_values['taxonomy'] ?? '';
							$terms    = $sub_values['terms'] ?? [];

							if ( ! empty( $terms ) ) {
								$terms = array_map( $this->translate_object_ids( 'TERM' ), $terms );
							}

							if ( $taxonomy && ! empty( $terms ) ) {
								$post_terms = get_the_terms( $post->ID, $taxonomy );

								if ( is_array( $post_terms ) ) {
									$post_terms = wp_list_pluck( $post_terms, 'term_id' );

									foreach ( $terms as $term_id ) {
										$term_children = get_term_children( $term_id, $taxonomy );

										if ( is_array( $term_children ) && ! empty( $term_children ) ) {
											$terms = array_merge( $terms, $term_children );
										}
									}

									return $this->compare( ! empty( array_intersect( $terms, $post_terms ) ), $operator, true );
								}
							}

							return false;

						case 'post-sticky':
							$is_sticky = is_sticky( $post->ID );

							return $this->compare( $is_sticky, $operator, true );
					}
				}

				return false;

			// User
			case 'user':
				if ( 'logged-in' === $value ) {
					return $this->compare( is_user_logged_in(), $operator, true );
				} elseif ( 'logged-out' === $value ) {
					return $this->compare( ! is_user_logged_in(), $operator, true );
				} elseif ( 'user-role' === $value && is_user_logged_in() ) {
					$current_user = wp_get_current_user();
					$role         = $sub_values['role'] ?? '';
					$capabilities = $sub_values['capabilities'] ?? [];

					$has_role = in_array( $role, $current_user->roles, true );

					if ( ! empty( $capabilities ) && is_array( $capabilities ) ) {
						foreach ( $capabilities as $capability ) {
							if ( $this->compare( $current_user->has_cap( $capability ), $operator, true ) ) {
								return true;
							}
						}

						return false;
					}

					return $this->compare( $has_role, $operator, true );
				}

				return false;

			// WooCommerce
			case 'woocommerce':
				if ( kalium()->is->woocommerce_active() ) {
					$checks = [
						'product-page'     => 'is_product',
						'products-page'    => 'is_shop',
						'product-category' => 'is_product_category',
						'product-tag'      => 'is_product_tag',
						'cart-page'        => 'is_cart',
						'checkout-page'    => 'is_checkout',
						'account-page'     => 'is_account_page',
					];

					if ( isset( $checks[ $value ] ) && function_exists( $checks[ $value ] ) ) {
						return $this->compare( call_user_func( $checks[ $value ] ), $operator, true );
					}
				}

				return false;

			// Date & Time
			case 'date-time':
				if ( 'date' === $value && ! empty( $sub_values['date'] ) ) {
					$date    = strtotime( substr( $sub_values['date'], 0, 10 ) );
					$current = strtotime( wp_date( 'Y-m-d' ) );

					if ( false === $date || false === $current ) {
						return false;
					}

					return $this->compare( $current, $operator, $date );
				} elseif ( 'time' === $value && ! empty( $sub_values['time'] ) ) {
					$hours   = absint( $sub_values['time']['hours'] ?? 0 );
					$minutes = absint( $sub_values['time']['minutes'] ?? 0 );

					$now    = (int) wp_date( 'G' ) * 3600 + (int) wp_date( 'i' ) * 60;
					$target = $hours * 3600 + $minutes * 60;

					return $this->compare( $now, $operator, $target );
				} elseif ( 'days' === $value && ! empty( $sub_values['days'] ) && is_array( $sub_values['days'] ) ) {
					$current_day = strtolower( gmdate( 'l', current_time( 'timestamp' ) ) ); // Full day name in lowercase

					return $this->compare( in_array( $current_day, $sub_values['days'], true ), $operator, true );
				}

				return false;

			// Custom condition
			case 'custom-condition':
				if ( 'php-callback' === $value ) {
					$callback = $sub_values['callback'] ?? '';

					if ( is_string( $callback ) && is_callable( $callback ) ) {
						$disallowed = [ 'eval', 'exec', 'passthru', 'shell_exec', 'system', 'proc_open', 'popen' ];

						if ( in_array( strtolower( $callback ), $disallowed, true ) ) {
							return false;
						}

						return $this->compare( (bool) call_user_func( $callback, $condition ), $operator, true );
					}
				} elseif ( 'url-query-string' === $value ) {
					if ( isset( $sub_values['key'], $sub_values['value'] ) ) {
						$actual = $_GET[ $sub_values['key'] ] ?? null;

						return $this->compare( $actual, $operator, $sub_values['value'] );
					}
				} elseif ( 'referrer' === $value ) {
					if ( isset( $_SERVER['HTTP_REFERER'], $sub_values['url'] ) ) {
						return $this->compare( $_SERVER['HTTP_REFERER'], $operator, $sub_values['url'] );
					}
				}

				return false;
		}

		return true;
	}

	/**
	 * Group display conditions into AND/OR nested structure.
	 *
	 * @param array $conditions
	 *
	 * @return array
	 */
	public function group_conditions( $conditions ) {
		$groups      = [];
		$current_set = [];

		if ( is_array( $conditions ) ) {
			foreach ( $conditions as $index => $condition ) {
				$relation = strtoupper( $condition['relation'] ?? 'OR' );
				unset( $condition['relation'] );

				if ( 0 === $index ) {
					$current_set[] = $condition;
					continue;
				}

				if ( $relation === 'AND' ) {
					$current_set[] = $condition;
				} else {
					$groups[]    = $current_set;
					$current_set = [ $condition ];
				}
			}

			if ( ! empty( $current_set ) ) {
				$groups[] = $current_set;
			}
		}

		return $groups;
	}

	/**
	 * Check display conditions.
	 *
	 * @param array $condition_groups
	 *
	 * @return bool
	 */
	public function check_conditions( $conditions ) {
		$condition_groups = $this->group_conditions( $conditions );

		foreach ( $condition_groups as $group ) {
			$results = array_map( [ $this, 'check' ], $group );

			if ( ! in_array( false, $results, true ) ) {
				return true;
			}
		}

		return false;
	}

	/**
	 * Get available post types for select list.
	 *
	 * @param bool $include_builtin
	 * @param bool $singular_label
	 *
	 * @return array
	 */
	public function get_post_types_choices( $include_builtin = false, $singular_label = false ) {
		$args = [
			'public'      => true,
			'has_archive' => true,
			'_builtin'    => false,
		];

		$post_types_objects = get_post_types( $args, 'objects' );

		// Explicitly include 'post' and 'page' if $include_builtin is true
		if ( $include_builtin ) {
			foreach ( [ 'page', 'post' ] as $builtin ) {
				if ( post_type_exists( $builtin ) && ! isset( $post_types_objects[ $builtin ] ) ) {
					$post_types_objects = [ $builtin => get_post_type_object( $builtin ) ] + $post_types_objects;
				}
			}
		}

		// Do not include template parts post type
		unset( $post_types_objects[ Template_Parts::POST_TYPE ] );

		$post_types = array_map(
			function ( $post_type ) use ( $singular_label ) {
				return [
					'value' => $post_type->name,
					'label' => $singular_label ? $post_type->labels->singular_name : $post_type->labels->name,
				];
			},
			$post_types_objects
		);

		$post_types = array_values( $post_types );

		return apply_filters( 'kalium_display_conditions_post_types_choices', $post_types );
	}

	/**
	 * Get available taxonomies for select list.
	 *
	 * @param bool $include_builtin
	 *
	 * @return array
	 */
	public function get_taxonomy_choices( $include_builtin = false ) {
		$args = [
			'public'            => true,
			'show_ui'           => true,
			'show_in_nav_menus' => true,
		];

		if ( ! $include_builtin ) {
			$args['_builtin'] = false;
		}

		$taxonomy_objects = get_taxonomies( $args, 'objects' );

		$taxonomy_objects = array_filter(
			$taxonomy_objects,
			function ( $taxonomy ) {
				return $taxonomy->rewrite && ! empty( $taxonomy->rewrite['slug'] );
			}
		);

		$taxonomies = array_map(
			function ( $taxonomy ) {
				return [
					'value' => $taxonomy->name,
					'label' => $taxonomy->labels->name,
				];
			},
			$taxonomy_objects
		);

		$taxonomies = array_values( $taxonomies );

		return apply_filters( 'kalium_display_conditions_taxonomy_choices', $taxonomies );
	}

	/**
	 * Register REST API endpoints for dynamic data used in conditions.
	 */
	public function rest_endpoints() {
		register_rest_route(
			kalium_rest_namespace(),
			'/display-conditions/source',
			[
				'methods'             => [
					'GET',
					'POST',
				],
				'callback'            => [ $this, 'handle_dynamic_data_request' ],
				'permission_callback' => kalium_hook_return_value( 'install_themes' ),
			]
		);
	}

	/**
	 * Handle dynamic data request for object types.
	 *
	 * @param \WP_REST_Request $request
	 *
	 * @return \WP_Error|\WP_REST_Response
	 */
	public function handle_dynamic_data_request( $request ) {
		$object_type = $request->get_param( 'object_type' );

		switch ( $object_type ) {
			case 'post_type':
				return $this->get_post_type_items( $request );

			case 'taxonomy':
				return $this->get_taxonomy_items( $request );

			case 'user':
				return $this->get_user_items( $request );

			default:
				return new \WP_Error( 'invalid_object_type', 'Invalid object type.', [ 'status' => 400 ] );
		}
	}

	/**
	 * Get post type items for the specified post type.
	 *
	 * @param \WP_REST_Request $request
	 *
	 * @return \WP_Error|\WP_REST_Response
	 */
	protected function get_post_type_items( $request ) {
		$post_type = $request->get_param( 'post_type' );

		if ( ! post_type_exists( $post_type ) ) {
			return new \WP_Error( 'invalid_post_type', 'Invalid post type.', [ 'status' => 400 ] );
		}

		$args = wp_parse_args(
			$request->get_param( 'object_args' ),
			[
				'post_type'      => $post_type,
				'post_status'    => 'publish',
				'posts_per_page' => 100,
				'orderby'        => 'title',
				'order'          => 'ASC',
			]
		);

		$query = new \WP_Query( $args );

		$items = [];

		foreach ( $query->posts as $post ) {
			$items[] = [
				'value' => $post->ID,
				'label' => html_entity_decode( get_the_title( $post ) ),
			];
		}

		return rest_ensure_response( $items );
	}

	/**
	 * Get taxonomy items for the specified taxonomy.
	 *
	 * @param \WP_REST_Request $request
	 *
	 * @return \WP_Error|\WP_REST_Response
	 */
	protected function get_taxonomy_items( $request ) {
		$taxonomy = $request->get_param( 'taxonomy' );

		if ( ! taxonomy_exists( $taxonomy ) ) {
			return new \WP_Error( 'invalid_taxonomy', 'Invalid taxonomy.', [ 'status' => 400 ] );
		}

		$args = wp_parse_args(
			$request->get_param( 'object_args' ),
			[
				'taxonomy'   => $taxonomy,
				'hide_empty' => false,
				'number'     => 100,
				'orderby'    => 'name',
				'order'      => 'ASC',
			]
		);

		$terms = get_terms( $args );

		$items = [];

		foreach ( $terms as $term ) {
			$items[] = [
				'value' => $term->term_id,
				'label' => $term->name,
			];
		}

		return rest_ensure_response( $items );
	}

	/**
	 * Get use items for the specified query.
	 *
	 * @param \WP_REST_Request $request
	 *
	 * @return \WP_Error|\WP_REST_Response
	 */
	protected function get_user_items( $request ) {
		$args = wp_parse_args(
			$request->get_param( 'object_args' ),
			[
				'orderby' => 'display_name',
				'order'   => 'ASC',
				'number'  => 100,
			]
		);

		$users_query = new \WP_User_Query( $args );
		$users       = $users_query->get_results();

		$items = [];

		foreach ( $users as $user ) {
			$items[] = [
				'value' => $user->ID,
				'label' => $user->display_name,
			];
		}

		return rest_ensure_response( $items );
	}

	/**
	 * Auto translate object IDs.
	 *
	 * @param string $type
	 *
	 * @return callable
	 * @since 4.2.2
	 */
	public function translate_object_ids( $type = 'POST' ) {
		$type = strtoupper( $type );

		return function ( $id ) use ( $type ) {
			if ( kalium()->is->multilingual() ) {
				if ( $id instanceof \WP_Post || $id instanceof \WP_Term ) {
					$id = kalium_get_translated_object_id( $id->ID, $id, true );
				} elseif ( 'POST' === $type && is_numeric( $id ) ) {
					$id = kalium_get_translated_object_id( $id, get_post( $id ), true );
				} elseif ( 'TERM' === $type && is_numeric( $id ) ) {
					$id = kalium_get_translated_object_id( $id, get_term( $id ), true );
				}
			}

			return $id;
		};
	}
}
