<?php
/**
 * Hook locations.
 *
 * @author Laborator
 * @link   https://kaliumtheme.com
 * @since 4.2
 */
namespace Kalium\Template_Parts;

use Kalium\Admin\Dashboard;

class Hook_Locations {

	/**
	 * Parsed hook locations.
	 *
	 * @var array
	 */
	private $hook_locations = [];

	/**
	 * Construct.
	 */
	public function __construct() {
		add_action( 'after_setup_theme', [ $this, 'init' ] );
		add_action( 'init', [ $this, 'view_hook_locations' ] );
	}

	/**
	 * Init.
	 */
	public function init() {
		// Parse registered hooks
		foreach ( $this->get_locations() as $group ) {
			foreach ( $group['locations'] as $location ) {
				if ( empty( $location['hooks'] ) ) {
					$location['hooks'] = [
						$location['name'],
					];
				}

				$this->hook_locations[ $location['name'] ] = [
					'label'         => $location['label'],
					'isolated_hook' => $this->get_isolated_hook_name( $location['name'] ),
					'hooks'         => array_map(
						function ( $hook_name, $args ) {
							if ( is_numeric( $hook_name ) && is_string( $args ) ) {
									$hook_name = $args;
									$args      = [];
							} elseif ( is_numeric( $args ) ) {
								$args = [
									'priority' => $args,
								];
							}

							$args = wp_parse_args(
								$args,
								[
									'priority'  => 10,
									'isolated'  => false,
									'container' => [],
								]
							);

							$container = wp_parse_args(
								$args['container'],
								[
									'before' => '',
									'after'  => '',
								]
							);

							return [
								'name'      => $hook_name,
								'priority'  => $args['priority'],
								'isolated'  => $args['isolated'],
								'container' => $container,
							];
						},
						array_keys( $location['hooks'] ),
						array_values( $location['hooks'] )
					),
				];
			}
		}

		// Create isolated hooks locations
		foreach ( $this->hook_locations as $hook_location ) {
			foreach ( $hook_location['hooks'] as $hook ) {
				if ( $hook['isolated'] ) {
					add_action( $hook['name'], $this->create_hook_location( $hook_location['isolated_hook'] ), $hook['priority'] );
				}
			}
		}
	}

	/**
	 * Add section hook.
	 *
	 * @param string $name
	 * @param int    $priority
	 * @param string $section_id
	 * @param array  $check_conditions
	 */
	public function add_section_hook( $name, $priority, $section_id, $check_conditions = true ) {
		// Custom hook location
		if ( ! isset( $this->hook_locations[ $name ] ) ) {
			$this->hook_locations[ $name ] = [
				'hooks' => [
					[
						'isolated' => false,
						'name'     => $name,
					],
				],
			];
		}

		$location    = $this->hook_locations[ $name ];
		$added_hooks = [];

		foreach ( $location['hooks'] as $hook ) {
			$hook_name = $hook['isolated'] ? $location['isolated_hook'] : $hook['name'];

			if ( $hook['isolated'] && in_array( $hook_name, $added_hooks, true ) ) {
				continue;
			}

			add_action( $hook_name, $this->display_section_part( $section_id, [ 'check_conditions' => $check_conditions ] ), $priority );

			$added_hooks[] = $hook_name;
		}
	}

	/**
	 * Display section callable.
	 *
	 * @param string $section_id
	 * @param array  $args
	 *
	 * @return callable
	 */
	public function display_section_part( $section_id, $args ) {
		$section_hook_title = $this->is_view_theme_hooks() ? ( '[Template Part Section] ' . get_the_title( $section_id ) . ' #' . $section_id ) : '';

		return static function () use ( $section_id, $args, $section_hook_title ) {
			kalium()->template_parts->render_template_part( $section_id, $args );
		};
	}

	/**
	 * View theme hooks mode.
	 *
	 * @return bool
	 */
	public function is_view_theme_hooks() {
		return kalium()->request->has( 'view-theme-hooks' ) && ! is_admin() && current_user_can( Dashboard::MENU_CAP );
	}

	/**
	 * View hook locations.
	 */
	public function view_hook_locations() {
		if ( false === $this->is_view_theme_hooks() ) {
			return;
		}

		kalium_enqueue( 'template-parts-frontend' );

		$assigned_hooks = [];

		foreach ( $this->hook_locations as $location_name => $hook_location ) {
			foreach ( $hook_location['hooks'] as $hook ) {
				$hook_name = $hook['isolated'] ? $hook_location['isolated_hook'] : $hook['name'];

				if ( ! in_array( $hook_name, $assigned_hooks, true ) ) {
					add_action(
						$hook_name,
						$this->view_hook_location(
							$location_name,
							$hook_location,
							$hook,
							$hook_location['label']
						),
						$hook['priority']
					);

					$assigned_hooks[] = $hook_name;
				}
			}
		}

		// Disable sticky header when viewing hook locations
		kalium_replace_theme_option( 'sticky_header', '__return_false', 1000 );
	}

	/**
	 * Returns qualified hook name used for isolated hook locations.
	 *
	 * @param string $name
	 *
	 * @return string
	 */
	public function get_isolated_hook_name( $name ) {
		return "_kalium_hook_location_{$name}_isolated";
	}

	/**
	 * Create isolated hook location.
	 *
	 * @param string $location_name
	 *
	 * @return callable
	 */
	public function create_hook_location( $location_name ) {
		$callback = static function ( $value ) use ( $location_name ) {
			do_action( $location_name );
			return $value;
		};

		return \Closure::bind( $callback, null, null );
	}

	/**
	 * View hook location.
	 *
	 * @param string $location_name
	 * @param array  $hook_location
	 * @param array  $hook
	 * @param string $label
	 *
	 * @return callable
	 */
	public function view_hook_location( $location_name, $hook_location, $hook, $label ) {
		$class_instance = & $this;

		return $view_hook_location = static function ( $value ) use ( $location_name, $hook_location, $hook, $label, &$class_instance, &$view_hook_location ) {
			global $wp_filter;

			$is_isolated = $hook['isolated'];
			$hook_name   = $is_isolated ? $hook_location['isolated_hook'] : $hook['name'];
			$hooks_count = 0;

			// Remove self
			remove_action( $hook['name'], $view_hook_location, $hook['priority'] );

			$hooks = isset( $wp_filter[ $hook_name ] ) ? $wp_filter[ $hook_name ]->callbacks : [];

			// Remove isolated hook closure from display
			if ( $is_isolated ) {
				foreach ( $hooks as $priority => $callbacks ) {
					$hooks[ $priority ] = array_filter(
						$callbacks,
						function ( $callback ) use ( $hook_name, &$class_instance ) {
							if ( ! empty( $callback['function'] ) && $callback['function'] instanceof \Closure ) {
								$reflection = new \ReflectionFunction( $callback['function'] );
								$class_vars = $reflection->getStaticVariables();

								if ( isset( $class_vars['location_name'] ) ) {
									$isolated_hook_name = $class_instance->get_isolated_hook_name( $class_vars['location_name'] );

									return $isolated_hook_name !== $hook_name;
								}
							}

							return true;
						}
					);
				}
			}

			// Count hooks
			foreach ( $hooks as $callbacks ) {
				$hooks_count += count( $callbacks );
			}

			// Container before
			$class_instance->container_before( $hook );

			printf(
				'<div class="hook-location-view" data-location="%2$s" data-hook="%3$s">
							<div class="hook-location-view__inner">
								<div class="hook-location-view__label">%1$s</div>
								<div class="hook-location-view__actions">
									<button type="button" class="badge badge-outline badge-info" data-hooks="%4$s">Hooks</button>
									<button type="button" class="badge badge-outline badge-info" data-select="%2$s">Select</button>
								</div>
								%5$s
							</div>
						</div>',
				esc_html( $label ),
				esc_attr( $location_name ),
				esc_attr( $hook['name'] ),
				esc_attr( $hooks_count ),
				$class_instance->list_hooked_callbacks( $hook['name'], $hooks )
			);

			// Container After
			$class_instance->container_after( $hook );

			// Show again in non-isolated hooks
			if ( ! $is_isolated ) {
				add_action( $hook['name'], $view_hook_location, $hook['priority'] );
			}

			return $value;
		};
	}

	/**
	 * List hooked callbacks.
	 *
	 * @param string $hook_name
	 * @param array  $hooks
	 *
	 * @return string
	 */
	public function list_hooked_callbacks( $hook_name, $hooks ) {
		if ( empty( $hooks ) || ! is_array( $hooks ) ) {
			return '';
		}

		return kalium()->helpers->capture_output(
			static function () use ( $hook_name, $hooks ) {
				echo '<ul class="hook-location-view__callback-list">';

				foreach ( $hooks as $priority => $callbacks ) {
					foreach ( $callbacks as $callback ) {
						if ( is_array( $callback['function'] ) ) {
							$function_name = is_object( $callback['function'][0] ) ? get_class( $callback['function'][0] ) . '->' . $callback['function'][1] : $callback['function'][0] . '::' . $callback['function'][1];
						} elseif ( is_string( $callback['function'] ) ) {
							$function_name = $callback['function'];
						} elseif ( $callback['function'] instanceof \Closure ) {
							$reflection  = new \ReflectionFunction( $callback['function'] );
							$static_vars = $reflection->getStaticVariables();

							$function_name = $static_vars['section_hook_title'] ?? 'Closure or anonymous function';
						} else {
							$function_name = 'Unknown callback type';
						}

						printf(
							'<li class="hook-location-view__callback-item">
										<div class="hook-location-view__callback-name">%1$s</div>
										<div class="hook-location-view__callback-priority">%2$d</div>
									</li>',
							esc_html( $function_name ),
							esc_html( $priority )
						);
					}
				}

				echo '</ul>';
			}
		);
	}

	/**
	 * Get available hook locations.
	 *
	 * @return array
	 */
	public function get_locations() {
		return apply_filters(
			'kalium_template_parts_hook_locations',
			[
				// Head
				[
					'name'      => 'Head',
					'locations' => [
						[
							'name'  => 'wp_head',
							'label' => 'WP Head',
						],
						[
							'name'  => 'wp_body_open',
							'label' => 'WP Body Open',
						],
					],
				],

				// Body
				[
					'name'      => 'Body',
					'locations' => [
						[
							'name'  => 'wrapper_before',
							'label' => 'Wrapper Before',
							'hooks' => [
								'kalium_before_wrapper',
							],
						],
						[
							'name'  => 'wrapper_after',
							'label' => 'Wrapper After',
							'hooks' => [
								'kalium_after_wrapper',
							],
						],
					],
				],

				// Header
				[
					'name'      => 'Header',
					'locations' => [
						[
							'name'  => 'header_before',
							'label' => 'Header Before',
							'hooks' => [
								'kalium_wrapper_start' => [
									'priority' => 5,
									'isolated' => true,
								],
							],
						],
						[
							'name'  => 'header_after',
							'label' => 'Header After',
							'hooks' => [
								'kalium_wrapper_start' => [
									'priority' => 15,
									'isolated' => true,
								],
							],
						],
					],
				],

				// Footer
				[
					'name'      => 'Footer',
					'locations' => [
						[
							'name'  => 'footer_before',
							'label' => 'Footer Before',
							'hooks' => [
								'kalium_wrapper_end' => [
									'priority' => 5,
									'isolated' => true,
								],
							],
						],
						[
							'name'  => 'footer_after',
							'label' => 'Footer After',
							'hooks' => [
								'kalium_wrapper_end' => [
									'priority' => 15,
									'isolated' => true,
								],
							],
						],
						[
							'name'  => 'footer_start',
							'label' => 'Footer Start',
							'hooks' => [
								'kalium_footer' => [
									'priority' => 5,
									'isolated' => true,
								],
							],
						],
						[
							'name'  => 'footer_end',
							'label' => 'Footer End',
							'hooks' => [
								'kalium_footer' => [
									'priority' => 90,
									'isolated' => true,
								],
							],
						],
						[
							'name'  => 'wp_footer',
							'label' => 'WP Footer',
						],
					],
				],

				// Content
				[
					'name'      => 'Content',
					'locations' => [
						[
							'name'  => 'content_before',
							'label' => 'Main Content Before',
							'hooks' => [
								'kalium_page_content_wrapper_before' => 10,
								'kalium_blog_archive_content_wrapper_before' => 10,
								'kalium_blog_single_content_wrapper_before' => 10,
								'kalium_search_content_wrapper_before' => 10,
							],
						],
						[
							'name'  => 'content_after',
							'label' => 'Main Content After',
							'hooks' => [
								'kalium_page_content_wrapper_after' => 90,
								'kalium_blog_archive_content_wrapper_after' => 90,
								'kalium_blog_single_content_wrapper_after' => 90,
								'kalium_search_content_wrapper_after' => 90,
							],
						],
					],
				],

				// Loop
				[
					'name'      => 'Loop',
					'locations' => [
						[
							'name'  => 'loop_before',
							'label' => 'Loop Before',
							'hooks' => [
								'kalium_blog_loop_before' => [
									'priority' => 10,
								],
								'kalium_portfolio_loop_before' => [
									'priority' => 10,
								],
							],
						],
						[
							'name'  => 'loop_after',
							'label' => 'Loop After',
							'hooks' => [
								'kalium_blog_loop_after' => [
									'priority' => 90,
								],
								'kalium_portfolio_loop_after' => [
									'priority' => 90,
								],
							],
						],
						[
							'name'  => 'loop_card_before',
							'label' => 'Loop Card Before',
							'hooks' => [
								'kalium_blog_loop_post_before' => [
									'priority' => 0,
									'isolated' => true,
								],
								'kalium_portfolio_loop_item' => [
									'priority' => 0,
									'isolated' => true,
								],
							],
						],
						[
							'name'  => 'loop_card_after',
							'label' => 'Loop Card After',
							'hooks' => [
								'kalium_blog_loop_post_after' => [
									'priority' => 90,
									'isolated' => true,
								],
								'kalium_portfolio_loop_item' => [
									'priority' => 90,
									'isolated' => true,
								],
							],
						],
					],
				],

				// Sidebar
				[
					'name'      => 'Sidebar',
					'locations' => [
						[
							'name'  => 'sidebar_before',
							'label' => 'Sidebar Before',
							'hooks' => [
								'kalium_dynamic_sidebar_before',
							],
						],
						[
							'name'  => 'sidebar_after',
							'label' => 'Sidebar After',
							'hooks' => [
								'kalium_dynamic_sidebar_after',
							],
						],
					],
				],

				// Blog - Archive
				[
					'name'      => 'Blog - Archive',
					'locations' => [
						[
							'name'  => 'blog_loop_post_thumbnail_before',
							'label' => 'Thumbnail Before',
							'hooks' => [
								'kalium_blog_loop_before_post_thumbnail' => 0,
							],
						],
						[
							'name'  => 'blog_loop_post_thumbnail_after',
							'label' => 'Thumbnail After',
							'hooks' => [
								'kalium_blog_loop_after_post_thumbnail' => [
									'priority' => 90,
									'isolated' => true,
								],
							],
						],
						[
							'name'  => 'blog_loop_post_details',
							'label' => 'Post Details',
							'hooks' => [
								'kalium_blog_loop_post_details' => 0,
							],
						],
					],
				],

				// Blog - Single
				[
					'name'      => 'Blog - Single',
					'locations' => [
						[
							'name'  => 'blog_single_post_start',
							'label' => 'Post Start',
							'hooks' => [
								'kalium_blog_single_post_start' => 0,
							],
						],
						[
							'name'  => 'blog_single_post_end',
							'label' => 'Post End',
							'hooks' => [
								'kalium_blog_single_post_end' => 90,
							],
						],
						[
							'name'  => 'blog_single_post_thumbnail_before',
							'label' => 'Featured Image Before',
							'hooks' => [
								'kalium_blog_single_before_post_image' => 0,
							],
						],
						[
							'name'  => 'blog_single_post_thumbnail_after',
							'label' => 'Featured Image After',
							'hooks' => [
								'kalium_blog_single_after_post_image' => 90,
							],
						],
						[
							'name'  => 'blog_single_post_header',
							'label' => 'Post Header',
							'hooks' => [
								'kalium_blog_single_post_header' => 0,
							],
						],
						[
							'name'  => 'blog_single_post_footer',
							'label' => 'Post Footer',
							'hooks' => [
								'kalium_blog_single_post_footer' => 0,
							],
						],
						[
							'name'  => 'blog_single_post_content_before',
							'label' => 'Post Content Before',
							'hooks' => [
								'kalium_blog_single_post_content' => [
									'priority' => 0,
									'isolated' => true,
								],
							],
						],
						[
							'name'  => 'blog_single_post_content_after',
							'label' => 'Post Content After',
							'hooks' => [
								'kalium_blog_single_post_content' => [
									'priority' => 90,
									'isolated' => true,
								],
							],
						],
					],
				],

				// Portfolio - Archive
				[
					'name'      => 'Portfolio - General',
					'locations' => [
						[
							'name'  => 'portfolio_loop_item_details',
							'label' => 'Loop Item Details',
							'hooks' => [
								'kalium_portfolio_loop_item_details' => 0,
							],
						],
						[
							'name'  => 'kalium_portfolio_lightbox_item_description',
							'label' => 'Lightbox Item Description',
							'hooks' => [
								'kalium_portfolio_lightbox_item_description' => 0,
							],
						],
					],
				],

				// Portfolio - Single
				[
					'name'      => 'Portfolio - Single',
					'locations' => [
						[
							'name'  => 'portfolio_single_before',
							'label' => 'Portfolio Item Before',
							'hooks' => [
								'kalium_portfolio_single_item_type-1' => [
									'priority' => 0,
									'isolated' => true,
								],
								'kalium_portfolio_single_item_type-2' => [
									'priority' => 0,
									'isolated' => true,
								],
								'kalium_portfolio_single_item_type-3' => [
									'priority' => 0,
									'isolated' => true,
								],
								'kalium_portfolio_single_item_type-4' => [
									'priority' => 0,
									'isolated' => true,
								],
								'kalium_portfolio_single_item_type-5' => [
									'priority' => 0,
									'isolated' => true,
								],
								'kalium_portfolio_single_item_type-6' => [
									'priority' => 0,
									'isolated' => true,
								],
								'kalium_portfolio_single_item_type-7' => [
									'priority' => 0,
									'isolated' => true,
								],
							],
						],
						[
							'name'  => 'portfolio_single_after',
							'label' => 'Portfolio Item After',
							'hooks' => [
								'kalium_portfolio_single_item_type-1' => [
									'priority' => 90,
									'isolated' => true,
								],
								'kalium_portfolio_single_item_type-2' => [
									'priority' => 90,
									'isolated' => true,
								],
								'kalium_portfolio_single_item_type-3' => [
									'priority' => 90,
									'isolated' => true,
								],
								'kalium_portfolio_single_item_type-4' => [
									'priority' => 90,
									'isolated' => true,
								],
								'kalium_portfolio_single_item_type-5' => [
									'priority' => 90,
									'isolated' => true,
								],
								'kalium_portfolio_single_item_type-6' => [
									'priority' => 90,
									'isolated' => true,
								],
								'kalium_portfolio_single_item_type-7' => [
									'priority' => 90,
									'isolated' => true,
								],
							],
						],
						[
							'name'  => 'portfolio_single_content',
							'label' => 'Portfolio Content',
							'hooks' => [
								'kalium_portfolio_single_type-1_content' => 0,
								'kalium_portfolio_single_type-2_content' => 0,
								'kalium_portfolio_single_type-3_content' => 0,
								'kalium_portfolio_single_type-4_content' => 0,
								'kalium_portfolio_single_type-5_content' => 0,
							],
						],
						[
							'name'  => 'portfolio_single_gallery_before',
							'label' => 'Portfolio Gallery Before',
							'hooks' => [
								'kalium_portfolio_single_gallery_before',
							],
						],
						[
							'name'  => 'portfolio_single_gallery_after',
							'label' => 'Portfolio Gallery After',
							'hooks' => [
								'kalium_portfolio_single_gallery_after',
							],
						],
					],
				],

				// Comments
				[
					'name'      => 'Comments',
					'locations' => [
						[
							'name'  => 'comments_before',
							'label' => 'Comments Before',
							'hooks' => [
								'kalium_comments_before',
							],
						],
						[
							'name'  => 'comments_after',
							'label' => 'Comments After',
							'hooks' => [
								'kalium_comments_after',
							],
						],
						[
							'name'  => 'comments_list_before',
							'label' => 'Comments List Before',
							'hooks' => [
								'kalium_comments_list_before',
							],
						],
						[
							'name'  => 'comments_text_before',
							'label' => 'Comments Text Before',
							'hooks' => [
								'kalium_blog_post_comment_text_before' => 10,
							],
						],
						[
							'name'  => 'comments_text_after',
							'label' => 'Comments Text After',
							'hooks' => [
								'kalium_blog_post_comment_text_after' => 90,
							],
						],
						[
							'name'  => 'comment_fields_before',
							'label' => 'Comment Fields Before',
							'hooks' => [
								'comment_form_before_fields',
								'comment_form_field_comment',
							],
						],
						[
							'name'  => 'comment_fields_after',
							'label' => 'Comment Fields After',
							'hooks' => [
								'comment_form_submit_button',
							],
						],
						[
							'name'  => 'comment_form_before',
							'label' => 'Comment Form Before',
							'hooks' => [
								'comment_form_before',
							],
						],
						[
							'name'  => 'comment_form_after',
							'label' => 'Comment Form After',
							'hooks' => [
								'comment_form_after',
							],
						],
					],
				],

				// Search
				[
					'name'      => 'Search Page',
					'locations' => [
						[
							'name'  => 'search_page_start',
							'label' => 'Search Page Start',
							'hooks' => [
								'kalium_search_page' => [
									'priority' => 15,
									'isolated' => true,
								],
							],
						],
						[
							'name'  => 'search_page_end',
							'label' => 'Search Page End',
							'hooks' => [
								'kalium_search_page' => [
									'priority' => 45,
									'isolated' => true,
								],
							],
						],
						[
							'name'  => 'search_results_before',
							'label' => 'Search Results Before',
							'hooks' => [
								'kalium_search_page' => [
									'priority' => 35,
									'isolated' => true,
								],
							],
						],
					],
				],

				// WooCommerce - General
				[
					'name'      => 'WooCommerce - General',
					'locations' => [
						[
							'name'  => 'woocommerce_before_main_content',
							'label' => 'Main Content Before',
							'hooks' => [
								'woocommerce_before_main_content' => [
									'priority' => 0,
									'isolated' => true,
								],
							],
						],
						[
							'name'  => 'woocommerce_after_main_content',
							'label' => 'Main Content After',
							'hooks' => [
								'woocommerce_after_main_content' => [
									'priority' => 10000,
									'isolated' => true,
								],
							],
						],
					],
				],

				// WooCommerce - Archive
				[
					'name'      => 'WooCommerce - Archive',
					'locations' => [
						[
							'name'  => 'woocommerce_heading_before',
							'label' => 'Shop Heading Before',
							'hooks' => [
								'kalium_woocommerce_shop_heading' => [
									'priority' => 5,
									'isolated' => true,
								],
							],
						],
						[
							'name'  => 'woocommerce_heading_after',
							'label' => 'Shop Heading After',
							'hooks' => [
								'kalium_woocommerce_shop_heading' => [
									'priority' => 10000,
									'isolated' => true,
								],
							],
						],
						[
							'name'  => 'woocommerce_loop_before',
							'label' => 'Loop Before',
							'hooks' => [
								'woocommerce_before_shop_loop' => [
									'priority' => 5,
									'isolated' => true,
								],
							],
						],
						[
							'name'  => 'woocommerce_loop_after',
							'label' => 'Loop After',
							'hooks' => [
								'woocommerce_after_shop_loop' => [
									'priority' => 90,
									'isolated' => true,
								],
							],
						],
						[
							'name'  => 'woocommerce_loop_card_before',
							'label' => 'Loop Card Before',
							'hooks' => [
								'kalium_woocommerce_shop_loop_item' => [
									'priority' => 0,
									'isolated' => true,
								],
							],
						],
						[
							'name'  => 'woocommerce_loop_card_after',
							'label' => 'Loop Card After',
							'hooks' => [
								'kalium_woocommerce_shop_loop_item' => [
									'priority' => 90,
									'isolated' => true,
								],
							],
						],
					],
				],

				// WooCommerce - Single
				[
					'name'      => 'WooCommerce - Single',
					'locations' => [
						[
							'name'  => 'woocommerce_before_single_product',
							'label' => 'Product Before',
						],
						[
							'name'  => 'woocommerce_after_single_product',
							'label' => 'Product After',
						],
						[
							'name'  => 'woocommerce_single_product_gallery_before',
							'label' => 'Product Gallery Before',
							'hooks' => [
								'woocommerce_before_single_product_summary' => [
									'priority' => 4,
									'isolated' => true,
								],
							],
						],
						[
							'name'  => 'woocommerce_single_product_gallery_after',
							'label' => 'Product Gallery After',
							'hooks' => [
								'woocommerce_before_single_product_summary' => [
									'priority' => 900,
									'isolated' => true,
								],
							],
						],
						[
							'name'  => 'woocommerce_single_product_summary',
							'label' => 'Product Summary',
							'hooks' => [
								'woocommerce_single_product_summary' => 0,
							],
						],
						[
							'name'  => 'woocommerce_product_summary_after',
							'label' => 'Product Summary After',
							'hooks' => [
								'woocommerce_after_single_product_summary' => [
									'priority' => 1,
									'isolated' => true,
								],
							],
						],
						[
							'name'  => 'woocommerce_product_meta_end',
							'label' => 'Product Meta',
						],
						[
							'name'  => 'woocommerce_share',
							'label' => 'Product Share',
						],
						[
							'name'  => 'product_tabs_before',
							'label' => 'Product Tabs Before',
							'hooks' => [
								'woocommerce_product_tabs' => [
									'priority' => 10,
									'isolated' => true,
								],
							],
						],
						[
							'name'  => 'product_tabs_after',
							'label' => 'Product Tabs After',
							'hooks' => [
								'woocommerce_product_after_tabs' => 15,
							],
						],
					],
				],

				// WooCommerce - Cart
				[
					'name'      => 'WooCommerce - Cart',
					'locations' => [
						[
							'name'  => 'woocommerce_before_cart',
							'label' => 'Cart Before',
							'hooks' => [
								'woocommerce_before_cart' => 10,
							],
						],
						[
							'name'  => 'woocommerce_after_cart',
							'label' => 'Cart After',
							'hooks' => [
								'woocommerce_after_cart' => [
									'priority' => 990,
									'isolated' => true,
								],
							],
						],
						[
							'name'  => 'woocommerce_before_cart_table',
							'label' => 'Cart Table Before',
						],
						[
							'name'  => 'woocommerce_after_cart_table',
							'label' => 'Cart Table After',
						],
						[
							'name'  => 'woocommerce_cart_collaterals_before',
							'label' => 'Cart Collaterals Before',
							'hooks' => [
								'woocommerce_cart_collaterals' => [
									'priority' => 5,
									'isolated' => true,
								],
							],
						],
						[
							'name'  => 'woocommerce_cart_collaterals_after',
							'label' => 'Cart Collaterals After',
							'hooks' => [
								'woocommerce_cart_collaterals' => [
									'priority' => 90,
									'isolated' => true,
								],
							],
						],
						[
							'name'  => 'woocommerce_before_cart_totals',
							'label' => 'Cart Totals Before',
						],
						[
							'name'  => 'woocommerce_proceed_to_checkout',
							'label' => 'Proceed to Checkout',
						],
						[
							'name'  => 'woocommerce_cart_is_empty',
							'label' => 'Cart Empty',
							'hooks' => [
								'woocommerce_cart_is_empty' => 0,
							],
						],
					],
				],

				// WooCommerce - Checkout
				[
					'name'      => 'WooCommerce - Checkout',
					'locations' => [
						[
							'name'  => 'woocommerce_before_checkout_form',
							'label' => 'Checkout Form Before',
						],
						[
							'name'  => 'woocommerce_after_checkout_form',
							'label' => 'Checkout Form After',
						],
						[
							'name'  => 'woocommerce_checkout_billing',
							'label' => 'Billing Details',
						],
						[
							'name'  => 'woocommerce_checkout_shipping',
							'label' => 'Shipping Details',
						],
						[
							'name'  => 'woocommerce_before_order_notes',
							'label' => 'Order Notes Before',
						],
						[
							'name'  => 'woocommerce_after_order_notes',
							'label' => 'Order Notes After',
						],
						[
							'name'  => 'woocommerce_checkout_order_review_before',
							'label' => 'Order Review Before',
							'hooks' => [
								'woocommerce_checkout_order_review' => [
									'priority' => 5,
									'isolated' => true,
								],
							],
						],
						[
							'name'  => 'woocommerce_review_order_before_order_total',
							'label' => 'Order Review Total Before',
							'hooks' => [
								'woocommerce_review_order_before_order_total' => [
									'container' => [
										'before' => '<tr><td colspan="2">',
										'after'  => '</td></tr>',
									],
								],
							],
						],
						[
							'name'  => 'woocommerce_review_order_after_order_total',
							'label' => 'Order Review Total After',
							'hooks' => [
								'woocommerce_review_order_after_order_total' => [
									'container' => [
										'before' => '<tr><td colspan="2">',
										'after'  => '</td></tr>',
									],
								],
							],
						],
						[
							'name'  => 'woocommerce_review_order_before_submit',
							'label' => 'Order Review Submit Before',
							'hooks' => [
								'woocommerce_review_order_before_submit',
							],
						],
						[
							'name'  => 'woocommerce_review_order_after_submit',
							'label' => 'Order Review Submit After',
							'hooks' => [
								'woocommerce_review_order_after_submit',
							],
						],
					],
				],

				// WooCommerce - My Account
				[
					'name'      => 'WooCommerce - My Account',
					'locations' => [
						[
							'name'  => 'woocommerce_account_content_before',
							'label' => 'Account Content Before',
							'hooks' => [
								'woocommerce_account_content' => [
									'priority' => 0,
									'isolated' => true,
								],
							],
						],
						[
							'name'  => 'woocommerce_account_content_after',
							'label' => 'Account Content After',
							'hooks' => [
								'woocommerce_account_content' => [
									'priority' => 900,
									'isolated' => true,
								],
							],
						],
						[
							'name'  => 'woocommerce_account_navigation_before',
							'label' => 'Navigation Before',
							'hooks' => [
								'woocommerce_before_account_navigation' => [
									'priority' => 10,
									'isolated' => true,
								],
							],
						],
						[
							'name'  => 'woocommerce_account_navigation_after',
							'label' => 'Navigation After',
							'hooks' => [
								'woocommerce_after_account_navigation' => [
									'priority' => 900,
									'isolated' => true,
								],
							],
						],
						[
							'name'  => 'woocommerce_login_form_start',
							'label' => 'Login Form Start',
						],
						[
							'name'  => 'woocommerce_login_form_end',
							'label' => 'Login Form End',
						],
						[
							'name'  => 'woocommerce_login_form',
							'label' => 'Login Form',
						],
						[
							'name'  => 'woocommerce_register_form_start',
							'label' => 'Register Form Start',
						],
						[
							'name'  => 'woocommerce_register_form',
							'label' => 'Register Form End',
						],
						[
							'name'  => 'woocommerce_register_form_end',
							'label' => 'Register Form',
						],
					],
				],

				// Other
				[
					'name'      => 'Other',
					'locations' => [
						[
							'name'  => 'kalium_header_fullscreen_menu_before',
							'label' => 'Fullscreen Menu Before',
						],
						[
							'name'  => 'kalium_header_fullscreen_menu_after',
							'label' => 'Fullscreen Menu After',
						],
						[
							'name'  => 'kalium_header_off_canvas_side_menu_before',
							'label' => 'Off-Canvas Side Menu Before',
						],
						[
							'name'  => 'kalium_header_off_canvas_side_menu_after',
							'label' => 'Off-Canvas Side Menu After',
						],
						[
							'name'  => 'kalium_header_off_canvas_top_menu_before',
							'label' => 'Off-Canvas Top Menu Before',
						],
						[
							'name'  => 'kalium_header_off_canvas_top_menu_after',
							'label' => 'Off-Canvas Top Menu After',
						],
					],
				],
			]
		);
	}

	/**
	 * View hooks URL.
	 *
	 * @return string
	 */
	public function get_view_hooks_url() {
		return add_query_arg( 'view-theme-hooks', 1, home_url() );
	}

	/**
	 * Container before.
	 *
	 * @param array $hook
	 */
	private function container_before( $hook ) {
		if ( ! empty( $hook['container']['before'] ) ) {
			echo $hook['container']['before'];
		}
	}

	/**
	 * Container after.
	 *
	 * @param array $hook
	 */
	private function container_after( $hook ) {
		if ( ! empty( $hook['container']['after'] ) ) {
			echo $hook['container']['after'];
		}
	}
}
