<?php
/**
 * Template parts.
 *
 * @author Laborator
 * @link   https://kaliumtheme.com
 * @since 4.2
 */
namespace Kalium\Template_Parts;

use Kalium\Core\Settings;
use Kalium\Admin\Dashboard;
use Elementor\Frontend;
use Elementor\Plugin;
use WP_Style_Engine_CSS_Rules_Store;

class Template_Parts {

	/**
	 * Post type slug.
	 *
	 * @var string
	 */
	const POST_TYPE = 'kalium_template_part';

	/**
	 * Meta keys.
	 *
	 * @var array
	 */
	const META_KEYS = [
		'TYPE'               => 'type',
		'DISPLAY_CONDITIONS' => 'display_conditions',
		'PLACEMENT'          => 'placement',
		'CONTAINER_SETTINGS' => 'container_settings',
		'HEADER_SETTINGS'    => 'header_settings',
		'FOOTER_SETTINGS'    => 'footer_settings',
	];

	/**
	 * Option name to store the template part settings.
	 *
	 * @var string
	 */
	const OPTION_NAME = 'kalium_template_parts';

	/**
	 * Template parts settings.
	 *
	 * @return array
	 */
	public static function get_settings() {
		return wp_parse_args(
			get_option( self::OPTION_NAME ),
			[
				'enabled' => true,
			]
		);
	}

	/**
	 * Get single setting.
	 *
	 * @param string $name
	 *
	 * @return mixed
	 */
	public static function get_setting( $name ) {
		return self::get_settings()[ $name ] ?? null;
	}

	/**
	 * Template_Parts constructor.
	 */
	public function __construct() {
		// Settings
		add_action( 'kalium_settings_init', [ $this, 'register_settings' ] );
		add_action( 'kalium_dashboard_settings_sections', [ $this, 'register_settings_section' ] );
		add_action( 'kalium_settings_section_template-parts', [ $this, 'settings' ] );

		if ( kalium()->is->template_parts_active() ) {
			// Register post type
			add_action( 'init', [ $this, 'register_post_type' ] );

			// Register dedicated post meta
			add_action( 'init', [ $this, 'register_post_meta' ] );

			// Rest endpoints
			add_action( 'rest_api_init', [ $this, 'rest_endpoints' ] );

			// Display sections
			add_action( 'template_redirect', [ $this, 'display_sections' ] );
			add_action( 'template_redirect', [ $this, 'display_header' ] );
			add_action( 'template_redirect', [ $this, 'display_footer' ] );

			// Preview template part
			add_action( 'template_redirect', [ $this, 'preview' ], 10000 );

			// Register shortcode for using section template parts
			add_shortcode( 'kalium_section', [ $this, 'shortcode_section' ] );

			// Add the menu item
			add_action( 'admin_menu', [ $this, 'menu_item' ], 11 );

			// Shortcut in the site name links
			add_action( 'admin_bar_menu', [ $this, 'add_admin_bar_link' ], 40 );

			// Template parts type tabs
			add_action( 'views_edit-' . self::POST_TYPE, [ $this, 'display_tabs' ] );

			// Template parts admin columns
			add_filter( 'manage_edit-' . self::POST_TYPE . '_columns', [ $this, 'edit_columns' ], 1000 );
			add_action( 'manage_' . self::POST_TYPE . '_posts_custom_column', [ $this, 'posts_custom_column' ], 1000, 2 );

			// Add toggle active/inactive row action
			add_filter( 'post_row_actions', [ $this, 'action_toggle_status' ], 10, 2 );

			// Register bulk actions for activate/deactivate
			add_filter( 'bulk_actions-edit-' . self::POST_TYPE, [ $this, 'register_bulk_actions' ] );
			add_filter( 'handle_bulk_actions-edit-' . self::POST_TYPE, [ $this, 'handle_bulk_actions' ], 10, 3 );

			// Enqueue assets for the admin list page
			add_action( 'admin_enqueue_scripts', [ $this, 'enqueue_assets' ] );

			// Enqueue assets for the admin edit
			add_action( 'enqueue_block_editor_assets', [ $this, 'enqueue_post_edit_assets' ] );

			// Remove "Filter by date" dropdown in post type list screen
			add_filter( 'disable_months_dropdown', [ $this, 'disable_date_filter' ], 10, 2 );

			// Handle toggle status action for template parts
			add_action( 'admin_init', [ $this, 'handle_toggle_status_action' ] );

			// Filter main query to only show posts of the current type in admin list.
			add_action( 'pre_get_posts', [ $this, 'filter_type_query' ] );

			// Clear parts cache
			add_action( 'save_post_' . self::POST_TYPE, [ $this, 'clear_parts_cache' ] );
			add_action( 'deleted_post', [ $this, 'clear_parts_cache' ] );

			// Elementor support
			add_action( 'elementor/init', [ $this, 'add_elementor_support' ] );
			add_action( 'elementor/documents/register', [ $this, 'register_elementor_document' ] );

			// WPML compatibility
			add_filter( 'wpml_config_array', [ $this, 'wpml_config_array' ] );

			// Set the template part type on creation based on the current tp_type query string
			add_action( 'wp_insert_post', [ $this, 'assign_part_type' ], 10, 3 );
		}
	}

	/**
	 * Retrieve template parts.
	 *
	 * @param array $args
	 *
	 * @return array
	 */
	public function get_template_parts( $args = [] ) {
		$args = wp_parse_args(
			$args,
			[
				'type'           => null,
				'has_conditions' => false,
				'has_placement'  => false,
				'cache'          => true,
			]
		);

		// Use cache
		if ( $args['cache'] ) {
			$cache_key = self::OPTION_NAME . '_' . md5( serialize( $args ) );
			$entries   = get_transient( $cache_key );

			if ( false !== $entries ) {
				return $entries;
			}
		}

		// Query args
		$query_args = [
			'post_type'      => self::POST_TYPE,
			'post_status'    => 'publish',
			'posts_per_page' => -1,
			'orderby'        => 'menu_order',
			'order'          => 'ASC',
		];

		// Filter by conditions
		if ( $args['has_conditions'] ) {
			$query_args['meta_query'][] = [
				'key'     => self::META_KEYS['DISPLAY_CONDITIONS'],
				'value'   => 'a:0:{}',
				'compare' => '!=',
			];
		}

		// Filter by placement
		if ( $args['has_placement'] ) {
			$query_args['meta_query'][] = [
				'key'     => self::META_KEYS['PLACEMENT'],
				'value'   => 'a:0:{}',
				'compare' => '!=',
			];
		}

		// Query entries
		$query = new \WP_Query( $query_args );

		// Build entries
		$entries = [];

		foreach ( $query->posts as $post ) {
			$id      = $post->ID;
			$content = $post->post_content;
			$type    = $this->get_type( $id );

			// Filter by type
			if ( $args['type'] && $type !== $args['type'] ) {
				continue;
			}

			$entry = [
				'id'         => $id,
				'type'       => $type,
				'title'      => get_the_title( $id ),
				'content'    => $content,
				'conditions' => $this->get_display_conditions( $id ),
			];

			if ( 'section' === $type ) {
				$entry['placement'] = $this->get_placement( $id );
			}

			$entries[] = $entry;
		}

		if ( $args['cache'] ) {
			set_transient( $cache_key, $entries );
		}

		return $entries;
	}

	/**
	 * Template part types.
	 *
	 * @return array
	 */
	public function get_types() {
		return [
			'section' => [
				'label' => 'Section',
			],
			'header'  => [
				'label' => 'Header',
			],
			'footer'  => [
				'label' => 'Footer',
			],
			'pages'   => [
				'label'    => 'Pages',
				'disabled' => true,
			],
		];
	}

	/**
	 * Get type options.
	 *
	 * @param string $id
	 *
	 * @return array
	 */
	public function get_type_options( $id ) {
		return $this->get_types()[ $id ] ?? null;
	}

	/**
	 * Get the current viewing template part.
	 *
	 * @return string|null
	 */
	public function get_current_type() {
		$current_type = kalium()->request->query( 'tp_type' );

		return $current_type && in_array( $current_type, array_keys( $this->get_types() ), true ) ? $current_type : null;
	}

	/**
	 * Get type.
	 *
	 * @param int $post_id
	 *
	 * @return string
	 */
	public function get_type( $post_id ) {
		return get_post_meta( $post_id, self::META_KEYS['TYPE'], true );
	}

	/**
	 * Get display conditions.
	 *
	 * @param int  $post_id
	 * @param bool $grouped
	 *
	 * @return array
	 */
	public function get_display_conditions( $post_id, $grouped = false ) {
		$display_conditions = get_post_meta( $post_id, self::META_KEYS['DISPLAY_CONDITIONS'], true );

		if ( $grouped ) {
			return kalium()->display_conditions->group_conditions( $display_conditions );
		}

		return $display_conditions;
	}

	/**
	 * Get placement.
	 *
	 * @param int $post_id
	 *
	 * @return array
	 */
	public function get_placement( $post_id ) {
		return get_post_meta( $post_id, self::META_KEYS['PLACEMENT'], true );
	}

	/**
	 * Get container settings.
	 *
	 * @param int $post_id
	 *
	 * @return array
	 */
	public function get_container_settings( $post_id ) {
		return wp_parse_args(
			get_post_meta( $post_id, self::META_KEYS['CONTAINER_SETTINGS'], true ),
			[
				'container'  => true,
				'visibility' => null,
				'classes'    => '',
				'tag_name'   => '',
			]
		);
	}

	/**
	 * Get header settings.
	 *
	 * @param int $post_id
	 *
	 * @return array
	 */
	public function get_header_settings( $post_id ) {
		return wp_parse_args(
			get_post_meta( $post_id, self::META_KEYS['HEADER_SETTINGS'], true ),
			[
				'sticky'                      => false,
				'sticky_effect'               => 'none',
				'sticky_autohide'             => false,
				'sticky_progress_with_scroll' => true,
				'sticky_animation_duration'   => 0.3,
				'over_content'                => false,
				'content_spacing'             => '',
				'content_spacing_tablet'      => '',
				'content_spacing_mobile'      => '',
			]
		);
	}

	/**
	 * Get footer settings.
	 *
	 * @param int $post_id
	 *
	 * @return array
	 */
	public function get_footer_settings( $post_id ) {
		return wp_parse_args(
			get_post_meta( $post_id, self::META_KEYS['FOOTER_SETTINGS'], true ),
			[
				'fixed'  => false,
				'effect' => 'none',
			]
		);
	}

	/**
	 * Checks if current page is template parts edit page.
	 *
	 * @return bool
	 */
	public function is_edit_page() {
		global $pagenow;

		return 'edit.php' === $pagenow && self::POST_TYPE === kalium()->request->query( 'post_type' );
	}

	/**
	 * Register the custom post type for Template Parts.
	 */
	public function register_post_type() {
		register_post_type(
			self::POST_TYPE,
			[
				'label'               => 'Template Parts',
				'labels'              => [
					'singular_name' => 'Template Part',
					'add_new_item'  => 'Add New',
					'search_items'  => 'Search',
				],
				'supports'            => [ 'title', 'editor', 'custom-fields' ],
				'hierarchical'        => false,
				'public'              => true,
				'show_ui'             => true,
				'show_in_menu'        => false,
				'show_in_admin_bar'   => true,
				'show_in_nav_menus'   => false,
				'show_in_rest'        => true,
				'can_export'          => true,
				'has_archive'         => true,
				'exclude_from_search' => true,
				'publicly_queryable'  => true,
				'capability_type'     => 'page',
			]
		);

		if ( $this->is_edit_page() ) {
			$this->rename_draft_status_label();
		}
	}

	/**
	 * Register post meta.
	 */
	public function register_post_meta() {
		$auth_callback = static function () {
			return current_user_can( Dashboard::MENU_CAP );
		};

		// Type
		register_post_meta(
			self::POST_TYPE,
			self::META_KEYS['TYPE'],
			[
				'type'          => 'string',
				'show_in_rest'  => true,
				'single'        => true,
				'default'       => current( array_keys( $this->get_types() ) ),
				'auth_callback' => $auth_callback,
			]
		);

		// Display conditions
		register_post_meta(
			self::POST_TYPE,
			self::META_KEYS['DISPLAY_CONDITIONS'],
			[
				'type'          => 'array',
				'show_in_rest'  => [
					'schema' => [
						'type'  => 'array',
						'items' => [
							'type'       => 'object',
							'properties' => [
								'type'       => [ 'type' => 'string' ],
								'operator'   => [ 'type' => 'string' ],
								'value'      => [ 'type' => 'string' ],
								'relation'   => [ 'type' => 'string' ],
								'sub_values' => [
									'type'                 => [ 'object', 'null' ],
									'additionalProperties' => true,
								],
							],
							'required'   => [ 'type', 'operator', 'value' ],
						],
					],
				],
				'single'        => true,
				'auth_callback' => $auth_callback,
			]
		);

		// Placement
		register_post_meta(
			self::POST_TYPE,
			self::META_KEYS['PLACEMENT'],
			[
				'type'          => 'array',
				'show_in_rest'  => [
					'schema' => [
						'type'  => 'array',
						'items' => [
							'type'       => 'object',
							'properties' => [
								'name'     => [ 'type' => 'string' ],
								'priority' => [ 'type' => 'integer' ],
							],
							'required'   => [ 'name' ],
						],
					],
				],
				'single'        => true,
				'auth_callback' => $auth_callback,
			]
		);

		// Container settings
		register_post_meta(
			self::POST_TYPE,
			self::META_KEYS['CONTAINER_SETTINGS'],
			[
				'type'          => 'object',
				'show_in_rest'  => [
					'schema' => [
						'type'       => 'object',
						'properties' => [
							'container'  => [
								'type' => 'boolean',
							],
							'visibility' => [
								'type'  => 'array',
								'items' => [
									'type' => 'string',
								],
							],
							'classes'    => [
								'type' => 'string',
							],
							'tag_name'   => [
								'type' => 'string',
							],
						],
					],
				],
				'default'       => [
					'container'  => true,
					'visibility' => [
						'desktop',
						'tablet',
						'mobile',
					],
					'classes'    => '',
					'tag_name'   => '',
				],
				'single'        => true,
				'auth_callback' => $auth_callback,
			]
		);

		// Register header settings meta
		register_post_meta(
			self::POST_TYPE,
			self::META_KEYS['HEADER_SETTINGS'],
			[
				'type'          => 'object',
				'show_in_rest'  => [
					'schema' => [
						'type'       => 'object',
						'properties' => [
							'sticky'                      => [
								'type' => 'boolean',
							],
							'sticky_effect'               => [
								'type' => 'string',
							],
							'sticky_autohide'             => [
								'type' => 'boolean',
							],
							'sticky_progress_with_scroll' => [
								'type' => 'boolean',
							],
							'sticky_animation_duration'   => [
								'type' => 'number',
							],
							'over_content'                => [
								'type' => 'boolean',
							],
							'content_spacing'             => [
								'type' => 'string',
							],
							'content_spacing_tablet'      => [
								'type' => 'string',
							],
							'content_spacing_mobile'      => [
								'type' => 'string',
							],
						],
					],
				],
				'default'       => [
					'sticky'                      => false,
					'sticky_effect'               => 'none',
					'sticky_autohide'             => false,
					'sticky_progress_with_scroll' => true,
					'sticky_animation_duration'   => 0.3,
					'over_content'                => false,
					'content_spacing'             => '',
					'content_spacing_tablet'      => '',
					'content_spacing_mobile'      => '',
				],
				'single'        => true,
				'auth_callback' => $auth_callback,
			]
		);

		// Register footer settings meta
		register_post_meta(
			self::POST_TYPE,
			self::META_KEYS['FOOTER_SETTINGS'],
			[
				'type'          => 'object',
				'show_in_rest'  => [
					'schema' => [
						'type'       => 'object',
						'properties' => [
							'fixed'  => [
								'type' => 'boolean',
							],
							'effect' => [
								'type' => 'string',
							],
						],
					],
				],
				'default'       => [
					'fixed'  => false,
					'effect' => 'none',
				],
				'single'        => true,
				'auth_callback' => $auth_callback,
			]
		);
	}

	/**
	 * Register rest endpoints.
	 */
	public function rest_endpoints() {
		register_rest_route(
			kalium_rest_namespace(),
			'/template-parts/sections',
			[
				'methods'             => 'GET',
				'callback'            => [ $this, 'rest_response_sections' ],
				'permission_callback' => kalium_hook_return_value( 'install_themes' ),
			]
		);
	}

	/**
	 * Retrieve sections on REST.
	 *
	 * @return \WP_REST_Response
	 */
	public function rest_response_sections() {
		$sections = $this->get_template_parts( [ 'type' => 'section' ] );

		return rest_ensure_response( $sections );
	}

	/**
	 * Display active sections.
	 */
	public function display_sections() {
		$sections_parts = $this->get_template_parts(
			[
				'type'           => 'section',
				'has_conditions' => true,
				'has_placement'  => true,
			]
		);

		foreach ( $sections_parts as $sections_part ) {
			foreach ( $sections_part['placement'] as $location ) {
				// Process enqueues
				$this->process_enqueues( $sections_part['id'], true );

				// Display section
				kalium()->hook_locations->add_section_hook( $location['name'], $location['priority'], $sections_part['id'] );
			}
		}
	}

	/**
	 * Display active header.
	 *
	 * @since 4.3
	 */
	public function display_header() {
		$header_parts = $this->get_template_parts(
			[
				'type'           => 'header',
				'has_conditions' => true,
			]
		);

		// First one to match replaces the header
		foreach ( $header_parts as $header_part ) {
			$id = $header_part['id'];

			if ( kalium()->display_conditions->check_conditions( $this->get_display_conditions( $id ) ) ) {
				add_filter( 'kalium_show_header', '__return_false' );
				add_action(
					'kalium_wrapper_start',
					function () use ( $id ) {
						$this->render_template_part( $id );
					}
				);
				return;
			}
		}
	}

	/**
	 * Display active footer.
	 *
	 * @since 4.3
	 */
	public function display_footer() {
		$footer_parts = $this->get_template_parts(
			[
				'type'           => 'footer',
				'has_conditions' => true,
			]
		);

		// First one to match replaces the footer
		foreach ( $footer_parts as $footer_part ) {
			$id = $footer_part['id'];

			if ( kalium()->display_conditions->check_conditions( $this->get_display_conditions( $id ) ) ) {
				add_filter( 'kalium_show_footer', '__return_false' );
				add_action(
					'kalium_wrapper_end',
					function () use ( $id ) {
						$this->render_template_part( $id );
					}
				);
				return;
			}
		}
	}

	/**
	 * Preview template part.
	 */
	public function preview() {
		if ( ! is_singular( self::POST_TYPE ) ) {
			return;
		}

		the_post();

		?>
		<!DOCTYPE html>
		<html <?php language_attributes(); ?>>
		<head>
			<?php wp_head(); ?>
		</head>
		<body <?php body_class( 'template-part-preview' ); ?>>
			<main class="template-part-preview__main">
				<?php $this->render_template_part( get_the_ID() ); ?>
			</main>
			<?php wp_footer(); ?>
		</body>
		</html>
		<?php

		exit;
	}

	/**
	 * Render template part content by ID.
	 *
	 * @param int   $post_id
	 * @param array $args
	 *
	 * @return string|void
	 */
	public function render_template_part( $post_id, $args = [] ) {
		if ( ! $post_id || get_post_type( $post_id ) !== self::POST_TYPE ) {
			return;
		}

		// Args
		$args = wp_parse_args(
			$args,
			[
				'check_conditions' => false,
				'container'        => true,
				'echo'             => true,
			]
		);

		// Check conditions
		if ( $args['check_conditions'] && false === kalium()->display_conditions->check_conditions( $this->get_display_conditions( $post_id ) ) ) {
			return;
		}

		// Template part type
		$type = $this->get_type( $post_id );

		// Object ID
		$object_id = kalium_get_translated_object_id( $post_id, self::POST_TYPE, true );

		// Selector
		$selector = '.template-part[data-id="' . $post_id . '"]';

		// Custom styles
		$custom_styles = [
			'selector' => $selector,
			'vars'     => [],
			'props'    => [],
		];

		// Container
		$container = [
			'tag_name'   => 'div',
			'attributes' => [
				'data-type' => $type,
				'data-id'   => $post_id,
				'class'     => [
					'template-part',
				],
			],
		];

		$use_container = $args['container'];

		if ( $use_container ) {
			$container_settings = $this->get_container_settings( $post_id );

			$use_container = kalium_validate_boolean( $container_settings['container'] );
		}

		// Container
		if ( $use_container ) {
			$container['attributes']['class'] = array_merge(
				$container['attributes']['class'],
				kalium_get_device_visibility_classes( $container_settings['visibility'] )
			);

			if ( ! empty( $container_settings['classes'] ) ) {
				$container['attributes']['class'][] = $container_settings['classes'];
			}

			if ( ! empty( $container_settings['tag_name'] ) ) {
				$container['tag_name'] = $container_settings['tag_name'];
			}

			// Apply header settings
			if ( 'header' === $type ) {
				$header_settings = $this->get_header_settings( $post_id );

				// Sticky header
				if ( $header_settings['sticky'] ) {
					kalium_create_sticky_header(
						$selector,
						[
							'mode'             => $header_settings['sticky_autohide'] ? 'autohide' : 'standard',
							'effect'           => $header_settings['sticky_effect'],
							'animate'          => ! $header_settings['sticky_progress_with_scroll'],
							'animate_duration' => $header_settings['sticky_animation_duration'],
							'offset_start'     => [
								'#wpadminbar'  => 'height',
								'.page-border' => 'padding-top',
							],
						]
					);
				}

				// Header over content
				if ( $header_settings['over_content'] ) {
					$container['attributes']['class'][] = 'header-over-content';

					$header_spacer = static function ( $output ) use ( $header_settings, $selector, &$header_spacer ) {
						remove_filter( 'kalium_template_part_content', $header_spacer, 10 );

						$output .= kalium_render_element(
							[
								'tag_name'   => 'div',
								'attributes' => [
									'class' => [
										'header-spacer',
									],
								],
								'content'    => kalium_print_inline_style(
									[
										'selector' => $selector . ' + .header-spacer',
										'vars'     => [
											'spacing' => kalium_parse_responsive_value(
												array_filter(
													[
														'desktop' => $header_settings['content_spacing'],
														'tablet'  => $header_settings['content_spacing_tablet'],
														'mobile'  => $header_settings['content_spacing_mobile'],
													]
												)
											),
										],
									],
									false
								),
							],
							false
						);

						return $output;
					};

					add_filter( 'kalium_template_part_content', $header_spacer, 10 );
				}
			}

			// Apply footer settings
			if ( 'footer' === $type ) {
				$footer_settings = $this->get_footer_settings( $post_id );

				// Fixed footer
				if ( $footer_settings['fixed'] ) {
					$container['attributes']['class'][] = 'fixed-footer';

					if ( in_array( $footer_settings['effect'], [ 'fade', 'slide' ] ) ) {
						$container['attributes']['class'][] = 'fixed-footer--' . $footer_settings['effect'];
					}
				}
			}
		}

		// Post content
		$post_content = get_post_field( 'post_content', $object_id );

		// Handle differently on Elementor
		if ( kalium()->is->elementor_active() && get_queried_object_id() !== $object_id ) {
			$elementor_filters = [
				[
					'the_content',
					[
						Plugin::$instance->frontend,
						'apply_builder_in_content',
					],
					Frontend::THE_CONTENT_FILTER_PRIORITY,
				],
				[
					'the_content',
					[
						Plugin::$instance->preview,
						'builder_wrapper',
					],
					999999,
				],
			];

			// Remove filters
			foreach ( $elementor_filters as $filter ) {
				call_user_func_array( 'remove_filter', $filter );
			}

			// Render content
			if ( kalium()->elementor->is_built_with_elementor( $object_id ) && ! kalium()->elementor->is_preview( $object_id ) ) {
				$content = kalium()->elementor->get_builder_content( $object_id, true );
			} else {
				$content = apply_filters( 'the_content', $post_content );
			}

			// Restore filters
			foreach ( $elementor_filters as $filter ) {
				call_user_func_array( 'add_filter', $filter );
			}
		} else {
			$content = apply_filters( 'the_content', $post_content );
		}

		// Rendered content
		$output = kalium()->helpers->capture_output(
			function () use ( $container, $use_container, $post_id, $content, $custom_styles ) {
				if ( 'publish' !== get_post_status( $post_id ) || ! trim( $content ) ) {
					return;
				}

				// Wrapper start
				if ( $use_container ) {
					kalium_render_element( array_merge( $container, [ 'close' => false ] ) );
				}

				// Content
				echo $content;

				// Print custom styles
				kalium_print_inline_style( $custom_styles );

				// Print block supports styles
				$this->maybe_print_block_supports_styles();

				// Print GreenShift block styles
				$this->maybe_print_greenshift_block_styles( $post_id );

				// Print WPBakery element styles
				$this->maybe_print_wpbakery_block_styles( $post_id );

				// Wrapper end
				if ( $use_container ) {
					echo '</' . $container['tag_name'] . '>';
				}
			}
		);

		/**
		 * Filter template part content.
		 *
		 * @param string $output
		 * @param int    $post_id
		 * @param array  $args
		 *
		 * @since 4.2.2
		 */
		$output = apply_filters( 'kalium_template_part_content', $output, $post_id, $args );

		if ( $args['echo'] ) {
			echo $output;
			return;
		}

		return $output;
	}

	/**
	 * Process enqueues.
	 *
	 * @param int  $post_id
	 * @param bool $check_conditions
	 */
	public function process_enqueues( $post_id, $check_conditions = false ) {
		if ( ! $post_id || get_post_type( $post_id ) !== self::POST_TYPE ) {
			return;
		}

		// Check conditions
		if ( $check_conditions && false === kalium()->display_conditions->check_conditions( $this->get_display_conditions( $post_id ) ) ) {
			return;
		}

		// Enqueue elementor frontend styles
		if ( kalium()->is->elementor_active() && kalium()->elementor->is_built_with_elementor( $post_id ) ) {
			Plugin::$instance->frontend->enqueue_styles();
		}
	}

	/**
	 * Shortcode to render a template part section.
	 *
	 * @param array $attrs
	 *
	 * @return string
	 */
	public function shortcode_section( $attrs = [] ) {
		$attrs = shortcode_atts(
			[
				'id'               => 0,
				'check_conditions' => 'no',
			],
			$attrs,
			'kalium_section'
		);

		$post_id = absint( $attrs['id'] );

		// Not a template part or section type
		if ( ! $post_id || get_post_type( $post_id ) !== self::POST_TYPE || 'section' !== $this->get_type( $post_id ) ) {
			return '';
		}

		return $this->render_template_part(
			$post_id,
			[
				'check_conditions' => kalium_validate_boolean( $attrs['check_conditions'] ),
				'echo'             => false,
			]
		);
	}

	/**
	 * Clear cached template parts transients.
	 */
	public function clear_parts_cache() {
		global $wpdb;

		$transient_prefix = '_transient_' . self::OPTION_NAME . '_';

		$transients = $wpdb->get_col(
			$wpdb->prepare(
				"SELECT option_name FROM $wpdb->options WHERE option_name LIKE %s",
				$wpdb->esc_like( $transient_prefix ) . '%'
			)
		);

		foreach ( $transients as $transient ) {
			$key = str_replace( '_transient_', '', $transient );
			delete_transient( $key );
		}
	}

	/**
	 * Get edit posts page link.
	 *
	 * @return string
	 * @since 4.2.1
	 */
	public function get_edit_posts_link() {
		$link = 'edit.php?post_type=' . self::POST_TYPE;

		if ( ! is_admin() ) {
			$link = admin_url( $link );
		}

		return $link;
	}

	/**
	 * Add the Template Parts post type to the Kalium admin menu.
	 */
	public function menu_item() {
		add_submenu_page( Dashboard::MENU_SLUG, 'Template Parts', 'Template Parts', Dashboard::MENU_CAP, $this->get_edit_posts_link() );
	}

	/**
	 * Add admin bar link.
	 *
	 * @param \WP_Admin_Bar $wp_admin_bar
	 *
	 * @since 4.2.1
	 */
	public function add_admin_bar_link( $wp_admin_bar ) {
		if ( is_admin() ) {
			return;
		}

		$wp_admin_bar->add_node(
			[
				'parent' => 'site-name',
				'id'     => self::POST_TYPE,
				'title'  => 'Template Parts',
				'href'   => $this->get_edit_posts_link(),
			]
		);
	}

	/**
	 * Edit columns.
	 *
	 * @param array $columns
	 *
	 * @return array
	 */
	public function edit_columns( $columns ) {
		$columns = kalium()->helpers->insert_at(
			$columns,
			'after',
			'title',
			[
				'type'       => 'Type',
				'conditions' => 'Conditions',
				'status'     => 'Status',
			]
		);

		// Section type - specific columns
		$current_type = $this->get_current_type();

		if ( 'section' === $current_type || ! $current_type ) {
			$columns = kalium()->helpers->insert_at(
				$columns,
				'after',
				'type',
				[
					'placement' => 'Placement',
				]
			);

			$columns = kalium()->helpers->insert_at(
				$columns,
				'after',
				'conditions',
				[
					'shortcode' => 'Shortcode',
				]
			);
		}

		return $columns;
	}

	/**
	 * Custom column content.
	 *
	 * @param string $column
	 * @param int    $post_id
	 */
	public function posts_custom_column( $column, $post_id ) {
		$type = $this->get_type( $post_id );

		switch ( $column ) {
			// Type
			case 'type':
				$type_entry = $this->get_type_options( $type );

				if ( $type_entry ) {
					echo $type_entry['label'];
				}
				break;

			// Placement
			case 'placement':
				$placement = $this->get_placement( $post_id );

				if ( ! empty( $placement ) ) {
					echo '<ul class="template-parts__describe template-parts__describe--placement">';

					foreach ( $placement as $placement_entry ) {
						$this->describe_placement( $placement_entry );
					}

					echo '</ul>';
				}
				break;

			// Conditions
			case 'conditions':
				$conditions = $this->get_display_conditions( $post_id, true );

				if ( empty( $conditions ) ) {
					echo '<em>None</em>';
				} else {
					echo '<ul class="template-parts__describe template-parts__describe--conditions">';

					foreach ( $conditions as $condition ) {
						$this->describe_conditions( $condition );
					}

					echo '</ul>';
				}
				break;

			// Shortcode
			case 'shortcode':
				if ( 'section' === $type ) {
					echo '<code>[kalium_section id="' . esc_attr( kalium_get_original_object_id( $post_id, self::POST_TYPE ) ) . '"]</code>';
				}
				break;

			// Status
			case 'status':
				$status = get_post_status( $post_id );

				printf(
					'<strong class="template-parts__status template-parts__status--%s">%s</strong>',
					$status,
					$status === 'publish' ? 'Enabled' : 'Disabled'
				);
				break;
		}
	}

	/**
	 * Add toggle active/inactive row action.
	 *
	 * @param array   $actions
	 * @param WP_Post $post
	 *
	 * @return array
	 */
	public function action_toggle_status( $actions, $post ) {
		if ( self::POST_TYPE !== $post->post_type ) {
			return $actions;
		}

		$url = wp_nonce_url(
			add_query_arg(
				[
					'action'  => 'tp_toggle_status',
					'post_id' => $post->ID,
				]
			),
			'tp_toggle_status_' . $post->ID
		);

		$actions['toggle_status'] = sprintf(
			'<a href="%s">%s</a>',
			esc_url( $url ),
			esc_html( ( 'publish' === get_post_status( $post ) ) ? 'Disable' : 'Enable' )
		);

		// Remove quick edit
		unset( $actions['inline hide-if-no-js'] );

		return $actions;
	}

	/**
	 * Disable "Filter by date" dropdown in admin list table.
	 *
	 * @param bool   $disable
	 * @param string $post_type
	 * @return bool
	 */
	public function disable_date_filter( $disable, $post_type ) {
		if ( self::POST_TYPE === $post_type ) {
			return true;
		}

		return $disable;
	}

	/**
	 * Handle the toggle status action for template parts.
	 */
	public function handle_toggle_status_action() {
		if (
			! isset( $_GET['action'], $_GET['post_id'] ) ||
			'tp_toggle_status' !== $_GET['action'] ||
			! current_user_can( 'edit_post', $_GET['post_id'] ) ||
			! wp_verify_nonce( $_GET['_wpnonce'] ?? '', 'tp_toggle_status_' . $_GET['post_id'] )
		) {
			return;
		}

		$post_id = absint( $_GET['post_id'] );
		$post    = get_post( $post_id );

		if ( $post && self::POST_TYPE === $post->post_type ) {
			$new_status = ( 'publish' === $post->post_status ) ? 'draft' : 'publish';

			wp_update_post(
				[
					'ID'          => $post_id,
					'post_status' => $new_status,
				]
			);
		}

		wp_safe_redirect( remove_query_arg( [ 'action', '_wpnonce', 'post_id' ] ) );
		exit;
	}

	/**
	 * Filter the main query to only show posts of the current type in admin list.
	 *
	 * @param \WP_Query $query
	 */
	public function filter_type_query( $query ) {
		global $pagenow, $wp_meta_keys;

		if ( is_admin() && $query->is_main_query() && $this->is_edit_page() && ( $current_type = $this->get_current_type() ) ) {
			$type_key  = 'type';
			$type_meta = $wp_meta_keys['post'][ self::POST_TYPE ][ $type_key ];

			$meta_query = [
				[
					'key'   => $type_key,
					'value' => $current_type,
				],
			];

			if ( isset( $type_meta['default'] ) && $type_meta['default'] === $current_type ) {
				$meta_query['relation'] = 'OR';

				$meta_query[] = [
					'key'     => $type_key,
					'compare' => 'NOT EXISTS',
				];
			}

			$query->set( 'meta_query', $meta_query );
		}
	}

	/**
	 * Register bulk actions for activate/deactivate.
	 *
	 * @param array $actions
	 *
	 * @return array
	 */
	public function register_bulk_actions( $actions ) {
		return array_merge(
			$actions,
			[
				'enable_template_part'  => 'Enable',
				'disable_template_part' => 'Disable',
			]
		);
	}

	/**
	 * Handle bulk actions for activate/deactivate.
	 *
	 * @param string $redirect_to
	 * @param string $doaction
	 * @param array  $post_ids
	 *
	 * @return string
	 */
	public function handle_bulk_actions( $redirect_to, $doaction, $post_ids ) {
		if ( ! in_array( $doaction, [ 'enable_template_part', 'disable_template_part' ], true ) ) {
			return $redirect_to;
		}

		$new_status = ( 'enable_template_part' === $doaction ) ? 'publish' : 'draft';

		foreach ( $post_ids as $post_id ) {
			$post = get_post( $post_id );

			if ( $post && self::POST_TYPE === $post->post_type && current_user_can( 'edit_post', $post_id ) ) {
				wp_update_post(
					[
						'ID'          => $post_id,
						'post_status' => $new_status,
					]
				);
			}
		}

		return add_query_arg(
			[
				"bulk_{$doaction}" => count( $post_ids ),
			],
			$redirect_to
		);
	}

	/**
	 * Enqueue assets.
	 */
	public function enqueue_assets() {
		$screen = get_current_screen();

		if ( ! $screen || self::POST_TYPE !== $screen->post_type || kalium()->is->block_editor() ) {
			return;
		}

		kalium_enqueue( 'template-parts' );

		// Template parts JS data
		$data = [
			'type' => $this->get_types(),
		];

		kalium_add_inline_script(
			'template-parts',
			'kaliumAdmin.vars.templateParts = ' . wp_json_encode( $data ),
			'before'
		);
	}

	/**
	 * Enqueue post assets.
	 */
	public function enqueue_post_edit_assets() {
		$screen = get_current_screen();

		if ( ! $screen || self::POST_TYPE !== $screen->post_type ) {
			return;
		}

		// Edit page
		kalium_enqueue( 'template-parts-edit' );

		// Template parts edit JS data
		$data = [
			'types'             => $this->get_types(),
			'hookLocations'     => kalium()->hook_locations->get_locations(),
			'viewHooksUrl'      => kalium()->hook_locations->get_view_hooks_url(),
			'displayConditions' => [
				'types'     => kalium()->display_conditions->get_types(),
				'operators' => kalium()->display_conditions->get_operators(),
			],
			'metaKeys'          => self::META_KEYS,
		];

		kalium_add_inline_script(
			'template-parts-edit',
			'kaliumAdmin.vars.templatePartsEdit = ' . wp_json_encode( $data ),
			'before'
		);
	}

	/**
	 * Display template parts tabs.
	 *
	 * @param array $views
	 *
	 * @return array
	 */
	public function display_tabs( $views ) {
		$types        = $this->get_types();
		$type_ids     = array_keys( $types );
		$current_type = $this->get_current_type();

		?>
		<div class="nav-tab-wrapper">
			<?php
			// All
			kalium_render_element(
				[
					'tag_name'   => 'a',
					'attributes' => [
						'href'  => remove_query_arg( 'tp_type' ),
						'class' => [
							'nav-tab',
							! in_array( $current_type, $type_ids, true ) ? 'nav-tab-active' : '',
						],
					],
					'content'    => 'All',
				]
			);

			// Types
			foreach ( $types as $id => $type ) {
				$label = $type['label'];
				$url   = add_query_arg( 'tp_type', $id );

				$classes = [
					'nav-tab',
				];

				if ( $current_type === $id ) {
					$classes[] = 'nav-tab-active';
				}

				if ( ! empty( $type['disabled'] ) ) {
					$classes[] = 'disabled';
				}

				kalium_render_element(
					[
						'tag_name'   => 'a',
						'attributes' => [
							'href'  => $url,
							'class' => $classes,
						],
						'content'    => esc_html( $label ),
					]
				);
			}
			?>
		</div>
		<?php

		return $views;
	}

	/**
	 * Register setting.
	 */
	public function register_settings() {

		// Enabled state
		kalium()->settings->register(
			'template-parts',
			[
				'setting'           => [ self::OPTION_NAME, 'enabled' ],
				'input_type'        => 'switch',
				'default'           => true,
				'sanitize_callback' => 'kalium_validate_boolean',
			]
		);
	}

	/**
	 * Register setting.
	 *
	 * @param array $sections
	 *
	 * @return array
	 */
	public function register_settings_section( $sections ) {
		$sections[] = [
			'id'       => 'template-parts',
			'title'    => 'Template Parts',
			'icon'     => 'kalium-admin-icon-po-layout-type',
			'priority' => 10,
		];

		return $sections;
	}

	/**
	 * Portfolio settings.
	 */
	public function settings() {
		?>
		<div class="kalium-dashboard__setting">
			<div class="kalium-dashboard__setting__label">
				<h3>Template Parts</h3>
				<p>Enable the Template Parts system to manage custom content, replace templates like Archives and Single Pages, create pop-ups, and customize loop cards.</p>
			</div>
			<div class="kalium-dashboard__setting__control">
				<?php Settings::setting( 'template-parts' ); ?>
			</div>
		</div>
		<?php
	}

	/**
	 * Generate a human-readable description of a condition group.
	 *
	 * @param array $condition_group
	 */
	public function describe_conditions( $condition_group ) {
		static $types = null, $operators = null;

		if ( empty( $condition_group ) || ! is_array( $condition_group ) ) {
			return;
		}

		if ( null === $types ) {
			$types = kalium()->display_conditions->get_types();
		}

		if ( null === $operators ) {
			$operators = array_merge(
				kalium()->display_conditions->get_operators(),
				kalium()->display_conditions->get_additional_operators( 'all' )
			);
		}

		$descriptions = [];

		/**
		 * Get description for choice sub value.
		 *
		 * @param array $choice_sub_value
		 * @param array $sub_values
		 * @param array $sub_value_desc
		 */
		$get_sub_value_desc = static function ( $sub_value_entry, $sub_values, &$desc_arr ) use ( &$get_sub_value_desc ) {
			$name   = $sub_value_entry['name'] ?? '';
			$source = $sub_value_entry['source'] ?? null;
			$value  = $sub_values[ $name ];

			switch ( $sub_value_entry['type'] ) {
				case 'select':
					if ( $source ) {
						$items = [];

						if ( 'array' === $source['type'] ) {
							foreach ( (array) $value as $v ) {
								foreach ( $source['choices'] as $choice ) {
									if ( $choice['value'] == $v ) {
										$items[] = $choice['label'];
										break;
									}
								}
							}
						} elseif ( 'dynamic' === $source['type'] ) {
							switch ( $source['object_type'] ?? '' ) {
								case 'post_type':
									foreach ( (array) $value as $post_id ) {
										$post = get_post( $post_id );
										if ( $post ) {
											$items[] = get_the_title( $post );
										}
									}
									break;
								case 'taxonomy':
									foreach ( (array) $value as $term_id ) {
										$term = get_term( $term_id );
										if ( $term && ! is_wp_error( $term ) ) {
											$items[] = $term->name;
										}
									}
									break;
							}
						}

						$items_count = count( $items );

						if ( 3 <= $items_count ) {
							$desc_arr[] = $items_count . ' selected';
						} else {
							$desc_arr[] = implode( ', ', $items );
						}
					}
					break;

				case 'date':
					$parsed = strtotime( $value );

					if ( $parsed ) {
						$desc_arr[] = date_i18n( get_option( 'date_format' ), $parsed );
					}
					break;

				case 'time':
					if ( is_array( $value ) && isset( $value['hours'], $value['minutes'] ) ) {
						$time = sprintf( '%02d:%02d', (int) $value['hours'], (int) $value['minutes'] );

						$desc_arr[] = date_i18n( get_option( 'time_format' ), strtotime( $time ) );
					}
					break;

				default:
					$desc_arr[] = is_array( $value ) ? implode( ', ', $value ) : $value;
			}

			// Recursive get sub value desc
			if ( ! empty( $source['choices'] ) ) {
				$sub_choice = wp_list_filter( $source['choices'], [ 'value' => $value ] );
				$sub_choice = array_shift( $sub_choice );

				if ( ! empty( $sub_choice['sub_value'] ) ) {
					$get_sub_value_desc( $sub_choice['sub_value'], $sub_values, $desc_arr );
				}
			}
		};

		foreach ( $condition_group as $condition ) {
			$type       = $condition['type'] ?? '';
			$operator   = $condition['operator'] ?? '==';
			$value      = $condition['value'] ?? '';
			$sub_values = $condition['sub_values'] ?? [];

			$type_data = wp_list_filter( $types, [ 'name' => $type ] );
			$type_data = array_shift( $type_data );

			if ( ! $type_data || ! isset( $type_data['choices'][ $value ] ) ) {
				continue;
			}

			$choice         = $type_data['choices'][ $value ];
			$choice_label   = $choice['label'] ?? $value;
			$operator_label = strtolower( $operators[ $operator ] ?? $operator );

			$sub_value_desc = [];

			if ( ! empty( $choice['sub_value'] ) && ! empty( $sub_values ) ) {
				$choice_sub_values = $choice['sub_value'];
				$choice_sub_values = is_array( $choice_sub_values ) && isset( $choice_sub_values[0] ) ? $choice_sub_values : [ $choice_sub_values ];

				foreach ( $choice_sub_values as $choice_sub_value ) {
					$name = $choice_sub_value['name'] ?? '';

					if ( empty( $name ) || ! isset( $sub_values[ $name ] ) ) {
						continue;
					}

					$get_sub_value_desc( $choice_sub_value, $sub_values, $sub_value_desc );
				}
			}

			// Predefined description formats
			if ( 'custom-condition' === $type && 'php-callback' === $value ) {
				$description_format = '<strong>%3$s (%4$s)</strong> %2$s <strong>true</strong>';
				$sub_value_desc     = reset( $sub_value_desc );
			} elseif ( 'custom-condition' === $type && in_array( $value, [ 'post-meta', 'url-query-string' ], true ) ) {
				$description_format = '<strong>%3$s</strong>: <strong>' . $sub_value_desc[0] . '</strong> %2$s <strong>%4$s</strong>';
				$sub_value_desc     = $sub_value_desc[1];
			} elseif ( 'custom-condition' === $type && 'referrer' === $value ) {
				$description_format = '<strong>%3$s</strong> %2$s <strong>%4$s</strong>';
				$sub_value_desc     = reset( $sub_value_desc );
			} elseif ( 'date-time' === $type ) {
				$description_format = '<strong>%3$s</strong> %2$s <strong>%4$s</strong>';
				$sub_value_desc     = reset( $sub_value_desc );
			} elseif ( 'user-role' === $value ) {
				$description_format = '<strong>%3$s</strong> %2$s <strong>%4$s</strong>';

				if ( ! empty( $sub_value_desc[1] ) ) {
					$sub_value_desc = $sub_value_desc[0] . ' (Caps: ' . $sub_value_desc[1] . ')';
				} else {
					$sub_value_desc = reset( $sub_value_desc );
				}
			} elseif ( 'singular' === $type && 'custom-post-type' === $value ) {
				$description_format = '<strong>%3$s</strong> %2$s <strong>%4$s</strong>';

				if ( ! empty( $sub_value_desc[1] ) ) {
					$sub_value_desc = $sub_value_desc[0] . ' (' . $sub_value_desc[1] . ')';
				} else {
					$sub_value_desc = reset( $sub_value_desc );
				}
			}

			// Default description format
			if ( empty( $description_format ) ) {
				$description_format = '<strong>%1$s</strong> %2$s <strong>%3$s</strong>';
			}

			if ( is_array( $sub_value_desc ) && ! empty( $sub_value_desc ) ) {
				$sub_value_desc      = ' (' . implode( ', ', $sub_value_desc ) . ')';
				$description_format .= ' <strong>%4$s</strong>';
			}

			$descriptions[] = sprintf(
				$description_format,
				$type_data['label'],
				$operator_label,
				$choice_label,
				$sub_value_desc
			);
		}

		printf( '<li class="template-parts__describe-entry">%s</li>', trim( implode( ' & ', $descriptions ) ) );
	}

	/**
	 * Add support for elementor.
	 */
	public function add_elementor_support() {
		add_post_type_support( self::POST_TYPE, 'elementor' );
	}

	/**
	 * Register elementor document.
	 *
	 * @param \Elementor\Core\Documents_Manager $documents_manager
	 */
	public function register_elementor_document( $documents_manager ) {
		$document_type = self::POST_TYPE;

		$documents_manager->register_document_type( $document_type . '_document', Elementor_Document::class );
	}

	/**
	 * WPML config array.
	 *
	 * @param array $config
	 *
	 * @return array
	 */
	public function wpml_config_array( $config ) {
		$config['wpml-config']['custom-types']['custom-type'][] = [
			'value' => self::POST_TYPE,
			'attr'  => [
				'translate' => 2,
			],
		];

		foreach ( self::META_KEYS as $meta_key ) {
			$config['wpml-config']['custom-fields']['custom-field'][] = [
				'value' => $meta_key,
				'attr'  => [
					'action' => 'copy',
				],
			];
		}

		return $config;
	}

	/**
	 * Set the template part type on creation based on the current tp_type query string.
	 *
	 * @param int     $post_id
	 * @param WP_Post $post
	 * @param bool    $update
	 *
	 * @since 4.3
	 */
	public function assign_part_type( $post_id, $post, $update ) {
		global $pagenow;

		if ( $update || 'post-new.php' !== $pagenow || self::POST_TYPE !== $post->post_type ) {
			return;
		}

		$type = $this->get_current_type();

		if ( empty( $type ) ) {
			return;
		}

		if ( isset( $this->get_types()[ $type ] ) ) {
			if ( ! metadata_exists( 'post', $post_id, 'type' ) ) {
				add_post_meta( $post_id, 'type', $type, true );

				// Extra options
				if ( in_array( $type, [ 'header', 'footer' ], true ) ) {
					$container_settings = array_merge(
						$this->get_container_settings( $post_id ),
						[
							'tag_name' => $type,
						]
					);

					add_post_meta( $post_id, 'container_settings', $container_settings, true );
				}
			}
		}
	}

	/**
	 * Generate a human-readable description for a placement entry.
	 *
	 * @param array $placement_entry
	 */
	private function describe_placement( $placement_entry ) {
		static $locations;

		if ( is_null( $locations ) ) {
			$locations = wp_list_pluck(
				array_merge(
					...wp_list_pluck(
						kalium()->hook_locations->get_locations(),
						'locations'
					)
				),
				'label',
				'name'
			);
		}

		$label = $locations[ $placement_entry['name'] ] ?? null;

		if ( $label ) {
			printf( '<li class="template-parts__describe-entry">%s</li>', esc_html( $label ), esc_html( $placement_entry['priority'] ) );
		}
	}

	/**
	 * Rename the 'draft' post status label to 'Disabled'.
	 */
	private function rename_draft_status_label() {
		add_action(
			'init',
			function () {
				global $wp_post_statuses;

				if ( isset( $wp_post_statuses['draft'] ) ) {
					$wp_post_statuses['draft']->label       = 'Disabled';
					$wp_post_statuses['draft']->label_count = _n_noop( 'Disabled <span class="count">(%s)</span>', 'Disabled <span class="count">(%s)</span>' );
				}
			},
			11
		);

		add_filter(
			'display_post_states',
			function ( $states, $post ) {
				if ( 'draft' === $post->post_status && $post->post_type === self::POST_TYPE ) {
					foreach ( $states as &$state ) {
						if ( $state === 'Draft' ) {
							$state = 'Disabled';
						}
					}
				}
				return $states;
			},
			10,
			2
		);
	}

	/**
	 * Print block supports styles.
	 *
	 * @since 4.3
	 */
	private function maybe_print_block_supports_styles() {
		if ( ! function_exists( 'wp_style_engine_get_stylesheet_from_context' ) ) {
			return;
		}

		$store_name    = 'block-supports';
		$inline_styles = wp_style_engine_get_stylesheet_from_context( $store_name );

		if ( ! empty( $inline_styles ) ) {
			kalium_print_inline_style( [ 'css' => $inline_styles ] );

			// Remove rules from store
			$block_supports_store = WP_Style_Engine_CSS_Rules_Store::get_store( $store_name );

			if ( ! empty( $rules = $block_supports_store->get_all_rules() ) ) {
				foreach ( $rules as $rule ) {
					$block_supports_store->remove_rule( $rule->get_selector() );
				}
			}
		}
	}

	/**
	 * Print Greenshift block styles.
	 *
	 * @param int $post_id
	 *
	 * @since 4.3
	 */
	private function maybe_print_greenshift_block_styles( $post_id ) {
		if ( ! kalium()->is->greenshift_active() ) {
			return;
		}

		$gspb_css_content = get_post_meta( $post_id, '_gspb_post_css', true );

		if ( $gspb_css_content ) {
			$gspb_saved_css_content = gspb_get_final_css( $gspb_css_content );

			kalium_print_inline_style(
				[
					'id'  => 'gspb-inline-style-' . $post_id,
					'css' => $gspb_saved_css_content,
				]
			);
		}
	}

	/**
	 * Print WPBakery element styles.
	 *
	 * @param int $post_id
	 *
	 * @since 4.3
	 */
	private function maybe_print_wpbakery_block_styles( $post_id ) {
		if ( ! kalium()->is->wpbakery_active() ) {
			return;
		}

		vc_manager()->vc()->addShortcodesCss( $post_id );
	}
}
