<?php
/**
 * Kalium WordPress Theme
 *
 * Blog core functions.
 *
 * @author Laborator
 * @link   https://kaliumtheme.com
 */
if ( ! defined( 'ABSPATH' ) ) {
	exit; // Direct access not allowed.
}

/**
 * Initialize blog options.
 *
 * @param string $id
 * @param array  $extend_options
 *
 * @return array
 */
function kalium_init_blog_options( $id = null, $extend_options = [] ) {
	global $blog_options;

	// Blog template and layout
	$blog_post_layout = kalium_get_theme_option( 'blog_post_layout' );

	// Other settings
	$pagination_settings = kalium()->customize->get_pagination_settings( 'blog_pagination' );

	// Blog Options Array
	$blog_options = [

		// Instance ID
		'id'     => 'blog-posts',

		// Layout
		'layout' => kalium_get_theme_option( 'blog_layout' ),

		// Loop blog options
		'loop'   => [

			// Blog header
			'header'                    => [
				'show'        => kalium_get_theme_option( 'blog_show_header_title' ),
				'title'       => kalium_get_theme_option( 'blog_title' ),
				'description' => kalium_get_theme_option( 'blog_description' ),
			],

			// Container classes
			'container_classes'         => [
				'blog-posts',
				'grid',
			],

			// Sidebar
			'sidebar'                   => [

				// Visibility
				'visible'   => kalium_get_theme_option( 'blog_sidebar' ),

				// Alignment
				'alignment' => kalium_get_theme_option( 'blog_sidebar_align' ),
			],

			// Post formats support
			'post_formats'              => kalium_get_theme_option( 'blog_post_formats' ),

			// Post title
			'post_title'                => kalium_get_theme_option( 'blog_post_title' ),

			// Post excerpt
			'post_excerpt'              => kalium_get_theme_option( 'blog_post_excerpt' ),

			// Post excerpt length
			'post_excerpt_length'       => kalium_get_theme_option( 'blog_post_excerpt_length' ),

			// Post thumbnail
			'post_thumbnail'            => [

				// Visibility
				'visible'        => kalium_get_theme_option( 'blog_thumbnails' ),

				// Image size
				'size'           => kalium()->customize->get_image_size_value( 'blog_thumbnail' ),

				// Border radius
				'border_radius'  => kalium_get_theme_option( 'blog_thumbnail_border_radius' ),

				// Boundless
				'boundless'      => kalium_get_theme_option( 'blog_thumbnail_boundless' ),

				// Vertical alignment
				'content_valign' => kalium_get_theme_option( 'blog_thumbnail_content_valign' ),

				// Placeholder
				'placeholder'    => [
					'enabled'  => kalium_get_theme_option( 'blog_thumbnails_placeholder' ),
					'bg_color' => kalium_get_theme_option( 'blog_thumbnail_placeholder_bg' ),
					'image'    => kalium_get_attachment_id_from_theme_option( 'blog_thumbnail_placeholder_image' ),
				],

				// Aspect ratio
				'aspect_ratio'   => kalium()->customize->get_aspect_ratio_value( 'blog_thumbnail' ),

				// Hover layer
				'hover'          => [
					'visible'     => kalium_get_theme_option( 'blog_thumbnail_hover_effect' ),
					'background'  => kalium_get_theme_option( 'blog_thumbnail_hover_bg' ),
					'padding'     => kalium_get_theme_option( 'blog_thumbnail_hover_padding' ),
					'icon'        => kalium_get_theme_option( 'blog_thumbnail_hover_icon' ),

					// Eye icon
					'eye_icon'    => kalium_get_theme_option( 'blog_thumbnail_hover_icon_eye' ),

					// Custom icon
					'custom_icon' => [
						'image_id' => kalium_get_attachment_id_from_theme_option( 'blog_thumbnail_hover_icon_custom' ),
						'width'    => kalium_get_theme_option( 'blog_thumbnail_hover_icon_custom_width' ),
					],
				],
			],

			// Post meta
			'post_meta'                 => [

				// Visibility
				'visible'       => kalium_get_theme_option( 'blog_post_meta' ),

				// Icon
				'icon'          => kalium_get_theme_option( 'blog_post_meta_icon' ),

				// Post date
				'post_date'     => kalium_get_theme_option( 'blog_post_date' ),

				// Post Category
				'post_category' => kalium_get_theme_option( 'blog_category' ),
			],

			// Post format icon
			'post_format_icon'          => kalium_get_theme_option( 'blog_post_type_icon' ),

			// Post format icon position
			'post_format_icon_position' => kalium_get_theme_option( 'blog_post_type_icon_position' ),

			// Columns
			'columns'                   => kalium_get_theme_option( 'blog_grid_columns' ),

			// Gap
			'gap'                       => kalium_get_theme_option( 'blog_grid_columns_gap' ),

			// Row layout mode
			'masonry_mode'              => kalium_get_theme_option( 'blog_grid_masonry_mode' ),

			// Pagination
			'pagination'                => [
				'enabled'                   => true,
				'type'                      => $pagination_settings['type'],
				'align'                     => $pagination_settings['align'],
				'numbers_display'           => $pagination_settings['numbers_display'],
				'load_more_infinite_scroll' => $pagination_settings['infinite_scroll_auto_reveal'],
				'load_more_loading_effect'  => $pagination_settings['infinite_scroll_loading_effect'],
			],
		],

		// Single blog options
		'single' => [

			// Layout
			'narrow'                   => 'narrow' === $blog_post_layout,

			// Share story
			'share'                    => [

				// Visibility
				'visible'  => kalium_get_theme_option( 'blog_post_social_share' ),

				// Share networks
				'networks' => kalium_get_theme_option( 'blog_post_social_share_networks' ),

				// Icons style
				'style'    => kalium_get_theme_option( 'blog_post_social_share_style' ),
			],

			// Sidebar
			'sidebar'                  => [

				// Visibility
				'visible'   => in_array( $blog_post_layout, [ 'sidebar-left', 'sidebar-right' ], true ),

				// Alignment
				'alignment' => 'sidebar-left' === $blog_post_layout ? 'left' : 'right',
			],

			// Post formats
			'post_formats'             => true,

			// Post image
			'post_image'               => [

				// Visibility
				'visible'      => kalium_get_theme_option( 'blog_single_thumbnails' ),

				// Image size
				'size'         => kalium()->customize->get_image_size_value( 'blog_featured_image' ),

				// Aspect ratio
				'aspect_ratio' => kalium()->customize->get_aspect_ratio_value( 'blog_featured_image' ),

				// Image placement
				'placement'    => kalium_get_theme_option( 'blog_featured_image_placement' ),
			],

			// Post title
			'post_title'               => kalium_get_theme_option( 'blog_single_title' ),

			// Post content
			'post_content'             => true,

			// Post meta
			'post_meta'                => [

				// Visibility
				'visible'       => kalium_get_theme_option( 'blog_single_post_meta' ),

				// Icon
				'icon'          => kalium_get_theme_option( 'blog_single_post_meta_icon' ),

				// Post category
				'post_category' => kalium_get_theme_option( 'blog_category_single' ),

				// Post date
				'post_date'     => kalium_get_theme_option( 'blog_post_date_single' ),

				// Post tags
				'post_tags'     => kalium_get_theme_option( 'blog_tags' ),
			],

			// Post comments
			'post_comments'            => kalium_get_theme_option( 'blog_comments' ),

			// Author
			'author'                   => [

				// Visibility
				'visible' => kalium_get_theme_option( 'blog_author_info' ),
			],

			// Post navigation
			'post_navigation'          => [

				// Enabled state
				'enabled' => kalium_get_theme_option( 'blog_post_navigation' ),

				// Options
				'options' => kalium()->customize->get_post_navigation_settings(
					'blog_post_navigation',
					[
						'post_type' => 'post',
					]
				),
			],

			// Gallery carousel auto switch
			'gallery_autoswitch_image' => absint( kalium_get_theme_option( 'blog_gallery_autoswitch' ) ),
		],
	];

	// Extend/replace blog options
	if ( is_array( $extend_options ) ) {
		$blog_options = array_replace_recursive( $blog_options, $extend_options );
	}

	// Blog instance ID
	if ( $id ) {
		$blog_options['id'] = sprintf( 'blog-posts-%s', esc_attr( $id ) );
	}

	// CSS selector
	$blog_options['selector'] = sprintf( '.%s', esc_attr( $blog_options['id'] ) );

	// Selector class in container
	$blog_options['loop']['container_classes'][] = $blog_options['id'];

	// Blog settings based on blog template
	if ( 'grid' === $blog_options['layout'] || isset( $blog_options['force_grid_columns'] ) ) {
		$blog_options['loop']['container_classes'] = array_merge( $blog_options['loop']['container_classes'], kalium_responsive_classname_infix( $blog_options['loop']['columns'], 'columns' ) );
	}

	// Force default pagination on "load-more" pagination type
	if ( 'load-more' === $blog_options['loop']['pagination']['type'] ) {
		$blog_options['loop']['pagination']['numbers_display'] = null;
	}

	// When its assigned as blog page
	if ( is_home() && ( $post = get_queried_object() ) ) {
		$heading_title = kalium_get_field( 'heading_title', $post->ID );

		// Show heading title
		if ( $heading_title ) {
			$heading_title_type         = kalium_get_field( 'page_heading_title_type', $post->ID );
			$heading_custom_title       = kalium_get_field( 'page_heading_custom_title', $post->ID );
			$heading_description_type   = kalium_get_field( 'page_heading_description_type', $post->ID );
			$heading_custom_description = kalium_get_field( 'page_heading_custom_description', $post->ID );

			$blog_options['loop']['header']['show']        = true;
			$blog_options['loop']['header']['title']       = 'post_title' == $heading_title_type ? get_the_title( $post ) : $heading_custom_title;
			$blog_options['loop']['header']['description'] = 'post_content' == $heading_description_type ? apply_filters( 'the_content', $post->post_content ) : $heading_custom_description;
		}
	}

	// Single
	if ( is_single() ) {
		$post_id = get_queried_object_id();

		// Featured image placement
		$featured_image_placement = kalium_get_field( 'featured_image_placing', $post_id );

		if ( $featured_image_placement && in_array(
			$featured_image_placement,
			[
				'container',
				'container-wide',
				'full-width',
				'hide',
			]
		) ) {
			$blog_options['single']['post_image']['placement'] = $featured_image_placement;

			if ( 'hide' == $featured_image_placement ) {
				$blog_options['single']['post_image']['visible'] = false;
			}
		}

		// Featured image size
		$post_image_size = kalium_get_field( 'post_image_size', $post_id );

		if ( in_array( $post_image_size, [ 'default', 'full' ] ) ) {
			$blog_options['single']['post_image']['size'] = 'default' == $post_image_size ? 'blog-single-1' : 'full';
		}

		// Custom container width
		if ( kalium_get_field( 'custom_grid_container_width', $post_id ) ) {
			$blog_options['single']['narrow'] = false;
		}

		// Password protected post or attachment page disable few sections
		if ( post_password_required() || is_attachment() ) {
			$blog_options['single']['post_image']['visible'] = false;
			$blog_options['single']['author']['visible']     = false;
			$blog_options['single']['share']['visible']      = false;
			$blog_options['single']['post_tags']             = false;
			$blog_options['single']['prev_next']             = false;
		}
	}

	return apply_filters( 'kalium_blog_options', $blog_options );
}

/**
 * Reset blog options global.
 */
function kalium_reset_blog_options() {
	global $blog_options;
	$blog_options = [];
}

/**
 * Get blog option value.
 *
 * @param string $name
 *
 * @return mixed|WP_Error|null
 */
function kalium_get_blog_option( $name ) {
	global $blog_options;

	// If blog options are not initialized
	if ( empty( $blog_options ) ) {
		kalium_init_blog_options();
	}

	return apply_filters( 'kalium_blog_option_' . $name, kalium_get_array_key( $blog_options, $name ) );
}

/**
 * Get current blog posts layout.
 *
 * @return string
 */
function kalium_get_blog_layout() {
	return kalium_get_blog_option( 'layout' );
}

/**
 * Is narrow layout.
 *
 * @return bool
 */
function kalium_is_narrow_blog_layout() {
	return kalium_get_blog_option( 'single/narrow' );
}

/**
 * Get current instance ID of blog.
 *
 * @return string
 */
function kalium_get_blog_instance_id() {
	return kalium_get_blog_option( 'id' );
}

/**
 * Check if current post is viewed as single post.
 *
 * @return bool
 * @since 4.0
 */
function kalium_is_single_post() {
	global $post;

	return is_single() && kalium_get_queried_object_id() === $post->ID;
}

/**
 * Check if post meta icons are visible.
 *
 * @return bool
 * @since 4.0
 */
function kalium_is_post_meta_icon_visible() {
	if ( kalium_is_single_post() ) {
		return kalium_get_blog_option( 'single/post_meta/icon' );
	}

	return kalium_get_blog_option( 'loop/post_meta/icon' );
}

/**
 * Get post featured image link or return post image link instead.
 *
 * @param WP_Post $post
 *
 * @return bool|false|string|WP_Error
 */
function kalium_get_post_image_link( $post ) {
	if ( has_post_thumbnail( $post ) ) {
		return get_the_post_thumbnail_url( $post, 'full' );
	} elseif ( is_object( $post ) && 'attachment' == $post->post_type ) {
		return $post->guid;
	}

	return get_permalink( $post );
}

/**
 * Extract post format content.
 *
 * @param WP_Post $post
 *
 * @return array|null
 */
function kalium_extract_post_format_content( $post = null ) {
	$post = get_post( $post );

	if ( ! $post ) {
		return null;
	}

	// Result holder
	$result = null;

	// Plain post content
	$post_content = $post->post_content;

	// Thumbnail size to use
	if ( is_single() ) {
		$thumbnail_size = kalium_get_blog_option( 'single/post_image/size' );
	} else {
		$thumbnail_size = kalium_get_blog_option( 'loop/post_thumbnail/size' );
	}

	// Lookup in blocks
	$blocks = parse_blocks( $post_content );

	// Extract post format
	$post_format = get_post_format( $post );

	switch ( $post_format ) {

		// Gallery post format
		case 'gallery':
			$gallery_images = kalium_get_field( 'post_slider_images', $post->ID );

			// Assign featured image as well
			if ( has_post_thumbnail( $post ) ) {
				$featured_image = [
					'id' => get_post_thumbnail_id( $post ),
				];

				if ( ! is_array( $gallery_images ) ) {
					$gallery_images = [];
				}

				if ( apply_filters( 'kalium_blog_post_gallery_format_include_featured_image', true ) ) {
					array_unshift( $gallery_images, $featured_image );
				}
			}

			// Only when has gallery items
			if ( ! empty( $gallery_images ) ) {
				$gallery_html = '';

				foreach ( $gallery_images as $gallery_image ) {
					if ( ! empty( $gallery_image['id'] ) ) {
						$image      = kalium_get_attachment_image( $gallery_image['id'], $thumbnail_size );
						$image_link = is_single() ? kalium_get_post_image_link( get_post( $gallery_image['id'] ) ) : get_permalink( $post );

						$gallery_html .= sprintf( '<div class="swiper-slide"><a href="%s">%s</a></div>', $image_link, $image );
					}
				}

				// Gallery has items
				if ( $gallery_html ) {
					$gallery_autoswitch_image = kalium_get_blog_option( 'single/gallery_autoswitch_image' );

					$slider_options = [
						'theme'       => 'light',
						'class'       => [
							'post-gallery-images',
							kalium_get_lightbox_class(),
						],
						'navigation'  => true,
						'loop'        => true,
						'slides'      => $gallery_html,
						'auto_height' => true,
						'in_grid'     => true,
						'echo'        => false,
					];

					if ( is_single() && $gallery_autoswitch_image > 0 ) {
						$slider_options['autoplay'] = $gallery_autoswitch_image;
					}

					$result = [
						'type'    => $post_format,
						'content' => '',
						'gallery' => kalium_slider( $slider_options ),
					];
				}
			}
			break;

		// Audio
		case 'video':
		case 'audio':
			if ( 'audio' === $post_format ) {
				$autoplay = is_single() && kalium_get_field( 'auto_play_audio', $post->ID );
			} else {
				$autoplay   = is_single() && kalium_get_field( 'auto_play_video', $post->ID );
				$resolution = kalium_get_field( 'video_resolution', $post->ID );
			}

			/**
			 * Get media attributes from provided content.
			 *
			 * @param string $content
			 *
			 * @return array|false
			 */
			$get_media_attrs = function ( $content ) use ( $post_format ) {
				if ( preg_match( '/<(' . $post_format . ')([^>]*)?>\s*(?<sources>.*?)\s*<\/(?:' . $post_format . ')>/is', $content, $matches ) ) {
					$attrs         = [
						'markup' => $matches[0],
					];
					$element       = kalium()->helpers->parse_html_element( do_shortcode( $matches[0] ) );
					$element_attrs = $element['attributes'];

					foreach ( [
						'src',
						'autoplay',
						'muted',
						'loop',
						'playsinline',
						'poster',
						'width',
						'height',
					] as $attr_name ) {
						if ( isset( $element_attrs[ $attr_name ] ) ) {
							$attrs[ $attr_name ] = $element_attrs[ $attr_name ];
						}
					}

					if ( empty( $element_attrs['src'] ) ) {
						$urls = wp_extract_urls( $matches['sources'] );

						if ( $media_src = current( $urls ) ) {
							$attrs['src'] = $media_src;
						}
					}

					return $attrs;
				}

				return false;
			};

			// Media attributes
			$media_args = [];

			// Poster
			if ( has_post_thumbnail( $post ) && ( $featured_image = wp_get_attachment_image_src( get_post_thumbnail_id( $post ), $thumbnail_size ) ) ) {
				$media_args['poster'] = $featured_image[0];
				$media_args['width']  = $featured_image[1];
				$media_args['height'] = $featured_image[2];
			}

			// Autoplay
			if ( $autoplay ) {
				$media_args['autoplay'] = true;
			}

			// Video resolution
			if ( ! empty( $resolution ) ) {
				$resolution = kalium_extract_aspect_ratio( $resolution );
				$media_args = array_merge( $media_args, $resolution );
			}

			// Try parsing from blocks
			foreach ( $blocks as $block ) {
				$block_attrs = $block['attrs'];

				// Video or audio block
				if ( in_array( $block['blockName'], [ 'core/video', 'core/audio' ], true ) ) {
					if ( $shortcode_media_args = $get_media_attrs( $block['innerHTML'] ) ) {
						$original_content = $block['innerHTML'];
						$media_args       = array_merge( $media_args, $shortcode_media_args );
					}
				}
				// Embed block
				elseif ( 'core/embed' === $block['blockName'] ) {
					$original_content  = $block['innerHTML'];
					$media_args['src'] = $block_attrs['url'] ?? null;
				}
				// Shortcode block
				elseif ( 'core/shortcode' === $block['blockName'] ) {
					if ( $shortcode_media_args = $get_media_attrs( $block['innerHTML'] ) ) {
						$original_content = $shortcode_media_args['markup'];
						$media_args       = array_merge( $media_args, $shortcode_media_args );
					}
				}

				// Break loop if media source is defined
				if ( ! empty( $media_args['src'] ) ) {
					break;
				}
			}

			// Try parsing [video] and [audio] shortcode
			if ( empty( $media_args['src'] ) && preg_match( '/' . get_shortcode_regex( [ 'video', 'audio' ] ) . '/', $post_content, $matches ) ) {
				$shortcode_atts = shortcode_parse_atts( $matches[3] ?? '' );

				$media_args['src'] = $shortcode_atts['src'] ?? ( $shortcode_atts['mp3'] ?? ( $shortcode_atts['mp4'] ?? null ) );

				if ( ! empty( $media_args['src'] ) ) {
					$original_content = $matches[0];
				}
			}

			// Try to extract from post content
			if ( empty( $media_args['src'] ) && ( $extracted_media_args = $get_media_attrs( $post_content ) ) ) {
				$original_content = $extracted_media_args['markup'];
				$media_args       = array_merge( $media_args, $extracted_media_args );
			}

			// Last resort, lookup for media (video/audio) URLs in content
			if ( empty( $media_args['src'] ) ) {
				$urls = wp_extract_urls( $post_content );

				if ( $first_url = current( $urls ) ) {
					if ( kalium()->media->is_video( $first_url ) || kalium()->media->is_audio( $first_url ) ) {
						$media_args['src'] = $first_url;

						// Link is not wrapped
						if ( ! preg_match( sprintf( '#("|\')%s("|\')#', preg_quote( $first_url ) ), $post_content ) ) {
							$original_content = $first_url;
						}
					}
				}
			}

			// Render media
			if ( ! empty( $media_args['src'] ) ) {
				unset( $media_args['markup'] );

				if ( kalium()->media->is_video( $media_args['src'] ) ) {
					$media = kalium_video_element( $media_args );
				} elseif ( kalium()->media->is_audio( $media_args['src'] ) ) {
					// Remove autoplay for audio elements in loop
					if ( false === is_single() && kalium_get_queried_object_id() !== $post->ID ) {
						unset( $media_args['autoplay'] );
					}

					$media = kalium_audio_element( $media_args );
				}

				// Translate CSS var "ratio" to "pi-ratio"
				$translate_ratio = function ( $matches ) {
					return sprintf( '%s; %s: %s', $matches[0], kalium_css_var_name( 'pi-ratio' ), $matches[1] );
				};

				$media = preg_replace_callback( '/' . kalium_css_var_name( 'ratio' ) . ':\s*([\d.]+)/', $translate_ratio, $media, 1 );
			}

			if ( isset( $media ) ) {
				$result = [
					'type'    => $post_format,
					'content' => $original_content ?? null,
					'media'   => $media,
				];
			}
			break;

		// Quotes
		case 'quote':
			$blockquote_pattern = '/<blockquote(?:\s+[^>]*)?>\s*(?P<quote>.*?)\s*(?:<cite>\s*(?P<author>.*?)\s*<\/cite>)?\s*<\/blockquote>/is';

			foreach ( $blocks as $block ) {
				if ( 'core/pullquote' === $block['blockName'] && preg_match( $blockquote_pattern, $block['innerHTML'], $quote_matches ) ) {
					$result = [
						'type'    => $post_format,
						'content' => $block['innerHTML'],
						'quote'   => $quote_matches['quote'],
						'author'  => $quote_matches['author'],
					];
					break;
				}
			}

			// Search in post_content
			if ( empty( $result ) && preg_match( $blockquote_pattern, $post_content, $matches ) ) {
				$content = $matches[0];
				$quote   = $matches['quote'];
				$author  = kalium_get_array_key( $matches, 'author' );

				$result = [
					'type'    => $post_format,
					'content' => $content,
					'quote'   => $quote,
					'author'  => $author,
				];
			}

			break;
	}

	return $result;
}

/**
 * Show post content format media.
 *
 * @param array $result
 * @param bool  $return
 *
 * @return string|void
 */
function kalium_show_post_format_content( $result, $return = false ) {
	$html = '';

	// Check if its valid result from extracted post format content
	if ( is_array( $result ) && isset( $result['type'] ) ) {

		switch ( $result['type'] ) {

			// Gallery
			case 'gallery':
				$html = $result['gallery'];
				break;

			// Video + audio
			case 'video':
			case 'audio':
				$html = $result['media'];
				break;

			// Quote
			case 'quote':
				$quote   = $result['quote'];
				$author  = $result['author'];
				$classes = [
					'post-quote',
				];

				// Single page
				if ( is_single() ) {
					$classes = array_merge(
						$classes,
						[
							'container-full',
							'container-child',
						]
					);
				}

				if ( $author ) {
					$quote .= "<cite>{$author}</cite>";
				}

				$html = sprintf( '<div class="%s"><blockquote>%s</blockquote></div>', kalium_tokenize_list( $classes ), $quote );
				break;
		}
	}

	if ( $return ) {
		return $html;
	}

	echo $html;
}

/**
 * Checks if given post is external post.
 *
 * @param WP_Post $post
 *
 * @return bool
 */
function kalium_is_external_url_post( $post ) {
	if ( $post instanceof WP_Post && 'link' === get_post_format( $post ) ) {
		$links = wp_extract_urls( apply_filters( 'the_content', $post->post_content ) );

		return ! empty( $links );
	}

	return false;
}

/**
 * Returns post link target.
 *
 * @retun string
 *
 * @param WP_Post $post
 *
 * @return mixed|string
 */
function kalium_get_post_link_target( $post = null ) {
	$_post       = get_post( $post );
	$link_target = '';

	// External post
	if ( $_post instanceof WP_Post && kalium_is_external_url_post( $_post ) && preg_match( '#href=("|\')' . preg_quote( get_permalink( $_post ) ) . '("|\')[^>]+?target=("|\')(?<target>[a-z_]+)("|\')#i', get_the_content( null, false, $_post ), $matches ) ) {
		$link_target = $matches['target'];
	}

	return apply_filters( 'kalium_blog_post_link_target', $link_target );
}

/**
 * Blog posts with masonry container.
 *
 * @since 4.0
 */
function kalium_make_masonry_posts_container() {
	if ( 'grid' !== kalium_get_blog_option( 'layout' ) || ! kalium_get_blog_option( 'loop/masonry_mode' ) ) {
		return;
	}

	kalium_create_masonry_container(
		[
			'options' => [
				'container' => '.blog-posts',
				'item'      => '.type-post',
			],
		]
	);
}
