<?php
/**
 * Kalium WordPress Theme
 *
 * Core hook functions.
 *
 * @author Laborator
 * @link   https://kaliumtheme.com
 */
if ( ! defined( 'ABSPATH' ) ) {
	exit; // Direct access not allowed.
}

/**
 * After setup theme.
 */
function kalium_theme_setup() {

	// Theme supports
	add_theme_support(
		'html5',
		[
			'comment-list',
			'comment-form',
			'search-form',
			'gallery',
			'caption',
			'style',
		]
	);
	add_theme_support( 'automatic-feed-links' );
	add_theme_support( 'post-thumbnails' );
	add_theme_support( 'featured-image' );
	add_theme_support(
		'post-formats',
		[
			'video',
			'quote',
			'image',
			'link',
			'gallery',
			'audio',
		]
	);

	// Title tag
	add_theme_support( 'title-tag' );

	// Adds support for responsive embeds
	add_theme_support( 'responsive-embeds' );

	// Color palette
	add_theme_support( 'editor-color-palette', kalium_get_editor_color_palette() );

	// Block styles
	add_theme_support( 'wp-block-styles' );

	// Editor styles
	add_theme_support( 'editor-styles' );

	// Align wide
	add_theme_support( 'align-wide' );

	// Editor style
	add_editor_style(
		array_map(
			function ( $handle ) {
				return str_replace( trailingslashit( get_template_directory_uri() ), '', kalium()->assets->get_style( $handle )['src'] );
			},
			array_merge(
				kalium()->assets->get_styles_list( 'editor-style' ),
				kalium()->assets->get_styles_list( 'theme-frontend-styles' )
			)
		)
	);

	// Register menus
	register_nav_menus(
		[
			'main-menu'   => 'Main Menu',
			'mobile-menu' => 'Mobile Menu',
		]
	);
}

add_action( 'after_setup_theme', 'kalium_theme_setup' );

/**
 * Enqueue styles.
 */
function kalium_enqueue_styles() {

	// Enqueue theme styles
	kalium_enqueue( 'theme-frontend-styles' );

	// Admin icons
	if ( is_admin_bar_showing() ) {
		kalium_enqueue( 'theme-icons-admin' );
	}

	// CSS loaders
	if ( 'preselected' === kalium_get_theme_option( 'image_loading_placeholder_type' ) ) {
		kalium_enqueue( 'css-loaders' );
	}

	// Theme style.css is optional
	if ( kalium_get_theme_option( 'performance_enqueue_style_css' ) ) {
		kalium_enqueue( 'theme-style' );
	}
}

add_action( 'wp_enqueue_scripts', 'kalium_enqueue_styles', 100 );

/**
 * Enqueue Kalium scripts and libraries.
 *
 * @return void
 */
function kalium_enqueue_scripts() {
	// Force jQuery enqueue
	if ( kalium_get_theme_option( 'performance_force_jquery' ) || isset( $_GET['force_load_jquery'] ) ) {
		wp_enqueue_script( 'jquery' );
	}

	// Single post enqueue
	if ( is_singular( 'post' ) ) {

		// Comment reply script
		if ( comments_open() ) {
			wp_enqueue_script( 'comment-reply' );
		}
	}
}

add_action( 'wp_enqueue_scripts', 'kalium_enqueue_scripts', 100 );

/**
 * Maybe load theme options.
 *
 * @since 4.0
 */
function kalium_maybe_load_theme_options() {
	kalium()->theme_options->maybe_load_theme_options();
}

add_action( 'init', 'kalium_maybe_load_theme_options' );

/**
 * Enqueue theme scripts.
 *
 * @since 4.0
 */
function kalium_enqueue_theme_scripts() {
	kalium_enqueue( 'theme-frontend-scripts' );
}

add_action( 'wp_footer', 'kalium_enqueue_theme_scripts' );

/**
 * Custom JavaScript in header.
 *
 * @return void
 */
function kalium_head_custom_js() {
	$user_custom_js_head = kalium_get_theme_option( 'user_custom_js_head' );

	if ( ! empty( $user_custom_js_head ) ) {
		// Auto wrap with <script> tag (maybe not a good idea!)
		if ( ! preg_match( '/\<\w+/', $user_custom_js_head ) ) {
			$user_custom_js_head = '<script> ' . $user_custom_js_head . ' </script>';
		}

		echo $user_custom_js_head;
	}
}

add_action( 'wp_head', 'kalium_head_custom_js', 100 );

/**
 * Custom User JavaScript right before body end tag.
 *
 * @return void
 */
function kalium_footer_custom_js() {
	$user_custom_js = kalium_get_theme_option( 'user_custom_js' );

	if ( ! empty( $user_custom_js ) ) {
		// Auto wrap with <script> tag (maybe not a good idea!)
		if ( ! preg_match( '/\<\w+/', $user_custom_js ) ) {
			$user_custom_js = sprintf( '<script>%s</script>', $user_custom_js );
		}

		echo $user_custom_js;
	}
}

add_action( 'wp_print_footer_scripts', 'kalium_footer_custom_js' );

/**
 * Theme widgets init.
 *
 * @return void
 */
function kalium_widgets_init() {
	$sidebars = [
		// Blog Sidebar
		[
			'id'   => 'blog_sidebar',
			'name' => 'Blog Archive',
		],
		// Sidebar on single post
		[
			'id'   => 'blog_sidebar_single',
			'name' => 'Single Post',
		],
		// Footer Widget Area
		[
			'id'   => 'footer_sidebar',
			'name' => 'Footer Widget Area 1',
		],
		[
			'id'   => 'footer_sidebar_2',
			'name' => 'Footer Widget Area 2',
		],
		[
			'id'   => 'footer_sidebar_3',
			'name' => 'Footer Widget Area 3',
		],
		[
			'id'   => 'footer_sidebar_4',
			'name' => 'Footer Widget Area 4',
		],
		[
			'id'   => 'footer_sidebar_5',
			'name' => 'Footer Widget Area 5',
		],
		[
			'id'   => 'footer_sidebar_6',
			'name' => 'Footer Widget Area 6',
		],
		// Top Menu Sidebar
		[
			'id'   => 'top_menu_sidebar',
			'name' => 'Off-Canvas Top',
		],
		// Sidebar Menu Widgets
		[
			'id'   => 'sidebar_menu_sidebar',
			'name' => 'Off-Canvas Side',
		],
		// Shop Sidebar
		[
			'id'   => 'shop_sidebar',
			'name' => 'Shop Archive',
		],
		// Sidebar on single post
		[
			'id'   => 'shop_sidebar_single',
			'name' => 'Single Product',
		],
	];

	// Sidebars array
	$sidebars = apply_filters( 'kalium_sidebars_array', $sidebars );

	// Register sidebars
	foreach ( $sidebars as $widget ) {
		register_sidebar(
			[
				'id'            => $widget['id'],
				'name'          => $widget['name'],
				'before_widget' => '<div id="%1$s" class="widget %2$s">',
				'after_widget'  => '</div>',
				'before_title'  => '<h2 class="widget-title">',
				'after_title'   => '</h2>',
				'description'   => kalium_get_array_key( $widget, 'description' ),
			]
		);
	}
}

add_action( 'widgets_init', 'kalium_widgets_init' );

/**
 * Creates default footer widgets when theme gets installed for the first time.
 *
 * @since 4.0
 */
function kalium_create_default_footer_widgets() {
	$sidebars_widgets = wp_get_sidebars_widgets();
	$widget_block     = get_option( 'widget_block' );

	// Content to insert
	$left_column  = '[kalium_site_info display="{copyright} {year} {site_title} – {theme_credits}"]';
	$right_column = '[kalium_social_icons icon="yes" label="yes"]';

	// Create widgets
	$last_id = array_keys( $widget_block );
	$last_id = $last_id[ count( $last_id ) - 2 ] ?? 1;

	$left_column_id  = $last_id + 1;
	$right_column_id = $last_id + 2;

	$widget_block[ $left_column_id ] = [
		'content' => $left_column,
	];

	$widget_block[ $right_column_id ] = [
		'content' => $right_column,
	];

	$sidebars_widgets['footer_sidebar'][]   = 'block-' . $left_column_id;
	$sidebars_widgets['footer_sidebar_2'][] = 'block-' . $right_column_id;

	unset( $widget_block['_multiwidget'] );
	$widget_block['_multiwidget'] = 1;

	wp_set_sidebars_widgets( $sidebars_widgets );
	update_option( 'widget_block', $widget_block );
}

add_action( 'kalium_first_install', 'kalium_create_default_footer_widgets' );

/**
 * Print appended CSS on footer.
 *
 * @return void
 */
function kalium_legacy_print_footer_css() {
	global $kalium_append_custom_css;

	if ( empty( $kalium_append_custom_css ) ) {
		return;
	}

	printf( '<style data-appended-custom-css="true">%s</style>', kalium_compress_text( implode( "\n\n", $kalium_append_custom_css ) ) );
}

add_action( 'wp_footer', 'kalium_legacy_print_footer_css' );

/**
 * Define JS variables such as "ajaxurl" and "icl_language_code" when WPML is active.
 */
function kalium_js_define_global_vars() {
	$vars = [
		sprintf( 'var ajaxurl = ajaxurl || %s;', wp_json_encode( admin_url( 'admin-ajax.php' ) ) ),
	];

	if ( defined( 'ICL_LANGUAGE_CODE' ) ) {
		$vars[] = sprintf( 'var icl_language_code = %s;', wp_json_encode( ICL_LANGUAGE_CODE ) );
	}

	wp_print_inline_script_tag( implode( PHP_EOL, $vars ) );
}

add_action( 'wp_print_scripts', 'kalium_js_define_global_vars' );

/**
 * Kalium get default excerpt length.
 *
 * @return int
 */
function kalium_default_excerpt_length() {
	return apply_filters( 'kalium_default_excerpt_length', 55 );
}

/**
 * Excerpt more dots.
 *
 * @return string
 */
function kalium_default_excerpt_more() {
	return apply_filters( 'kalium_default_excerpt_more', '&hellip;' );
}

/**
 * Scroll to top.
 */
function kalium_scroll_to_top() {
	if ( ! kalium_get_theme_option( 'scroll_to_top' ) ) {
		return;
	}

	$trigger_pos = kalium_get_theme_option( 'scroll_to_top_activate' );

	if ( kalium_get_theme_option( 'scroll_to_top_activate_footer' ) ) {
		$trigger_pos = 'footer';
	}

	// Position
	list( $position_y, $position_x ) = explode( '-', kalium_get_theme_option( 'scroll_to_top_position' ) );

	// Classes
	$classes = [
		'scroll-to-top',
		'scroll-to-top--pos-' . $position_y,
		'scroll-to-top--pos-' . $position_x,
	];

	// Visibility classes
	$classes = array_merge( $classes, kalium_get_device_visibility_classes( kalium_get_theme_option( 'scroll_to_top_visibility' ) ) );

	// Icon
	switch ( kalium_get_theme_option( 'scroll_to_top_icon' ) ) {
		case 'icon-2':
			$icon = '<i class="kalium-icon-long-arrow-up"></i>';
			break;

		default:
			$icon = '<i class="kalium-icon-arrow-up"></i>';
	}

	// Colors
	$text       = kalium_get_theme_option( 'scroll_to_top_color' );
	$background = kalium_get_theme_option( 'scroll_to_top_bg' );

	// Print styles
	kalium_print_inline_style(
		[
			'selector' => '.scroll-to-top',
			'vars'     => [
				'stt-size'             => kalium_map_responsive_value(
					kalium_get_theme_option( 'scroll_to_top_size' ),
					function ( $value ) {
						if ( is_numeric( $value ) ) {
							return $value . 'px';
						}
					}
				),
				'stt-spacing'          => kalium_map_responsive_value(
					kalium_get_theme_option( 'scroll_to_top_spacing' ),
					function ( $value ) {
						if ( is_numeric( $value ) ) {
							return $value . 'px';
						}
					}
				),
				'button-border-radius' => kalium_to_length( kalium_get_theme_option( 'scroll_to_top_border_radius' ) ),
				'button-color'         => kalium_get_array_key( $text, 'normal' ),
				'button-hover-color'   => kalium_get_array_key( $text, 'hover' ),
				'button-active-color'  => kalium_get_array_key( $text, 'hover' ),
				'button-bg'            => kalium_get_array_key( $background, 'normal' ),
				'button-hover-bg'      => kalium_get_array_key( $background, 'hover' ),
				'button-active-bg'     => kalium_get_array_key( $background, 'hover' ),
			],
		]
	)

	?>
	<button <?php kalium_class_attr( $classes ); ?> type="button" data-trigger-pos="<?php echo esc_attr( $trigger_pos ); ?>">
		<?php echo $icon; ?>
	</button>
	<?php
}

add_action( 'wp_footer', 'kalium_scroll_to_top' );

/**
 * Page Custom CSS.
 */
function kalium_page_custom_css() {
	$queried_object_id = kalium_get_queried_object_id();

	if ( $queried_object_id && ( $page_custom_css = kalium()->acf->get_field( 'page_custom_css', $queried_object_id ) ) ) {
		$page_custom_css = str_replace( '{{ID}}', $queried_object_id, $page_custom_css );
		$page_custom_css = sprintf( '<style data-page-custom-css="true">%s</style>', $page_custom_css );

		if ( apply_filters( 'kalium_page_custom_css_append_header', true ) ) {
			add_action( 'wp_print_styles', kalium_hook_echo_value( $page_custom_css ) );
		} else {
			add_action( 'wp_footer', kalium_hook_echo_value( $page_custom_css ) );
		}
	}
}

add_action( 'wp', 'kalium_page_custom_css' );

/**
 * Add open graph meta in header.
 */
function kalium_open_graph_meta() {

	// Only show if open graph meta is allowed
	if ( ! apply_filters( 'kalium_open_graph_meta', true ) || ! is_singular() ) {
		return;
	}

	// Current post
	$post = get_queried_object();

	// Excerpt, clean styles
	$excerpt = kalium_clean_excerpt( get_the_excerpt(), true );
	?>
	<meta property="og:type" content="article">
	<meta property="og:title" content="<?php echo esc_attr( get_the_title() ); ?>">
	<meta property="og:url" content="<?php echo esc_url( get_permalink() ); ?>">
	<meta property="og:site_name" content="<?php echo esc_attr( get_bloginfo( 'name' ) ); ?>">

	<?php if ( $excerpt ) : ?>
		<meta property="og:description" content="<?php echo esc_attr( $excerpt ); ?>">
	<?php endif; ?>

	<?php
	if ( has_post_thumbnail( $post ) ) :
		$image = wp_get_attachment_image_src( get_post_thumbnail_id( $post ), 'full' );
		?>
		<meta property="og:image" content="<?php echo esc_url( $image[0] ); ?>">
		<link itemprop="image" href="<?php echo esc_url( $image[0] ); ?>">

		<?php
		if ( apply_filters( 'kalium_meta_google_thumbnail', true ) ) :
			$image = wp_get_attachment_image_src( get_post_thumbnail_id( $post ), 'thumbnail' );
			?>
			<!--
			<PageMap>
			<DataObject type="thumbnail">
				<Attribute name="src" value="<?php echo esc_url( $image[0] ); ?>"/>
				<Attribute name="width" value="<?php echo esc_attr( $image[1] ); ?>"/>
				<Attribute name="height" value="<?php echo esc_attr( $image[2] ); ?>"/>
			</DataObject>
			</PageMap>
		-->
		<?php endif; ?>

		<?php
	endif;
}

add_action( 'wp_head', 'kalium_open_graph_meta', 5 );

/**
 * Disable Kalium Open Graph data output when SEO plugins are enabled.
 *
 * @param bool $enabled
 *
 * @return bool
 * @since 4.0
 */
function kalium_open_graph_meta_disable_conditionally( $enabled ) {
	if ( defined( 'WPSEO_VERSION' ) ) {
		$social = WPSEO_Options::get_option( 'wpseo_social' );

		if ( isset( $social['opengraph'] ) ) {
			return false;
		}
	} elseif ( function_exists( 'rank_math' ) || defined( 'AIOSEO_PHP_VERSION_DIR' ) || defined( 'SEOPRESS_VERSION' ) ) {
		return false;
	}

	return $enabled;
}

add_filter( 'kalium_open_graph_meta', 'kalium_open_graph_meta_disable_conditionally' );

/**
 * Grid container custom width.
 *
 * @since 4.0
 */
function kalium_set_container_max_width() {
	if ( $max_width = kalium_get_container_max_width() ) {
		kalium_set_css_var( 'content-width', $max_width );
	}
}

add_action( 'wp', 'kalium_set_container_max_width' );

/**
 * Block spacing.
 *
 * @since 4.0
 */
function kalium_set_block_spacing() {
	$spacing_sizes = wp_list_pluck(
		kalium_get_spacing_sizes(),
		'size',
		'slug'
	);

	$block_spacing = kalium_get_theme_option( 'block_spacing', true );

	if ( is_numeric( $block_spacing ) ) {
		if ( isset( $spacing_sizes[ $block_spacing ] ) ) {
			$block_spacing = $spacing_sizes[ $block_spacing ];
		} elseif ( 80 === $block_spacing ) {
			$block_spacing = kalium_get_theme_option( 'block_spacing_custom' );
		} else {
			$block_spacing = '0px';
		}

		if ( $block_spacing ) {
			kalium_set_css_var( 'block-spacing', $block_spacing );
		}
	}
}

add_action( 'wp', 'kalium_set_block_spacing' );

/**
 * Container width defined for specific post item.
 *
 * @param string $max_width
 *
 * @return string
 */
function kalium_custom_container_max_width( $max_width ) {
	if ( is_singular() ) {
		$queried_object_id = kalium_get_queried_object_id();
	} elseif ( is_admin() ) {
		if ( kalium_is_block_editor() ) {
			global $post;

			if ( $post ) {
				$queried_object_id = $post->ID;
			}
		}
	}

	// Custom in-page container width
	if ( ! empty( $queried_object_id ) && kalium_get_field( 'custom_grid_container_width', $queried_object_id ) ) {
		$container_width = kalium_get_field( 'grid_container_width', $queried_object_id );

		// Full width
		if ( kalium_get_field( 'grid_container_fullwidth', $queried_object_id ) ) {
			$max_width = '100%';
		} // Fallback compatibility with previous implementation
		elseif ( is_numeric( $container_width ) ) {
			$max_width = $container_width . 'px';
		}
		// Custom width
		elseif ( 'custom' === $container_width ) {
			$custom_width = kalium_get_field( 'grid_container_width_custom', $queried_object_id );

			if ( is_numeric( $custom_width ) ) {
				$max_width = $custom_width . 'px';
			}
		}
		// Predefined width
		elseif ( $predefined_width = kalium_get_predefined_container_max_width( $container_width ) ) {
			$max_width = $predefined_width;
		}
	}

	return $max_width;
}

add_filter( 'kalium_container_max_width', 'kalium_custom_container_max_width' );

/**
 * Show breadcrumb in the specified location of pages.
 *
 * @return void
 * @since 3.2
 */
function kalium_legacy_breadcrumb() {
	if ( ! kalium()->is->breadcrumb_navxt_active() ) {
		return;
	}

	// Current Object ID
	$object_id = kalium_get_queried_object_id();

	// Display or not
	$breadcrumb_show = kalium_validate_boolean( kalium_get_theme_option( 'breadcrumbs' ) );

	// Check breadcrumb visibility on certain pages
	if ( $breadcrumb_show ) {
		$visibility = array_map(
			'kalium_validate_boolean',
			[
				'home'            => kalium_get_theme_option( 'breadcrumb_visibility_homepage' ),
				'blog'            => kalium_get_theme_option( 'breadcrumb_visibility_blog' ),
				'portfolio'       => kalium_get_theme_option( 'breadcrumb_visibility_portfolio' ),
				'search'          => kalium_get_theme_option( 'breadcrumb_visibility_search' ),
				'not_found'       => kalium_get_theme_option( 'breadcrumb_visibility_404' ),
				'header_absolute' => kalium_get_theme_option( 'breadcrumb_visibility_absolute_header' ),
			]
		);

		if ( ! $visibility['home'] && is_front_page() ) {
			$breadcrumb_show = false;
		} elseif ( ! $visibility['portfolio'] && ( is_post_type_archive( 'portfolio' ) || is_singular( 'portfolio' ) ) ) {
			$breadcrumb_show = false;
		} elseif ( ! $visibility['blog'] && ( is_post_type_archive( 'post' ) || is_singular( 'post' ) || is_home() || is_category() || is_tag() ) ) {
			$breadcrumb_show = false;
		} elseif ( ! $visibility['search'] && is_search() ) {
			$breadcrumb_show = false;
		} elseif ( ! $visibility['not_found'] && is_404() ) {
			$breadcrumb_show = false;
		} elseif ( ! $visibility['header_absolute'] && 'absolute' === kalium_get_header_option( 'position' ) ) {
			$breadcrumb_show = false;
		}
	}

	// Single page options
	if ( is_singular() ) {
		$single_breadcrumb_show = kalium_get_field( 'breadcrumb', $object_id );

		// Force disabled for current page
		if ( 'disable' === $single_breadcrumb_show ) {
			$breadcrumb_show = false;
		} // Force enabled for current page
		elseif ( 'enable' === $single_breadcrumb_show ) {
			$breadcrumb_show = true;
		}
	}

	// Breadcrumb can display
	if ( apply_filters( 'kalium_breadcrumb_display', $breadcrumb_show ) ) {

		// Default placement in wrapper start
		$breadcrumb_hook_tag      = 'kalium_wrapper_start';
		$breadcrumb_hook_priority = 15;

		// Single portfolio pages
		if ( is_singular( 'portfolio' ) ) {
			$item_type = kalium_get_field( 'item_type', $object_id );

			// Side type
			if ( 'type-1' === $item_type ) {

				// Full background
				if ( 'fullbg' === kalium_get_field( 'gallery_type' ) ) {
					$breadcrumb_hook_tag      = 'kalium_portfolio_type_side_portfolio_before_title';
					$breadcrumb_hook_priority = 10;
				}
			} // Columned type
			elseif ( 'type-2' === $item_type ) {
				if ( kalium_get_field( 'show_featured_image', $object_id ) && 'absolute' === kalium_get_header_option( 'position' ) ) {
					$breadcrumb_hook_tag      = 'kalium_portfolio_type_columned_before_gallery';
					$breadcrumb_hook_priority = 10;
				}
			} // Fullscreen type
			elseif ( 'type-5' === $item_type ) {
				$breadcrumb_hook_tag      = 'kalium_portfolio_type_fullscreen_before_title';
				$breadcrumb_hook_priority = 10;
			}
		} // Single post
		elseif ( is_singular( 'post' ) ) {

			// Featured image placement
			$post_image_placement = kalium_get_blog_option( 'single/post_image/placement' );

			// Fullwidth image
			if ( 'full-width' === $post_image_placement ) {
				$breadcrumb_hook_tag      = 'kalium_blog_single_post_content';
				$breadcrumb_hook_priority = 5;
			}
		} // Single product
		elseif ( is_singular( 'product' ) ) {

			// Product images that are stretched to the edge of browser in plain gallery type
			if ( in_array(
				kalium_get_theme_option( 'shop_single_product_images_layout' ),
				[
					'plain',
					'plain-sticky',
				]
			) && kalium_validate_boolean( kalium_get_theme_option( 'shop_single_plain_image_stretch' ) ) ) {
				$breadcrumb_hook_tag      = 'woocommerce_single_product_summary';
				$breadcrumb_hook_priority = 0;
			}
		}

		// Insert breadcrumb hook
		add_action( $breadcrumb_hook_tag, 'kalium_breadcrumb', $breadcrumb_hook_priority );
	}
}

add_action( 'wp', 'kalium_legacy_breadcrumb' );

/**
 * Exclude post types from search.
 *
 * @param WP_Query $query
 *
 * @since 3.1.3
 */
function kalium_exclude_post_types_from_search( $query ) {
	if ( is_admin() || ! $query->is_main_query() || ! $query->is_search() ) {
		return;
	}

	if ( $exclude_post_types = kalium_get_theme_option( 'exclude_search_post_types' ) ) {
		if ( ! empty( $exclude_post_types ) ) {
			$public_post_types = array_merge(
				[ 'post', 'page' ],
				array_values(
					get_post_types(
						[
							'public'   => true,
							'_builtin' => false,
						]
					)
				)
			);

			// Allowed post types
			$post_types = array_diff( $public_post_types, $exclude_post_types );

			// Specified post type only
			$specified_post_type = kalium()->request->query( 'post_type' );

			if ( ! empty( $specified_post_type ) && in_array( $specified_post_type, $post_types ) ) {
				$post_types = $specified_post_type;
			}

			// No post type, disable search
			if ( empty( $post_types ) ) {
				$post_types = 'none';
			}

			// Set allowed post types
			$query->set( 'post_type', $post_types );
		}
	}
}

add_filter( 'pre_get_posts', 'kalium_exclude_post_types_from_search' );

/**
 * Disable jQuery migrate script.
 *
 * @param WP_Scripts $scripts
 *
 * @since 3.4
 */
function kalium_performance_disable_jquery_migrate( $scripts ) {
	if ( false === kalium_validate_boolean( kalium_get_theme_option( 'performance_jquery_migrate' ) ) ) {
		if ( ! is_admin() && isset( $scripts->registered['jquery'] ) ) {
			$script = $scripts->registered['jquery'];

			if ( $script->deps ) {
				$script->deps = array_diff(
					$script->deps,
					[
						'jquery-migrate',
					]
				);
			}
		}
	}
}

add_action( 'wp_default_scripts', 'kalium_performance_disable_jquery_migrate' );

/**
 * Disable WordPress Emoji library.
 *
 * @since 3.4
 */
function kalium_performance_disable_wp_emoji() {
	if ( false === kalium_get_theme_option( 'performance_wp_emoji' ) ) {
		remove_action( 'wp_head', 'print_emoji_detection_script', 7 );
		remove_action( 'wp_print_styles', 'print_emoji_styles' );
	}
}

add_action( 'wp_enqueue_scripts', 'kalium_performance_disable_wp_emoji', 0 );

/**
 * Disable WordPress embed library.
 *
 * @since 3.4
 */
function kalium_performance_disable_wp_embed() {
	if ( false === kalium_get_theme_option( 'performance_wp_embed' ) ) {
		remove_action( 'wp_head', 'wp_oembed_add_host_js' ); // Deprecated
		remove_filter( 'embed_oembed_html', 'wp_maybe_enqueue_oembed_host_js' );
	}
}

add_action( 'wp_enqueue_scripts', 'kalium_performance_disable_wp_embed' );

/**
 * JPEG quality.
 *
 * @param int $quality
 *
 * @return int
 * @since 3.4
 */
function kalium_performance_jpeg_quality( $quality ) {
	if ( is_numeric( $jpeg_quality = kalium_get_theme_option( 'performance_jpeg_quality' ) ) ) {
		return intval( $jpeg_quality );
	}

	return $quality;
}

add_filter( 'jpeg_quality', 'kalium_performance_jpeg_quality' );

/**
 * Kalium JS API.
 *
 * @param bool $force
 *
 * @since 4.0
 * @since 4.3 - $force parameter added
 */
function kalium_print_js_api( $force = false ) {
	if ( ! doing_action( 'wp_head' ) && ! $force ) {
		return;
	}

	$attributes = [];

	// Add this when WPRocket is active
	if ( defined( 'WP_ROCKET_VERSION' ) ) {
		$attributes['nowprocket'] = true;
	}

	wp_print_inline_script_tag( file_get_contents( kalium()->locate_file( 'assets/js/kalium.min.js' ) ), $attributes );
}

add_action( 'wp_print_scripts', 'kalium_print_js_api', 0 );

/**
 * Kalium JS vars.
 *
 * @since 4.0
 */
function kalium_print_js_vars() {
	kalium_js_define_vars(
		[
			// Assets directory
			'assetsUrl'            => kalium()->assets_url(),

			// Lightbox settings
			'lightboxSettings'     => kalium_get_lightbox_settings(),

			// Mobile menu breakpoint
			'mobileMenuBreakpoint' => kalium_get_mobile_menu_breakpoint(),

			// Debug mode
			'debugMode'            => kalium()->is->debugging(),
		]
	);
}

add_action( 'wp_head', 'kalium_print_js_vars' );

/**
 * Kalium JS API libraries.
 *
 * @since 4.0
 */
function kalium_print_js_libraries() {
	$libraries = [];

	$map_styles_src = function ( $handle ) {
		$item = kalium()->assets->get_style( $handle );

		return add_query_arg( 'ver', $item['ver'], $item['src'] );
	};

	$map_scripts_src = function ( $handle ) {
		$item = kalium()->assets->get_script( $handle );

		return add_query_arg( 'ver', $item['ver'], $item['src'] );
	};

	// Register libraries for JS API
	foreach ( kalium()->assets->get_js_api_libraries() as $library_name => $items ) {
		if ( ! is_array( $items ) ) {
			$items = [ $items ];
		}

		// Map enqueues
		$items = array_map(
			function ( $handle ) use ( &$map_styles_src, &$map_scripts_src ) {

				return array_merge(
					array_map(
						$map_styles_src,
						kalium()->assets->get_styles_list( $handle )
					),
					array_map(
						$map_scripts_src,
						kalium()->assets->get_scripts_list( $handle )
					)
				);
			},
			$items
		);

		// Flatten values
		$items = array_merge( ...array_values( $items ) );

		if ( ! empty( $items ) ) {
			$libraries[ $library_name ] = $items;
		}
	}

	kalium_js_define_vars(
		[
			'libraries' => $libraries,
		]
	);
}

add_action( 'wp_footer', 'kalium_print_js_libraries' );

/**
 * Init image placeholder handler.
 */
function kalium_image_placeholder_init() {

	// Use image dominant color
	if ( 'dominant-color' === kalium_get_theme_option( 'image_loading_placeholder_bg_type' ) ) {
		kalium()->images->use_dominant_color( true );
	}

	// Placeholder Type
	switch ( kalium_get_theme_option( 'image_loading_placeholder_type' ) ) {

		// Loading animation
		case 'loading-animation':
			$loading_animation = kalium_get_css_loading_animation( kalium_get_theme_option( 'image_loading_placeholder_loading_animation' ) );

			if ( $loading_animation ) {

				// Enqueue loaders
				kalium_enqueue( 'css-loaders' );

				// Assign CSS loading animation
				kalium()->images->set_before_content( sprintf( '<span class="loader">%s</span>', $loading_animation['content'] ) );
			}
			break;

		// Custom icon
		case 'custom-icon':
			$custom_icon = kalium_get_attachment_id_from_theme_option( 'image_loading_placeholder_custom_icon' );

			if ( false !== wp_get_attachment_image_src( $custom_icon, 'full' ) ) {
				kalium()->images->set_before_content( '<span class="loader"><span class="loader-el custom-icon"></span></span>' );
			}
			break;
	}
}

add_action( 'wp_loaded', 'kalium_image_placeholder_init' );

/**
 * Image placeholder style.
 *
 * @since 4.0
 */
function kalium_print_image_placeholder_style() {
	$placeholder_type = kalium_get_theme_option( 'image_loading_placeholder_type' );
	$loader_color     = kalium_get_theme_option( 'image_loading_placeholder_loader_color' );

	// CSS vars
	$vars = [];

	// Background
	$bg_type = kalium_get_theme_option( 'image_loading_placeholder_bg_type' );

	if ( 'color' === $bg_type ) {
		$vars['placeholder-bg'] = kalium_get_theme_option( 'image_loading_placeholder_bg_color' );
	} elseif ( 'gradient' === $bg_type ) {
		$vars['placeholder-bg'] = kalium_get_theme_option( 'image_loading_placeholder_bg_gradient' );
	}

	// When Loading Animation and Custom Icon placeholder type are selected
	if ( in_array( $placeholder_type, [ 'loading-animation', 'custom-icon' ] ) ) {

		// Loader Color
		if ( 'loading-animation' === $placeholder_type ) {
			$vars['loader-color'] = $loader_color;
		}

		// Custom Icon
		if ( 'custom-icon' === $placeholder_type ) {
			$custom_icon = kalium_get_attachment_id_from_theme_option( 'image_loading_placeholder_custom_icon' );

			if ( $image = wp_get_attachment_image_src( $custom_icon, 'full' ) ) {
				$vars['loader-icon'] = sprintf( 'url(%s)', esc_attr( $image[0] ) );
			}
		}

		// Alignment
		if ( $align = kalium_get_theme_option( 'image_loading_placeholder_align' ) ) {
			list( $align_vertical, $align_horizontal ) = explode( ' ', $align );

			if ( in_array( $align_vertical, [ 'top', 'bottom' ] ) ) {
				$vars['loader-align-y'] = str_replace( [ 'top', 'bottom' ], [ 'start', 'end' ], $align_vertical );
			}

			if ( in_array( $align_horizontal, [ 'left', 'right' ] ) ) {
				$vars['loader-align-x'] = str_replace( [ 'left', 'right' ], [ 'start', 'end' ], $align_horizontal );
			}

			// Apply spacing
			if ( 'center center' !== $align ) {
				$spacing = kalium_get_theme_option( 'image_loading_placeholder_spacing' );

				if ( is_numeric( $spacing ) ) {
					$vars['loader-spacing'] = $spacing . 'px';
				}
			}
		}

		// Size
		if ( $size = kalium_get_theme_option( 'image_loading_placeholder_size' ) ) {
			$vars['loader-size'] = $size . 'px';
		}
	}

	// Print styles
	kalium_print_inline_style(
		[
			'selector' => '.image-placeholder',
			'vars'     => $vars,
		]
	);
}

add_action( 'wp_head', 'kalium_print_image_placeholder_style', 20 );

/**
 * Replace color references for color type attributes in Laborator Builder.
 *
 * @param \Laborator_Builder\Element $element
 *
 * @since 4.0
 */
function kalium_laborator_builder_element_replace_color_references( $element ) {
	foreach ( $element->get_attributes() as $attribute ) {
		if ( 'color' === $attribute->get_type() ) {
			$attribute->get_value_object()->map(
				function ( $color ) {
					return kalium_replace_color_references( $color );
				},
				true
			);
		}
	}
}

add_action( 'laborator_builder_element', 'kalium_laborator_builder_element_replace_color_references' );

/**
 * Footer visibility on single item.
 *
 * @param bool $visibility
 *
 * @return bool
 * @since 4.0
 */
function kalium_footer_visibility_single_page( $visibility ) {
	$queried_object = get_queried_object();

	if ( $queried_object instanceof WP_Post ) {
		switch ( kalium_get_field( 'footer_visibility', $queried_object ) ) {
			case 'show':
				$visibility = true;
				break;

			case 'hide':
				$visibility = false;
				break;
		}
	}

	return $visibility;
}

add_action( 'kalium_theme_option_footer_enabled', 'kalium_footer_visibility_single_page' );

/**
 * Social networks shortcode.
 *
 * This shortcode is part of Kalium since very first release and for
 * backwards compatibility reasons it will be still part of the theme.
 *
 * @param array $attributes
 *
 * @return string
 * @since 1.0
 */
function lab_social_networks_shortcode( $attributes ) {
	$args = wp_parse_args(
		$attributes,
		[
			'icon'  => false,
			'label' => true,
			'echo'  => false,
		]
	);

	/**
	 * Legacy attributes.
	 */
	$legacy_attributes = wp_parse_args(
		$attributes,
		[
			'rounded'    => '',
			'colored'    => '',
			'colored-bg' => '',
		]
	);

	// Rounded
	if ( in_array( 'rounded', $legacy_attributes ) ) {
		$args['icon']         = true;
		$args['label']        = false;
		$args['shape_type']   = 'rounded';
		$args['color']        = 'custom';
		$args['color_custom'] = kalium_color_reference( 'theme_colors_headings' );
	}

	// Colored
	if ( in_array( 'colored', $legacy_attributes ) ) {
		$args['color'] = 'brand';
	}

	// Colored hover
	if ( 'hover' === $legacy_attributes['colored'] ) {
		$args['color_hover'] = 'brand';
	}

	// Colored background
	if ( in_array( 'colored-bg', $legacy_attributes ) || ! empty( $legacy_attributes['colored-bg'] ) ) {
		$args['icon']       = true;
		$args['label']      = false;
		$args['background'] = 'brand';

		if ( empty( $args['shape_type'] ) ) {
			$args['shape_type'] = 'square';
		}
	}

	// Colored background hover
	if ( 'hover' === $legacy_attributes['colored-bg'] ) {
		$args['background']         = null;
		$args['background_hover']   = 'brand';
		$args['color_hover']        = 'custom';
		$args['color_hover_custom'] = '#fff';
	}

	return kalium_social_networks( $args );
}

add_shortcode( 'lab_social_networks', 'lab_social_networks_shortcode' );

/**
 * The new Social icons shortcode that replaces the previous [lab_social_networks].
 *
 * @param array $atts
 *
 * @return string
 * @since 4.0
 */
function kalium_social_icons_shortcode( $atts ) {
	$color_atts = [
		'color'            => '',
		'color_hover'      => '',
		'label_color'      => '',
		'label_hover'      => '',
		'background'       => '',
		'background_hover' => '',
	];

	$atts = shortcode_atts(
		array_merge(
			[
				'icon'      => 'yes',
				'label'     => 'no',
				'spacing'   => '',
				'size'      => '',
				'radius'    => '',
				'outline'   => '',
				'new_tab'   => '',
				'no_follow' => '',
			],
			$color_atts
		),
		$atts
	);

	// Social networks args
	$social_networks = [
		'echo'    => false,
		'icon'    => kalium_validate_boolean( $atts['icon'] ),
		'label'   => kalium_validate_boolean( $atts['label'] ),
		'spacing' => $atts['spacing'],
		'size'    => $atts['size'],
	];

	// Radius
	if ( in_array( $atts['radius'], [ 'rounded', 'square' ] ) ) {
		$social_networks['shape_type'] = $atts['radius'];
	} elseif ( $atts['radius'] ) {
		$social_networks['shape_type']   = 'custom';
		$social_networks['shape_radius'] = $atts['radius'];
	}

	// Outline
	if ( ! empty( $social_networks['shape_type'] ) && kalium_validate_boolean( $atts['outline'] ) ) {
		$social_networks['shape_fill'] = 'outline';
	}

	// New window
	if ( '' !== $atts['new_tab'] ) {
		$social_networks['new_tab'] = kalium_validate_boolean( $atts['new_tab'] );
	}

	// No follow
	if ( '' !== $atts['no_follow'] ) {
		$social_networks['no_follow'] = kalium_validate_boolean( $atts['no_follow'] );
	}

	// Colors
	foreach ( $color_atts as $color_attr => $default_value ) {
		if ( empty( $atts[ $color_attr ] ) ) {
			continue;
		}

		$attr_value = $atts[ $color_attr ];

		if ( 'brand' === $attr_value ) {
			$social_networks[ $color_attr ] = $attr_value;
		} else {
			$social_networks[ $color_attr ]             = 'custom';
			$social_networks[ $color_attr . '_custom' ] = $attr_value;
		}
	}

	return kalium_social_networks( $social_networks );
}

add_shortcode( 'kalium_social_icons', 'kalium_social_icons_shortcode' );

/**
 * Page heading for single page.
 */
function kalium_single_page_heading() {
	$queried_object_id = kalium_get_queried_object_id();

	// Do not show on archive pages
	if ( ! is_singular() || ! $queried_object_id || kalium_is_maintenance_mode() ) {
		return;
	}

	// Custom page heading (defined on post)
	if ( kalium_get_field( 'heading_title', $queried_object_id ) ) {
		$current_post       = get_post( $queried_object_id );
		$title_type         = kalium_get_field( 'page_heading_title_type', $queried_object_id );
		$description_type   = kalium_get_field( 'page_heading_description_type', $queried_object_id );
		$custom_title       = kalium_get_field( 'page_heading_custom_title', $queried_object_id ) ?: '';
		$custom_description = kalium_get_field( 'page_heading_custom_description', $queried_object_id ) ?: '';

		// Sanitize title and description
		$custom_title       = wp_kses_post( $custom_title );
		$custom_description = kalium_format_content( wp_kses_post( $custom_description ) );

		// Inherit from post title
		if ( 'post_title' === $title_type ) {
			$custom_title = get_the_title( $current_post );
		}

		// Inherit from post content
		if ( 'post_content' === $description_type ) {
			$custom_description = apply_filters( 'the_content', get_the_content( null, false, $current_post ) );
		}

		// Page heading visibility
		kalium()->page->show_page_heading( true );

		// Page heading data
		kalium()->page->set_page_heading(
			[
				'title'       => $custom_title,
				'description' => $custom_description,
			]
		);
	}
}

add_action( 'wp_head', 'kalium_single_page_heading' );

/**
 * General body class.
 *
 * @param array $classes
 *
 * @return array
 */
function kalium_body_classes( $classes ) {
	if ( kalium_get_theme_option( 'theme_borders' ) ) {
		$classes[] = 'page-borders';

		$page_border_visibility = array_map(
			function ( $hidden_class ) {
				return 'page-borders--' . $hidden_class;
			},
			kalium_get_device_visibility_classes(
				kalium_get_theme_option( 'theme_borders_visibility' )
			)
		);

		$classes = array_merge( $classes, $page_border_visibility );
	}

	if ( 'fixed' === kalium_get_theme_option( 'footer_type' ) ) {
		$classes[] = 'has-fixed-footer';
	}

	return $classes;
}

add_filter( 'body_class', 'kalium_body_classes' );

/**
 * Link style.
 *
 * @since 4.0
 */
function kalium_print_link_styles() {
	$link_style = kalium_get_theme_option( 'link_style' );

	/**
	 * Prefixes link style var name.
	 *
	 * @param string $name
	 *
	 * @return string
	 */
	$link_style_var_name = function ( $name ) {
		return "ls-{$name}";
	};

	/**
	 * Get link style content.
	 *
	 * @param string $name
	 *
	 * @return string
	 */
	$get_link_style = function ( $name ) {
		$file_path = kalium()->assets_path( 'css/link-styles/' . $name . '.min.css' );

		if ( file_exists( $file_path ) ) {
			return file_get_contents( $file_path );
		}
		return null;
	};

	// Vars and style
	$style = null;
	$vars  = [];

	// Reverse state
	$is_reverse = kalium_get_theme_option( 'link_style_reverse_state' );

	// Link styles
	switch ( $link_style ) {

		// Underline
		case 'underline':
			$style     = 'underline-animated';
			$is_static = 'static' === kalium_get_theme_option( 'link_style_underline_type' );
			$thickness = kalium_get_theme_option( 'link_style_underline_thickness' );
			$position  = kalium_get_theme_option( 'link_style_underline_position' );
			$color     = kalium_get_theme_option( 'link_style_underline_color' );

			// Static
			if ( $is_static ) {
				$style = 'underline-static';
			} else {

				// Animation
				switch ( kalium_get_theme_option( 'link_style_underline_animation' ) ) {

					// Left
					case 'left':
						$vars[ $link_style_var_name( 'line-direction-value' ) ]       = '100%';
						$vars[ $link_style_var_name( 'line-direction-hover-value' ) ] = '100%';
						break;

					// Left to right
					case 'left-to-right':
						$vars[ $link_style_var_name( 'line-direction-hover-value' ) ] = '100%';
						$vars[ $link_style_var_name( 'transition-value' ) ]           = kalium_css_var_reference( $link_style_var_name( 'transition-left-to-right' ) );
						break;

					// Inside Out
					case 'inside-out':
						$vars[ $link_style_var_name( 'line-direction-value' ) ] = '50%';
						break;

					// Outside In
					case 'outside-in':
						$vars[ $link_style_var_name( 'is-outside-in' ) ]          = kalium_css_var_reference( 'on' );
						$vars[ $link_style_var_name( 'line-width-hover-value' ) ] = '50%';
						break;
				}
			}

			// Thickness
			if ( ! empty( $thickness ) ) {
				$vars[ $link_style_var_name( 'line-thickness-value' ) ] = $thickness;
			}

			// Position
			if ( 'center' === $position ) {
				if ( $is_static ) {
					$vars[ $link_style_var_name( 'line-hover-value' ) ] = 'line-through';
				} else {
					$vars[ $link_style_var_name( 'line-position-value' ) ] = '55%';
				}
			}

			// Color
			if ( ! empty( $color ) ) {
				$vars[ $link_style_var_name( 'color-value' ) ] = $vars[ $link_style_var_name( 'color-hover-value' ) ] = $color;
			}
			break;

		// Swap
		case 'swap':
			$style      = 'swap';
			$is_reverse = false;
			$direction  = kalium_get_theme_option( 'link_style_swap_direction' );

			if ( 'bottom' === $direction ) {
				$vars[ $link_style_var_name( 'swap-direction-1-hover-value' ) ] = '1';
				$vars[ $link_style_var_name( 'swap-direction-2-value' ) ]       = '-1';
			}
			break;

		// Fill
		case 'fill':
			$style = 'fill-text';

			// Fill background
			if ( 'background' === kalium_get_theme_option( 'link_style_fill_type' ) ) {
				$style = 'fill-background';

				switch ( kalium_get_theme_option( 'link_style_fill_bg_direction' ) ) {

					// Left
					case 'left':
						$vars[ $link_style_var_name( 'fill-bg-height-value' ) ] = '100%';
						$vars[ $link_style_var_name( 'fill-bg-width-value' ) ]  = '0%';
						$vars[ $link_style_var_name( 'fill-bg-pos-x-value' ) ]  = '100%';
						break;

					// Right
					case 'right':
						$vars[ $link_style_var_name( 'fill-bg-height-value' ) ] = '100%';
						$vars[ $link_style_var_name( 'fill-bg-width-value' ) ]  = '0%';
						break;

					// Bottom
					case 'bottom':
						$vars[ $link_style_var_name( 'fill-bg-pos-y-value' ) ] = '0%';
						break;

					// Top
					default:
						// Underline
						$show_underline      = kalium_get_theme_option( 'link_style_fill_bg_underline' );
						$underline_thickness = kalium_get_theme_option( 'link_style_fill_bg_underline_thickness' );

						if ( $show_underline ) {

							// Default underline height
							if ( empty( $underline_thickness ) ) {
								$underline_thickness = '.1em';
							}

							$vars[ $link_style_var_name( 'fill-bg-height-value' ) ] = $underline_thickness;
						}
				}
			} else {
				$direction = kalium_get_theme_option( 'link_style_fill_text_direction' );

				if ( 'backwards' === $direction ) {
					$vars[ $link_style_var_name( 'is-backwards' ) ] = kalium_css_var_reference( 'on' );

					// Change direction
					$vars[ $link_style_var_name( 'fill-direction-value' ) ]       = '0%';
					$vars[ $link_style_var_name( 'fill-direction-hover-value' ) ] = '100%';
				}
			}
			break;
	}

	// Link style
	if ( $style ) {

		// Link style
		kalium_print_inline_style(
			[
				'id'  => 'link-style',
				'css' => $get_link_style( $style ),
			]
		);

		// Link style options
		kalium_print_inline_style(
			[
				'id'       => 'link-style-options',
				'selector' => ':root',
				'vars'     => $vars,
			]
		);

		// Link style class
		kalium()->helpers->add_body_class( 'link-style-' . $style );

		// Reverse state
		if ( $is_reverse ) {
			kalium()->helpers->add_body_class( 'link-style-reverse' );
		}
	}
}

add_action( 'wp_head', 'kalium_print_link_styles', 20 );

/**
 * Button style.
 *
 * @since 4.0
 */
function kalium_print_button_style() {
	$vars = [];

	// Border
	if ( $button_border = kalium_get_theme_option( 'button_border' ) ) {
		$button_border = kalium_to_border_box( $button_border );

		if ( is_array( $button_border ) ) {
			foreach ( $button_border as $prop => $value ) {
				if ( ! empty( $value ) ) {
					$vars[ "button-border-{$prop}" ] = $value;
				}
			}
		}

		// Hover border
		$vars['button-hover-border-color'] = kalium_get_theme_option( 'button_border_hover' );

		// Active border
		$vars['button-active-border-color'] = kalium_get_theme_option( 'button_border_active' );
	}

	// Shadow
	$button_shadow = kalium_get_theme_option( 'button_shadow' );

	$vars['button-shadow-color'] = kalium_to_box_shadow_color( $button_shadow );

	if ( kalium_has_box_shadow_size( $button_shadow ) ) {
		$vars['button-shadow-size'] = kalium_to_box_shadow_size( $button_shadow, '0 0 0' );
	}

	// Shadow on hover
	$button_hover_shadow = kalium_get_theme_option( 'button_shadow_hover' );

	$vars['button-hover-shadow-color'] = kalium_to_box_shadow_color( $button_hover_shadow );

	if ( kalium_has_box_shadow_size( $button_hover_shadow ) ) {
		$vars['button-hover-shadow-size'] = kalium_to_box_shadow_size( $button_hover_shadow, '0 2px 8px' );
	}

	// Shadow on active
	$button_active_shadow = kalium_get_theme_option( 'button_shadow_active' );

	$vars['button-active-shadow-color'] = kalium_to_box_shadow_color( $button_active_shadow );

	if ( kalium_has_box_shadow_size( $button_active_shadow ) ) {
		$vars['button-active-shadow-size'] = kalium_to_box_shadow_size( $button_active_shadow, '0 2px 8px' );
	}

	// Padding
	$vars['button-padding'] = kalium_map_responsive_value(
		kalium_get_theme_option( 'button_padding' ),
		function ( $padding ) {
			return kalium_to_length( $padding, '.375rem .75rem' );
		}
	);

	// Border radius
	$vars['button-border-radius'] = kalium_map_responsive_value(
		kalium_get_theme_option( 'button_radius' ),
		function ( $radius ) {
			return kalium_to_length( $radius, kalium_css_var_reference( 'border-radius' ) );
		}
	);

	// Print button style
	kalium_print_inline_style(
		[
			'id'       => 'button-style',
			'selector' => kalium_get_root_selector(),
			'vars'     => $vars,
		]
	);
}

add_action( 'wp_head', 'kalium_print_button_style', 20 );

/**
 * Form style.
 *
 * @since 4.0
 */
function kalium_print_form_style() {
	$vars = [];

	// Form style - Type 2
	if ( 'type-2' === kalium_get_theme_option( 'form_style' ) ) {
		kalium()->helpers->add_body_class( 'form-style-2' );
	}

	// Placeholder color
	$vars['input-placeholder-color'] = kalium_get_theme_option( 'form_input_placeholder_color' );

	// Text color
	$input_color = kalium_get_theme_option( 'form_input_color' );
	$text_colors = [
		'input-color'       => 'normal',
		'input-hover-color' => 'hover',
		'input-focus-color' => 'focus',
	];

	foreach ( $text_colors as $var_name => $prop ) {
		if ( ! empty( $input_color[ $prop ] ) ) {
			$vars[ $var_name ] = $input_color[ $prop ];
		}
	}

	// Background color
	$input_bg  = kalium_get_theme_option( 'form_input_bg' );
	$bg_colors = [
		'input-bg'       => 'normal',
		'input-hover-bg' => 'hover',
		'input-focus-bg' => 'focus',
	];

	foreach ( $bg_colors as $var_name => $prop ) {
		if ( ! empty( $input_bg[ $prop ] ) ) {
			$vars[ $var_name ] = $input_bg[ $prop ];
		}
	}

	// Border color
	$input_border  = kalium_get_theme_option( 'form_input_border' );
	$border_colors = [
		'input-border-color'       => 'normal',
		'input-hover-border'       => 'hover',
		'input-focus-border-color' => 'focus',
	];

	foreach ( $border_colors as $var_name => $prop ) {
		if ( ! empty( $input_border[ $prop ] ) ) {
			$vars[ $var_name ] = $input_border[ $prop ];
		}
	}

	// Border width
	$vars['input-border-width'] = kalium_map_responsive_value(
		kalium_get_theme_option( 'form_input_border_width' ),
		function ( $value ) {
			if ( is_numeric( $value ) ) {
				return $value . 'px';
			}
		}
	);

	// Padding
	$vars['input-padding'] = kalium_map_responsive_value(
		kalium_get_theme_option( 'form_input_padding' ),
		function ( $padding ) {
			return kalium_to_length( $padding, '8px 12px' );
		}
	);

	// Border radius
	$vars['input-border-radius'] = kalium_map_responsive_value(
		kalium_get_theme_option( 'form_input_radius' ),
		function ( $radius ) {
			return kalium_to_length( $radius, kalium_css_var_reference( 'border-radius' ) );
		}
	);

	// Checkbox border radius
	$vars['checkbox-border-radius'] = kalium_map_responsive_value(
		kalium_get_theme_option( 'form_checkbox_radius' ),
		function ( $radius ) {
			return kalium_to_length( $radius, kalium_css_var_reference( 'border-radius' ) );
		}
	);

	// Checkbox color
	$form_checkbox_color            = kalium_get_theme_option( 'form_checkbox_color' );
	$vars['checkbox-checked-color'] = $form_checkbox_color;

	// Shadow
	$form_input_shadow = kalium_get_theme_option( 'form_input_shadow' );

	$vars['input-shadow-color'] = kalium_to_box_shadow_color( $form_input_shadow );

	if ( kalium_has_box_shadow_size( $form_input_shadow ) ) {
		$vars['input-shadow-size'] = kalium_to_box_shadow_size( $form_input_shadow, '0 1px 2px' );
	}

	// Shadow on hover
	$form_input_hover_shadow = kalium_get_theme_option( 'form_input_shadow_hover' );

	$vars['input-hover-shadow-color'] = kalium_to_box_shadow_color( $form_input_hover_shadow );

	if ( kalium_has_box_shadow_size( $form_input_hover_shadow ) ) {
		$vars['input-hover-shadow-size'] = kalium_to_box_shadow_size( $form_input_hover_shadow, $vars['input-shadow-size'] ?? '0 1px 2px' );
	}

	// Shadow on focus
	$form_input_focus_shadow = kalium_get_theme_option( 'form_input_shadow_focus' );

	$vars['input-focus-shadow-color'] = kalium_to_box_shadow_color( $form_input_focus_shadow );

	if ( kalium_has_box_shadow_size( $form_input_focus_shadow ) ) {
		$vars['input-focus-shadow-size'] = kalium_to_box_shadow_size( $form_input_focus_shadow, '0 0 0 .25rem' );
	}

	// Textarea height
	$vars['textarea-height'] = kalium_map_responsive_value(
		kalium_get_theme_option( 'form_textarea_height' ),
		function ( $height ) {
			if ( is_numeric( $height ) ) {
				return $height . 'px';
			}
		}
	);

	// Print form style
	kalium_print_inline_style(
		[
			'id'       => 'form-style',
			'selector' => ':root',
			'vars'     => $vars,
		]
	);
}

add_action( 'wp_head', 'kalium_print_form_style', 20 );

/**
 * Dropdown menu caret.
 *
 * @param string   $item_output
 * @param WP_Post  $menu_item
 * @param int      $depth
 * @param stdClass $args
 *
 * @return string
 * @since 4.0
 */
function kalium_menu_add_dropdown_toggle( $item_output, $menu_item, $depth, $args ) {
	if ( ! empty( $args->dropdown_toggle ) && ! empty( $menu_item->classes ) && in_array( 'menu-item-has-children', $menu_item->classes ) ) {
		$icon = ! empty( $args->dropdown_submenu_alt ) && $depth > 0 ? 'arrow-right' : 'arrow-down';

		$dropdown_icon = sprintf( '<span class="dropdown-toggle">%s</span>', kalium_get_icon( $icon ) );
		$item_output   = str_replace( '</a>', $dropdown_icon . '</a>', $item_output );
	}

	return $item_output;
}

add_filter( 'walker_nav_menu_start_el', 'kalium_menu_add_dropdown_toggle', 10, 4 );

/**
 * Image placeholder load event.
 *
 * @since 4.0
 */
function kalium_image_placeholder_load_event() {
	/*
	document.addEventListener( 'load', function( ev ) {
		if ( 'loading' === document.readyState && 'IMG' === ev.target.tagName && ev.target.complete ) {
			var parentEl = ev.target.parentElement;

			if ( 'PICTURE' === parentEl.tagName ) {
				parentEl = parentEl.parentElement;
			}

			if ( parentEl.classList.contains( 'image-placeholder' ) ) {
				parentEl.classList.add( 'loaded' );
			}
		}
	}, true );
	*/

	?>
	<script>
		document.addEventListener("load",(function(e){if("loading"===document.readyState&&"IMG"===e.target.tagName&&e.target.complete){var t=e.target.parentElement;"PICTURE"===t.tagName&&(t=t.parentElement),t.classList.contains("image-placeholder")&&t.classList.add("loaded")}}),!0);
	</script>
	<?php
}

add_action( 'wp_print_scripts', 'kalium_image_placeholder_load_event' );

/**
 * Return correct image dimensions for SVG files.
 */
function kalium_images_correct_svg_dimensions( $image, $attachment_id = null ) {
	if ( is_array( $image ) && kalium()->is->svg( $image[0] ) && $image[1] === $image[2] && 2 >= $image[1] + $image[2] ) {
		list( $image[1], $image[2] ) = kalium()->helpers->get_svg_dimensions( $attachment_id );
	}

	return $image;
}

add_filter( 'wp_get_attachment_image_src', 'kalium_images_correct_svg_dimensions', 10, 2 );

/**
 * Disable comments in attachment pages.
 *
 * @param bool $comments_enabled
 *
 * @return bool
 * @since 4.0
 */
function kalium_attachment_disable_comments( $comments_enabled ) {
	return is_attachment() ? false : $comments_enabled;
}

add_filter( 'kalium_blog_comments', 'kalium_attachment_disable_comments', 1000 );

/**
 * Sticky sidebar.
 *
 * @param array $classes
 * @param array $sidebar_options
 *
 * @return array
 * @since 4.0
 */
function kalium_sidebar_sticky_behavior( $classes, $sidebar_options ) {

	// Sticky sidebar
	if ( $sidebar_options['sticky'] ) {

		// Sticky behavior
		switch ( $sidebar_options['sticky_behavior'] ) {
			// Widgets
			case 'widgets':
				if ( is_numeric( $last_widgets = $sidebar_options['sticky_last_widgets'] ) ) {

					/**
					 * Widget wrapper.
					 *
					 * @param array $widget
					 * @param array $content
					 */
					$wrap_widget = function ( &$widget, $content ) {
						if ( ! $widget ) {
							return;
						}

						$content = wp_parse_args(
							$content,
							[
								'before' => '',
								'after'  => '',
							]
						);

						$callback = $widget['callback'];

						$widget['callback'] = function () use ( &$content, &$callback ) {
							echo $content['before'];
							call_user_func_array( $callback, func_get_args() );
							echo $content['after'];
						};
					};

					/**
					 * Last sticky widgets wrapper.
					 *
					 * @param string $index
					 */
					$wrap_sticky_last_widgets = function ( $index ) use ( &$wrap_widget, $last_widgets ) {
						global $wp_registered_widgets;

						$sidebars_widgets = wp_get_sidebars_widgets()[ $index ];
						$last_widgets_arr = array_slice( $sidebars_widgets, -1 * $last_widgets );

						if ( ! empty( $last_widgets_arr ) ) {
							$first_widget = & $wp_registered_widgets[ reset( $last_widgets_arr ) ];
							$last_widget  = & $wp_registered_widgets[ end( $last_widgets_arr ) ];

							// Starting widget
							$wrap_widget( $first_widget, [ 'before' => '<div class="sidebar-sticky">' ] );

							// Ending widget
							$wrap_widget( $last_widget, [ 'after' => '</div>' ] );
						}
					};

					add_action( 'dynamic_sidebar_before', $wrap_sticky_last_widgets );
					add_action(
						'dynamic_sidebar_after',
						function () use ( $wrap_sticky_last_widgets ) {
							remove_action( 'dynamic_sidebar_before', $wrap_sticky_last_widgets );
						}
					);
				}
				break;

			// Sidebar
			default:
				$classes[] = 'sidebar-sticky';
		}

		// Sticky offset
		if ( is_numeric( $sticky_offset = $sidebar_options['sticky_offset'] ) ) {
			$print_sticky_offset = function () use ( &$print_sticky_offset, $sticky_offset ) {
				kalium_print_inline_style(
					[
						'selector' => '.sidebar',
						'vars'     => [
							'sticky-offset' => $sticky_offset . 'px',
						],
					]
				);

				remove_action( 'dynamic_sidebar_before', $print_sticky_offset );
			};

			add_action( 'dynamic_sidebar_before', $print_sticky_offset );
		}
	}

	return $classes;
}

add_filter( 'kalium_sidebar_class', 'kalium_sidebar_sticky_behavior', 10, 2 );

/**
 * Preload Kalium theme icons.
 *
 * @since 4.0
 */
function kalium_preload_theme_icons_font() {
	if ( kalium_get_theme_option( 'kalium_preload_theme_icons' ) ) {
		printf( '<link rel="preload" href="%s" as="font" type="font/woff" crossorigin>', esc_attr( kalium()->assets_url( 'icons/main/KaliumIcons.woff' ) ) );
	}
}

add_action( 'wp_print_styles', 'kalium_preload_theme_icons_font' );

/**
 * Add do_shortcode support for text widgets.
 *
 * @param string $text
 *
 * @return string
 */
function kalium_add_support_for_shortcodes_in_text_widget( $text ) {
	return do_shortcode( $text );
}

add_filter( 'widget_text', 'kalium_add_support_for_shortcodes_in_text_widget' );

/**
 * Date shortcode.
 *
 * @param array $atts
 *
 * @return string
 */
function kalium_shortcode_date( $atts = [] ) {
	$atts = shortcode_atts(
		[
			'format' => get_option( 'date_format' ),
		],
		$atts
	);

	return date_i18n( $atts['format'] );
}

if ( ! shortcode_exists( 'date' ) ) {
	add_shortcode( 'date', 'kalium_shortcode_date' );
}

/**
 * Disable adaptive images for custom image sizes only.
 *
 * @since 4.0
 */
function kalium_disable_adaptive_for_custom_image_sizes() {
	static $current_image_size;

	add_filter(
		'wp_get_attachment_image_src',
		function ( $image, $attachment_id, $size ) use ( &$current_image_size ) {
			$current_image_size = $size;
			return $image;
		},
		1000,
		3
	);

	add_filter(
		'wp_calculate_image_srcset_meta',
		function ( $image_meta ) use ( &$current_image_size ) {

			if ( is_array( $current_image_size ) ) {
				return false;
			}

			return $image_meta;
		},
		1000,
		4
	);
}

add_action( 'after_setup_theme', 'kalium_disable_adaptive_for_custom_image_sizes' );

/**
 * Save widgets locations when switching the theme.
 *
 * @param string   $new_name
 * @param WP_Theme $new_theme
 * @param WP_Theme $old_theme
 *
 * @since 4.0
 */
function kalium_save_current_sidebars_widgets( $new_name, $new_theme, $old_theme ) {
	$transient = '_kalium_sidebars_widgets_' . $old_theme->get_stylesheet();

	if ( false === get_transient( $transient ) ) {
		set_transient(
			$transient,
			[
				'timestamp' => time(),
				'data'      => wp_get_sidebars_widgets(),
			]
		);
	}
}

add_action( 'switch_theme', 'kalium_save_current_sidebars_widgets', 10, 3 );

/**
 * Restore sidebars widgets when the theme is activated.
 *
 * @since 4.0
 */
function kalium_restore_sidebars_widgets() {
	$transient = '_kalium_sidebars_widgets_' . get_stylesheet();
	$widgets   = get_transient( $transient );

	if ( $widgets && time() - $widgets['timestamp'] < DAY_IN_SECONDS ) {
		wp_set_sidebars_widgets( $widgets['data'] );
		delete_transient( $transient );
	}
}

add_action( 'after_switch_theme', 'kalium_restore_sidebars_widgets' );

/**
 * Password protected page or any post type in singular page.
 *
 * @since 4.0.6
 */
function kalium_password_protected_page( $template ) {
	if ( is_singular() && post_password_required() ) {
		$wrapper_id = 'password-protected';

		// Theme header
		get_header();

		// Wrapper start
		kalium_content_wrapper_start( $wrapper_id );

		// Password form
		kalium_post_password_form();

		// Wrapper end
		kalium_content_wrapper_end( $wrapper_id );

		// Theme footer
		get_footer();

		exit;
	}
}

add_filter( 'template_redirect', 'kalium_password_protected_page' );

/**
 * Define scrollbar width CSS var.
 *
 * @since 4.0.7
 */
function kalium_define_scrollbar_width_var() {
	$var_name = esc_js( kalium_css_var_name( 'scrollbar-width' ) );

	/*
		( function( varName ) {
			var styleEl = document.createElement('style'),
				getScrollbarWidth = function () {
					return window.innerWidth - document.documentElement.clientWidth;
				},
				resetScrollbarWidth = function () {
					styleEl.innerHTML = ':root { ' + varName + ': ' + getScrollbarWidth() + 'px; }';
				};

			if ( getScrollbarWidth() ) {
				document.head.appendChild( styleEl );

				resetScrollbarWidth();
				window.addEventListener('resize', resetScrollbarWidth, { passive: true });
			}
		} )( '<?php echo $var_name; ?>' );
	 */

	?>
	<script>!function(e){var n=document.createElement("style"),t=function(){return window.innerWidth-document.documentElement.clientWidth},i=function(){n.innerHTML=":root { "+e+": "+t()+"px; }"};t()&&(document.head.appendChild(n),i(),window.addEventListener("resize",i,{passive:!0}))}('<?php echo $var_name; ?>');</script>
	<?php
}

add_action( 'wp_footer', 'kalium_define_scrollbar_width_var', 0 );

/**
 * Theme REST namespace.
 *
 * @since 4.0.9
 */
function kalium_admin_js_define_rest_namespace() {
	kalium_admin_js_define_vars(
		[
			'rest_namespace' => kalium_rest_namespace( false ),
		]
	);
}

add_action( 'admin_footer', 'kalium_admin_js_define_rest_namespace' );

/**
 * Adjacent post handler.
 *
 * @param WP_Post $adjacent_post
 * @param bool    $previous
 * @param array   $args
 *
 * @return WP_Post
 * @since 4.1.2
 */
function kalium_adjacent_post_handler( $adjacent_post, $previous, $args ) {
	global $post;

	// Adjacent products
	if ( 'product' === $post->post_type ) {
		$product = \Kalium\Integrations\WooCommerce\Adjacent_Products::get_adjacent_product(
			[
				'previous'     => $previous,
				'in_same_term' => $args['in_same_term'],
				'taxonomy'     => $args['taxonomy'],
				'loop'         => $args['loop'],
			],
			$post
		);

		return $product ? get_post( $product->get_id() ) : null;
	}

	// Custom post assigned in Parameters & Options
	if ( kalium_get_field( 'custom_prevnext_links' ) ) {
		$field_name  = $previous ? 'prevnext_previous_id' : 'prevnext_next_id';
		$custom_post = kalium_get_field( $field_name );

		if ( $custom_post instanceof WP_Post ) {
			return $custom_post;
		}
	}

	// Loop
	if ( ! $adjacent_post && $args['loop'] ) {
		$query_args = [
			'post_type'      => $post->post_type,
			'status'         => 'publish',
			'exclude'        => [ $post->ID ],
			'orderby'        => 'date',
			'order'          => $previous ? 'DESC' : 'ASC',
			'posts_per_page' => 2,
		];

		if ( $args['in_same_term'] ) {
			$terms = get_the_terms( $post->ID, $args['taxonomy'] );

			if ( ! empty( $terms ) && ! is_wp_error( $terms ) ) {
				$query_args['tax_query'] = [
					[
						'taxonomy' => $args['taxonomy'],
						'field'    => 'slug',
						'terms'    => wp_list_pluck( $terms, 'slug' ),
					],
				];
			}
		}

		$posts = get_posts( $query_args );

		if ( count( $posts ) >= 2 ) {
			return $posts[0];
		}
	}

	return $adjacent_post;
}

add_filter( 'kalium_adjacent_post_object', 'kalium_adjacent_post_handler', 10, 3 );
