<?php
/**
 * Kalium WordPress Theme
 *
 * Header hook functions.
 *
 * @author Laborator
 * @link   https://kaliumtheme.com
 */
if ( ! defined( 'ABSPATH' ) ) {
	exit; // Direct access not allowed.
}

/**
 * Custom header style options on page.
 */
function kalium_header_custom_style_on_page() {
	if ( ! is_singular() && ! is_front_page() || kalium_is_maintenance_mode() ) {
		return;
	}

	$queried_object_id = kalium_get_queried_object_id();

	// Header styling
	switch ( kalium_get_field( 'page_header_styling', $queried_object_id ) ) {
		case 'yes':
			$header_style = kalium_get_field( 'header_style', $queried_object_id );

			if ( $container_background = kalium_get_array_key( $header_style, 'container_background' ) ) {
				kalium_replace_theme_option( 'header_background', $container_background );
			}

			if ( $header_border_color = kalium_get_array_key( $header_style, 'container_border_color' ) ) {
				kalium_replace_theme_option(
					'header_border',
					function ( $border ) use ( &$header_border_color ) {
						if ( ! empty( $border['color'] ) ) {
							$border['color'] = $header_border_color;
						} elseif ( is_array( $border ) ) {
							foreach ( $border as $prop => $border_entry ) {
								if ( ! empty( $border_entry['color'] ) ) {
									$border[ $prop ]['color'] = $header_border_color;
								}
							}
						}

						return $border;
					}
				);
			}

			if ( ! empty( $header_padding = array_filter( kalium_get_array_key( $header_style, 'container_padding', [] ), 'is_numeric' ) ) ) {
				$header_padding = array_merge(
					$header_padding,
					[
						'unit' => 'px',
					]
				);

				kalium_replace_theme_option( 'header_vpadding', kalium_hook_array_merge( $header_padding ) );
			}

			if ( is_numeric( $container_bottom_spacing = kalium_get_array_key( $header_style, 'container_bottom_spacing' ) ) ) {
				kalium_replace_theme_option( 'header_bottom_spacing', $container_bottom_spacing );
			}

			if ( ! empty( $header_links = kalium_filter_array_items( kalium_get_array_key( $header_style, 'links' ) ) ) ) {
				kalium_replace_theme_option( 'header_style_links', kalium_hook_array_merge( $header_links ) );
			}

			if ( ! empty( $header_hamburger_menu = kalium_filter_array_items( kalium_get_array_key( $header_style, 'hamburger_menu' ) ) ) ) {
				kalium_replace_theme_option( 'header_style_menu_trigger', kalium_hook_array_merge( $header_hamburger_menu ) );
			}

			if ( $header_text = kalium_get_array_key( $header_style, 'text' ) ) {
				kalium_replace_theme_option( 'header_style_text', $header_text );
			}

			if ( ! empty( $header_pill_background = kalium_filter_array_items( kalium_get_array_key( $header_style, 'pill_background' ) ) ) ) {
				kalium_replace_theme_option( 'header_menu_pill_background', kalium_hook_array_merge( $header_pill_background ) );
			}

			if ( ! empty( $header_pill_color = kalium_filter_array_items( kalium_get_array_key( $header_style, 'pill_color' ) ) ) ) {
				kalium_replace_theme_option( 'header_menu_pill_color', kalium_hook_array_merge( $header_pill_color ) );
			}

			if ( $submenu_bg = kalium_get_array_key( $header_style, 'submenu_background' ) ) {
				kalium_replace_theme_option( 'header_style_submenu_bg', $submenu_bg );
			}

			if ( ! empty( $header_submenu_links = kalium_filter_array_items( kalium_get_array_key( $header_style, 'submenu_links' ) ) ) ) {
				kalium_replace_theme_option( 'header_style_submenu_links', kalium_hook_array_merge( $header_submenu_links ) );
			}

			if ( ! empty( $header_submenu_pill_bg = kalium_filter_array_items( kalium_get_array_key( $header_style, 'submenu_pill_background' ) ) ) ) {
				kalium_replace_theme_option( 'header_menu_pill_submenu_bg', kalium_hook_array_merge( $header_submenu_pill_bg ) );
			}

			if ( $header_submenu_border_color = kalium_get_array_key( $header_style, 'submenu_border_color' ) ) {
				kalium_replace_theme_option(
					'header_border',
					function ( $border ) use ( &$header_submenu_border_color ) {
						if ( ! empty( $border['color'] ) ) {
							$border['color'] = $header_submenu_border_color;
						} elseif ( is_array( $border ) ) {
							foreach ( $border as $prop => $border_entry ) {
								if ( ! empty( $border_entry['color'] ) ) {
									$border[ $prop ]['color'] = $header_submenu_border_color;
								}
							}
						}

						return $border;
					}
				);
			}

			// Set custom colors for Mini Cart
			if ( ! empty( $header_links['normal'] ) && kalium()->is->woocommerce_active() && ! kalium()->style->is_dark_color( $header_links['normal'] ) ) {
				add_action(
					'wp_body_open',
					function () {
						kalium_print_inline_style(
							[
								'styles' => [
									[
										'selector' => '.header-block .mini-cart-trigger .items-count',
										'vars'     => [
											'badge-color' => kalium_css_var_reference( 'heading-color' ),
										],
									],
									[
										'selector' => '.header-block .mini-cart-trigger .mini-cart-popup',
										'vars'     => [
											'link-color' => kalium_css_var_reference( 'heading-color' ),
											'link-hover-color' => kalium_css_var_reference( 'heading-color' ),
										],
									],
								],
							]
						);
					}
				);
			}

			// Mobile menu
			if ( ! empty( $mobile_menu_links = kalium_filter_array_items( kalium_get_array_key( $header_style, 'mobile_menu_links' ) ) ) ) {
				kalium_replace_theme_option( 'mobile_menu_link_colors', kalium_hook_array_merge( $mobile_menu_links ) );
			}

			if ( $mobile_menu_text = kalium_get_array_key( $header_style, 'mobile_menu_text' ) ) {
				kalium_replace_theme_option( 'mobile_menu_text', $mobile_menu_text );
			}

			if ( $mobile_menu_close_button = kalium_get_array_key( $header_style, 'mobile_menu_close_button' ) ) {
				kalium_replace_theme_option(
					'mobile_menu_close_button_color',
					[
						'normal' => $mobile_menu_close_button,
						'hover'  => $mobile_menu_close_button,
					]
				);
			}
			break;

		case 'no':
			kalium_replace_theme_option( 'header_background', '__return_null' );
			kalium_replace_theme_option( 'header_border', '__return_null' );
			kalium_replace_theme_option( 'header_bottom_spacing', '__return_null' );
			break;
	}
}

add_action( 'wp', 'kalium_header_custom_style_on_page' );

/**
 * Custom logo and menu options for current page.
 */
function kalium_header_custom_logo_and_menu_on_page() {
	if ( ! is_singular() || kalium_is_maintenance_mode() ) {
		return;
	}

	$queried_object_id = kalium_get_queried_object_id();

	// Custom sticky logo
	if ( $custom_sticky_logo  = kalium_get_field( 'custom_sticky_logo', $queried_object_id ) ) {
		kalium_replace_theme_option( 'sticky_header_custom_logo', '__return_true' );
		kalium_replace_theme_option( 'sticky_header_custom_logo_image', $custom_sticky_logo );
	}

	// Overwrite sticky header option for current page
	$sticky_menu_on_page = kalium_get_field( 'sticky_menu_on_page', $queried_object_id );

	if ( in_array( $sticky_menu_on_page, [ 'enable', 'disable' ] ) ) {
		kalium_replace_theme_option( 'sticky_header', ( 'enable' === $sticky_menu_on_page ? '__return_true' : '__return_false' ) );

		// Add unit
		$add_unit = function ( &$arr, $props ) {
			foreach ( $props as $prop ) {
				if ( isset( $arr[ $prop ] ) && is_numeric( $arr[ $prop ] ) ) {
					$arr[ $prop ] .= 'px';
				}
			}
		};

		$sticky_style = kalium_get_field( 'sticky_header_style', $queried_object_id );

		if ( $sticky_bg = kalium_get_array_key( $sticky_style, 'container_background' ) ) {
			kalium_replace_theme_option( 'sticky_header_background_color', $sticky_bg );
		}

		if ( ! empty( $sticky_border = kalium_filter_array_items( kalium_get_array_key( $sticky_style, 'container_bottom_border' ) ) ) ) {
			$add_unit( $sticky_border, [ 'width' ] );

			if ( ! empty( $sticky_border['color'] ) || ! empty( $sticky_border['width'] ) ) {
				kalium_replace_theme_option( 'sticky_header_border', kalium_hook_array_merge( $sticky_border ) );
			}
		}

		if ( ! empty( $sticky_shadow = kalium_filter_array_items( kalium_get_array_key( $sticky_style, 'container_shadow' ) ) ) ) {
			$add_unit( $sticky_shadow, [ 'offset_x', 'offset_y', 'blur_radius', 'spread_radius' ] );

			kalium_replace_theme_option( 'sticky_header_shadow', kalium_hook_array_merge( $sticky_shadow ) );
		}

		if ( ! empty( $sticky_padding = kalium_filter_array_items( kalium_get_array_key( $sticky_style, 'container_padding' ) ) ) ) {
			$sticky_padding = array_merge(
				$sticky_padding,
				[
					'unit' => 'px',
				]
			);

			kalium_replace_theme_option( 'sticky_header_vertical_padding', kalium_hook_array_merge( $sticky_padding ) );
		}

		if ( ! empty( $sticky_links = kalium_filter_array_items( kalium_get_array_key( $sticky_style, 'links' ) ) ) ) {
			kalium_replace_theme_option( 'sticky_header_links_color', kalium_hook_array_merge( $sticky_links ) );
		}

		if ( ! empty( $sticky_menu_trigger = kalium_filter_array_items( kalium_get_array_key( $sticky_style, 'menu_trigger' ) ) ) ) {
			kalium_replace_theme_option( 'sticky_header_menu_trigger', kalium_hook_array_merge( $sticky_menu_trigger ) );
		}

		if ( $sticky_text = kalium_get_array_key( $sticky_style, 'text' ) ) {
			kalium_replace_theme_option( 'sticky_header_text_color', $sticky_text );
		}

		if ( ! empty( $sticky_pill_bg = kalium_filter_array_items( kalium_get_array_key( $sticky_style, 'pill_background' ) ) ) ) {
			kalium_replace_theme_option( 'sticky_header_pill_bg_color', kalium_hook_array_merge( $sticky_pill_bg ) );
		}

		if ( ! empty( $sticky_pill_color = kalium_filter_array_items( kalium_get_array_key( $sticky_style, 'pill_color' ) ) ) ) {
			kalium_replace_theme_option( 'sticky_header_pill_color', kalium_hook_array_merge( $sticky_pill_color ) );
		}
	}

	// Custom logo
	if ( is_numeric( $custom_logo = kalium_get_field( 'custom_logo', $queried_object_id ) ) ) {
		$custom_logo_width = kalium_get_field( 'custom_logo_width' );

		kalium_replace_theme_option( 'use_uploaded_logo', '__return_true' );
		kalium_replace_theme_option( 'custom_logo_image', $custom_logo );

		if ( is_numeric( $custom_logo_width ) && $custom_logo_width > 0 ) {
			kalium_replace_theme_option( 'custom_logo_max_width', $custom_logo_width );
		}
	}
}

add_action( 'wp', 'kalium_header_custom_logo_and_menu_on_page' );

/**
 * Sticky Header init.
 *
 * @since 4.4
 */
function kalium_sticky_header_init() {
	if ( ! kalium_show_header() || ! kalium_get_theme_option( 'sticky_header' ) ) {
		return;
	}

	$transform_styles = $swap_elements = $header_block_styles = [];

	// Vertical padding
	list( $padding_top, $padding_bottom ) = array_values( (array) kalium_get_theme_option( 'sticky_header_vertical_padding' ) );

	if ( is_numeric( $padding_top ) ) {
		$header_block_styles['paddingTop'] = $padding_top . 'px';
	}

	if ( is_numeric( $padding_bottom ) ) {
		$header_block_styles['paddingBottom'] = $padding_bottom . 'px';
	}

	// Background color
	$background_color = kalium_get_theme_option( 'sticky_header_background_color' );

	if ( ! empty( $background_color ) ) {
		$header_block_styles['backgroundColor'] = kalium_replace_color_references( $background_color );
	}

	// Border color
	$border_color = kalium_get_theme_option( 'sticky_header_border' );

	if ( kalium_has_border( $border_color ) ) {
		$header_block_styles['borderBottom'] = implode( ' ', array_values( kalium_to_border_box( $border_color ) ) );
	}

	// Box shadow
	$box_shadow = kalium_get_theme_option( 'sticky_header_shadow' );

	if ( ! empty( $box_shadow['color'] ) && kalium_has_box_shadow_size( $box_shadow ) ) {
		$box_shadow_color = kalium_replace_color_references( $box_shadow['color'] );
		$box_shadow_size  = kalium_to_box_shadow_size( $box_shadow, '0 0 0' );

		$header_block_styles['boxShadow'] = "{$box_shadow_color} {$box_shadow_size}";
	}

	// Add header block styles
	if ( ! empty( $header_block_styles ) ) {
		$transform_styles['.header-block'] = $header_block_styles;
	}

	// Shrink logo
	$shrink_logo = kalium_get_theme_option( 'sticky_header_shrink_logo' );

	if ( is_numeric( $shrink_logo ) ) {
		$shrink_logo /= 100;

		$transform_styles['.header-logo.logo-image'] = [
			'width'  => sprintf( 'calc(%s * %s)', kalium_css_var_reference( 'logo-width' ), $shrink_logo ),
			'height' => sprintf( 'calc(%s * %s)', kalium_css_var_reference( 'logo-height' ), $shrink_logo ),
		];
	}

	// Custom logo
	if ( kalium_get_theme_option( 'sticky_header_custom_logo' ) ) {
		if ( $sticky_logo = wp_get_attachment_image( kalium_get_attachment_id_from_theme_option( 'sticky_header_custom_logo_image' ), 'full' ) ) {
			$swap_elements['sticky'] = [
				'element'   => $sticky_logo,
				'container' => [
					'selector' => '.header-logo.logo-image',
				],
				'stage'     => 'active',
			];

			// Generate new logo shrink size
			$sticky_logo_element = kalium()->helpers->parse_html_element( $sticky_logo );
			$sticky_logo_width   = intval( $sticky_logo_element['attributes']['width'] ?? 0 );
			$sticky_logo_height  = intval( $sticky_logo_element['attributes']['height'] ?? 0 );
			$site_logo           = wp_get_attachment_image_src( kalium_get_attachment_id_from_theme_option( 'custom_logo_image' ), 'full' );

			if ( 0 < $sticky_logo_width && 0 < $sticky_logo_height && is_array( $site_logo ) ) {
				$max_logo_width = kalium_get_responsive_value( kalium_get_theme_option( 'custom_logo_max_width' ) ) ?: $site_logo[1];

				list( $site_logo_width, $site_logo_height )     = kalium()->helpers->resize_by_width( $site_logo[1], $site_logo[2], $max_logo_width );
				list( $sticky_logo_width, $sticky_logo_height ) = kalium()->helpers->resize_by_width( $sticky_logo_width, $sticky_logo_height, $max_logo_width );

				// Shrink custom sticky logo
				if ( is_numeric( $shrink_logo ) ) {
					$sticky_logo_width  *= $shrink_logo;
					$sticky_logo_height *= $shrink_logo;
				}

				$transform_styles['.header-logo.logo-image'] = [
					'width'  => sprintf( 'calc(%s * %s)', kalium_css_var_reference( 'logo-width' ), $sticky_logo_width / $site_logo_width ),
					'height' => sprintf( 'calc(%s * %s)', kalium_css_var_reference( 'logo-height' ), $sticky_logo_height / $site_logo_height ),
				];
			}
		}
	}

	// Create sticky header instance
	kalium_create_sticky_header(
		'.site-header',
		[
			'sections'         => kalium_enum_value(
				kalium_get_theme_option( 'sticky_header_rows' ),
				[
					'all'  => false,
					'main' => '.header-block',
					'top'  => '.top-header-bar',
				],
				'main'
			) ,
			'mode'             => kalium_get_theme_option( 'sticky_header_mode' ),
			'effect'           => kalium_get_theme_option( 'sticky_header_effect' ),
			'animate'          => false === (bool) kalium_get_theme_option( 'sticky_header_progress_with_scroll' ),
			'animate_duration' => kalium_get_theme_option( 'sticky_header_animation_duration' ),
			'offset_start'     => [
				'#wpadminbar'  => 'height',
				'.page-border' => 'padding-top',
			],
			'offset'           => kalium_get_theme_option( 'sticky_header_offset' ),
			'transform_styles' => $transform_styles,
			'swap_elements'    => $swap_elements,
			'devices'          => kalium_get_theme_option( 'sticky_header_responsive' ),
		]
	);

	// Maybe replace color reference
	$maybe_replace_color_reference = function ( $color ) {
		return kalium_color_reference_value( $color ) ?: $color;
	};

	// Styles applied on fully active state
	$sticky_state_class = '.sticky-element-fully-active';

	if ( $style_links = kalium_get_theme_option( 'sticky_header_links_color' ) ) {
		$styles[] = [
			'selector' => $sticky_state_class,
			'vars'     => [
				'header-link-color'        => $maybe_replace_color_reference( kalium_get_array_key( $style_links, 'normal' ) ),
				'header-link-hover-color'  => kalium_get_array_key( $style_links, 'hover' ),
				'header-link-active-color' => kalium_get_array_key( $style_links, 'active' ),
			],
		];
	}

	if ( $style_menu_trigger = kalium_get_theme_option( 'sticky_header_menu_trigger' ) ) {
		$styles[] = [
			'selector' => $sticky_state_class,
			'vars'     => [
				'header-menu-trigger-color'        => $maybe_replace_color_reference( kalium_get_array_key( $style_menu_trigger, 'normal' ) ),
				'header-menu-trigger-hover-color'  => kalium_get_array_key( $style_menu_trigger, 'hover' ),
				'header-menu-trigger-active-color' => kalium_get_array_key( $style_menu_trigger, 'active' ),
			],
		];
	}

	if ( $style_text  = kalium_get_theme_option( 'sticky_header_text_color' ) ) {
		$styles[] = [
			'selector' => $sticky_state_class,
			'vars'     => [
				'header-color' => $style_text,
			],
		];
	}

	if ( $style_pill_bg = kalium_get_theme_option( 'sticky_header_pill_bg_color' ) ) {
		$styles[] = [
			'selector' => "{$sticky_state_class} .nav-pills",
			'vars'     => [
				'pill-bg'        => kalium_get_array_key( $style_pill_bg, 'normal' ),
				'pill-bg-hover'  => kalium_get_array_key( $style_pill_bg, 'hover' ),
				'pill-bg-active' => kalium_get_array_key( $style_pill_bg, 'active' ),
			],
		];
	}

	if ( $style_pill_color = kalium_get_theme_option( 'sticky_header_pill_color' ) ) {
		$styles[] = [
			'selector' => "{$sticky_state_class} .nav-pills",
			'vars'     => [
				'pill-color'        => $maybe_replace_color_reference( kalium_get_array_key( $style_pill_color, 'normal' ) ),
				'pill-color-hover'  => kalium_get_array_key( $style_pill_color, 'hover' ),
				'pill-color-active' => kalium_get_array_key( $style_pill_color, 'active' ),
			],
		];
	}

	// Print styles
	if ( ! empty( $styles ) ) {
		kalium_print_inline_style(
			[
				'styles' => $styles,
			]
		);
	}
}

add_action( 'wp_footer', 'kalium_sticky_header_init' );

/**
 * Initialize transparent header.
 *
 * @since 4.4
 */
function kalium_transparent_header_init() {
	$queried_object_id = kalium_get_queried_object_id();
	$header_position   = kalium_get_header_option( 'position' );

	if ( 'absolute' !== $header_position || post_password_required( $queried_object_id ) || kalium_is_maintenance_mode() ) {
		return;
	}

	// Check for display conditions
	$display_conditions      = kalium_get_theme_option( 'header_transparent_display_conditions' );
	$show_without_conditions = is_singular() && in_array( kalium_get_field( 'header_position', kalium_get_queried_object_id() ), [ 'static', 'absolute' ] );

	if ( ! kalium_check_display_conditions( $display_conditions, $show_without_conditions ) ) {
		return;
	}

	// Add body class
	add_filter( 'body_class', kalium_hook_add_array_value( 'header-absolute' ) );

	// Header spacing
	$header_spacing = kalium_get_header_option( 'spacing' );

	if ( ! kalium_is_empty_responsive( $header_spacing ) ) {
		kalium_print_inline_style(
			[
				'id'       => 'header-spacing',
				'selector' => '.wrapper',
				'props'    => [
					'padding-top' => kalium_map_responsive_value( $header_spacing, 'kalium_to_pixel_unit' ),
				],
			]
		);
	}

	// Custom logo
	if ( kalium_get_theme_option( 'header_transparent_custom_logo' ) ) {
		kalium_replace_theme_option( 'use_uploaded_logo', '__return_true', true );

		if ( $custom_logo_image = kalium_get_theme_option( 'header_transparent_custom_logo_image', true ) ) {
			kalium_replace_theme_option( 'custom_logo_image', $custom_logo_image, true );
		}

		if ( ! kalium_is_empty_responsive( $custom_logo_width = kalium_get_theme_option( 'header_transparent_logo_width', true ) ) ) {
			kalium_replace_theme_option( 'custom_logo_max_width', $custom_logo_width, true );
		}
	}

	// Header links
	if ( $header_links = kalium_get_theme_option( 'header_transparent_style_links' ) ) {
		kalium_replace_theme_option( 'header_style_links', kalium_hook_array_merge( $header_links ), true );
	}

	if ( $header_menu_trigger = kalium_get_theme_option( 'header_transparent_style_menu_trigger' ) ) {
		kalium_replace_theme_option( 'header_style_menu_trigger', kalium_hook_array_merge( $header_menu_trigger ), true );
	}

	if ( $header_text = kalium_get_theme_option( 'header_transparent_style_text' ) ) {
		kalium_replace_theme_option( 'header_style_text', $header_text, 10, true );
	}

	if ( $header_pill_background = kalium_get_theme_option( 'header_transparent_pill_background' ) ) {
		kalium_replace_theme_option( 'header_menu_pill_background', kalium_hook_array_merge( $header_pill_background ), true );
	}

	if ( $header_pill_color = kalium_get_theme_option( 'header_transparent_pill_color' ) ) {
		kalium_replace_theme_option( 'header_menu_pill_color', kalium_hook_array_merge( $header_pill_color ), true );
	}

	if ( $header_submenu_bg = kalium_get_theme_option( 'header_transparent_style_submenu_bg' ) ) {
		kalium_replace_theme_option( 'header_style_submenu_bg', $header_submenu_bg, true );
	}

	if ( $header_submenu_pill_bg = kalium_get_theme_option( 'header_transparent_pill_submenu_bg' ) ) {
		kalium_replace_theme_option( 'header_menu_pill_submenu_bg', kalium_hook_array_merge( $header_submenu_pill_bg ), true );
	}

	if ( $header_submenu_links = kalium_get_theme_option( 'header_transparent_style_submenu_links' ) ) {
		kalium_replace_theme_option( 'header_style_submenu_links', kalium_hook_array_merge( $header_submenu_links ), true );
	}

	if ( $header_submenu_border = kalium_get_theme_option( 'header_transparent_style_submenu_border' ) ) {
		kalium_replace_theme_option( 'header_style_submenu_border', kalium_hook_array_merge( $header_submenu_border ), true );
	}
}

add_action( 'wp_print_scripts', 'kalium_transparent_header_init' );

/**
 * Convert header types (standard, fullscreen, off-canvas top and off-canvas side) to Custom Header content.
 *
 * @param array $content
 *
 * @return array
 */
function kalium_header_convert_static_types_to_custom_header_content( $content ) {
	$header_content = [];

	// Vars
	$menu_type      = kalium_get_theme_option( 'main_menu_type' );
	$submenu_arrows = kalium_get_theme_option( 'submenu_dropdown_indicator' );

	if ( 'custom-header' !== $menu_type ) {

		// Language Switcher
		if ( kalium_get_theme_option( 'header_wpml_language_switcher' ) ) {
			$header_content[] = [
				'name'       => 'legacy_header_wpml_language_switcher',
				'attributes' => [
					[
						'name'  => 'flag',
						'value' => kalium_get_theme_option( 'header_wpml_language_flag' ),
					],
					[
						'name'  => 'flag_position',
						'value' => kalium_get_theme_option( 'header_wpml_language_flag_position' ),
					],
					[
						'name'  => 'display_text',
						'value' => kalium_get_theme_option( 'header_wpml_language_switcher_text' ),
					],
					[
						'name'  => 'display_text_format',
						'value' => kalium_get_theme_option( 'header_wpml_language_switcher_text_display_type' ),
					],
				],
			];
		}

		// Search field
		if ( kalium_get_theme_option( 'header_search_field' ) ) {
			$animate_icon = kalium_get_theme_option( 'header_search_field_animate_icon' );

			$header_content[] = [
				'name'       => 'legacy_header_search_field',
				'attributes' => [
					[
						'name'  => 'input_visibility',
						'value' => $animate_icon ? 'click' : 'visible',
					],
					[
						'name'  => 'icon_animation',
						'value' => $animate_icon ? kalium_get_theme_option( 'header_search_field_icon_animation' ) : 'none',
					],
				],
			];
		}

		// Mini Cart
		if ( kalium_get_theme_option( 'header_mini_cart' ) ) {
			$header_content[] = [
				'name'       => 'legacy_header_woocommerce_mini_cart',
				'attributes' => [
					[
						'name'  => 'icon',
						'value' => kalium_get_theme_option( 'header_cart_icon' ),
					],
					[
						'name'  => 'hide_empty',
						'value' => kalium_get_theme_option( 'header_cart_hide_empty' ),
					],
					[
						'name'  => 'counter_badge',
						'value' => kalium_get_theme_option( 'header_cart_counter_badge' ),
					],
					[
						'name'  => 'title',
						'value' => kalium_get_theme_option( 'header_mini_cart_title' ),
					],
					[
						'name'  => 'click_action',
						'value' => kalium_get_theme_option( 'header_cart_click_action' ),
					],
					[
						'name'  => 'popup_show_on',
						'value' => kalium_get_theme_option( 'header_cart_popup_show_on' ),
					],
					[
						'name'  => 'popup_align',
						'value' => kalium_get_theme_option( 'header_mini_popup_align' ),
					],
				],
			];
		}
	}

	switch ( $menu_type ) {

		// Standard menu
		case 'standard-menu':
			$menu_item_spacing = kalium_get_theme_option( 'header_menu_item_spacing' );

			if ( kalium_get_theme_option( 'menu_standard_menu_bar_visible' ) ) {
				$header_content[] = [
					'name'       => 'legacy_header_menu_trigger',
					'visible'    => true,
					'attributes' => [
						[
							'name'  => 'standard_menu_position',
							'value' => 'start',
						],
						[
							'name'  => 'standard_menu_animation',
							'value' => kalium_get_theme_option( 'menu_standard_menu_bar_effect' ),
						],
						[
							'name'  => 'standard_menu_dropdown_caret',
							'value' => $submenu_arrows,
						],
						[
							'name'  => 'menu_item_spacing',
							'value' => $menu_item_spacing,
						],
					],
				];
			} else {
				array_unshift(
					$header_content,
					[
						'name'       => 'legacy_header_menu',
						'visible'    => true,
						'attributes' => [
							[
								'name'  => 'submenu_dropdown_caret',
								'value' => $submenu_arrows,
							],
							[
								'name'  => 'menu_item_spacing',
								'value' => $menu_item_spacing,
							],
							[
								'name'  => 'pill_style',
								'value' => kalium_get_theme_option( 'header_menu_pills' ),
							],
							[
								'name'  => 'pill_background',
								'value' => kalium_get_theme_option( 'header_menu_pill_background' ),
							],
							[
								'name'  => 'pill_color',
								'value' => kalium_get_theme_option( 'header_menu_pill_color' ),
							],
							[
								'name'  => 'pill_padding',
								'value' => kalium_get_theme_option( 'header_menu_pill_padding' ),
							],
							[
								'name'  => 'pill_radius',
								'value' => kalium_get_theme_option( 'header_menu_pill_radius' ),
							],
						],
					]
				);
			}
			break;

		// Fullscreen menu
		case 'full-bg-menu':
			$fullscreen_toggle = [
				'name'       => 'legacy_header_menu_trigger',
				'visible'    => true,
				'attributes' => [
					[
						'name'  => 'trigger_type',
						'value' => 'fullscreen',
					],
					[
						'name'  => 'fullscreen_alignment',
						'value' => kalium_get_theme_option( 'menu_full_bg_alignment' ),
					],
					[
						'name'  => 'fullscreen_search',
						'value' => kalium_get_theme_option( 'menu_full_bg_search_field' ),
					],
				],
			];

			// Custom background
			if ( kalium_get_theme_option( 'fullscreen_menu_bg' ) ) {
				$fullscreen_toggle['attributes'] = array_merge(
					$fullscreen_toggle['attributes'],
					[
						[
							'name'  => 'fullscreen_bg_color',
							'value' => kalium_get_theme_option( 'fullscreen_menu_bg_color' ),
						],
						[
							'name'  => 'fullscreen_bg_image',
							'value' => kalium_get_attachment_id_from_theme_option( 'fullscreen_menu_bg_image' ),
						],
						[
							'name'  => 'fullscreen_bg_position',
							'value' => kalium_get_theme_option( 'fullscreen_menu_bg_position' ),
						],
						[
							'name'  => 'fullscreen_bg_repeat',
							'value' => kalium_get_theme_option( 'fullscreen_menu_bg_repeat' ),
						],
						[
							'name'  => 'fullscreen_bg_size',
							'value' => kalium_get_theme_option( 'fullscreen_menu_bg_size' ),
						],
						[
							'name'  => 'fullscreen_bg_size_custom',
							'value' => kalium_get_theme_option( 'fullscreen_menu_bg_size_custom' ),
						],
					]
				);
			}

			$header_content[] = $fullscreen_toggle;
			break;

		// Off-Canvas Side
		case 'sidebar-menu':
			$header_content[] = [
				'name'       => 'legacy_header_menu_trigger',
				'visible'    => true,
				'attributes' => [
					[
						'name'  => 'trigger_type',
						'value' => 'off-canvas-side',
					],
					[
						'name'  => 'menu_sidebar_menu_id',
						'value' => kalium_get_theme_option( 'menu_sidebar_menu_id' ),
					],
					[
						'name'  => 'off_canvas_side_show_widgets',
						'value' => kalium_get_theme_option( 'menu_sidebar_show_widgets' ),
					],
					[
						'name'  => 'off_canvas_side_alignment',
						'value' => kalium_get_theme_option( 'menu_sidebar_alignment' ),
					],
				],
			];
			break;

		// Off-Canvas Top
		case 'top-menu':
			$header_content[] = [
				'name'       => 'legacy_header_menu_trigger',
				'visible'    => true,
				'attributes' => [
					[
						'name'  => 'trigger_type',
						'value' => 'off-canvas-top',
					],
					[
						'name'  => 'menu_id',
						'value' => kalium_get_theme_option( 'menu_top_menu_id' ),
					],
					[
						'name'  => 'off_canvas_top_items_per_row',
						'value' => kalium_get_theme_option( 'menu_top_items_per_row' ),
					],
					[
						'name'  => 'off_canvas_top_widgets',
						'value' => kalium_get_theme_option( 'menu_top_show_widgets' ),
					],
					[
						'name'  => 'off_canvas_top_widgets_per_row',
						'value' => kalium_get_theme_option( 'menu_top_widgets_per_row' ),
					],
					[
						'name'  => 'off_canvas_top_widgets_container_width',
						'value' => kalium_get_theme_option( 'menu_top_widgets_container_width' ),
					],
				],
			];
			break;
	}

	// Modified header content
	if ( ! empty( $header_content ) ) {
		return array_merge(
			$content,
			[
				'left'   => null,
				'bottom' => null,
				'right'  => array_merge(
					$content['right'] ?? [],
					[
						'visible'    => true,
						'attributes' => [],
						'elements'   => $header_content,
					]
				),
			]
		);
	}

	return $content;
}

add_filter( 'kalium_header_custom_header_content', 'kalium_header_convert_static_types_to_custom_header_content' );

/**
 * Standard menu toggle setup hook.
 *
 * @param array $content
 *
 * @return array
 * @since 4.0
 */
function kalium_header_standard_menu_toggle_fix( $content ) {
	return array_map( 'kalium_header_standard_menu_toggle_setup', $content );
}

add_filter( 'kalium_header_custom_header_content', 'kalium_header_standard_menu_toggle_fix' );

/**
 * Mobile menu content.
 */
function kalium_mobile_menu_content() {
	laborator_builder_render(
		[
			'content_type'   => 'mobile_menu',
			'content_source' => 'mobile_menu_content',
			'wrapper'        => '<div class="mobile-menu__content">%s</div>',
		],
		[
			'off_canvas_view' => true,
		],
	);
}

add_action( 'kalium_mobile_menu', 'kalium_mobile_menu_content' );

/**
 * Mobile menu breakpoint.
 */
function kalium_mobile_menu_breakpoint() {
	$breakpoint = kalium_get_mobile_menu_breakpoint();

	// Outside mobile breakpoint
	$outside_breakpoint = [
		'selector' => [
			'.mobile-menu',
			'.mobile-menu-close',
			'.mobile-menu-overlay',
			'.header-block .header-block__item--mobile-menu-toggle',
		],
		'props'    => [
			'display' => 'none',
		],
		'media'    => sprintf( '(min-width: %dpx)', $breakpoint + 1 ),
	];

	// Inside mobile breakpoint
	$inside_breakpoint = [
		'selector' => [
			'.header-block .header-block__item--standard-menu',
		],
		'props'    => [
			'display' => 'none',
		],
		'media'    => sprintf( '(max-width: %dpx)', $breakpoint ),
	];

	// Style
	kalium_print_inline_style(
		[
			'id'     => 'mobile-menu-breakpoint',
			'styles' => [
				$outside_breakpoint,
				$inside_breakpoint,
			],
		]
	);
}

add_action( 'wp_head', 'kalium_mobile_menu_breakpoint' );
