<?php
/**
 * Kalium WordPress Theme
 *
 * Header template functions.
 *
 * @author Laborator
 * @link   https://kaliumtheme.com
 */
if ( ! defined( 'ABSPATH' ) ) {
	exit; // Direct access not allowed.
}

if ( ! function_exists( 'kalium_header_default_template' ) ) {

	/**
	 * Default header template.
	 */
	function kalium_header_default_template() {
		if ( ! kalium_show_header() ) {
			return;
		}

		// Load header template
		kalium_get_template(
			'header/container.php',
			[
				'attributes' => [
					'class' => kalium_header_get_classes(),
				],
			]
		);
	}
}

if ( ! function_exists( 'kalium_header_content_display' ) ) {

	/**
	 * Display header content.
	 */
	function kalium_header_content_display() {
		$args = [
			'attributes' => [
				'class' => [
					'header-block',
				],
			],
		];

		// Pills navigation
		if ( kalium_get_theme_option( 'header_menu_pills' ) ) {
			$args['attributes']['class'][] = 'nav-pills';

			$pill_background = kalium_get_theme_option( 'header_menu_pill_background' );
			$pill_color      = kalium_get_theme_option( 'header_menu_pill_color' );
			$pill_submenu_bg = kalium_get_theme_option( 'header_menu_pill_submenu_bg' );
			$pill_padding    = kalium_get_theme_option( 'header_menu_pill_padding' );
			$pill_radius     = kalium_get_theme_option( 'header_menu_pill_radius' );

			// Pill CSS vars
			$vars = [];

			$vars['pill-bg']        = kalium_get_array_key( $pill_background, 'normal' );
			$vars['pill-bg-hover']  = kalium_get_array_key( $pill_background, 'hover' );
			$vars['pill-bg-active'] = kalium_get_array_key( $pill_background, 'active' );

			$vars['pill-color']        = kalium_get_array_key( $pill_color, 'normal' );
			$vars['pill-color-hover']  = kalium_get_array_key( $pill_color, 'hover' );
			$vars['pill-color-active'] = kalium_get_array_key( $pill_color, 'active' );

			$vars['pill-submenu-bg']        = kalium_get_array_key( $pill_submenu_bg, 'normal' );
			$vars['pill-submenu-bg-hover']  = kalium_get_array_key( $pill_submenu_bg, 'hover' );
			$vars['pill-submenu-bg-active'] = kalium_get_array_key( $pill_submenu_bg, 'active' );

			if ( $padding = kalium_to_length( $pill_padding, 0 ) ) {
				$vars['pill-padding'] = $padding;
			}

			if ( $radius = kalium_to_length( $pill_radius, 0 ) ) {
				$vars['pill-radius'] = $radius;
			}

			// Print CSS
			kalium_print_inline_style(
				[
					'id'       => 'nav-pills',
					'selector' => '.header-block',
					'vars'     => $vars,
				]
			);
		}

		// Display custom header template
		kalium_get_template( 'header/header-block.php', $args );
	}
}

if ( ! function_exists( 'kalium_header_top_bar_display' ) ) {

	/**
	 * Top header bar.
	 */
	function kalium_header_top_bar_display() {
		if ( ! kalium_header_has_top_bar() ) {
			return;
		}

		// Top Bar Content
		$top_bar_content = kalium_header_get_top_bar_content();

		// Content blocks
		$content_left  = $top_bar_content['left'];
		$content_right = $top_bar_content['right'];

		// Alignment
		$content_left_alignment  = laborator_builder_get_attribute_value( $content_left, 'align' );
		$content_right_alignment = laborator_builder_get_attribute_value( $content_right, 'align', 'right' );

		// Responsive visibility
		$responsive = kalium_get_theme_option( 'top_bar_responsive' );

		$show_on = [
			'desktop' => in_array( 'desktop', $responsive ),
			'tablet'  => in_array( 'tablet', $responsive ),
			'mobile'  => in_array( 'mobile', $responsive ),
		];

		// Container classes
		$classes = [
			'top-header-bar',
		];

		// Responsive
		foreach ( $show_on as $device_type => $visible ) {
			if ( ! $visible ) {
				$classes[] = 'top-header-bar--hide-on-' . $device_type;
			}
		}

		// Row container classes
		$row_container_classes = [
			'top-header-bar__row-container',
			'container',
		];

		// Left column class
		$left_column_classes = [
			'top-header-bar__column',
			'top-header-bar__column--content-left',
		];

		if ( $content_left_alignment ) {
			$left_column_classes[] = 'top-header-bar__column--alignment-' . $content_left_alignment;
		}

		// Left column class
		$right_column_classes = [
			'top-header-bar__column',
			'top-header-bar__column--content-right',
		];

		if ( $content_right_alignment ) {
			$right_column_classes[] = 'top-header-bar__column--alignment-' . $content_right_alignment;
		}

		// Custom background and border color
		$container_colors = kalium_get_theme_option( 'top_header_bar_container_colors' );

		// Border
		if ( ! empty( $container_colors['border'] ) ) {
			$classes[] = 'top-header-bar--has-border';
		}

		// Separator
		if ( ! empty( $container_colors['separator'] ) ) {
			$classes[] = 'top-header-bar--with-separators';
		}

		?>
		<div <?php kalium_class_attr( $classes ); ?>>

			<div <?php kalium_class_attr( $row_container_classes ); ?>>

				<div class="top-header-bar__row top-header-bar--row">

					<?php if ( ! empty( $content_left['elements'] && $content_left['visible'] ) ) : ?>
						<div <?php kalium_class_attr( $left_column_classes ); ?>>
							<?php
							// Left content entries
							laborator_builder_render(
								[
									'content_type'   => 'legacy_header_top_bar',
									'content_source' => [
										'type'    => 'direct',
										'content' => $content_left,
									],
								],
								[
									'parent_selector' => '.top-header-bar__column--content-left',
									'menu_depth'      => 0,
								]
							);
							?>
						</div>
					<?php endif; ?>

					<?php if ( ! empty( $content_right['elements'] && $content_right['visible'] ) ) : ?>
						<div <?php kalium_class_attr( $right_column_classes ); ?>>
							<?php
							// Right content entries
							laborator_builder_render(
								[
									'content_type'   => 'legacy_header_top_bar',
									'content_source' => [
										'type'    => 'direct',
										'content' => $content_right,
									],
								],
								[
									'parent_selector' => '.top-header-bar__column--content-right',
									'menu_depth'      => 0,
								]
							);
							?>
						</div>
					<?php endif; ?>

				</div>

			</div>

		</div>
		<?php
	}
}

if ( ! function_exists( 'kalium_header_content_left' ) ) {

	/**
	 * Header left menu content (inline with the logo).
	 */
	function kalium_header_content_left() {
		$header_content = kalium_header_get_content();
		$content_left   = $header_content['left'];

		// Do not show if centered logo header is not selected
		if ( ! $content_left || ! $content_left['visible'] || empty( $content_left['elements'] ) ) {
			return;
		}

		$classes = [
			'header-block__column',
			'header-block--content-left',
		];

		// Alignment
		$classes[] = 'header-block--align-' . laborator_builder_get_attribute_value( $content_left, 'align', 'left' );

		?>
		<div <?php kalium_class_attr( $classes ); ?>>

			<div class="header-block__items-row header-block__items-row--content-left">
				<?php
				// Render Content
				laborator_builder_render(
					[
						'content_type'   => 'legacy_header',
						'content_source' => [
							'type'    => 'direct',
							'content' => $content_left,
						],
					],
					[
						'parent_selector' => '.header-block__items-row--content-left',
					]
				);
				?>
			</div>

		</div>
		<?php
	}
}

if ( ! function_exists( 'kalium_header_content_right' ) ) {

	/**
	 * Centered header right menu content (inline with the logo).
	 */
	function kalium_header_content_right() {
		$header_content = kalium_header_get_content();
		$content_right  = $header_content['right'];

		// Do not show if centered logo header is not selected
		if ( ! $content_right || ! $content_right['visible'] || empty( $content_right['elements'] ) ) {
			return;
		}

		$classes = [
			'header-block__column',
			'header-block--content-right',
		];

		// Alignment
		$classes[] = 'header-block--align-' . laborator_builder_get_attribute_value( $content_right, 'align', 'right' );
		?>
		<div <?php kalium_class_attr( $classes ); ?>>

			<div class="header-block__items-row  header-block__items-row--content-right">
				<?php
				// Render Content
				laborator_builder_render(
					[
						'content_type'   => 'legacy_header',
						'content_source' => [
							'type'    => 'direct',
							'content' => $content_right,
						],
					],
					[
						'parent_selector' => '.header-block__items-row--content-right',
					]
				);
				?>
			</div>

		</div>
		<?php
	}
}

if ( ! function_exists( 'kalium_header_content_below' ) ) {

	/**
	 * Header content below the logo.
	 */
	function kalium_header_content_below() {
		$header_content = kalium_header_get_content();
		$content_bottom = $header_content['bottom'];

		// Do not show if centered logo header is not selected
		if ( ! $content_bottom || ! $content_bottom['visible'] || empty( $content_bottom['elements'] ) ) {
			return;
		}

		$classes = [
			'header-block__column',
			'header-block--content-below',
		];

		// Alignment
		$classes[] = 'header-block--align-' . laborator_builder_get_attribute_value( $content_bottom, 'align', 'left' );

		?>
		<div class="header-block__row-container container">

			<div class="header-block__row header-block__row--secondary">

				<div <?php kalium_class_attr( $classes ); ?>>

					<div class="header-block__items-row header-block__items-row--content-bottom">
						<?php
						// Render Content
						laborator_builder_render(
							[
								'content_type'   => 'legacy_header',
								'content_source' => [
									'type'    => 'direct',
									'content' => $content_bottom,
								],
							],
							[
								'parent_selector' => '.header-block__items-row--content-bottom',
							]
						);
						?>
					</div>

				</div>

			</div>

		</div>
		<?php
	}
}

if ( ! function_exists( 'kalium_header_content_logo' ) ) {

	/**
	 * Centered header logo.
	 */
	function kalium_header_content_logo() {

		?>
		<div class="header-block__column header-block__logo header-block--auto-grow">
			<?php
			// Logo element
			kalium_logo_element();
			?>
		</div>
		<?php
	}
}

if ( ! function_exists( 'kalium_header_fullscreen_menu' ) ) {

	/**
	 * Fullscreen Menu container.
	 */
	function kalium_header_fullscreen_menu() {
		$args = kalium_header_get_fullscreen_menu_options();

		// Nav menu
		$menu_args = [
			'container'       => '',
			'echo'            => true,
			'dropdown_toggle' => kalium_get_theme_option( 'submenu_dropdown_indicator' ),
		];

		if ( is_numeric( $args['menu_id'] ) ) {
			$menu_args['menu'] = $args['menu_id'];
		} elseif ( is_string( $args['menu_id'] ) ) {
			$menu_args['theme_location'] = $args['menu_id'];
		}

		// Full bg menu classes
		$classes = [
			'fullscreen-menu',
			kalium_get_link_plain_class(),
		];

		if ( 'centered-horizontal' === $args['align'] ) {
			$classes[] = 'menu-horizontally-center';
		}

		if ( in_array( $args['align'], [ 'centered', 'centered-horizontal' ] ) ) {
			$classes[] = 'menu-aligned-center';
		}

		// Custom background
		if ( $args['custom_bg'] ) {
			$background = [];

			// Background props
			$background_image       = $args['background']['image'];
			$background_position    = $args['background']['position'];
			$background_repeat      = $args['background']['repeat'];
			$background_size        = $args['background']['size'];
			$background_size_custom = $args['background']['size_custom'];

			if ( 'custom' === $background_size ) {
				$background_size = $background_size_custom;
			}

			// Background image
			if ( ! empty( $background_image ) && ( $image_file = wp_get_attachment_image_src( $background_image, 'full' ) ) ) {
				$background['background-image'] = 'url("' . esc_url( $image_file[0] ) . '")';

				// Background position
				if ( ! empty( $background_position ) ) {
					$background['background-position'] = str_replace( '-', ' ', $background_position );
				}

				// Background repeat
				if ( ! empty( $background_repeat ) ) {
					$background['background-repeat'] = $background_repeat;
				}

				// Background size
				if ( ! empty( $background_size ) ) {
					if ( is_numeric( $background_size ) ) {
						$background_size .= 'px';
					}

					$background['background-size'] = $background_size;
				}
			}

			kalium_print_inline_style(
				[
					'id'       => 'fullscreen-menu-bg',
					'selector' => '.site-header > .fullscreen-menu',
					'props'    => $background,
				]
			);
		}
		?>
		<div <?php kalium_class_attr( $classes ); ?>>

			<?php
			/**
			 * Before fullscreen menu.
			 */
			do_action( 'kalium_header_fullscreen_menu_before' );
			?>

			<div <?php kalium_class_attr( [ 'fullscreen-menu-navigation', kalium_get_link_style_class_by_context( 'fullscreen-menu' ) ] ); ?>>
				<div class="container">
					<nav>
						<?php
						// Navigation
						kalium_nav_menu( $menu_args );

						// Search field
						if ( $args['search_field'] ) :
							?>
							<form class="search-form" method="get" action="<?php echo esc_url( kalium_search_url() ); ?>" enctype="application/x-www-form-urlencoded">
								<input id="full-bg-search-inp" type="search" class="search-field" value="<?php echo get_search_query(); ?>" name="s" autocomplete="off"/>
								<label for="full-bg-search-inp">
									<?php
									// Search placeholder
									printf( esc_html__( 'Search', 'kalium' ) );
									?>
								</label>
							</form>
						<?php endif; ?>
					</nav>
				</div>
			</div>

			<?php
			/**
			 * After fullscreen menu.
			 */
			do_action( 'kalium_header_fullscreen_menu_after' );
			?>

		</div>
		<?php
	}
}

if ( ! function_exists( 'kalium_header_search_field' ) ) {

	/**
	 * Header search field.
	 */
	function kalium_header_search_field( $args = [] ) {
		$args = wp_parse_args(
			$args,
			[
				'skin'             => '',
				'align'            => '',
				'input_visibility' => '',
				'icon_animation'   => '', // [shift,scale]
			]
		);

		// Classes
		$classes = [
			'header-search-input',
		];

		// Skin class
		if ( ! empty( $args['skin'] ) ) {
			$classes[] = $args['skin'];
		}

		// Right aligned
		if ( 'right' === $args['align'] ) {
			$classes[] = 'header-search-input--align-right';
		}

		// Search input visibility
		if ( 'visible' === $args['input_visibility'] ) {
			$classes[] = 'header-search-input--input-visible';
		}
		?>
		<div <?php kalium_class_attr( $classes ); ?>>
			<form role="search" method="get" action="<?php echo esc_url( kalium_search_url() ); ?>">
				<div class="search-field">
					<span><?php esc_html_e( 'Search site...', 'kalium' ); ?></span>
					<input type="search" value="" autocomplete="off" name="s" aria-label="Search" />
				</div>

				<div class="search-icon">
					<a href="#" data-animation="<?php echo esc_attr( $args['icon_animation'] ); ?>" aria-label="Search site">
						<?php echo kalium_get_icon( 'search' ); ?>
					</a>
				</div>

				<?php
				// Lang attribute
				if ( kalium()->request->has( 'lang' ) ) {
					printf( '<input type="hidden" name="lang" value="%s" />', esc_attr( kalium()->request->query( 'lang' ) ) );
				}
				?>
			</form>

		</div>
		<?php
	}
}

if ( ! function_exists( 'kalium_toggle_bars' ) ) {

	/**
	 * Toggle bars element.
	 *
	 * @param array $args
	 *
	 * @since 4.1.2
	 */
	function kalium_toggle_bars( $args = [] ) {
		$args = wp_parse_args(
			$args,
			[
				'action'        => null,
				'attributes'    => [],
				'icon'          => kalium_get_theme_option( 'menu_hamburger_custom_icon' ),
				'icon_position' => kalium_get_theme_option( 'menu_hamburger_custom_icon_position' ),
				'label'         => kalium_get_theme_option( 'menu_hamburger_custom_label' ),
				'label_open'    => kalium_get_theme_option( 'menu_hamburger_custom_label_text' ),
				'label_close'   => kalium_get_theme_option( 'menu_hamburger_custom_label_close_text' ),
			]
		);

		// Fallback when both label and icon are disabled
		$args['icon'] = ( $args['icon'] || $args['label'] ) ? $args['icon'] : true;

		$attributes = wp_parse_args(
			$args['attributes'],
			[
				'type'        => 'button',
				'class'       => [],
				'data-action' => $args['action'],
				'aria-label'  => 'Toggle navigation',
			]
		);

		array_unshift( $attributes['class'], 'toggle-bars' );

		if ( $args['icon'] ) {
			$attributes['class'][] = 'toggle-bars--icon-' . kalium_enum_value(
				$args['icon_position'],
				[
					'left'  => 'start',
					'right' => 'end',
				]
			);
		}

		?>
		<button <?php kalium_render_attributes( $attributes ); ?>>
			<?php if ( $args['icon'] ) : ?>
				<span class="toggle-bars__bars">
					<span class="toggle-bars__bar-line toggle-bars__bar-line"></span>
					<span class="toggle-bars__bar-line toggle-bars__bar-line--mid"></span>
					<span class="toggle-bars__bar-line toggle-bars__bar-line"></span>
				</span>
			<?php endif; ?>
			<?php if ( $args['label'] ) : ?>
				<span class="toggle-bars__label" data-open="<?php echo esc_html( $args['label_open'] ); ?>" data-close="<?php echo esc_html( $args['label_close'] ); ?>"></span>
			<?php endif; ?>
		</button>
		<?php
	}
}

if ( ! function_exists( 'kalium_header_fullscreen_menu_display' ) ) {

	/**
	 * Display fullscreen menu.
	 */
	function kalium_header_fullscreen_menu_display() {
		if ( ! kalium_show_header() ) {
			return;
		}

		// Fullscreen
		if ( kalium_header_has_menu_type( 'fullscreen' ) ) {
			kalium_header_fullscreen_menu();
		}
	}
}

if ( ! function_exists( 'kalium_header_off_canvas_top_menu_default_template' ) ) {

	/**
	 * Display header top menu.
	 */
	function kalium_header_off_canvas_top_menu_default_template() {
		if ( ! kalium_show_header() ) {
			return;
		}

		// Top menu
		if ( kalium_get_theme_option( 'menu_top_force_include' ) || kalium_header_has_menu_type( 'off-canvas-top' ) ) {
			kalium_off_canvas_top_menu();
		}
	}
}

if ( ! function_exists( 'kalium_off_canvas_top_menu' ) ) {

	/**
	 * Off-canvas top menu.
	 *
	 * @since 4.0
	 */
	function kalium_off_canvas_top_menu() {
		$top_menu_options = kalium_header_get_off_canvas_top_options();

		// Nav
		$nav = kalium_nav_menu( $top_menu_options['menu_id'] );

		$menu_container_width    = 'col-sm-12';
		$widgets_container_width = 'col-sm-12';

		if ( $top_menu_options['widgets'] ) {
			switch ( $top_menu_options['widgets_container_width'] ) {
				case 'col-3':
					$menu_container_width    = 'col-sm-9';
					$widgets_container_width = 'col-sm-3';
					break;

				case 'col-4':
					$menu_container_width    = 'col-sm-8';
					$widgets_container_width = 'col-sm-4';
					break;

				case 'col-5':
					$menu_container_width    = 'col-sm-7';
					$widgets_container_width = 'col-sm-5';
					break;

				case 'col-7':
					$menu_container_width    = 'col-sm-5';
					$widgets_container_width = 'col-sm-7';
					break;

				case 'col-8':
					$menu_container_width    = 'col-sm-4';
					$widgets_container_width = 'col-sm-8';
					break;

				case 'col-9':
					$menu_container_width    = 'col-sm-3';
					$widgets_container_width = 'col-sm-9';
					break;

				default:
					$menu_container_width    = 'col-sm-6';
					$widgets_container_width = 'col-sm-6';
			}

			if ( ! $nav ) {
				$widgets_container_width = 'col-sm-12';
			}
		}

		// Widgets columns
		$widgets_columns = is_numeric( $top_menu_options['widgets_per_row'] ) ? $top_menu_options['widgets_per_row'] : 3;

		// Classes
		$classes = [
			'top-menu-container',
			'menu-type-' . kalium_get_theme_option( 'main_menu_type' ),
		];

		?>
		<div <?php kalium_class_attr( $classes ); ?>>
			<?php
			/**
			 * Before off-canvas top menu.
			 */
			do_action( 'kalium_header_off_canvas_top_menu_before' );
			?>

			<div class="<?php echo esc_attr( kalium_if_else( kalium_is_fullwidth_header(), 'container-full', 'container' ) ); ?>">
				<div class="row">
					<?php
					if ( ! empty( $nav ) ) :
						$nav_classes = [
							'top-menu',
							'menu-row-items-' . $top_menu_options['items_per_row'],
							kalium_get_link_style_class_by_context( 'top-menu' ),
						];
						?>
						<div class="<?php echo esc_attr( $menu_container_width ); ?>">
							<nav <?php kalium_class_attr( $nav_classes ); ?>>
								<?php echo $nav; ?>
							</nav>
						</div>
					<?php endif; ?>

					<?php if ( $top_menu_options['widgets'] && is_active_sidebar( 'top_menu_sidebar' ) ) : ?>
						<div class="<?php echo esc_attr( $widgets_container_width ); ?>">
							<div <?php kalium_class_attr( array_merge( [ 'widget-area' ], kalium_responsive_classname_infix( $widgets_columns, 'columns' ) ) ); ?> role="complementary">
								<?php dynamic_sidebar( 'top_menu_sidebar' ); ?>
							</div>
						</div>
					<?php endif; ?>
				</div>
			</div>

			<?php
			/**
			 * After off-canvas top menu.
			 */
			do_action( 'kalium_header_off_canvas_top_menu_after' );
			?>
		</div>
		<?php
	}
}

if ( ! function_exists( 'kalium_header_off_canvas_side_menu_default_template' ) ) {

	/**
	 * Display header side menu.
	 */
	function kalium_header_off_canvas_side_menu_default_template() {
		if ( ! kalium_show_header() ) {
			return;
		}

		// Sidebar menu
		if ( kalium_get_theme_option( 'menu_sidebar_force_include' ) || kalium_header_has_menu_type( 'off-canvas-side' ) ) {
			kalium_off_canvas_side_menu();
		}
	}
}

if ( ! function_exists( 'kalium_off_canvas_side_menu' ) ) {

	/**
	 * Off-canvas side menu.
	 *
	 * @since 4.0
	 */
	function kalium_off_canvas_side_menu() {
		$side_menu_options = kalium_header_get_off_canvas_side_options();

		// Menu args
		$menu_id   = $side_menu_options['menu_id'];
		$menu_args = [
			'menu'            => $menu_id,
			'dropdown_toggle' => kalium_get_theme_option( 'submenu_dropdown_indicator' ),
		];

		$nav = kalium_nav_menu( $menu_args );

		// Container classes
		$classes = [
			'sidebar-menu-wrapper',
			'menu-type-sidebar-menu',
			'sidebar-alignment-' . $side_menu_options['alignment'],
		];

		?>
		<div <?php kalium_class_attr( $classes ); ?>>
			<div class="sidebar-menu-container">
				<?php
				/**
				 * Before off-canvas side menu.
				 */
				do_action( 'kalium_header_off_canvas_side_menu_before' );
				?>

				<a class="sidebar-menu-close" href="#"></a>

				<?php if ( $nav ) : ?>
					<div <?php kalium_class_attr( [ 'sidebar-main-menu', kalium_get_link_style_class_by_context( 'sidebar-menu' ) ] ); ?>>
						<?php
						/**
						 * Navigation menu.
						 */
						echo $nav;
						?>
					</div>
				<?php endif; ?>

				<?php if ( $side_menu_options['widgets'] && is_active_sidebar( 'sidebar_menu_sidebar' ) ) : ?>
					<div class="sidebar-menu-widgets blog-sidebar">
						<?php
						/**
						 * Side menu widgets.
						 */
						dynamic_sidebar( 'sidebar_menu_sidebar' );
						?>
					</div>
				<?php endif; ?>

				<?php
				/**
				 * After off-canvas side menu.
				 */
				do_action( 'kalium_header_off_canvas_side_menu_after' );
				?>
			</div>
		</div>

		<div class="sidebar-menu-disabler"></div>
		<?php
	}
}

if ( ! function_exists( 'kalium_header_woocommerce_cart' ) ) {

	/**
	 * Header mini cart item.
	 *
	 * @param array $args
	 */
	function kalium_header_woocommerce_cart( $args = [] ) {
		if ( ! kalium()->is->woocommerce_active() ) {
			return;
		}

		$args = wp_parse_args(
			$args,
			[
				'icon'          => null,
				'title'         => null,
				'hide_empty'    => null,
				'counter_badge' => null,
				'click_action'  => null,
				'popup_show_on' => null,
				'popup_align'   => null,
			]
		);

		// Defined icon types
		$icons = [
			'icon-1' => 'shopping-bag',
			'icon-2' => 'shopping-cart',
			'icon-3' => 'shopping-basket',
		];

		// Vars
		$icon         = $icons[ $args['icon'] ] ?? null;
		$title        = $args['title'];
		$click_action = $args['click_action'];
		$is_popup     = 'mini-cart' === $click_action && 'popup' === kalium_get_theme_option( 'shop_mini_cart_type' );

		// Container classes
		$classes = [
			'mini-cart-trigger',
		];

		// Hide cart icon when empty
		if ( $args['hide_empty'] && ! kalium_woocommerce_get_items_in_cart() ) {
			$classes[] = 'mini-cart-trigger--hide-empty';
		}

		// Mini cart popup
		if ( $is_popup ) {
			$classes[] = 'mini-cart-trigger--dropdown-align-' . $args['popup_align'];

			// Show on hover
			if ( 'hover' === $args['popup_show_on'] ) {
				$classes[] = 'mini-cart-trigger--hover-show';
			}
		}

		// Enqueue cart fragments
		wp_enqueue_script( 'wc-cart-fragments' );

		?>
		<div <?php kalium_class_attr( $classes ); ?> data-click-action="<?php echo esc_attr( $click_action ); ?>">
			<a href="<?php echo esc_url( wc_get_cart_url() ); ?>" <?php kalium_class_attr( [ 'mini-cart-trigger__link' ] ); ?> aria-label="Cart">
				<?php if ( $icon ) : ?>
					<span class="mini-cart-trigger__icon">
						<?php
						echo kalium_get_icon( $icon );

						if ( $args['counter_badge'] ) {
							echo '<span class="badge items-count">&hellip;</span>';
						}
						?>
					</span>
				<?php endif; ?>

				<?php if ( $title ) : ?>
					<span class="mini-cart-trigger__title">
						<?php echo esc_html( $title ); ?>
					</span>
				<?php endif; ?>
			</a>

			<?php if ( $is_popup ) : ?>
			<div class="mini-cart-popup">
				<div class="empty-loading-cart-contents">
					<?php esc_html_e( 'Loading cart contents...', 'kalium' ); ?>
				</div>
			</div>
			<?php endif; ?>
		</div>
		<?php
	}
}

if ( ! function_exists( 'kalium_mobile_menu_default_template' ) ) {

	/**
	 * Display mobile menu template.
	 *
	 * @since 4.0
	 */
	function kalium_mobile_menu_default_template() {
		if ( ! kalium_show_header() ) {
			return;
		}

		$args = [
			'attributes' => [
				'class' => [
					'mobile-menu',
				],
			],
		];

		// Vars
		$vars = [];

		// Padding
		if ( $content_padding = kalium_get_theme_option( 'mobile_menu_content_padding' ) ) {
			$vars['mobile-menu-content-padding'] = kalium_to_length( $content_padding );
		}

		// Item spacing
		if ( $item_spacing = kalium_get_theme_option( 'mobile_menu_item_spacing' ) ) {
			$vars['mobile-menu-item-spacing'] = $item_spacing;
		}

		// Overlay blur
		if ( $overlay_blur = kalium_get_theme_option( 'mobile_menu_overlay_blur' ) ) {
			$vars['mobile-menu-overlay-blur'] = $overlay_blur;
			$args['attributes']['class'][]    = 'mobile-menu--has-overlay-blur';
		}

		// Menu type (style)
		switch ( kalium_get_theme_option( 'menu_mobile_type' ) ) {
			case 'fullscreen':
				$args['attributes']['class'][] = 'mobile-menu--fullscreen';
				break;

			default:
				$args['attributes']['class'][] = 'mobile-menu--slide';

				// Slide from right
				if ( 'left' === kalium_get_theme_option( 'menu_mobile_slide_direction' ) ) {
					$args['attributes']['class'][] = 'mobile-menu--slide-left';
				}

				// Content width
				if ( $content_width = kalium_get_theme_option( 'mobile_menu_content_width' ) ) {
					$vars['mobile-menu-content-width'] = $content_width;
				}
		}

		// Style
		kalium_print_inline_style(
			[
				'selector' => kalium_get_root_selector(),
				'vars'     => $vars,
			]
		);

		// Mobile menu template
		kalium_get_template( 'mobile-menu.php', $args );

		// Close button
		kalium_mobile_menu_close_button();
	}
}

if ( ! function_exists( 'kalium_mobile_menu_close_button' ) ) {

	/**
	 * Mobile menu close button.
	 *
	 * @since 4.0
	 */
	function kalium_mobile_menu_close_button() {
		$close_button = kalium_get_theme_option( 'menu_mobile_close_button' );

		if ( in_array( $close_button, [ 'left', 'right' ], true ) ) {
			$classes = [
				'mobile-menu-close',
			];

			if ( 'left' === $close_button ) {
				$classes[] = 'mobile-menu-close--left';
			}
			?>
			<button type="button" aria-label="Close" <?php kalium_class_attr( $classes ); ?>>
				<span aria-hidden="true" class="kalium-icon-remove"></span>
			</button>
			<?php
		}
	}
}
