<?php
/**
 * Kalium WordPress Theme
 *
 * Other/uncategorized functions.
 *
 * @author Laborator
 * @link   https://kaliumtheme.com
 */
if ( ! defined( 'ABSPATH' ) ) {
	exit; // Direct access not allowed.
}

/**
 * Null function.
 *
 * @return void
 */
function kalium_null_function() {
}

/**
 * Conditional return value.
 *
 * @param bool  $condition
 * @param mixed $if
 * @param mixed $else
 *
 * @return mixed
 * @since 4.3
 */
function kalium_if_else( $condition, $if, $else = null ) {
	return $condition ? $if : $else;
}

/**
 * Enqueue GSAP library.
 *
 * @return void
 */
function kalium_enqueue_gsap_library() {
	kalium_enqueue( 'gsap' );
}

/**
 * Enqueue Isotope & Packery library.
 *
 * @return void
 */
function kalium_enqueue_isotope_and_packery_library() {
	kalium_enqueue( 'isotope' );

	// Workaround for WPBakery
	if ( wp_script_is( 'isotope', 'registered' ) ) {
		wp_dequeue_script( 'isotope' );
		wp_deregister_script( 'isotope' );
	}
}

/**
 * Register dynamic translatable string for WPML.
 *
 * @param string $value
 * @param string $name
 */
function kalium_wpml_register_single_string( $value, $name ) {
	do_action( 'wpml_register_single_string', 'kalium', $name, $value );
}

/**
 * WPML dynamic translatable string.
 *
 * @param string $original_value
 * @param string $name
 *
 * @return string
 */
function kalium_wpml_translate_single_string( $original_value, $name ) {
	return apply_filters( 'wpml_translate_single_string', $original_value, 'kalium', $name );
}

/**
 * Get social network share post link.
 *
 * @param string $social_network
 * @param int    $post_id
 *
 * @return array|null
 * @since 4.0
 */
function kalium_get_social_network_share_post_link( $social_network, $post_id = null ) {
	static $networks;

	/**
	 * Filters list of providers for social networks share.
	 *
	 * @param array $networks
	 */
	if ( is_null( $networks ) ) {
		$networks = apply_filters(
			'kalium_social_networks_share_providers',
			[
				'facebook'  => [
					'element' => 'kalium_social_icon_facebook',
					'url'     => 'https://www.facebook.com/sharer.php?u={PERMALINK}',
					'label'   => 'Facebook',
				],
				'twitter'   => [
					'element' => 'kalium_social_icon_twitter_x',
					'url'     => 'https://twitter.com/share?text={TITLE}&url={PERMALINK}',
					'label'   => 'Twitter X',
				],
				'bluesky'   => [
					'element' => 'kalium_social_icon_bluesky',
					'url'     => 'https://bsky.app/intent/compose?text={TITLE}+-+{PERMALINK}',
					'label'   => 'Bluesky',
				],
				'threads'   => [
					'element' => 'kalium_social_icon_threads',
					'url'     => 'https://www.threads.net/intent/post?text={TITLE}&url={PERMALINK}',
					'label'   => 'Threads',
				],
				'tumblr'    => [
					'element' => 'kalium_social_icon_tumblr',
					'url'     => 'https://www.tumblr.com/share/link?url={PERMALINK}&name={TITLE}&description={EXCERPT}',
					'label'   => 'Tumblr',
				],
				'linkedin'  => [
					'element' => 'kalium_social_icon_linkedin',
					'url'     => 'https://linkedin.com/shareArticle?mini=true&url={PERMALINK}&title={TITLE}',
					'label'   => 'LinkedIn',
				],
				'pinterest' => [
					'element' => 'kalium_social_icon_pinterest',
					'url'     => 'https://pinterest.com/pin/create/button/?url={PERMALINK}&description={TITLE}&media={FEATURED_IMAGE}',
					'label'   => 'Pinterest',
				],
				'vkontakte' => [
					'element' => 'kalium_social_icon_vkontakte',
					'url'     => 'https://vkontakte.ru/share.php?url={PERMALINK}&title={TITLE}&description={EXCERPT}',
					'label'   => 'VKontakte',
				],
				'whatsapp'  => [
					'element' => 'kalium_social_icon_whatsapp',
					'url'     => 'https://api.whatsapp.com/send?text={TITLE} - {PERMALINK}',
					'label'   => 'WhatsApp',
				],
				'telegram'  => [
					'element' => 'kalium_social_icon_telegram',
					'url'     => 'https://t.me/share/url?url={PERMALINK}&text={TITLE}',
					'label'   => 'Telegram',
				],
				'xing'      => [
					'element' => 'kalium_social_icon_xing',
					'url'     => 'https://www.xing.com/spi/shares/new?url={PERMALINK}',
					'label'   => 'Xing',
				],
				'print'     => [
					'element' => 'kalium_social_icon_custom',
					'url'     => 'javascript:window.print();',
					'label'   => __( 'Print', 'kalium' ),
				],
				'email'     => [
					'element' => 'kalium_social_icon_email',
					'url'     => '?subject={TITLE}&body={EMAIL_BODY}',
					'label'   => __( 'Email', 'kalium' ),
				],
			]
		);
	}

	// Post
	$post = get_post( $post_id );

	// Valid network and post
	if ( $post instanceof WP_Post && isset( $networks[ $social_network ] ) ) {
		$network_entry = kalium_get_array_key( $networks, $social_network );

		// Share URL
		$url = $network_entry['url'];

		// URL vars to replace
		$url_vars = [
			'PERMALINK'      => get_permalink( $post ),
			'TITLE'          => get_the_title( $post ),
			'EXCERPT'        => wp_trim_words( kalium_clean_excerpt( $post->post_excerpt, true ), 40, '&hellip;' ),
			'FEATURED_IMAGE' => wp_get_attachment_url( get_post_thumbnail_id( $post ) ),
			'EMAIL_BODY'     => sprintf( __( 'Check out what I just spotted: %s', 'kalium' ), get_permalink( $post ) ),
		];

		foreach ( $url_vars as $var_name => $value ) {
			$url = str_replace( '{' . $var_name . '}', rawurlencode( $value ), $url );
		}

		/**
		 * Filters social network share link markup.
		 *
		 * @param array $network_entry
		 * @param int $post_id
		 */
		return apply_filters( 'kalium_social_network_share_post_link', array_merge( $network_entry, [ 'url' => $url ] ), $post_id );
	}

	return null;
}

/**
 * Search page url.
 *
 * @return string
 */
function kalium_search_url() {
	global $polylang;

	// Default search page URL
	$url = home_url( '/' );

	// Polylang Search URL
	if ( ! empty( $polylang ) ) {
		$url = $polylang->curlang->search_url;
	}

	return apply_filters( 'kalium_search_url', $url );
}

/**
 * Check if its maintenance mode.
 *
 * @return bool
 * @since 3.4.3
 */
function kalium_is_maintenance_mode() {
	$maintenance_mode = kalium_get_theme_option( 'maintenance_mode' );

	// Do not show for administrators
	if ( current_user_can( 'manage_options' ) ) {
		$maintenance_mode = kalium()->request->has( 'view-maintenance' );
	}

	return (bool) $maintenance_mode;
}

/**
 * Get CSS loading animations.
 *
 * @return array
 * @since 4.0
 */
function kalium_get_css_loading_animations() {
	return apply_filters(
		'kalium_css_loading_animations',
		[
			'modern-circular'  => [
				'title'   => 'Modern Circular',
				'content' => '<span class="loader-el modern-circular"><svg class="circular" viewBox="25 25 50 50"><circle class="path" cx="50" cy="50" r="20" fill="none" stroke-width="3" stroke-miterlimit="10"/></svg></span>',
			],
			'ball-scale'       => [
				'title'   => 'Ball Scale',
				'content' => '<span class="loader-el ball-scale"><span></span></span>',
			],
			'line-scale'       => [
				'title'   => 'Line Scale',
				'content' => '<span class="loader-el line-scale"><span></span><span></span><span></span><span></span><span></span></span>',
			],
			'ball-pulse'       => [
				'title'   => 'Ball Pulse',
				'content' => '<span class="loader-el ball-pulse"><span></span><span></span><span></span></span>',
			],
			'semi-circle-spin' => [
				'title'   => 'Semi Circle Spin',
				'content' => '<span class="loader-el semi-circle-spin"><span></span></span>',
			],
		]
	);
}

/**
 * Get CSS loading animation.
 *
 * @param string $id
 *
 * @return array
 * @since 4.0
 */
function kalium_get_css_loading_animation( $id ) {
	$loading_animations = kalium_get_css_loading_animations();

	if ( isset( $loading_animations[ $id ] ) ) {
		return $loading_animations[ $id ];
	}

	return current( $loading_animations );
}

/**
 * Get dominant colors from an image.
 *
 * @param string $file
 * @param int    $limit
 *
 * @return array|null
 * @since 4.0
 */
function kalium_get_image_dominant_colors( $file, $limit = 2 ) {
	try {
		$palette = \League\ColorExtractor\Palette::fromFilename( $file );

		return array_values(
			array_map(
				[ \League\ColorExtractor\Color::class, 'fromIntToHex' ],
				array_keys( $palette->getMostUsedColors( $limit ) )
			)
		);
	} catch ( Exception $e ) {
	}

	return null;
}

/**
 * Parse social share networks.
 *
 * @param array $content
 *
 * @return array
 * @since 4.0
 */
function kalium_parse_social_share_networks( $content ) {
	$networks = [];

	$networks_map = [
		'social_share_link_facebook'  => 'facebook',
		'social_share_link_twitter'   => 'twitter',
		'social_share_link_bluesky'   => 'bluesky',
		'social_share_link_threads'   => 'threads',
		'social_share_link_linkedin'  => 'linkedin',
		'social_share_link_tumblr'    => 'tumblr',
		'social_share_link_pinterest' => 'pinterest',
		'social_share_link_email'     => 'email',
		'social_share_link_vkontakte' => 'vkontakte',
		'social_share_link_whatsapp'  => 'whatsapp',
		'social_share_link_telegram'  => 'telegram',
		'social_share_link_print'     => 'print',
	];

	if ( isset( $content['elements'] ) ) {
		foreach ( $content['elements'] as $element ) {
			if ( ! empty( $networks_map[ $element['name'] ] ) ) {
				$networks[] = $networks_map[ $element['name'] ];
			}
		}
	}

	return $networks;
}

/**
 * Get image sizes.
 *
 * @param string $get_size
 *
 * @return array<array>|array|false
 * @since 4.0
 */
function kalium_get_image_sizes( $get_size = '' ) {
	global $_wp_additional_image_sizes;

	$default_image_sizes = [
		'thumbnail',
		'medium',
		'medium_large',
		'large',
	];

	$sizes = [];

	foreach ( $default_image_sizes as $size ) {
		$sizes[ $size ] = [
			'width'  => (int) get_option( $size . '_size_w' ),
			'height' => (int) get_option( $size . '_size_h' ),
			'crop'   => (bool) get_option( $size . '_crop' ),
		];
	}

	if ( $_wp_additional_image_sizes ) {
		$sizes = array_merge( $sizes, $_wp_additional_image_sizes );
	}

	// Get only 1 size
	if ( $get_size ) {
		return $sizes[ $get_size ] ?? false;
	}

	return apply_filters( 'image_size_names_choose', $sizes );
}

/**
 * Get image size names.
 *
 * @param array $args
 * @param bool  $refresh_sizes
 *
 * @return array
 * @since 4.0
 */
function kalium_get_image_size_names( $args = [] ) {
	static $size_names;

	$args = wp_parse_args(
		$args,
		[
			'include' => [],
			'exclude' => [],
			'custom'  => false,
		]
	);

	if ( is_null( $size_names ) ) {
		$size_names = apply_filters(
			'kalium_image_size_names',
			[
				'full'                          => 'Full Size',
				'thumbnail'                     => 'Thumbnail',
				'medium'                        => 'Medium',
				'medium_large'                  => 'Medium Large',
				'large'                         => 'Large',
				'woocommerce_thumbnail'         => 'WooCommerce Thumbnail',
				'woocommerce_single'            => 'WooCommerce Single',
				'woocommerce_gallery_thumbnail' => 'WooCommerce Gallery Thumbnail',
				'woocommerce_archive_thumbnail' => 'WooCommerce Archive Thumbnail',
			]
		);
	}

	// Image sizes
	$image_sizes     = kalium_get_image_sizes();
	$image_size_keys = array_keys( $image_sizes );

	// Set names
	$image_sizes = array_combine(
		$image_size_keys,
		array_map(
			function ( $size, $size_name ) use ( &$size_names ) {
				if ( isset( $size_names[ $size_name ] ) ) {
					$display_name = $size_names[ $size_name ];
				} else {
					$display_name = ucwords( str_replace( [ '-', '_' ], ' ', $size_name ) );
				}

				return sprintf( '%s - %s x %s', $display_name, kalium_get_array_key( $size, 'width' ) ?: 'Auto', kalium_get_array_key( $size, 'height' ) ?: 'Auto' );
			},
			$image_sizes,
			$image_size_keys
		)
	);

	// Full size
	$image_sizes['full'] = $size_names['full'];

	// Filters
	$include = is_array( $args['include'] ) ? $args['include'] : array_map( 'trim', explode( ',', $args['include'] ) );
	$exclude = is_array( $args['exclude'] ) ? $args['exclude'] : array_map( 'trim', explode( ',', $args['exclude'] ) );

	$available_sizes = array_filter(
		$image_sizes,
		function ( $size ) use ( $include, $exclude ) {
			if ( ! empty( $include ) ) {
				return in_array( $size, $include );
			} elseif ( ! empty( $exclude ) ) {
				return ! in_array( $size, $exclude );
			}

			return true;
		},
		ARRAY_FILTER_USE_KEY
	);

	// Custom image size entry
	if ( kalium_validate_boolean( $args['custom'] ) ) {
		$available_sizes['custom'] = 'Custom Image Size';
	}

	return $available_sizes;
}

/**
 * Get aspect ratio.
 *
 * @param array $args
 *
 * @return array
 * @since 4.0
 */
function kalium_get_aspect_ratio_names( $args = [] ) {
	static $available_ratios;

	$args = wp_parse_args(
		$args,
		[
			'include' => [],
			'exclude' => [],
			'custom'  => false,
		]
	);

	if ( is_null( $available_ratios ) ) {
		$available_ratios = [
			'auto' => 'Original',
			'1/1'  => 'Square - 1:1',
			'4/3'  => 'Standard - 4:3',
			'3/4'  => 'Portrait - 3:4',
			'3/2'  => 'Classic - 3:2',
			'2/3'  => 'Classic Portrait - 2:3',
			'16/9' => 'Wide - 16:9',
			'9/16' => 'Tall - 9:16',
		];
	}

	// Filters
	$include = is_array( $args['include'] ) ? $args['include'] : array_map( 'trim', explode( ',', $args['include'] ) );
	$exclude = is_array( $args['exclude'] ) ? $args['exclude'] : array_map( 'trim', explode( ',', $args['exclude'] ) );

	$ratios = array_filter(
		$available_ratios,
		function ( $size ) use ( $include, $exclude ) {
			if ( ! empty( $include ) ) {
				return in_array( $size, $include );
			} elseif ( ! empty( $exclude ) ) {
				return ! in_array( $size, $exclude );
			}

			return true;
		},
		ARRAY_FILTER_USE_KEY
	);

	// Custom aspect ratio entry
	if ( kalium_validate_boolean( $args['custom'] ) ) {
		$ratios['custom'] = 'Custom';
	}

	return $ratios;
}

/**
 * Get footer copyrights text.
 *
 * @return string
 */
function kalium_get_default_copyrights_text() {
	return sprintf( '&copy; %d %s &ndash; WordPress theme by <a href="%s">Laborator</a>', date( 'Y' ), get_bloginfo( 'name' ), 'https://laborator.co' );
}

/**
 * Map pixel unit values.
 *
 * @param numeric|string $value
 *
 * @return string|null
 */
function kalium_map_pixel_unit_values( $value ) {
	return is_numeric( $value ) ? "{$value}px" : ( ! empty( $value ) ? $value : null );
}

/**
 * Extract YouTube video ID from URL.
 *
 * @param string $url
 *
 * @return string|null
 */
function kalium_get_youtube_video_id_from_url( $url ) {
	preg_match( '/[\\?\\&]v=([^\\?\\&]+)/', $url, $matches );
	return $matches[1] ?? null;
}

/**
 * Helper function to retrieve attachment ID from media theme option.
 *
 * @param int|array $option
 *
 * @return int|false
 * @since 4.0
 */
function kalium_get_attachment_id_from_theme_option( $option ) {
	$media = kalium_get_theme_option( $option );

	if ( is_int( $media ) ) {
		return $media;
	} elseif ( is_array( $media ) && array_key_exists( 'id', $media ) ) {
		return $media['id'];
	}

	return false;
}

/**
 * Elementor switcher control value.
 *
 * @param bool|string $state
 * @param string      $return_value
 *
 * @since 4.0
 */
function kalium_elementor_switcher_control_value( $state, $return_value = 'yes' ) {
	if ( kalium_validate_boolean( $state ) ) {
		return $return_value;
	}

	return '';
}

/**
 * Get user IP.
 *
 * @return string|null
 */
function kalium_get_user_ip() {
	if ( ! isset( $_SERVER['REMOTE_ADDR'] ) ) {
		return null;
	}

	return $_SERVER['REMOTE_ADDR'];
}

/**
 * Get post likes.
 *
 * @param int $post_id
 *
 * @return array
 */
function kalium_get_post_likes( $post_id = null ) {
	$user_ip = kalium_get_user_ip();
	$post    = get_post( $post_id );
	$likes   = get_post_meta( $post->ID, 'post_likes', true );

	if ( ! is_array( $likes ) ) {
		$likes = [];
	}

	return [
		'liked' => in_array( $user_ip, $likes ),
		'count' => count( $likes ),
	];
}

/**
 * Build query vars from WPB query parameter string.
 *
 * @param string $query_data
 *
 * @return array
 * @since 4.0
 */
function kalium_wpb_build_query_args( $query_data ) {
	$query_data    = VcLoopSettings::parseData( $query_data );
	$query_builder = new VcLoopQueryBuilder( $query_data );

	$get_args = function () {
		return $this->args;
	};

	return $get_args->call( $query_builder );
}

/**
 * Get the translated object ID using WPML or Polylang.
 *
 * @param int                    $id
 * @param string|WP_Post|WP_Term $object
 * @param bool                   $return_original_if_missing
 * @param string|null            $language_code
 *
 * @return int|null
 * @since 4.2.2
 */
function kalium_get_translated_object_id( $id, $object = null, $return_original_if_missing = false, $language_code = null ) {
	$object_type = null;

	if ( is_string( $object ) ) {
		$object_type = $object;
	} elseif ( $object instanceof WP_Post ) {
		$object_type = $object->post_type;
	} elseif ( $object instanceof WP_Term ) {
		$object_type = $object->taxonomy;
	}

	// WPML
	if ( ! empty( $object_type ) && kalium()->is->wpml_active() ) {
		return apply_filters( 'wpml_object_id', $id, $object_type, $return_original_if_missing, $language_code ) ?: $id;
	}

	// Fallback
	return $id;
}

/**
 * Get original object ID.
 *
 * @param int                    $id
 * @param string|WP_Post|WP_Term $object
 *
 * @return int|null
 * @since 4.2.2
 */
function kalium_get_original_object_id( $id, $object = null ) {
	$language_code = null;

	// Default language
	if ( kalium()->is->wpml_active() ) {
		$language_code = wpml_get_default_language();
	}

	return kalium_get_translated_object_id( $id, $object, true, $language_code );
}

/**
 * Adds tooltip attributes to HTML element.
 *
 * @param string $content
 * @param array  $args
 *
 * @return array|null
 * @since 4.4.2
 */
function kalium_tooltip_attrs( $content, $args = [] ) {
	$args = wp_parse_args(
		$args,
		[
			'trigger'    => null,
			'placement'  => null,
			'echo'       => true,
			'echo_space' => true,
		]
	);

	$atts = [];

	if ( ! empty( $content ) ) {
		$atts['data-tippy-content'] = trim( $content );

		foreach ( [ 'trigger', 'placement' ] as $attr ) {
			if ( isset( $args[ $attr ] ) ) {
				$atts[ "data-tippy-$attr" ] = $args[ $attr ];
			}
		}

		// Enqueue Tippy library
		kalium_enqueue( 'tippy' );
	}

	if ( $args['echo'] ) {
		echo $args['echo_space'] ? ' ' : '';
		echo kalium()->helpers->render_attributes( $atts );
		return;
	}

	return $atts;
}
