<?php
/**
 * Kalium WordPress Theme
 *
 * Portfolio hook functions.
 *
 * @author Laborator
 * @link   https://kaliumtheme.com
 */
if ( ! defined( 'ABSPATH' ) ) {
	exit; // Direct access not allowed.
}

/**
 * Portfolio items per page.
 *
 * @param WP_Query $query
 *
 * @since 4.0
 */
function kalium_portfolio_posts_per_page( $query ) {
	if ( kalium()->portfolio->is_portfolio_query( $query ) ) {
		$posts_per_page = kalium_get_portfolio_loop_option( 'grid/items_per_page' );

		if ( empty( $query->get( 'posts_per_page' ) ) && is_numeric( $posts_per_page ) ) {
			$query->set( 'posts_per_page', $posts_per_page );
		}
	}
}

add_action( 'pre_get_posts', 'kalium_portfolio_posts_per_page' );

/**
 * Skip portfolio items that do not have featured image assigned.
 *
 * @param WP_Query $query
 *
 * @since 4.0
 */
function kalium_portfolio_skip_no_image_items( $query ) {
	if ( is_admin() && ! wp_doing_ajax() ) {
		return;
	}

	// Skip on singular pages
	if ( $query->is_singular() ) {
		$post_types = $query->get( 'post_type' );

		if ( ! empty( $post_types ) && ! is_array( $post_types ) ) {
			$post_types = explode( ',', $post_types );
		}

		if ( ! empty( $post_types ) ) {
			foreach ( $post_types as $post_type ) {
				if ( in_array( $post_type, kalium_get_portfolio_post_types() ) ) {
					return;
				}
			}
		}
	}

	if ( kalium()->portfolio->is_portfolio_query( $query ) ) {
		$current_meta_query = $query->get( 'meta_query' );

		$meta_query = [
			[
				'key'     => '_thumbnail_id',
				'compare' => 'EXISTS',
			],
		];

		if ( is_array( $current_meta_query ) ) {
			$meta_query = array_merge( $current_meta_query, $meta_query );
		}

		$query->set( 'meta_query', $meta_query );
	}
}

add_action( 'pre_get_posts', 'kalium_portfolio_skip_no_image_items' );

/**
 * Portfolio load more pagination query handler.
 *
 * @param WP_Query $query
 *
 * @since 4.0
 */
function kalium_portfolio_query_items( $query ) {
	if ( kalium()->portfolio->is_portfolio_query( $query ) && kalium_get_portfolio_loop_option( 'filtering/enabled' ) ) {
		$portfolio_query = kalium()->request->request( 'portfolio_query' );

		if ( is_string( $portfolio_query ) ) {
			$portfolio_query = kalium_decode_json( wp_unslash( $portfolio_query ), true );
		}

		// Validate query
		if ( ! is_array( $portfolio_query ) || empty( $portfolio_query ) ) {
			return;
		}

		$filters  = kalium_parse_portfolio_filters( $portfolio_query['filters'] ?? null );
		$item_ids = $portfolio_query['item_ids'] ?? null;

		// Build tax query
		$tax_query = $query->get( 'tax_query' );

		if ( ! is_array( $tax_query ) ) {
			$tax_query = [];
		}

		foreach ( kalium_get_portfolio_loop_option( 'filtering/taxonomies' ) as $taxonomy ) {
			if ( ! empty( $filters[ $taxonomy ] ) ) {
				$taxonomy_filters = [
					'relation' => 'OR',
				];

				unset( $query->query[ $taxonomy ] );
				unset( $query->query_vars[ $taxonomy ] );

				foreach ( $filters[ $taxonomy ] as $terms ) {
					$taxonomy_filters[] = [
						'taxonomy' => $taxonomy,
						'field'    => 'slug',
						'terms'    => $terms,
					];
				}

				$tax_query[] = $taxonomy_filters;
			}
		}

		$query->set( 'tax_query', $tax_query );

		// Query all items
		if ( ! empty( $portfolio_query['query_all'] ) ) {
			$query->tax_query = null;

			foreach ( kalium_get_portfolio_loop_option( 'filtering/taxonomies' ) as $taxonomy ) {
				unset( $query->query[ $taxonomy ] );
				unset( $query->query_vars[ $taxonomy ] );
			}

			// Force using post type
			$query->set( 'post_type', kalium_get_portfolio_loop_option( 'post_type' ) );
		}

		// Item IDs are provided, remove the pagination
		if ( ! empty( $item_ids ) ) {
			$query->set( 'post__not_in', $item_ids );
			$query->set( 'paged', 1 );

			$query->query['paged'] = 1;

			$GLOBALS['wp_query']->set( 'paged', 1 );

			// Remove the "p404_redirect" hook because it breaks the load-more pagination
			if ( function_exists( 'p404_redirect' ) ) {
				remove_action( 'wp', 'p404_redirect' );
			}
		}
	}
}

add_action( 'pre_get_posts', 'kalium_portfolio_query_items' );

/**
 * After deactivating Portfolio Post Type plugin, enable built-in portfolio post type.
 *
 * @since 4.0
 */
function kalium_portfolio_maybe_enable_portfolio_post_type( $plugin ) {
	if ( 'portfolio-post-type/portfolio-post-type.php' === $plugin && 0 < array_sum( (array) wp_count_posts( 'portfolio' ) ) ) {
		update_option( 'kalium_portfolio_post_type', true );
	}
}

add_action( 'deactivate_plugin', 'kalium_portfolio_maybe_enable_portfolio_post_type' );

/**
 * Share buttons
 *
 * @param array $atts
 *
 * @return string
 */
function kalium_shortcode_portfolio_share_buttons( $atts ) {
	global $post;

	ob_start();

	kalium_social_networks_share(
		[
			'post_id'  => kalium_get_array_key( $atts, 'id', $post->ID ),
			'networks' => kalium_get_theme_option( 'portfolio_share_item_networks' ),
			'style'    => 'default' === kalium_get_theme_option( 'portfolio_like_share_layout' ) ? 'text' : 'icons',
		]
	);

	return ob_get_clean();
}

add_shortcode( 'kalium_portfolio_share_buttons', 'kalium_shortcode_portfolio_share_buttons' );

/**
 * Add rewrite endpoints for taxonomies.
 */
function kalium_portfolio_taxonomy_rewrite_endpoints() {
	foreach ( kalium_get_portfolio_post_types() as $post_type ) {
		$taxonomies = get_object_taxonomies( $post_type, 'objects' );

		foreach ( $taxonomies as $taxonomy ) {
			$rewrite_slug = kalium_get_array_key( $taxonomy->rewrite, 'slug', $taxonomy->name );

			add_rewrite_endpoint( $rewrite_slug, EP_PERMALINK | EP_PAGES );
		}
	}
}

add_action( 'init', 'kalium_portfolio_taxonomy_rewrite_endpoints', 100 );

/**
 * Initialize main query in portfolio page.
 */
function kalium_portfolio_init_main_query() {

	// Portfolio page template
	if ( is_page_template( 'portfolio.php' ) ) {
		$custom_loop_options = [
			'url'        => get_permalink( get_queried_object() ),
			'query_vars' => [
				'post_type' => kalium_get_portfolio_loop_option( 'post_type' ),
			],
		];

		// Set dynamic image height
		$set_dynamic_image_height = function ( $state ) use ( &$custom_loop_options ) {
			if ( 'yes' === $state ) {
				$custom_loop_options['item']['thumbnail']['size'] = 'medium_large';
				$custom_loop_options['item']['aspect_ratio']      = null;
			} elseif ( 'no' === $state ) {
				$custom_loop_options['item']['thumbnail']['size'] = [ 655, 545 ];
				$custom_loop_options['item']['aspect_ratio']      = null;
			}
		};

		// Set hover effect
		$set_hover_effect = function ( $state ) use ( &$custom_loop_options ) {
			if ( 'none' === $state ) {
				$custom_loop_options['item']['thumbnail']['hover_overlay']['enabled'] = false;
			} elseif ( in_array( $state, [ 'full', 'distanced' ] ) ) {
				$custom_loop_options['item']['thumbnail']['hover_overlay']['enabled'] = true;

				if ( 'full' === $state ) {
					$custom_loop_options['item']['thumbnail']['hover_overlay']['offset'] = 0;
				} elseif ( 'distanced' === $state ) {
					$custom_loop_options['item']['thumbnail']['hover_overlay']['offset'] = 15;
				}
			}
		};

		// Set hover layer background color
		$set_hover_layer_bg = function ( $color ) {
			kalium_print_inline_style(
				[
					'selector' => kalium_get_portfolio_loop_option( 'selector' ) . ' .portfolio-item-entry',
					'vars'     => [
						'pi-overlay-bg' => $color,
					],
				]
			);
		};

		// Heading
		if ( ! kalium_get_field( 'show_title_description' ) ) {
			$custom_loop_options['heading']['title']['visible'] = false;
		}

		// Grid options
		$columns_count = kalium_get_field( 'columns_count' );

		if ( is_numeric( $columns_count ) ) {
			$custom_loop_options['grid']['columns'] = array_merge(
				kalium_get_portfolio_loop_option( 'grid/columns' ),
				[
					'desktop' => intval( $columns_count ),
				]
			);
		}

		// Reveal effect
		$reveal_effect = kalium_get_field( 'reveal_effect' );

		if ( 'inherit' !== $reveal_effect ) {
			$custom_loop_options['item']['aov']['enabled']          = 'none' !== $reveal_effect;
			$custom_loop_options['item']['aov']['legacy_animation'] = $reveal_effect;
			$custom_loop_options['item']['aov']['animation']        = null;
		}

		// Set default category
		if ( $default_category = kalium_get_field( 'default_filter_category' ) ) {
			$custom_loop_options['filtering']['current']['portfolio_category'] = $default_category->slug;

			if ( kalium_get_field( 'filter_category_hide_all' ) ) {
				$custom_loop_options['filtering']['reset'] = false;
			}
		}

		// Layout type options
		switch ( kalium_get_field( 'layout_type' ) ) {

			// Type 1
			case 'type-1':
				$custom_loop_options['layout'] = 'type-1';

				// Dynamic image height
				$set_dynamic_image_height( kalium_get_field( 'portfolio_type_1_dynamic_image_height' ) );

				// Hover effect
				$set_hover_effect( kalium_get_field( 'portfolio_type_1_thumbnail_hover_effect' ) );

				// Hover layer bg
				if ( $hover_layer_bg = kalium_get_field( 'portfolio_type_1_custom_hover_background_color' ) ) {
					$set_hover_layer_bg( $hover_layer_bg );
				}
				break;

			// Type 2
			case 'type-2':
				$custom_loop_options['layout'] = 'type-2';

				// Grid spacing
				$grid_spacing = kalium_get_field( 'portfolio_type_2_grid_spacing' );

				if ( 'normal' === $grid_spacing ) {
					$custom_loop_options['grid']['gap'] = 30;
				} elseif ( 'merged' === $grid_spacing ) {
					$custom_loop_options['grid']['gap'] = 0;
				}

				// Dynamic image height
				$set_dynamic_image_height( kalium_get_field( 'portfolio_type_2_dynamic_image_height' ) );

				// Hover effect
				$set_hover_effect( kalium_get_field( 'portfolio_type_2_thumbnail_hover_effect' ) );

				// Content position
				$content_position = kalium_get_field( 'portfolio_type_2_thumbnail_hover_text_position' );

				if ( 'inherit' !== $content_position ) {
					$custom_loop_options['item']['type-2']['content_position'] = $content_position;
				}

				// Hover layer bg
				$hover_layer_bg  = kalium_get_field( 'portfolio_type_2_thumbnail_hover_style' );
				$custom_hover_bg = kalium_get_field( 'portfolio_type_2_custom_hover_background_color' );

				if ( $custom_hover_bg ) {
					$set_hover_layer_bg( $custom_hover_bg );
				} elseif ( 'primary' === $hover_layer_bg ) {
					$set_hover_layer_bg( sprintf( 'color-mix(in srgb, var(%s) 85%, transparent)', kalium_css_var_name( 'color-1' ) ) );
				} elseif ( 'black' === $hover_layer_bg ) {
					$set_hover_layer_bg( 'rgba(10, 10, 10, 0.85)' );
				} elseif ( 'white' === $hover_layer_bg ) {
					$set_hover_layer_bg( 'rgba(255, 255, 255, 0.85)' );
				}
				break;
		}

		// Custom query
		if ( kalium_get_field( 'custom_query' ) ) {

			// Portfolio items
			$portfolio_items = kalium_get_field( 'portfolio_items' );

			if ( ! empty( $portfolio_items ) ) {
				$custom_loop_options['query_vars']['post__in'] = $portfolio_items;
			}

			// Taxonomy query
			$tax_query = [];

			// Category
			$portfolio_category = kalium_get_field( 'select_from_category' );

			if ( ! empty( $portfolio_category ) ) {
				$tax_query[] = [
					'taxonomy' => 'portfolio_category',
					'operator' => 'IN',
					'terms'    => $portfolio_category,
				];
			}

			// Tag
			$portfolio_tag = kalium_get_field( 'select_from_tags' );

			if ( ! empty( $portfolio_tag ) ) {
				$tax_query[] = [
					'taxonomy' => 'portfolio_tag',
					'operator' => 'IN',
					'terms'    => $portfolio_tag,
				];
			}

			// Add tax query
			if ( ! empty( $tax_query ) ) {
				$custom_loop_options['query_vars']['tax_query'] = $tax_query;

				// Unset individual iDs
				unset( $custom_loop_options['query_vars']['post__in'] );
			}

			// Order
			$order_by = kalium_get_field( 'order_by' );
			$order    = kalium_get_field( 'order' );

			if ( 'default' !== $order_by ) {
				$custom_loop_options['query_vars']['orderby'] = $order_by;
			}

			if ( in_array( $order, [ 'asc', 'desc' ] ) ) {
				$custom_loop_options['query_vars']['order'] = $order;
			}

			// Limit
			if ( is_numeric( $limit = kalium_get_field( 'items_per_page' ) ) ) {
				$custom_loop_options['query_vars']['posts_per_page'] = $limit;
			}
		}

		// Custom masonry items
		if ( kalium_get_field( 'masonry_style_portfolio' ) ) {
			$masonry_items = [];

			if ( is_array( $items_list = kalium_get_field( 'masonry_items_list' ) ) ) {
				foreach ( $items_list as $items_row ) {
					foreach ( $items_row['items_row'] as $item_entry ) {
						$masonry_items[] = [
							'id'       => $item_entry['item']->ID ?? null,
							'box_size' => $item_entry['box_size'],
						];
					}
				}
			}

			if ( ! empty( $masonry_items ) ) {
				$custom_loop_options['custom_masonry'] = $masonry_items;

				if ( is_numeric( $limit = kalium_get_field( 'masonry_items_per_page' ) ) ) {
					$custom_loop_options['query_vars']['posts_per_page'] = $limit;
				}
			}
		}

		// Execute query
		$custom_loop_options['query'] = new WP_Query(
			kalium_get_portfolio_loop_query_args(
				$custom_loop_options['query_vars'],
				false
			)
		);

		// Set query
		kalium_init_portfolio_loop_options( null, $custom_loop_options );
	}
}

add_action( 'kalium_portfolio_archive_before_content', 'kalium_portfolio_init_main_query', 0 );

/**
 * Custom masonry items handling.
 *
 * @since 4.0
 */
function kalium_portfolio_custom_masonry() {
	if ( $custom_masonry = kalium_get_portfolio_loop_option( 'custom_masonry' ) ) {
		$matched_entry = wp_list_filter( $custom_masonry, [ 'id' => get_the_ID() ] );

		if ( ! empty( $matched_entry ) ) {
			$box_entry = kalium_get_array_first( $matched_entry );
			$selector  = sprintf( '%s .portfolio-item-entry.post-%d', kalium_get_portfolio_loop_option( 'selector' ), get_the_ID() );

			$image_size   = wp_get_attachment_image_src( get_post_thumbnail_id(), 'full' );
			$columns      = $box_entry['columns'];
			$aspect_ratio = $box_entry['aspect_ratio'];
			$column_width = apply_filters( 'kalium_portfolio_custom_masonry_column_width', 550 * ( $columns / 3 ), $columns );

			// Generate image size for the column
			$resized_image_size = kalium()->helpers->aspect_ratio_fit( $image_size[1], $image_size[2], $aspect_ratio, $column_width );

			kalium_set_portfolio_loop_option( 'item/thumbnail/size', $resized_image_size );

			kalium_print_inline_style(
				[
					'selector' => $selector,
					'vars'     => [
						'grid-columns'     => kalium_parse_responsive_value(
							[
								'desktop' => kalium()->helpers->force_decimal_dot( 12 / $columns ),
								'mobile'  => 1,
							]
						),
						'pi-masonry-span'  => kalium()->helpers->force_decimal_dot( ceil( $columns / 4 ) - 1 ),
						'pi-masonry-ratio' => implode(
							' / ',
							array_map(
								[
									kalium()->helpers,
									'force_decimal_dot',
								],
								$aspect_ratio
							)
						),
					],
				]
			);
		}
	}
}

add_action( 'kalium_portfolio_loop_item', 'kalium_portfolio_custom_masonry', 0 );

/**
 * Apply portfolio permalinks.
 *
 * @since 4.0
 */
function kalium_portfolio_permalinks() {
	$permalinks = kalium_get_portfolio_permalinks();
	$post_types = kalium_get_portfolio_post_types();
	$empty_base = [];

	// Post types permalinks
	foreach ( $post_types as $post_type ) {
		$custom_prefix = kalium_get_array_key( $permalinks, $post_type, $post_type );

		// Change prefix
		add_filter(
			'register_post_type_args',
			function ( $args, $post_type_name ) use ( $post_type, $custom_prefix, &$empty_base ) {
				if ( $post_type_name === $post_type ) {
					$args['rewrite']['slug'] = $custom_prefix;

					// Add to empty base arr
					if ( ! $custom_prefix ) {
						$empty_base[] = $post_type_name;
					}
				}

				return $args;
			},
			10,
			2
		);

		// No prefix
		if ( empty( $custom_prefix ) ) {
			add_filter(
				'post_type_link',
				function ( $post_link, $post ) use ( $post_type ) {
					if ( $post_type === $post->post_type && 'publish' === $post->post_status ) {
						$post_link = str_replace( "/$post_type/", '/', $post_link );
					}

					return $post_link;
				},
				10,
				2
			);
		}
	}

	// Assign empty base post type
	add_action(
		'pre_get_posts',
		function ( $query ) use ( &$empty_base ) {
			/** @var WP_Query $query */
			if ( empty( $empty_base ) || ! $query->is_main_query() || ! isset( $query->query['page'] ) || is_home() || isset( $query->query['post_type'] ) ) {
				return;
			}

			$post_type = $query->get( 'post_type' );

			if ( empty( $post_type ) ) {
				$post_type = [ 'post', 'page' ];
			} elseif ( ! is_array( $post_type ) ) {
				$post_type = explode( ',', $post_type );
			}

			$query->set( 'post_type', array_merge( $post_type, $empty_base ) );
		},
		1000
	);

	// Taxonomy rewrite slug
	$taxonomy_slug = function ( $taxonomy ) use ( &$permalinks ) {
		$permalink = $permalinks[ $taxonomy ] ?? null;

		return function ( $args ) use ( &$taxonomy, &$permalink ) {
			$args['rewrite']['hierarchical'] = true;

			if ( ! empty( $permalink ) ) {
				$args['rewrite']['slug'] = $permalink;
			}

			return $args;
		};
	};

	// Taxonomies
	add_filter( 'portfolioposttype_category_args', $taxonomy_slug( 'portfolio_category' ) );
	add_filter( 'portfolioposttype_tag_args', $taxonomy_slug( 'portfolio_tag' ) );
}

add_action( 'init', 'kalium_portfolio_permalinks', 0 );

/**
 * Portfolio setup permalinks.
 *
 * @since 4.0
 */
function kalium_portfolio_setup_permalinks() {
	if ( ! kalium()->is->portfolio_active() ) {
		return;
	}

	$option_name = 'kalium_portfolio_permalinks';
	$post_types  = kalium_get_portfolio_post_types();
	$permalinks  = kalium_get_portfolio_permalinks();

	/**
	 * Permalink value sanitizer.
	 *
	 * @param string $permalink
	 *
	 * @return string
	 */
	$sanitize_permalink = function ( $permalink ) {
		$permalink = esc_url_raw( trim( $permalink ) );
		$permalink = str_replace( 'http://', '', $permalink );

		return untrailingslashit( $permalink );
	};

	/**
	 * Input field rendered.
	 *
	 * @param string $id
	 *
	 * @return callable
	 */
	$input_field = function ( $id ) {
		return function ( $args ) use ( &$id ) {
			$args = wp_parse_args(
				$args,
				[
					'placeholder' => null,
					'value'       => null,
					'description' => null,
				]
			);

			printf(
				'<input type="text" name="%1$s" id="%1$s" value="%2$s" class="regular-text code" placeholder="%3$s" />',
				esc_attr( $id ),
				esc_attr( $args['value'] ),
				esc_attr( $args['placeholder'] )
			);

			if ( $description = $args['description'] ) {
				printf( '<p class="description">%s</p>', trim( $description ) );
			}
		};
	};

	// Add settings section
	add_settings_section(
		'kalium_portfolio',
		'Portfolio',
		function () {
			echo wp_kses_post( wpautop( 'Adjust your portfolio\'s permalinks using the customization options below, optimizing your URLs for better organization and online performance.' ) );
		},
		'permalink'
	);

	// Save settings
	if (
		kalium()->request->has( $option_name, 'POST' ) &&
		current_user_can( 'manage_options' ) &&
		check_admin_referer( 'update-permalink' )
	) {
		if ( is_array( $permalinks = kalium()->request->input( $option_name ) ) ) {
			update_option( $option_name, array_map( $sanitize_permalink, $permalinks ) );
		}
	}

	// Portfolio permalink structure
	foreach ( $post_types as $post_type ) {
		$post_type_obj = get_post_type_object( $post_type );

		if ( $post_type_obj ) {
			$setting_id = "kalium_portfolio_permalinks[$post_type]";

			add_settings_field(
				$setting_id,
				sprintf( '%s base', $post_type_obj->label ),
				$input_field( $setting_id ),
				'permalink',
				'kalium_portfolio',
				[
					'value'       => kalium_get_array_key( $permalinks, $post_type, $post_type ),
					'description' => sprintf( 'Optional: Leave this field empty to remove the <strong>%s</strong> post type prefix.', $post_type ),
				]
			);
		}
	}

	// Taxonomy permalink structure
	$taxonomies = get_object_taxonomies( kalium_get_array_first( $post_types ), 'objects' );

	foreach ( $taxonomies as $taxonomy => $taxonomy_obj ) {
		$setting_id = "kalium_portfolio_permalinks[{$taxonomy}]";

		add_settings_field(
			$setting_id,
			$taxonomy_obj->label,
			$input_field( $setting_id ),
			'permalink',
			'kalium_portfolio',
			[
				'placeholder' => $taxonomy,
				'value'       => kalium_get_array_key( $permalinks, $taxonomy ),
			]
		);
	}
}

add_action( 'admin_init', 'kalium_portfolio_setup_permalinks' );

/**
 * Use custom term links endpoint on pages.
 *
 * @param string  $termlink
 * @param WP_Term $term
 * @param string  $taxonomy
 *
 * @return string
 * @since 4.0
 */
function kalium_portfolio_custom_term_link( $termlink, $term, $taxonomy ) {
	global $wp_rewrite;

	if (
		is_single() &&
		in_array( $taxonomy, kalium()->portfolio->get_taxonomies() ) &&
		kalium()->portfolio->is_portfolio_query( kalium_get_portfolio_loop_option( 'query' ) ) &&
		( $url = kalium_get_portfolio_loop_option( 'url' ) )
	) {
		$termlink = untrailingslashit( $url );

		// Add to permalink structure
		if ( $perma_struct = $wp_rewrite->get_extra_permastruct( $taxonomy ) ) {
			$termlink .= trailingslashit( str_replace( "%{$taxonomy}%", $term->slug, $perma_struct ) );
		} else {
			$termlink = add_query_arg( [ $taxonomy => $term->slug ], $termlink );
		}
	}

	return $termlink;
}

add_filter( 'term_link', 'kalium_portfolio_custom_term_link', 10, 3 );

/**
 * Add taxonomy classes for each portfolio loop item.
 *
 * @param string[] $classes
 *
 * @return array
 * @since 4.0
 */
function kalium_portfolio_loop_taxonomy_classes( $classes ) {
	global $post;

	if ( in_array( $post->post_type, kalium_get_portfolio_post_types() ) && kalium_get_portfolio_loop_option( 'filtering/enabled' ) ) {
		$taxonomy_terms = kalium_get_portfolio_loop_option( 'filtering/terms' );

		if ( is_array( $taxonomy_terms ) ) {
			foreach ( $taxonomy_terms as $taxonomy => $terms ) {
				foreach ( $terms as $term ) {
					$term_class = $taxonomy . '-' . $term['slug'];

					if ( in_array( $post->ID, $term['posts'] ) && ! in_array( $term_class, $classes ) ) {
						$classes[] = $term_class;
					}
				}
			}
		}
	}

	return $classes;
}

add_action( 'post_class', 'kalium_portfolio_loop_taxonomy_classes' );

/**
 * Custom portfolio options in single page.
 *
 * @param array $args
 *
 * @return array
 * @since 4.0
 */
function kalium_portfolio_single_custom_options( $args ) {
	// Full screen item type
	if ( 'type-5' === $args['type'] ) {
		$args['navigation']['prev_next']['layout']              = 'fixed';
		$args['navigation']['prev_next']['position_fixed_type'] = 'right';
	}

	return $args;
}

add_filter( 'kalium_portfolio_single_options', 'kalium_portfolio_single_custom_options' );

/**
 * Redirect portfolio items to external project link.
 *
 * @since 4.0
 */
function kalium_portfolio_single_redirect_to_project_link() {
	if ( is_singular( kalium_get_portfolio_post_types() ) && 'external' === kalium_get_portfolio_single_option( 'link/redirect_type' ) ) {
		$url = kalium_get_portfolio_single_option( 'link/url' );

		if ( $url && '#' !== $url ) {
			wp_redirect( $url, 301 );
			die();
		}
	}
}

add_action( 'template_redirect', 'kalium_portfolio_single_redirect_to_project_link' );

/**
 * Use portfolio item external link.
 *
 * @param string  $link
 * @param WP_Post $post
 *
 * @return string
 */
function kalium_portfolio_single_external_project_link( $link, $post ) {
	if ( in_array( $post->post_type, kalium_get_portfolio_post_types() ) && 'external' === kalium_get_field( 'item_linking', $post ) ) {
		return kalium_get_field( 'launch_link_href', $post ) ?: $link;
	}

	return $link;
}

add_filter( 'post_type_link', 'kalium_portfolio_single_external_project_link', 10, 2 );
