<?php
/**
 * Kalium WordPress Theme
 *
 * Portfolio template functions.
 *
 * @author Laborator
 * @link   https://kaliumtheme.com
 */
if ( ! defined( 'ABSPATH' ) ) {
	exit; // Direct access not allowed.
}

if ( ! function_exists( 'kalium_portfolio_archive_wrapper_start' ) ) {

	/**
	 * Portfolio archive wrapper start.
	 *
	 * @since 4.0
	 */
	function kalium_portfolio_archive_wrapper_start() {
		$args = [
			'classes' => [
				'portfolio',
			],
		];

		// Full width container
		$full_width         = kalium_get_theme_option( 'portfolio_full_width' );
		$full_width_heading = kalium_get_theme_option( 'portfolio_full_width_heading' );

		// Inherit options from "Portfolio Page" template
		if ( is_page() && is_page_template( 'portfolio.php' ) ) {
			$custom_full_width = kalium_get_field( 'portfolio_full_width' );

			if ( 'yes' === $custom_full_width ) {
				$full_width = true;

				// Page heading
				$custom_full_width_heading = kalium_get_field( 'portfolio_full_width_title_container' );

				if ( 'yes' === $custom_full_width_heading ) {
					$full_width_heading = 'contained';
				} elseif ( 'no' === $custom_full_width_heading ) {
					$full_width_heading = 'full-width';
				}
			} elseif ( 'no' === $custom_full_width ) {
				$full_width = false;
			}
		}

		if ( $full_width ) {
			$args['container_fullwidth'] = true;

			if ( 'contained' === $full_width_heading ) {
				$args['classes'][] = 'portfolio--heading-contained';
			}

			// Stretch to the edge
			if ( $stretch_edge = kalium_get_theme_option( 'portfolio_full_width_stretch_edge' ) ) {
				$args['classes'] = array_merge(
					$args['classes'],
					kalium_responsive_classname_infix(
						$stretch_edge,
						function ( $stretched, $bs_viewport, $viewport ) {
							if ( $stretched ) {
								return 'portfolio--stretch-items-' . $viewport;
							}
						}
					)
				);
			}
		}

		kalium_content_wrapper_start( 'portfolio_archive', $args );
	}
}

if ( ! function_exists( 'kalium_portfolio_archive_wrapper_end' ) ) {

	/**
	 * Portfolio archive wrapper end.
	 *
	 * @since 4.0
	 */
	function kalium_portfolio_archive_wrapper_end() {
		kalium_content_wrapper_end( 'portfolio_archive' );
	}
}

if ( ! function_exists( 'kalium_portfolio_loop_before' ) ) {

	/**
	 * Portfolio loop before.
	 *
	 * @since 4.2
	 */
	function kalium_portfolio_loop_before() {
		do_action( 'kalium_portfolio_loop_before' );
	}
}

if ( ! function_exists( 'kalium_portfolio_loop_after' ) ) {

	/**
	 * Portfolio loop after.
	 *
	 * @since 4.2
	 */
	function kalium_portfolio_loop_after() {
		do_action( 'kalium_portfolio_loop_after' );
	}
}

if ( ! function_exists( 'kalium_portfolio_items_loop' ) ) {

	/**
	 * Portfolio archive wrapper end.
	 *
	 * @since 4.0
	 */
	function kalium_portfolio_items_loop() {
		$id       = kalium_get_portfolio_loop_option( 'id' );
		$selector = kalium_get_portfolio_loop_option( 'selector' );
		$columns  = kalium_get_portfolio_loop_option( 'grid/columns' );

		// Portfolio is not enabled
		if ( ! kalium()->portfolio->is_enabled() ) {
			return;
		}

		// Container classes
		$classes = [
			'portfolio-items',
		];

		// ID class
		$classes[] = $id;

		// Grid columns
		$classes[] = 'grid';

		$classes = array_merge(
			$classes,
			kalium_responsive_classname_infix( $columns, 'columns' )
		);

		// Mark this container as modified columns when previewing in Customizer
		if ( is_customize_preview() ) {
			$compare_columns = array_map( 'kalium_parse_responsive_value', [ $columns, kalium_get_theme_option( 'portfolio_grid_columns' ) ] );

			if ( ! empty( array_diff( $compare_columns[0], $compare_columns[1] ) ) ) {
				$classes[] = 'portfolio-items--columns-modified';
			}
		}

		// Masonry layout
		$masonry_layout = kalium_get_portfolio_loop_option( 'grid/masonry/enabled' ) || 'isotope' === kalium_get_portfolio_loop_option( 'filtering/handler' );

		// Custom masonry mode
		if ( kalium_get_portfolio_loop_option( 'custom_masonry' ) ) {
			$classes[] = 'portfolio-items--custom-masonry';
		}

		// CSS vars
		$vars = [

			// Grid gap
			'grid-gap' => kalium_map_responsive_value(
				kalium_get_portfolio_loop_option( 'grid/gap' ),
				function ( $value ) {
					if ( is_numeric( $value ) ) {
						return $value . 'px';
					}
				}
			),
		];

		// Hover overlay
		if ( kalium_get_portfolio_loop_option( 'item/thumbnail/hover_overlay/enabled' ) ) {
			$offset = kalium_get_portfolio_loop_option( 'item/thumbnail/hover_overlay/offset' );

			// Hover overlay offset
			$vars['pi-overlay-offset'] = kalium_map_responsive_value(
				$offset,
				function ( $value ) {
					if ( is_numeric( $value ) ) {
						return $value . 'px';
					}
				}
			);

			// Hover overlay background color
			if ( 'color' === kalium_get_portfolio_loop_option( 'item/thumbnail/hover_overlay/color_type' ) ) {
				$vars['pi-overlay-bg'] = kalium_get_portfolio_loop_option( 'item/thumbnail/hover_overlay/color' );
			}
			// Hover overlay color alpha
			else {
				$vars['pi-overlay-bg-alpha'] = kalium_get_portfolio_loop_option( 'item/thumbnail/hover_overlay/color_alpha' ) / 100;
			}

			// Hover overlay text color
			if ( 'custom' === kalium_get_portfolio_loop_option( 'item/thumbnail/hover_overlay/text_color_type' ) ) {
				$vars['pi-overlay-color'] = kalium_get_portfolio_loop_option( 'item/thumbnail/hover_overlay/text_color' );
			}
		}

		// Aspect ratio
		$vars['custom-aspect-ratio'] = kalium_get_portfolio_loop_option( 'item/thumbnail/aspect_ratio' );

		// Border radius
		if ( $border_radius = kalium_get_portfolio_loop_option( 'item/thumbnail/border_radius' ) ) {
			$vars['pi-thumbnail-border-radius'] = $border_radius;
		}

		// Print JS args
		kalium_js_define_vars(
			[
				"portfolioContainers[{$id}]" => kalium_get_portfolio_container_js_options(),
			]
		);

		// Print style
		kalium_print_inline_style(
			[
				'styles' => [
					[
						'id'       => $id,
						'selector' => $selector,
						'vars'     => $vars,
					],
					// Hover overlay custom text color
					isset( $vars['pi-overlay-color'] ) ? [
						'selector' => $selector . ' .portfolio-item',
						'vars'     => [
							'link-hover-color' => $vars['pi-overlay-color'] ?? null,
						],
					] : null,
				],
			]
		);

		// Masonry container
		if ( $masonry_layout ) {
			kalium_create_masonry_container(
				[
					'options' => [
						'container'     => $selector,
						'item'          => '.portfolio-item-entry',
						'layout_mode'   => 'fit-rows' === kalium_get_portfolio_loop_option( 'grid/masonry/mode' ) ? 'fitRows' : 'packery',
						'hidden_style'  => [
							'opacity'   => 0,
							'transform' => 'scale(0.001)',
						],
						'visible_style' => [
							'opacity'   => 1,
							'transform' => 'scale(1)',
						],
					],
				]
			);
		}

		// Items loop
		kalium_get_template(
			'portfolio/items.php',
			[
				'attributes' => [
					'id'    => kalium_get_portfolio_loop_option( 'id' ),
					'class' => array_unique( $classes ),
				],
				'the_query'  => kalium_get_portfolio_loop_option( 'query' ),
			]
		);

		// Reset post data
		wp_reset_postdata();
	}
}

if ( ! function_exists( 'kalium_portfolio_loop_item_template' ) ) {

	/**
	 * Portfolio loop item template.
	 *
	 * @since 4.0
	 */
	function kalium_portfolio_loop_item_template() {
		$item_layout = kalium_get_portfolio_loop_option( 'layout' );

		// Attributes
		$attributes = [
			'class' => [
				'portfolio-item',
				'portfolio-item--' . $item_layout,
			],
		];

		// Animate on viewport
		$aov = kalium_get_portfolio_loop_option( 'item/aov' );

		if ( $aov['enabled'] && ! empty( $aov['animation'] ) ) {
			$attributes = array_merge(
				$attributes,
				kalium_animate_on_viewport(
					[
						'animation' => $aov['animation'],
						'duration'  => $aov['duration'],
						'stagger'   => $aov['stagger'],
						'delay'     => $aov['delay'],
					]
				)
			);
		}

		// Disabled
		if ( '#' === get_permalink() ) {
			$attributes['class'][] = 'portfolio-item--no-link';
		}

		// CSS vars
		$vars = [];

		// Dominant image color
		if ( kalium_get_portfolio_loop_option( 'item/thumbnail/hover_overlay/enabled' ) && 'dominant-color' === kalium_get_portfolio_loop_option( 'item/thumbnail/hover_overlay/color_type' ) ) {
			$vars['pi-thumb-color'] = kalium()->images->get_dominant_color( get_post_thumbnail_id() );
			$vars['pi-overlay-bg'] = sprintf(
				'color-mix(in srgb, var(%s) calc(%s * 100%%), transparent)',
				kalium_css_var_name( 'pi-thumb-color' ),
				kalium_css_var_reference( 'pi-overlay-bg-alpha' )
			);
		}

		// Print styles
		kalium_print_inline_style(
			[
				'selector' => sprintf(
					'%s .post-%d',
					kalium_get_portfolio_loop_option( 'selector' ),
					esc_attr( get_the_ID() )
				),
				'vars'     => $vars,
			]
		);

		// Item template
		kalium_get_template(
			'portfolio/loop/item-entry.php',
			[
				'attributes' => $attributes,
				'type'       => $item_layout,
			]
		);
	}
}

if ( ! function_exists( 'kalium_portfolio_item_default_template' ) ) {

	/**
	 * Portfolio loop item default template.
	 *
	 * @since 4.0
	 */
	function kalium_portfolio_item_default_template() {
		$type = kalium_get_portfolio_loop_option( 'layout' );

		/**
		 * Portfolio loop item type hook.
		 */
		do_action( "kalium_portfolio_loop_item_{$type}" );
	}
}

if ( ! function_exists( 'kalium_portfolio_loop_item_thumbnail' ) ) {

	/**
	 * Portfolio item featured image.
	 *
	 * @since 4.0
	 */
	function kalium_portfolio_loop_item_thumbnail() {
		if ( ! kalium_get_portfolio_loop_option( 'item/thumbnail/visible' ) || ! has_post_thumbnail() ) {
			return;
		}

		$attachment_id = get_post_thumbnail_id();
		$image_size    = kalium_get_portfolio_loop_option( 'item/thumbnail/size' );

		kalium_get_template(
			'portfolio/loop/item-thumbnail.php',
			[
				'attachment_id' => $attachment_id,
				'image_size'    => $image_size,
			]
		);
	}
}

if ( ! function_exists( 'kalium_portfolio_loop_maybe_display_featured_video' ) ) {

	/**
	 * Displays portfolio featured video if available.
	 *
	 * @since 4.0
	 */
	function kalium_portfolio_loop_maybe_display_featured_video() {
		$featured_video = kalium_get_array_first( kalium_get_field( 'featured_video' ) );

		if ( ! kalium_get_portfolio_loop_option( 'item/featured_video/enabled' ) || ! ( $video = kalium_get_array_key( $featured_video, 'src' ) ) ) {
			return;
		}

		// Video source
		$src    = kalium_get_array_key( $video, 'url' );
		$width  = kalium_get_array_key( $video, 'width' );
		$height = kalium_get_array_key( $video, 'height' );

		// Video props
		$autoplay = kalium_get_field( 'featured_video_autoplay' );
		$muted    = kalium_get_field( 'featured_video_mute' );
		$controls = kalium_get_field( 'featured_video_controls' );
		$loop     = kalium_get_field( 'featured_video_loop' );

		// Disable featured image
		kalium_set_portfolio_loop_option( 'item/thumbnail/visible', false, true );

		if ( ! $autoplay || $controls ) {
			kalium_set_portfolio_loop_option( 'item/thumbnail/hover_overlay/enabled', false, true );
		}

		kalium_render_element(
			[
				'tag_name'   => 'div',
				'attributes' => [
					'class' => [
						'portfolio-item__featured-video',
					],
				],
				'content'    => kalium_video_element(
					[
						'src'      => $src,
						'width'    => $width,
						'height'   => $height,
						'autoplay' => $autoplay,
						'muted'    => $muted || $autoplay,
						'controls' => $controls,
						'loop'     => $loop,
					]
				),
			]
		);
	}
}

if ( ! function_exists( 'kalium_portfolio_loop_item_title' ) ) {

	/**
	 * Portfolio loop item title.
	 *
	 * @since 4.0
	 */
	function kalium_portfolio_loop_item_title() {
		if ( ! kalium_get_portfolio_loop_option( 'item/title/visible' ) ) {
			return;
		}

		$title = get_the_title();

		// Link wrap
		if ( kalium_get_portfolio_loop_option( 'item/title/link' ) ) {
			$link_atts = [
				'href'       => get_permalink(),
				'aria-label' => $title,
			];

			// Open in new window
			if ( kalium_portfolio_external_link_opens_in_new_window() ) {
				$link_atts['target'] = '_blank';
			}

			$title = kalium_render_element(
				[
					'tag_name'   => 'a',
					'attributes' => $link_atts,
					'content'    => $title,
				],
				false
			);
		}

		// Title
		kalium_render_element(
			[
				'tag_name'   => kalium_get_portfolio_loop_option( 'item/title/tag_name' ),
				'attributes' => [
					'class' => [
						'portfolio-item__title',
						kalium_get_link_plain_class(),
					],
				],
				'content'    => $title,
			]
		);
	}
}

if ( ! function_exists( 'kalium_portfolio_loop_item_subtitle' ) ) {

	/**
	 * Portfolio item subtitle.
	 *
	 * @since 4.0
	 */
	function kalium_portfolio_loop_item_subtitle() {
		if ( ! kalium_get_portfolio_loop_option( 'item/subtitle/visible' ) ) {
			return;
		}

		$content_type = kalium_get_portfolio_loop_option( 'item/subtitle/content_type' );

		switch ( $content_type ) {

			// Categories / tags
			case 'categories':
			case 'categories-parent':
			case 'tags':
				$taxonomy = kalium_get_array_key(
					[
						'categories'        => 'portfolio_category',
						'categories-parent' => 'portfolio_category',
						'tags'              => 'portfolio_tag',
					],
					$content_type
				);

				$terms_list = get_the_terms( get_the_ID(), $taxonomy );

				if ( is_array( $terms_list ) ) {
					$subtitle = implode(
						kalium_get_portfolio_loop_option( 'item/subtitle/terms_separator' ),
						array_filter(
							array_map(
								function ( $term ) use ( $taxonomy, $content_type ) {
									if ( 'categories-parent' === $content_type && 0 !== $term->parent ) {
										return null;
									}

									return kalium_portfolio_term_filter_link( $term, false );
								},
								$terms_list
							)
						)
					);
				}
				break;

			// Item subtitle
			case 'subtitle':
				$subtitle = kalium_get_field( 'sub_title' );
				break;
		}

		// Display subtitle if available
		if ( ! empty( $subtitle ) ) {
			$classes = [
				'portfolio-item__subtitle',
				'portfolio-item__subtitle--' . $content_type,
				kalium_get_link_text_class(),
				kalium_get_link_style_class_by_context( 'portfolio-loop-item-categories', kalium_get_link_block_class() ),
			];

			printf( '<div %s>%s</div>', kalium_class_attr( $classes, false ), $subtitle );
		}
	}
}

if ( ! function_exists( 'kalium_portfolio_term_filter_link' ) ) {

	/**
	 * Portfolio filter term.
	 *
	 * @param WP_Term $term
	 * @param bool    $echo
	 *
	 * @return string|void
	 *
	 * @since 4.0
	 */
	function kalium_portfolio_term_filter_link( $term, $echo = true ) {
		if ( ! ( $term instanceof WP_Term ) ) {
			return;
		}

		$link = get_term_link( $term );

		if ( is_wp_error( $link ) ) {
			return null;
		}

		$term_link = sprintf(
			'<a href="%s" data-term="%s" rel="tag">%s</a>',
			esc_url( $link ),
			esc_attr( $term->taxonomy . ':' . $term->slug ),
			$term->name
		);

		if ( $echo ) {
			echo $term_link;
		} else {
			return $term_link;
		}
	}
}

if ( ! function_exists( 'kalium_portfolio_loop_item_details' ) ) {

	/**
	 * Portfolio item details.
	 *
	 * @since 4.0
	 */
	function kalium_portfolio_loop_item_details() {

		?>
		<div class="portfolio-item__details">

			<?php
			/**
			 * Portfolio loop item details hook.
			 *
			 * @hooked kalium_portfolio_loop_item_title - 10
			 * @hooked kalium_portfolio_loop_item_subtitle - 20
			 */
			do_action( 'kalium_portfolio_loop_item_details' );
			?>

		</div>
		<?php
	}
}

if ( ! function_exists( 'kalium_portfolio_loop_item_type_1_hover_layer' ) ) {

	/**
	 * Portfolio item type 1 thumbnail overlay.
	 *
	 * @since 4.0
	 */
	function kalium_portfolio_loop_item_type_1_thumbnail_overlay() {
		global $max_width_printed;

		if ( 'type-1' !== kalium_get_portfolio_loop_option( 'layout' ) || ! kalium_get_portfolio_loop_option( 'item/thumbnail/hover_overlay/enabled' ) ) {
			return;
		}

		$classes = [
			'portfolio-item__hover-overlay',
		];

		// CSS Vars
		$vars = [];

		// Icon
		if ( $icon_visible = kalium_get_portfolio_loop_option( 'item/type-1/icon/visible' ) ) {
			$icon_type         = kalium_get_portfolio_loop_option( 'item/type-1/icon/type' );
			$eye_icon          = kalium_get_portfolio_loop_option( 'item/type-1/icon/eye_icon' );
			$custom_icon       = kalium_get_portfolio_loop_option( 'item/type-1/icon/custom/attachment_id' );
			$custom_icon_width = kalium_get_portfolio_loop_option( 'item/type-1/icon/custom/width' );

			if ( is_null( $max_width_printed ) && is_numeric( $custom_icon_width ) ) {
				$vars['pi-overlay-icon-max-width'] = $custom_icon_width . 'px';

				$max_width_printed = true;
			}

			// Animated eye icon
			if ( 'eye' === $icon_type && 'animated' === $eye_icon ) {
				$classes[] = 'portfolio-item__hover-overlay--animated-eye';
			}
		}

		// Item styles
		kalium_print_inline_style(
			[
				'selector' => sprintf(
					'%s .portfolio-item__hover-overlay',
					kalium_get_portfolio_loop_option( 'selector' )
				),
				'vars'     => $vars,
			]
		);

		// Link target
		$link_target = kalium_portfolio_external_link_opens_in_new_window() ? '_blank' : '_self';

		?>
		<div <?php kalium_class_attr( $classes ); ?> data-url="<?php the_permalink(); ?>" data-target="<?php echo esc_attr( $link_target ); ?>">
			<?php
			if ( $icon_visible ) {
				// Eye icon
				if ( 'eye' === $icon_type && 'static' === $eye_icon ) {
					echo '<i class="kalium-icon-eye"></i>';
				} // Custom icon
				elseif ( 'custom' === $icon_type && ( $image = wp_get_attachment_image( $custom_icon, 'full', false, $max_width_attr ?? null ) ) ) {
					echo $image;
				}
			}
			?>
		</div>
		<?php
	}
}

if ( ! function_exists( 'kalium_portfolio_loop_item_like' ) ) {

	/**
	 * Portfolio loop item like button.
	 *
	 * @since 4.0
	 */
	function kalium_portfolio_loop_item_like() {
		if ( ! kalium_get_portfolio_loop_option( 'likes/enabled' ) ) {
			return;
		}

		$args = [
			'icon' => kalium_get_portfolio_loop_option( 'likes/icon' ),
		];

		?>
		<div class="portfolio-item__like">
			<?php
			// Like button
			kalium_like_button( $args );
			?>
		</div>
		<?php
	}
}

if ( ! function_exists( 'kalium_portfolio_loop_item_overlay_content' ) ) {

	/**
	 * Overlay content.
	 *
	 * @since 4.0
	 */
	function kalium_portfolio_loop_item_overlay_content() {
		if ( ! kalium_get_portfolio_loop_option( 'item/thumbnail/hover_overlay/enabled' ) ) {
			return;
		}

		$classes = [
			'portfolio-item__hover-overlay',
			'portfolio-item__hover-overlay--reveal-effect',
		];

		// Custom CSS vars
		$custom_vars = [];

		// Alignment
		$align = explode( '-', kalium_get_portfolio_loop_option( 'item/type-2/content_position' ) );

		if ( ! empty( $align[0] ) ) {
			$classes[] = 'portfolio-item__hover-overlay--align-' . $align[0];
		}

		if ( ! empty( $align[1] ) ) {
			$classes[] = 'portfolio-item__hover-overlay--align-' . $align[1];
		}

		// Dark color scheme
		$overlay_bg       = kalium_get_portfolio_loop_option( 'item/thumbnail/hover_overlay/color' );
		$overlay_bg_alpha = kalium()->style->color_alpha( $overlay_bg );

		if ( $custom_overlay_bg = kalium_get_field( 'custom_hover_background_color' ) ) {
			$overlay_bg       = $custom_overlay_bg;
			$overlay_bg_alpha = kalium()->style->color_alpha( $overlay_bg );

			$custom_vars['pi-overlay-bg'] = $overlay_bg;
		} elseif ( 'dominant-color' === kalium_get_portfolio_loop_option( 'item/thumbnail/hover_overlay/color_type' ) ) {
			$overlay_bg       = kalium()->images->get_dominant_color( get_post_thumbnail_id() );
			$overlay_bg_alpha = 1;
		}

		// Custom hover overlay text color
		$has_custom_text_color = 'custom' === kalium_get_portfolio_loop_option( 'item/thumbnail/hover_overlay/text_color_type' );

		if ( $custom_text_color = kalium_get_field( 'custom_hover_text_color' ) ) {
			$has_custom_text_color = true;

			$custom_vars['pi-overlay-color'] = $custom_vars['link-hover-color'] = $custom_text_color;
		}

		if ( $overlay_bg && ! kalium()->style->is_dark_color( $overlay_bg ) && 0.75 <= $overlay_bg_alpha && ! $has_custom_text_color ) {
			$classes[] = 'portfolio-item__hover-overlay--is-dark';
		}

		// Legacy: Item specific hover overlay options
		$legacy_overlay_style = kalium_get_field( 'hover_effect_style' );

		if ( $legacy_overlay_style && 'inherit' !== $legacy_overlay_style ) {

			// Hide hover overlay
			if ( 'none' === $legacy_overlay_style ) {
				return;
			}
			// No offset
			elseif ( 'full' === $legacy_overlay_style ) {
				$custom_vars['pi-overlay-offset'] = 0;
			}
			// With offset
			else {
				$custom_vars['pi-overlay-offset'] = '15px';
			}

			// Overlay initial state
			$overlay_initial_state = kalium_get_field( 'hover_layer_options' );

			if ( 'always-hover' === $overlay_initial_state ) {
				$classes[] = 'portfolio-item__hover-overlay--is-visible';
			} elseif ( 'hover-reverse' === $overlay_initial_state ) {
				$classes[] = 'portfolio-item__hover-overlay--is-reverse';
			}
		}

		// Custom style
		if ( ! empty( $custom_vars ) ) {
			kalium_print_inline_style(
				[
					'selector' => sprintf(
						'%s .post-%d .portfolio-item',
						kalium_get_portfolio_loop_option( 'selector' ),
						get_the_ID()
					),
					'vars'     => $custom_vars,
				]
			);
		}

		// Link target
		$link_target = kalium_portfolio_external_link_opens_in_new_window() ? '_blank' : '_self';

		?>
		<div <?php kalium_class_attr( $classes ); ?> data-url="<?php the_permalink(); ?>" data-target="<?php echo esc_attr( $link_target ); ?>">

			<?php
			/**
			 * Overlay content hooks.
			 *
			 * @hooked kalium_portfolio_loop_item_like - 10
			 * @hooked kalium_portfolio_loop_item_details - 20
			 */
			do_action( 'kalium_portfolio_loop_item_overlay_content' );
			?>

		</div>
		<?php
	}
}

if ( ! function_exists( 'kalium_portfolio_items_pagination' ) ) {

	/**
	 * Portfolio items pagination.
	 *
	 * @since 4.0
	 */
	function kalium_portfolio_items_pagination() {
		if ( ! kalium()->portfolio->is_enabled() || ! kalium_get_portfolio_loop_option( 'pagination/enabled' ) ) {
			return;
		}

		$pagination_type = kalium_get_portfolio_loop_option( 'pagination/type' );

		// More link
		if ( 'more-link' === $pagination_type ) {
			kalium_load_more_link(
				[
					'url'    => kalium_get_portfolio_loop_option( 'pagination/more_link/url' ),
					'target' => kalium_get_portfolio_loop_option( 'pagination/more_link/target' ),
					'text'   => kalium_get_portfolio_loop_option( 'pagination/more_link/text' ),
				]
			);
			return;
		}

		// Total pages
		$total_pages = kalium_get_portfolio_loop_option( 'query' )->max_num_pages;
		$total_posts = kalium_get_portfolio_loop_total_posts();
		$total_loop  = count( kalium_get_portfolio_loop_option( 'query' )->posts );

		if ( $total_posts <= $total_loop ) {
			return;
		}

		// Load more pagination
		if ( 'load-more' === $pagination_type ) {
			$maybe_hidden = 1 >= $total_pages ? 'hidden' : null;

			kalium_load_more_pagination(
				[
					'align'          => kalium_get_portfolio_loop_option( 'pagination/align' ),
					'loading_style'  => kalium_get_portfolio_loop_option( 'pagination/load_more/loading_style' ),
					'view_more_text' => kalium_get_portfolio_loop_option( 'pagination/load_more/view_more_text' ),
					'last_page_text' => kalium_get_portfolio_loop_option( 'pagination/load_more/last_page_text' ),
					'class'          => $maybe_hidden,
					'options'        => [
						'container'         => kalium_get_portfolio_loop_option( 'selector' ),
						'item'              => kalium_get_portfolio_loop_option( 'selector' ) . ' .portfolio-item-entry',
						'infinite_scroll'   => kalium_get_portfolio_loop_option( 'pagination/load_more/infinite_scroll' ),
						'current_page_path' => kalium_validate_boolean( $maybe_hidden ),
					],
				]
			);
		}

		// Pagination
		kalium_pagination(
			[
				'align'           => kalium_get_portfolio_loop_option( 'pagination/align' ),
				'display'         => kalium_get_portfolio_loop_option( 'pagination/numbers_display' ),
				'pagination_args' => [
					'total' => $total_pages,
				],
			]
		);
	}
}

if ( ! function_exists( 'kalium_portfolio_terms_filter' ) ) {

	/**
	 * Terms filter for portfolio loop.
	 *
	 * @param array $args
	 *
	 * @since 4.0
	 */
	function kalium_portfolio_terms_filter( $args = [] ) {
		if ( ! kalium_get_portfolio_loop_option( 'filtering/enabled' ) ) {
			return;
		}

		// Parse args
		$args = wp_parse_args(
			$args,
			[
				'taxonomy'     => null,
				'reset'        => kalium_get_portfolio_loop_option( 'filtering/reset' ),
				'reset_text'   => esc_html__( 'All', 'kalium' ),
				'active'       => null,
				'sub_terms'    => true,
				'align'        => null,
				'count'        => false,
				'count_style'  => null,
				'count_before' => null,
				'count_after'  => null,
			]
		);

		$taxonomy    = $args['taxonomy'];
		$taxonomies  = kalium_get_portfolio_loop_option( 'filtering/taxonomies' );
		$total_posts = kalium_get_portfolio_loop_total_posts();
		$label       = kalium_get_theme_option( $taxonomy . '_filter_label' );

		// Taxonomy not supported
		if ( empty( $taxonomies ) || ! in_array( $taxonomy, $taxonomies ) ) {
			return;
		}

		// Archive URL
		$archive_url = kalium_get_portfolio_loop_option( 'url' );

		if ( is_null( $archive_url ) ) {
			$archive_url = get_post_type_archive_link( kalium_get_portfolio_loop_option( 'post_type' ) );
		}

		// Terms
		$terms = kalium_get_portfolio_loop_option( 'filtering/terms' );

		// Taxonomy terms
		$taxonomy_terms = $terms[ $taxonomy ] ?? null;

		if ( empty( $taxonomy_terms ) ) {
			return;
		}

		// Term groups
		$term_groups = array_reduce(
			$taxonomy_terms,
			function ( $acc, $curr ) {
				if ( ! in_array( $curr['parent'], $acc ) ) {
					$acc[] = $curr['parent'];
				}

				return $acc;
			},
			[]
		);

		sort( $term_groups, SORT_NUMERIC );

		// Set active term
		$term_slugs = wp_list_pluck( $taxonomy_terms, 'parent', 'slug' );
		$term_ids   = wp_list_pluck( $taxonomy_terms, 'slug', 'id' );

		$active_terms = [];
		$active_term  = $args['active'];

		while ( $active_term && isset( $term_slugs[ $active_term ] ) ) {
			array_unshift( $active_terms, $active_term );

			$active_term = $term_ids[ $term_slugs[ $active_term ] ] ?? null;
		}

		// Active term group
		$active_term_group = $term_slugs[ end( $active_terms ) ] ?? 0;

		// Classes
		$classes = [
			'portfolio-filters',
			kalium_get_link_style_class(),
			kalium_get_link_text_class(),
			'all-links',
		];

		// Alignment
		if ( in_array( $args['align'], [ 'left', 'center', 'right' ] ) ) {
			$classes[] = 'portfolio-filters--align-' . $args['align'];
		}

		// Count style
		if ( $args['count'] && $args['count_style'] ) {
			$classes[] = 'portfolio-filters--count-' . $args['count_style'];
		}

		/**
		 * Show term count.
		 *
		 * @param int $num
		 *
		 * @return string
		 */
		$show_term_count = function ( $num ) use ( &$args ) {
			if ( $args['count'] ) {
				printf(
					'<span class="portfolio-filters__term__count" data-before="%1$s" data-after="%2$s" aria-hidden="true">%3$s</span><span class="screen-reader-text">%4$s</span>',
					esc_attr( $args['count_before'] ),
					esc_attr( $args['count_after'] ),
					esc_html( number_format_i18n( $num ) ),
					esc_html( sprintf( _n( '%s item', '%s items', $num, 'kalium' ), number_format_i18n( $num ) ) ),
				);
			}
		};

		?>
		<div <?php kalium_class_attr( $classes ); ?> data-portfolio-id="<?php echo esc_attr( kalium_get_portfolio_loop_option( 'id' ) ); ?>" data-taxonomy="<?php echo esc_attr( $taxonomy ); ?>">

			<?php
			// Filters
			foreach ( $term_groups as $term_group ) {
				$root_terms  = 0 === $term_group;
				$sub_terms   = ! $root_terms;
				$parent_term = $sub_terms ? get_term( $term_group, $taxonomy ) : null;

				// Term container attributes
				$terms_attributes = [
					'class' => [
						'portfolio-filters__terms',
					],
				];

				// Active terms
				if ( $term_group === $active_term_group ) {
					$terms_attributes['class'][] = 'portfolio-filters__terms--active';
				}

				// Sub terms disabled
				if ( $sub_terms && ! $args['sub_terms'] ) {
					continue;
				}

				// Data attributes for sub terms
				if ( $sub_terms ) {
					$terms_attributes = array_merge(
						$terms_attributes,
						[
							'data-parent' => $parent_term->slug,
						]
					);
				}

				?>
				<ul <?php kalium_render_attributes( $terms_attributes ); ?>>

					<?php
					// Label
					if ( $root_terms && $label ) {
						$filter_classes = [
							'portfolio-filters__term',
							'portfolio-filters__term--label',
						];

						?>
						<li <?php kalium_class_attr( $filter_classes ); ?>>
							<?php echo esc_html( $label ); ?>
						</li>
						<?php
					}

					// Reset
					if ( $root_terms && $args['reset'] ) {
						$reset_classes = [
							'portfolio-filters__term',
							'portfolio-filters__term--reset',
						];

						if ( empty( $active_terms ) ) {
							$reset_classes[] = 'portfolio-filters__term--active';
						}

						?>
						<li <?php kalium_class_attr( $reset_classes ); ?>>
							<?php
							kalium_render_element(
								[
									'tag_name'   => 'a',
									'attributes' => [
										'href'      => esc_url( $archive_url ),
										'data-term' => sprintf( '%s:*', esc_attr( $taxonomy ) ),
									],
									'content'    => esc_html( $args['reset_text'] ),
								]
							);

							// Term count
							$show_term_count( $total_posts );
							?>
						</li>
						<?php
					}
					// Back
					elseif ( $sub_terms ) {
						$back_classes = [
							'portfolio-filters__term',
							'portfolio-filters__term--back',
						];

						?>
						<li <?php kalium_class_attr( $back_classes ); ?>>
							<i class="kalium-icon-arrow-left"></i>

							<?php
							// Back to parent term
							kalium_portfolio_term_filter_link( $parent_term );
							?>
						</li>
						<?php
					}

					// Terms
					foreach ( $taxonomy_terms as $term_entry ) {
						if ( $term_entry['parent'] !== $term_group ) {
							continue;
						}

						$term_classes = [
							'portfolio-filters__term',
						];

						if ( in_array( $term_entry['slug'], $active_terms ) ) {
							$term_classes[] = 'portfolio-filters__term--active';
						}

						?>
						<li <?php kalium_class_attr( $term_classes ); ?>>
							<?php
							// Term filter link
							kalium_portfolio_term_filter_link( get_term( $term_entry['id'], $taxonomy ) );

							// Term count
							$show_term_count( $term_entry['count'] );
							?>
						</li>
						<?php
					}
					?>

				</ul>
				<?php
			}
			?>

		</div>
		<?php
	}
}

if ( ! function_exists( 'kalium_portfolio_loop_heading' ) ) {

	/**
	 * Portfolio loop heading.
	 *
	 * @since 4.0
	 */
	function kalium_portfolio_loop_heading() {
		$classes = [
			'portfolio-heading',
			'portfolio-heading--' . kalium_get_portfolio_loop_option( 'heading/layout' ),
		];
		?>
		<div <?php kalium_class_attr( $classes ); ?> data-portfolio-id="<?php echo esc_attr( kalium_get_portfolio_loop_option( 'id' ) ); ?>">
			<?php
			// Title
			if ( kalium_get_portfolio_loop_option( 'heading/title/visible' ) ) {
				$tag_name = kalium_get_portfolio_loop_option( 'heading/title/tag_name' );
				$title    = kalium_get_portfolio_loop_option( 'heading/title/text' );
				$content  = kalium_get_portfolio_loop_option( 'heading/title/content' );

				?>
				<div class="portfolio-heading__title">
					<?php
					// Title text
					if ( $title ) {
						kalium_render_element(
							[
								'tag_name' => $tag_name,
								'content'  => wp_kses_post( $title ),
							]
						);
					}

					// Content
					if ( $content ) {
						?>
						<div class="portfolio-heading__content entry-content">
							<?php echo $content; ?>
						</div>
						<?php
					}
					?>
				</div>
				<?php
			}

			// Filters
			if ( kalium_get_portfolio_loop_option( 'filtering/enabled' ) && is_array( $taxonomies = kalium_get_portfolio_loop_option( 'filtering/taxonomies' ) ) ) {

				?>
				<div class="portfolio-heading__filters">
					<?php
					foreach ( $taxonomies as $taxonomy ) {
						?>
						<div class="portfolio-heading__filters__filter">
							<?php
							kalium_portfolio_terms_filter(
								[
									'taxonomy'     => $taxonomy,
									'active'       => kalium_get_portfolio_loop_option( "filtering/current/{$taxonomy}" ),
									'sub_terms'    => kalium_get_portfolio_loop_option( 'filtering/sub_terms' ),
									'reset'        => kalium_get_portfolio_loop_option( 'filtering/reset' ),
									'reset_text'   => kalium_get_portfolio_loop_option( 'filtering/reset_text' ),
									'count'        => kalium_get_portfolio_loop_option( 'filtering/count/visible' ),
									'count_style'  => kalium_get_portfolio_loop_option( 'filtering/count/style' ),
									'count_before' => kalium_get_portfolio_loop_option( 'filtering/count/before' ),
									'count_after'  => kalium_get_portfolio_loop_option( 'filtering/count/after' ),
								]
							);
							?>
						</div>
						<?php
					}
					?>
				</div>
				<?php
			}
			?>

		</div>
		<?php
	}
}

if ( ! function_exists( 'kalium_portfolio_single_wrapper_start' ) ) {

	/**
	 * Portfolio single wrapper start.
	 *
	 * @since 4.0
	 */
	function kalium_portfolio_single_wrapper_start() {
		$args = [
			'classes' => [
				'single-portfolio',
				kalium_get_portfolio_single_option( 'id' ),
			],
		];

		// Custom item type
		if ( 'type-7' === kalium_get_portfolio_single_option( 'type' ) ) {
			$args['container_fullwidth'] = true;
		}

		kalium_content_wrapper_start( 'portfolio_single', $args );
	}
}

if ( ! function_exists( 'kalium_portfolio_single_wrapper_end' ) ) {

	/**
	 * Portfolio single wrapper end.
	 *
	 * @since 4.0
	 */
	function kalium_portfolio_single_wrapper_end() {
		kalium_content_wrapper_end( 'portfolio_single' );
	}
}

if ( ! function_exists( 'kalium_portfolio_single_item_type_display' ) ) {

	/**
	 * Item default template.
	 *
	 * @since 4.0
	 */
	function kalium_portfolio_single_item_default_template() {
		$id        = kalium_get_portfolio_single_option( 'id' );
		$item_type = kalium_get_portfolio_single_option( 'type' );

		// Item details in JS
		kalium_js_define_vars(
			[
				"portfolioSingle[{$id}]" => kalium_get_portfolio_single_js_options(),
			]
		);

		/**
		 * Portfolio single item type template.
		 */
		do_action( "kalium_portfolio_single_item_{$item_type}", $item_type );
	}
}

if ( ! function_exists( 'kalium_portfolio_single_type_side' ) ) {

	/**
	 * Side portfolio item type.
	 *
	 * @param string $item_type
	 *
	 * @since 4.0
	 */
	function kalium_portfolio_single_type_side( $item_type ) {
		$class_name = 'single-portfolio__type-side';

		$attributes = [
			'class' => [
				'single-portfolio__type',
			],
		];

		// Type class
		$attributes['class'][] = $class_name;

		// Content alignment
		if ( $content_align = kalium_get_portfolio_single_item_type_option( 'content/align' ) ) {
			$attributes['class'][] = $class_name . '--content-' . $content_align;
		}

		// Content width
		$content_width  = kalium_get_portfolio_single_item_type_option( 'content/width' );
		$defined_widths = [
			'1-4' => 'fourth',
			'1-2' => 'half',
		];

		if ( ! empty( $defined_widths[ $content_width ] ) ) {
			$attributes['class'][] = $class_name . '--content-' . $defined_widths[ $content_width ];
		}

		// Sticky description
		if ( kalium_get_portfolio_single_item_type_option( 'content/sticky' ) ) {
			$attributes['class'][] = $class_name . '--sticky';
		}

		kalium_get_template(
			'portfolio/single/item-type.php',
			[
				'item_type'  => $item_type,
				'attributes' => $attributes,
			]
		);
	}
}

if ( ! function_exists( 'kalium_portfolio_single_type_columned' ) ) {

	/**
	 * Columned portfolio item type.
	 *
	 * @param string $item_type
	 *
	 * @since 4.0
	 */
	function kalium_portfolio_single_type_columned( $item_type ) {
		$class_name = 'single-portfolio__type-columned';

		$attributes = [
			'class' => [
				'single-portfolio__type',
			],
		];

		// Layout class
		if ( $layout = kalium_get_portfolio_single_item_type_option( 'layout' ) ) {
			$attributes['class'][] = 'single-portfolio__type--' . $layout;
		}

		// Type class
		$attributes['class'][] = $class_name;

		// Gallery above description
		if ( 'above-description' === kalium_get_portfolio_single_item_type_option( 'gallery/position' ) ) {
			$attributes['class'][] = $class_name . '--gallery-first';
		}

		// Move share and like after gallery
		if ( 'after' === kalium_get_portfolio_single_item_type_option( 'share_like_position' ) ) {
			$like_and_share_centered = function () {
				kalium_portfolio_single_like_and_share( [ 'centered' => true ] );
			};

			remove_action( 'kalium_portfolio_single_type-2_content', 'kalium_portfolio_single_like_and_share', 90 );
			add_action( 'kalium_portfolio_single_type-2_gallery', $like_and_share_centered, 20 );
		}

		// Featured image
		if ( kalium_get_portfolio_single_item_type_option( 'featured_image/enabled' ) && kalium_get_portfolio_single_item_type_option( 'featured_image/full_width' ) ) {
			$attributes['class'][] = $class_name . '--full-width-featured-image';
		}

		// Full width gallery
		if ( kalium_get_portfolio_single_item_type_option( 'gallery/full_width' ) ) {
			$attributes['class'][] = $class_name . '--full-width-gallery';
		}

		kalium_get_template(
			'portfolio/single/item-type.php',
			[
				'item_type'  => $item_type,
				'attributes' => $attributes,
			]
		);
	}
}

if ( ! function_exists( 'kalium_portfolio_single_type_carousel' ) ) {

	/**
	 * Carousel portfolio item type.
	 *
	 * @param string $item_type
	 *
	 * @since 4.0
	 */
	function kalium_portfolio_single_type_carousel( $item_type ) {
		$class_name = 'single-portfolio__type-carousel';

		$attributes = [
			'class' => [
				'single-portfolio__type',
			],
		];

		// Layout class
		if ( $layout = kalium_get_portfolio_single_item_type_option( 'layout' ) ) {
			$attributes['class'][] = 'single-portfolio__type--' . $layout;
		}

		// Type class
		$attributes['class'][] = $class_name;

		// Reposition share and like
		if ( 'centered' === kalium_get_portfolio_single_item_type_option( 'layout' ) ) {
			remove_action( 'kalium_portfolio_single_type-3_content', 'kalium_portfolio_single_like_and_share', 90 );
			add_action( 'kalium_portfolio_single_type-3_content', 'kalium_portfolio_single_like_and_share', 35 );
		}

		kalium_get_template(
			'portfolio/single/item-type.php',
			[
				'item_type'  => $item_type,
				'attributes' => $attributes,
			]
		);
	}
}

if ( ! function_exists( 'kalium_portfolio_single_type_zig_zag' ) ) {

	/**
	 * Zig zag portfolio item type.
	 *
	 * @param string $item_type
	 *
	 * @since 4.0
	 */
	function kalium_portfolio_single_type_zig_zag( $item_type ) {
		$class_name = 'single-portfolio__type-zig-zag';

		$attributes = [
			'class' => [
				'single-portfolio__type',
			],
		];

		// Layout class
		if ( $layout = kalium_get_portfolio_single_item_type_option( 'layout' ) ) {
			$attributes['class'][] = 'single-portfolio__type--' . $layout;
		}

		// Type class
		$attributes['class'][] = $class_name;

		// Move share and like after gallery
		if ( 'after' === kalium_get_portfolio_single_item_type_option( 'share_like_position' ) ) {
			remove_action( 'kalium_portfolio_single_type-4_content', 'kalium_portfolio_single_like_and_share', 90 );
			add_action( 'kalium_portfolio_single_type-4_gallery', 'kalium_portfolio_single_like_and_share', 20 );
		}
		// Centered layout
		elseif ( 'centered' === kalium_get_portfolio_single_item_type_option( 'layout' ) ) {
			remove_action( 'kalium_portfolio_single_type-4_content', 'kalium_portfolio_single_like_and_share', 90 );
			add_action( 'kalium_portfolio_single_type-4_content', 'kalium_portfolio_single_like_and_share', 35 );
		}

		kalium_get_template(
			'portfolio/single/item-type.php',
			[
				'item_type'  => $item_type,
				'attributes' => $attributes,
			]
		);
	}
}

if ( ! function_exists( 'kalium_portfolio_single_type_full_screen' ) ) {

	/**
	 * Full screen portfolio item type.
	 *
	 * @param string $item_type
	 *
	 * @since 4.0
	 */
	function kalium_portfolio_single_type_full_screen( $item_type ) {
		$class_name = 'single-portfolio__type-full-screen';

		$attributes = [
			'class' => [
				'single-portfolio__type',
			],
		];

		// Type class
		$attributes['class'][] = $class_name;

		// Hide content
		if ( ! kalium_get_portfolio_single_item_type_option( 'content/enabled' ) ) {
			remove_all_actions( 'kalium_portfolio_single_type-5_content' );
			$attributes['class'][] = $class_name . '--content-hidden';
		}
		// Collapsed content
		elseif ( 'collapsed' === kalium_get_portfolio_single_item_type_option( 'content/toggle_state' ) ) {
			$attributes['class'][] = $class_name . '--content-collapsed';
		}

		// Disable footer
		add_filter( 'kalium_show_footer', '__return_false' );

		kalium_get_template(
			'portfolio/single/item-type.php',
			[
				'item_type'  => $item_type,
				'attributes' => $attributes,
			]
		);
	}
}

if ( ! function_exists( 'kalium_portfolio_single_type_lightbox' ) ) {

	/**
	 * Custom portfolio item type.
	 *
	 * @param string $item_type
	 *
	 * @since 4.0
	 */
	function kalium_portfolio_single_type_lightbox( $item_type ) {
		$class_name = 'single-portfolio__type-lightbox';

		$attributes = [
			'class' => [
				'single-portfolio__type',
			],
		];

		// Type class
		$attributes['class'][] = $class_name;

		kalium_get_template(
			'portfolio/single/item-type.php',
			[
				'item_type'  => $item_type,
				'attributes' => $attributes,
			]
		);
	}
}

if ( ! function_exists( 'kalium_portfolio_single_featured_image' ) ) {

	/**
	 * Featured image.
	 *
	 * @since 4.0
	 */
	function kalium_portfolio_single_featured_image( $args = [] ) {
		$args = wp_parse_args(
			$args,
			[
				'size'    => kalium_get_portfolio_single_option( 'gallery/image_size' ),
				'overlay' => null,
				'aov'     => null,
			]
		);

		$container_attributes = [
			'class' => [
				'single-portfolio__featured-image',
			],
		];

		// Lightbox
		if ( kalium_get_portfolio_single_option( 'gallery/lightbox' ) && 'type-6' !== kalium_get_portfolio_single_option( 'type' ) ) {
			$container_attributes['class'][] = kalium_get_lightbox_class();
		}

		if ( is_array( $args['aov'] ) ) {
			$container_attributes = array_merge( $container_attributes, $args['aov'] );
		}

		if ( has_post_thumbnail() ) {
			$content = kalium_portfolio_single_image(
				get_post_thumbnail_id(),
				[
					'size'    => $args['size'],
					'caption' => false,
					'overlay' => $args['overlay'],
					'return'  => true,
				]
			);

			kalium_render_element(
				[
					'tag_name'   => 'div',
					'attributes' => $container_attributes,
					'content'    => $content,
				]
			);
		}
	}
}

if ( ! function_exists( 'kalium_portfolio_single_display_featured_image' ) ) {

	/**
	 * Displays featured image.
	 *
	 * @since 4.0
	 */
	function kalium_portfolio_single_display_featured_image() {
		if ( ! kalium_get_portfolio_single_item_type_option( 'featured_image/enabled' ) ) {
			return;
		}

		$args = [];

		// Image size
		if ( kalium_get_portfolio_single_item_type_option( 'featured_image/full_width' ) ) {
			$args['size'] = 'full';
		}

		// Overlay
		if ( kalium_get_portfolio_single_item_type_option( 'featured_image/overlay/enabled' ) ) {
			$text = kalium_get_portfolio_single_item_type_option( 'featured_image/overlay/text' );

			if ( $icon = kalium_get_portfolio_single_item_type_option( 'featured_image/overlay/icon' ) ) {
				$text .= sprintf( '<i class="%s"></i>', $icon );
			}

			$args['overlay'] = "<span>{$text}</span>";
		}

		// Reveal effect
		if ( $reveal_effect = kalium_get_portfolio_single_option( 'gallery/reveal_effect' ) ) {
			$args['aov'] = $reveal_effect;

		}

		// Display image
		kalium_portfolio_single_featured_image( $args );
	}
}

if ( ! function_exists( 'kalium_portfolio_single_title_and_subtitle' ) ) {

	/**
	 * Title and subtitle.
	 *
	 * @since 4.0
	 */
	function kalium_portfolio_single_title_and_subtitle( $args = [] ) {
		$args = wp_parse_args(
			$args,
			[
				'centered' => 'centered' === kalium_get_portfolio_single_item_type_option( 'layout' ),
			]
		);

		$tag_name = kalium_get_portfolio_single_option( 'title/tag_name' );
		$title    = get_the_title();
		$subtitle = wpautop( trim( kalium_get_portfolio_single_option( 'subtitle' ) ) );

		$classes = [
			'single-portfolio__title',
		];

		if ( $args['centered'] ) {
			$classes[] = 'single-portfolio__title--centered';
		}

		kalium_get_template(
			'portfolio/single/title.php',
			[
				'attributes' => [
					'class' => $classes,
				],
				'tag_name'   => $tag_name,
				'title'      => $title,
				'subtitle'   => $subtitle,
			]
		);
	}
}

if ( ! function_exists( 'kalium_portfolio_single_entry_content' ) ) {

	/**
	 * Portfolio entry content.
	 *
	 * @param string $item_type
	 *
	 * @since 4.0
	 */
	function kalium_portfolio_single_entry_content() {
		$classes = [
			'single-portfolio__entry-content',
			'entry-content',
		];

		kalium_render_element(
			[
				'tag_name'   => 'div',
				'attributes' => [
					'class' => $classes,
				],
				'content'    => apply_filters( 'the_content', get_the_content() ),
			]
		);
	}
}

if ( ! function_exists( 'kalium_portfolio_single_content_services_wrapper_start' ) ) {

	/**
	 * Content and services wrapper start.
	 *
	 * @param string $item_type
	 *
	 * @since 4.0
	 */
	function kalium_portfolio_single_content_services_wrapper_start( $item_type ) {
		$classes = [
			'single-portfolio__content-services',
		];

		// Layout
		if ( $layout = kalium_get_portfolio_single_item_type_option( 'layout' ) ) {
			$classes[] = 'single-portfolio__content-services--' . $layout;
		}

		echo '<div ' . kalium_class_attr( $classes, false ) . '>';
	}
}

if ( ! function_exists( 'kalium_portfolio_single_content_services_wrapper_end' ) ) {

	/**
	 * Content and services wrapper end.
	 *
	 * @since 4.0
	 */
	function kalium_portfolio_single_content_services_wrapper_end() {
		echo '</div>';
	}
}

if ( ! function_exists( 'kalium_portfolio_single_checklist' ) ) {

	/**
	 * Portfolio checklist.
	 *
	 * @since 4.0
	 */
	function kalium_portfolio_single_checklist( $args = [] ) {
		$args = wp_parse_args(
			$args,
			[
				'inline' => 'centered' === kalium_get_portfolio_single_item_type_option( 'layout' ),
			]
		);

		// Checklists
		$checklists = kalium_get_portfolio_single_option( 'checklists' );

		if ( ! empty( $checklists ) ) {
			foreach ( $checklists as $checklist ) {
				$checklist_entries = explode( PHP_EOL, trim( kalium_get_array_key( $checklist, 'checklist' ) ) );

				if ( empty( $checklist_entries ) ) {
					continue;
				}

				$classes = [
					'single-portfolio__checklist',
					kalium_get_link_style_class(),
				];

				if ( $args['inline'] ) {
					$classes[] = 'single-portfolio__checklist--inline';
				} elseif ( '1-2' === kalium_get_array_key( $checklist, 'column_width' ) ) {
					$classes[] = 'single-portfolio__checklist--half';
				}

				?>
				<div <?php kalium_class_attr( $classes ); ?>>
					<?php
					if ( ! empty( $checklist['checklist'] ) ) {
						?>
						<?php if ( $title = kalium_get_array_key( $checklist, 'checklist_title' ) ) : ?>
						<h3><?php echo esc_html( $title ); ?></h3>
						<?php endif; ?>
						<?php
					}

					?>
					<ul>
						<?php
						foreach ( $checklist_entries as $checklist_entry ) {
							?>
							<li><?php echo wp_kses_post( $checklist_entry ); ?></li>
							<?php
						}
						?>
					</ul>
				</div>
				<?php
			}
		}
	}
}

if ( ! function_exists( 'kalium_portfolio_single_link' ) ) {

	/**
	 * Portfolio link.
	 *
	 * @since 4.0
	 */
	function kalium_portfolio_single_link() {
		$link = kalium_get_portfolio_single_option( 'link' );

		if ( empty( $link['title'] ) || empty( $link['url'] ) ) {
			return;
		}

		$link_attrs = [
			'href' => $link['url'],
		];

		// Open in new tab
		if ( $link['new_window'] ) {
			$link_attrs = array_merge(
				$link_attrs,
				[
					'target' => '_blank',
					'rel'    => 'noopener',
				]
			);
		}

		// Link
		kalium_render_element(
			[
				'tag_name'   => 'div',
				'attributes' => [
					'class' => [
						'single-portfolio__link',
						kalium_get_link_style_class(),
					],
				],
				'content'    => kalium_render_element(
					[
						'tag_name'   => 'a',
						'attributes' => $link_attrs,
						'content'    => esc_html( $link['title'] ),
					],
					false
				),
			]
		);
	}
}

if ( ! function_exists( 'kalium_portfolio_single_services_wrapper_start' ) ) {

	/**
	 * Services wrapper start.
	 *
	 * @since 4.0
	 */
	function kalium_portfolio_single_services_wrapper_start() {
		echo '<div class="single-portfolio__services">';
	}
}

if ( ! function_exists( 'kalium_portfolio_single_services_wrapper_end' ) ) {

	/**
	 * Services wrapper end.
	 *
	 * @since 4.0
	 */
	function kalium_portfolio_single_services_wrapper_end() {
		echo '</div>';
	}
}

if ( ! function_exists( 'kalium_portfolio_single_like' ) ) {

	/**
	 * Like button.
	 *
	 * @param array $args
	 *
	 * @since 4.0
	 */
	function kalium_portfolio_single_like( $args = [] ) {
		if ( ! kalium_get_portfolio_loop_option( 'likes/enabled' ) ) {
			return;
		}

		$args = wp_parse_args(
			$args,
			[
				'secondary' => false,
				'icon'      => kalium_get_portfolio_loop_option( 'likes/icon' ),
			]
		);

		$like_args = [
			'display_count' => ! $args['secondary'],
			'icon'          => $args['icon'],
		];

		$classes = [
			'single-portfolio__like',
			kalium_if_else( $args['secondary'], 'single-portfolio__like--type-2', 'single-portfolio__like--type-1' ),
		];

		// Secondary display
		if ( $args['secondary'] ) {
			$classes = array_merge(
				$classes,
				[
					'social-icons',
					'social-icons--with-icon-shape',
					'social-icons--icon-shape-rounded',
					'social-icons--background-hover-brand',
					'social-icons--instance-' . ( ! empty( $GLOBALS['kalium_social_networks_counter'] ) ? $GLOBALS['kalium_social_networks_counter'] : 1 ),
				]
			);

			$like_args = array_merge(
				$like_args,
				[
					'class'      => 'social-icon',
					'icon_class' => 'social-icon__icon',
				]
			);
		}
		?>
		<div <?php kalium_class_attr( $classes ); ?>>
			<?php
			// Like button
			kalium_like_button( $like_args );
			?>
		</div>
		<?php
	}
}

if ( ! function_exists( 'kalium_portfolio_single_share' ) ) {

	/**
	 * Like button.
	 *
	 * @param array $args
	 *
	 * @since 4.0
	 */
	function kalium_portfolio_single_share( $args = [] ) {
		if ( ! kalium_get_portfolio_single_option( 'share/enabled' ) ) {
			return;
		}

		$args = wp_parse_args(
			$args,
			[
				'secondary'  => false,
				'show_title' => true,
				'title'      => __( 'Share', 'kalium' ),
			]
		);

		$share_args = [
			'networks' => kalium_get_portfolio_single_option( 'share/social_networks' ),
		];

		if ( $args['secondary'] ) {
			$share_args['style'] = 'icons';
		}

		?>
		<div class="single-portfolio__share">
			<?php
			// Title
			if ( $args['show_title'] ) {
				printf( '<h4>%s</h4>', esc_html( $args['title'] ) );
			}

			// Social networks
			kalium_social_networks_share( $share_args );
			?>
		</div>
		<?php
	}
}

if ( ! function_exists( 'kalium_portfolio_single_like_and_share' ) ) {

	/**
	 * Single component that displays like and share together.
	 *
	 * @since 4.0
	 */
	function kalium_portfolio_single_like_and_share( $args = [] ) {
		$likes_enabled = kalium_get_portfolio_loop_option( 'likes/enabled' );
		$share_enabled = kalium_get_portfolio_single_option( 'share/enabled' );

		if ( ! $likes_enabled && ! $share_enabled ) {
			return;
		}

		$args = wp_parse_args(
			$args,
			[
				'secondary' => 'rounded' === kalium_get_portfolio_single_option( 'share/layout' ),
				'centered'  => 'centered' === kalium_get_portfolio_single_item_type_option( 'layout' ),
			]
		);

		$classes = [
			'single-portfolio__like-share',
		];

		if ( $args['centered'] ) {
			$classes[] = 'single-portfolio__like-share--centered';
		}

		if ( $args['secondary'] ) {
			$classes[] = 'single-portfolio__like-share--secondary';
		}
		?>
		<div <?php kalium_class_attr( $classes ); ?>>
			<?php
			if ( $likes_enabled ) {
				kalium_portfolio_single_like(
					[
						'secondary' => $args['secondary'],
					]
				);
			}

			if ( $share_enabled ) {
				kalium_portfolio_single_share(
					[
						'show_title' => ! $args['secondary'],
						'secondary'  => $args['secondary'],
					]
				);
			}
			?>
		</div>
		<?php
	}
}

if ( ! function_exists( 'kalium_portfolio_single_display_gallery_type' ) ) {

	/**
	 * Display portfolio gallery based on portfolio item type.
	 *
	 * @param string $item_type
	 *
	 * @since 4.0
	 */
	function kalium_portfolio_single_display_gallery_type( $item_type ) {
		$gallery = kalium_get_portfolio_single_option( 'gallery' );

		if ( empty( $items = $gallery['items'] ) ) {
			return;
		}

		$args = [
			'gap' => kalium_get_array_key( $gallery, 'gap' ),
		];

		switch ( $item_type ) {
			case 'type-1':
				$args = array_merge(
					$args,
					[
						'columned' => true,
						'masonry'  => kalium_get_portfolio_single_item_type_option( 'gallery/masonry' ),
					]
				);

				// Stretch container
				if ( 'fullbg' === kalium_get_portfolio_single_item_type_option( 'gallery/type' ) ) {
					$args = array_merge(
						$args,
						[
							'stretch'     => true,
							'stretch_top' => kalium_get_portfolio_single_item_type_option( 'gallery/stick_to_top' ),
						]
					);
				}
				break;

			case 'type-2':
				$args = array_merge(
					$args,
					[
						'columned' => true,
						'masonry'  => kalium_get_portfolio_single_item_type_option( 'gallery/masonry' ),
					]
				);
				break;

			case 'type-3':
				$args['carousel'] = [
					'keyboard'        => true,
					'auto_height'     => true,
					'slides_per_view' => 'auto',
				];

				if ( kalium_get_portfolio_single_option( 'gallery/slider_centered' ) ) {
					$args['carousel'] = array_merge(
						$args['carousel'],
						[
							'centered'             => true,
							'navigation_container' => true,
						]
					);
				}
				break;

			case 'type-4':
				$args['description'] = true;
				break;

			case 'type-5':
				$args['carousel'] = [
					'effect'     => 'fade',
					'cross_fade' => true,
					'loop'       => true,
					'keyboard'   => true,
					'navigation' => false,
					'pagination' => true,
					'theme'      => 'light',
				];
				break;
		}

		kalium_portfolio_single_gallery( $items, $args );
	}
}

if ( ! function_exists( 'kalium_portfolio_single_navigation' ) ) {

	/**
	 * Previous-next navigation.
	 *
	 * @since 4.0
	 */
	function kalium_portfolio_single_navigation( $args = [] ) {
		if ( ! kalium_get_portfolio_single_option( 'navigation/enabled' ) ) {
			return;
		}

		// Args
		$args = array_merge(
			wp_parse_args(
				$args,
				kalium_get_portfolio_single_option( 'navigation/options' )
			),
			[
				'echo' => false,
			]
		);

		// Classes
		$classes = [
			'single-portfolio__navigation',
		];

		if ( 'type-7' === kalium_get_portfolio_single_option( 'type' ) ) {
			$classes[] = 'container';
		}

		kalium_render_element(
			[
				'tag_name'   => 'div',
				'attributes' => [
					'class' => $classes,
				],
				'content'    => kalium_post_navigation( $args ),
			]
		);
	}
}

if ( ! function_exists( 'kalium_portfolio_single_image' ) ) {

	/**
	 * Display portfolio image for gallery item.
	 *
	 * @param int   $attachment_id
	 * @param array $args
	 *
	 * @return string|void
	 * @since 4.0
	 */
	function kalium_portfolio_single_image( $attachment_id, $args = [] ) {
		$args = wp_parse_args(
			$args,
			[
				'size'             => null,
				'caption'          => kalium_get_portfolio_single_option( 'gallery/captions/enabled' ),
				'caption_position' => kalium_get_portfolio_single_option( 'gallery/captions/position' ),
				'link'             => true,
				'overlay'          => null,
				'new_tab'          => false,
				'return'           => false,
			]
		);

		// Use general defined image size
		if ( empty( $args['size'] ) ) {
			$args['size'] = kalium_get_portfolio_single_option( 'gallery/image_size' );
		}

		if ( ! ( $image = kalium_get_attachment_image( $attachment_id, $args['size'] ) ) ) {
			return;
		}

		$classes = [
			'single-portfolio__image',
			'single-portfolio__image--caption-' . $args['caption_position'],
		];

		// Overlay
		if ( $args['overlay'] ) {
			$image .= kalium_render_element(
				[
					'tag_name'   => 'span',
					'attributes' => [
						'class' => 'single-portfolio__image__overlay',
					],
					'content'    => $args['overlay'],
				],
				false
			);
		}

		// Link
		if ( false !== $args['link'] ) {
			$link = wp_get_attachment_url( $attachment_id );

			if ( is_string( $args['link'] ) && 1 < strlen( $args['link'] ) ) {
				$link        = $args['link'];
				$link_target = $args['new_tab'] ? ' target="_blank" rel="noopener"' : '';
			}

			$image = sprintf(
				'<a href="%s"%s>%s</a>',
				esc_url( $link ),
				$link_target ?? '',
				$image
			);
		}

		// Caption
		if ( $args['caption'] && ( $caption_content = wp_get_attachment_caption( $attachment_id ) ) ) {
			$image .= kalium_render_element(
				[
					'tag_name' => 'figcaption',
					'content'  => nl2br( make_clickable( $caption_content ) ),
				],
				false
			);
		}

		// Image wrapper
		$image = kalium_render_element(
			[
				'tag_name'   => 'figure',
				'attributes' => [
					'class' => $classes,
				],
				'content'    => $image,
			],
			false
		);

		if ( $args['return'] ) {
			return $image;
		}

		echo $image; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
	}
}

if ( ! function_exists( 'kalium_portfolio_single_images_slider' ) ) {

	/**
	 * Display portfolio video for gallery item.
	 *
	 * @since 4.0
	 */
	function kalium_portfolio_single_images_slider( $attachment_ids, $args = [] ) {
		$args = wp_parse_args(
			$args,
			[
				'size'     => null,
				'autoplay' => false,
			]
		);

		// Use general defined image size
		if ( empty( $args['size'] ) ) {
			$args['size'] = kalium_get_portfolio_single_option( 'gallery/image_size' );
		}

		$slides = array_map(
			function ( $attachment_id ) use ( &$args ) {
				return kalium_portfolio_single_image(
					$attachment_id,
					[
						'size'   => $args['size'],
						'return' => true,
					]
				);
			},
			$attachment_ids
		);

		kalium_slider(
			[
				'class'       => 'single-portfolio__images',
				'slides'      => $slides,
				'autoplay'    => $args['autoplay'],
				'in_grid'     => true,
				'loop'        => true,
				'auto_height' => true,
				'theme'       => 'light',
			]
		);
	}
}

if ( ! function_exists( 'kalium_portfolio_single_video' ) ) {

	/**
	 * Video item.
	 *
	 * @param array $args
	 *
	 * @since 4.0
	 */
	function kalium_portfolio_single_video( $video_args, $args = [] ) {
		$args = wp_parse_args(
			$args,
			[
				'return' => false,
			]
		);

		$video_args = wp_parse_args(
			$video_args,
			[
				'source'       => null,
				'autoplay'     => null,
				'loop'         => null,
				'poster'       => null,
				'muted'        => null,
				'controls'     => null,
				'aspect_ratio' => null,
				'width'        => null,
				'height'       => null,
				'object_fit'   => null,
			]
		);

		if ( is_string( $video_args['aspect_ratio'] ) && preg_match( '/(\d+):(\d+)/', $video_args['aspect_ratio'], $matches ) ) {
			$video_args['width']  = $matches[1];
			$video_args['height'] = $matches[2];
		}

		$video = kalium_video_element(
			[
				'src'         => $video_args['source'],
				'autoplay'    => $video_args['autoplay'],
				'muted'       => $video_args['muted'],
				'loop'        => $video_args['loop'],
				'controls'    => $video_args['controls'],
				'poster'      => $video_args['poster'],
				'width'       => $video_args['width'],
				'height'      => $video_args['height'],
				'object_fit'  => $video_args['object_fit'],
				'playsinline' => true,
			]
		);

		if ( $args['return'] ) {
			return $video;
		}

		echo $video; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
	}
}

if ( ! function_exists( 'kalium_portfolio_single_comparison' ) ) {

	/**
	 * Comparison images.
	 *
	 * @param int   $image_1
	 * @param int   $image_2
	 * @param array $args
	 *
	 * @since 4.0
	 */
	function kalium_portfolio_single_comparison( $image_1, $image_2, $args = [] ) {
		$args = wp_parse_args(
			$args,
			[
				'size' => null,
			]
		);

		// Use general defined image size
		if ( empty( $args['size'] ) ) {
			$args['size'] = kalium_get_portfolio_single_option( 'gallery/image_size' );
		}

		$right_image = wp_get_attachment_image( $image_1, $args['size'] );
		$left_image  = wp_get_attachment_image( $image_2, $args['size'] );

		if ( ! $left_image || ! $right_image ) {
			return;
		}

		if ( kalium_get_portfolio_single_option( 'gallery/captions/enabled' ) ) {
			$right_image_title = get_the_title( $image_1 );
			$left_image_title  = get_the_title( $image_2 );
		}

		$attributes = [
			'class' => [
				'comparison-image-slider',
				'image-placeholder',
			],
		];

		// Add image ratio
		$image_attributes = kalium_get_array_key( kalium()->helpers->parse_html_element( $right_image ), 'attributes' );

		if ( ! empty( $image_attributes['width'] ) && ! empty( $image_attributes['height'] ) ) {
			$attributes['style'] = kalium()->helpers->css_props(
				[
					kalium_css_var_name( 'ratio' ) => number_format( $image_attributes['width'] / $image_attributes['height'], 6 ),
				]
			);
		}

		?>
		<figure <?php kalium_render_attributes( $attributes ); ?>>

			<?php
			/**
			 * Right image.
			 */
			echo $right_image; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped

			if ( ! empty( $right_image_title ) ) {
				?>
				<figcaption class="cd-image-label" data-type="original"><?php echo esc_html( $right_image_title ); ?></figcaption>
				<?php
			}
			?>

			<div class="cd-resize-img">
				<?php
				/**
				 * Left image.
				 */
				echo $left_image; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped

				if ( ! empty( $left_image_title ) ) {
					?>
					<figcaption class="cd-image-label" data-type="modified"><?php echo esc_html( $left_image_title ); ?></figcaption>
					<?php
				}
				?>
			</div>

			<span class="cd-handle"></span>
		</figure>
		<?php
	}
}

if ( ! function_exists( 'kalium_portfolio_single_quote' ) ) {

	/**
	 * Quote.
	 *
	 * @param string $quote
	 * @param string $author
	 *
	 * @since 4.0
	 */
	function kalium_portfolio_single_quote( $quote, $author ) {
		printf( '<blockquote class="single-portfolio__quote">%s<cite>%s</cite></blockquote>', wp_kses_post( kalium_format_content( $quote ) ), esc_html( $author ) );
	}
}

if ( ! function_exists( 'kalium_portfolio_single_custom_content' ) ) {

	/**
	 * Custom content.
	 *
	 * @param string $content
	 *
	 * @since 4.0
	 */
	function kalium_portfolio_single_custom_content( $content ) {
		$classes = [
			'single-portfolio__custom-content',
			'entry-content',
			kalium_get_link_style_class(),
		];

		printf( '<div %s>%s</div>', kalium_class_attr( $classes, false ), $content );
	}
}

if ( ! function_exists( 'kalium_portfolio_single_content_toggle' ) ) {

	/**
	 * Content toggle button used in full-screen portfolio item type.
	 *
	 * @since 4.0
	 */
	function kalium_portfolio_single_content_toggle() {
		$expand_text = __( 'Click here to show project info', 'kalium' );

		if ( $custom_expand_text = kalium_get_portfolio_single_item_type_option( 'content/show_project_info_text' ) ) {
			$expand_text = $custom_expand_text;
		}

		$classes = [
			'single-portfolio__content-toggle',
			'button-link',
		];

		?>
		<button <?php kalium_class_attr( $classes ); ?> type="button" data-content-action="expand">
			<span class="title h3"><?php the_title(); ?></span>
			<span class="text"><?php echo esc_html( $expand_text ); ?></span>
		</button>

		<button class="button button-link single-portfolio__content-toggle__icon single-portfolio__content-toggle__icon--expand" type="button" data-content-action="expand">
			<i class="kalium-icon-arrow-corner-right"></i>
		</button>

		<button class="button button-link single-portfolio__content-toggle__icon single-portfolio__content-toggle__icon--collapse" type="button" data-content-action="collapse">
			<i class="kalium-icon-arrow-corner-left"></i>
		</button>
		<?php
	}
}

if ( ! function_exists( 'kalium_portfolio_single_gallery' ) ) {

	/**
	 * Portfolio gallery.
	 *
	 * @param array $args
	 *
	 * @since 4.0
	 */
	function kalium_portfolio_single_gallery( $items, $args = [] ) {
		if ( empty( $items ) || ! is_array( $items ) ) {
			return;
		}

		$args = wp_parse_args(
			$args,
			[
				'columned'    => false,
				'carousel'    => false,
				'description' => false,
				'masonry'     => false,
				'stretch'     => false,
				'stretch_top' => true,
				'gap'         => null,
			]
		);

		$container_class    = 'single-portfolio__gallery';
		$container_selector = sprintf( '%s .%s', kalium_get_portfolio_single_option( 'selector' ), $container_class );

		// CSS vars
		$css_vars = [];

		// Container classes
		$classes = [
			$container_class,
		];

		// Lightbox
		if ( kalium_get_portfolio_single_option( 'gallery/lightbox' ) ) {
			$classes[] = kalium_get_lightbox_class();
		}

		// Columned
		if ( $args['columned'] ) {
			$classes[] = $container_class . '--columned';
			$classes[] = 'row';
		}

		// Carousel
		if ( $args['carousel'] ) {
			$slider_args = array_replace_recursive(
				[
					'class'      => $classes,
					'in_grid'    => true,
					'autoplay'   => kalium_get_portfolio_single_option( 'gallery/slider_autoplay' ),
					'loop'       => kalium_get_portfolio_single_option( 'gallery/slider_loop' ),
					'attributes' => kalium_get_portfolio_single_option( 'gallery/reveal_effect' ),
					'slides'     => function () use ( $items, $args ) {

						// Unset gallery reveal effect
						kalium_set_portfolio_single_option( 'gallery/reveal_effect', null );

						// TEMP FIX: Insufficient number of slides causes the Swiper to not work properly in Loop Mode
						if ( kalium_get_portfolio_single_option( 'gallery/slider_loop' ) && 3 >= count( $items ) ) {
							$items = array_merge( $items, $items );
						}

						// Gallery items
						kalium_portfolio_single_gallery_items( $items, $args );
					},
				],
				is_array( $args['carousel'] ) ? $args['carousel'] : []
			);

			if ( $max_height = kalium_get_portfolio_single_option( 'gallery/max_height' ) ) {
				$css_vars['sp-gallery-item-max-height'] = $max_height;

				$slider_args['class'][] = $container_class . '--constrained-height';
			}
		}

		// Gallery gap
		if ( ! is_null( $args['gap'] ) ) {
			$css_vars['sp-gallery-gap'] = is_numeric( $args['gap'] ) ? ( $args['gap'] . 'px' ) : $args['gap'];
		}

		// Stretched gallery
		if ( $args['stretch'] ) {
			kalium_js_stretch_container(
				kalium_get_portfolio_single_option( 'selector' ) . ' .single-portfolio__gallery-container',
				[
					'top' => $args['stretch_top'],
				]
			);
		}

		// Print styles
		kalium_print_inline_style(
			[
				'selector' => $container_selector,
				'vars'     => $css_vars,
			]
		);

		// Masonry container
		if ( $args['masonry'] ) {
			kalium_create_masonry_container(
				[
					'options' => [
						'item' => '.single-portfolio__gallery__item',
					],
				]
			);
		}

		// Carousel
		if ( ! empty( $slider_args ) ) {
			kalium_slider( $slider_args );
		} else {

			/**
			 * Portfolio single gallery before.
			 */
			do_action( 'kalium_portfolio_single_gallery_before' );

			?>
			<div <?php kalium_class_attr( $classes ); ?>>
				<?php

				// List gallery items
				kalium_portfolio_single_gallery_items( $items, $args );
				?>
			</div>
			<?php

			/**
			 * Portfolio single gallery after.
			 */
			do_action( 'kalium_portfolio_single_gallery_after' );
		}
	}
}

if ( ! function_exists( 'kalium_portfolio_single_gallery_items' ) ) {

	/**
	 * Parses gallery items.
	 *
	 * @param array $items
	 * @param array $args
	 *
	 * @since 4.0
	 */
	function kalium_portfolio_single_gallery_items( $items, $args = [] ) {
		foreach ( $items as $item ) {
			$item_attributes = [
				'class' => [
					'single-portfolio__gallery__item',
				],
			];

			// Columned
			if ( $args['columned'] ) {
				$item_attributes['class'][] = 'col';

				if ( ! empty( $item['column_width'] ) ) {
					switch ( $item['column_width'] ) {
						case '2-3':
							$item_attributes['class'][] = 'col-12 col-md-8';
							break;

						case '1-2':
							$item_attributes['class'][] = 'col-12 col-md-6';
							break;

						case '1-3':
							$item_attributes['class'][] = 'col-12 col-md-4';
							break;

						case '1-4':
							$item_attributes['class'][] = 'col-12 col-md-3';
							break;

						default:
							$item_attributes['class'][] = 'col-12';
					}
				}
			}

			// Carousel
			if ( $args['carousel'] ) {
				$item_attributes['class'][] = 'swiper-slide';
			}

			// Description
			if ( $args['description'] ) {
				$item_attributes['class'][] = 'single-portfolio__gallery__item--description';
				$item_attributes['class'][] = 'single-portfolio__gallery__item--description-width-' . kalium_get_array_key( $item, 'description_width' );
				$item_attributes['class'][] = 'single-portfolio__gallery__item--description-align-' . kalium_get_array_key( $item, 'description_align' );
			}

			// Reveal effect
			if ( $reveal_effect = kalium_get_portfolio_single_option( 'gallery/reveal_effect' ) ) {
				$item_attributes = array_merge( $item_attributes, $reveal_effect );
			}

			// Image size
			$image_size = kalium()->helpers->maybe_split_dimensions( kalium_get_array_key( $item, 'image_size' ) );

			?>
			<div <?php kalium_render_attributes( $item_attributes ); ?>>
				<?php
				switch ( $item['type'] ) {
					case 'image':
						kalium_portfolio_single_image(
							kalium_get_array_key( $item, 'image_id' ),
							[
								'link'    => kalium_get_array_key( $item, 'link' ),
								'new_tab' => kalium_get_array_key( $item, 'new_tab' ),
								'size'    => $image_size,
							]
						);
						break;

					case 'images':
						kalium_portfolio_single_images_slider(
							kalium_get_array_key( $item, 'images' ),
							[
								'autoplay' => kalium_get_array_key( $item, 'auto_switch' ),
								'size'     => $image_size,
							]
						);
						break;

					case 'comparison':
						kalium_portfolio_single_comparison(
							kalium_get_array_key( $item, 'left_image_id' ),
							kalium_get_array_key( $item, 'right_image_id' ),
							[
								'size' => $image_size,
							]
						);
						break;

					case 'video':
						kalium_portfolio_single_video( $item );
						break;

					case 'quote':
						kalium_portfolio_single_quote( kalium_get_array_key( $item, 'quote' ), kalium_get_array_key( $item, 'quote_author' ) );
						break;

					case 'html':
						kalium_portfolio_single_custom_content( kalium_get_array_key( $item, 'content' ) );
						break;
				}

				// Description
				if ( $args['description'] ) {
					kalium_render_element(
						[
							'tag_name'   => 'div',
							'attributes' => [
								'class' => [
									'single-portfolio__gallery__item__description',
								],
							],
							'content'    => kalium_render_element(
								[
									'tag_name'   => 'div',
									'attributes' => [
										'class' => [
											'single-portfolio__gallery__item__description__content',
											'entry-content',
											kalium_get_link_style_class(),
										],
									],
									'content'    => kalium_format_content( kalium_get_array_key( $item, 'description' ) ),
								],
								false
							),
						]
					);
				}
				?>
			</div>
			<?php
		}
	}
}

if ( ! function_exists( 'kalium_portfolio_lightbox_item_description_display' ) ) {

	/**
	 * Portfolio lightbox item description.
	 *
	 * @param WP_Post $post
	 *
	 * @since 4.2
	 */
	function kalium_portfolio_lightbox_item_description_display( $post ) {
		echo kalium_format_content( $post->post_content );
	}
}
