<?php
/**
 * Kalium WordPress Theme
 *
 * Theme style functions.
 *
 * @author Laborator
 * @link   https://kaliumtheme.com
 */
if ( ! defined( 'ABSPATH' ) ) {
	exit; // Direct access not allowed.
}

/**
 * Color reference regex.
 *
 * @var string
 */
const KALIUM_COLOR_REFERENCE_REGEX = '/kalium-color\((?<path>[^,]+)(\,\s?(?<opacity>[\d.]+))?\)/';

/**
 * Color variant regex.
 *
 * @var string
 */
const KALIUM_COLOR_VARIANT_REGEX = '/kalium-color-variant\(\s*(?<color>(#[a-fA-Z0-9]{3,}|kalium-color\(.*?\))),\s*\[(?<process_functions>.*?)\]\s*\)/';

/**
 * Prefixed CSS var name.
 *
 * @param ...string $args
 *
 * @return string
 */
function kalium_css_var_name() {
	return call_user_func_array( [ kalium()->style, 'var_name' ], func_get_args() );
}

/**
 * Returns prefixed CSS var reference.
 *
 * @param ...string $args
 *
 * @return string
 */
function kalium_css_var_reference() {
	return call_user_func_array( [ kalium()->style, 'var_reference' ], func_get_args() );
}

/**
 * Define custom CSS var that will be printed in <head>.
 *
 * @param string $name
 * @param string $value
 */
function kalium_set_css_var( $name, $value ) {
	kalium()->style->set_css_var( $name, $value );
}

/**
 * Check for self referencing variable.
 *
 * @param string $prop_name
 * @param string $prop_value
 *
 * @return bool
 * @since 4.0
 */
function kalium_is_self_referencing_css_var( $prop_name, $prop_value ) {
	return '--' === substr( $prop_name, 0, 2 ) && kalium()->helpers->str_starts_with( $prop_value, "var($prop_name" );
}

/**
 * Get theme color reference string with optional opacity.
 *
 * @param string $path
 * @param float  $opacity
 *
 * @return string
 */
function kalium_color_reference( $path, $opacity = null ) {
	$reference_func = 'kalium-color(' . esc_attr( $path );

	// Opacity
	if ( is_numeric( $opacity ) && $opacity < 1 && $opacity >= 0 ) {
		$reference_func .= ', ' . esc_attr( $opacity );
	}

	$reference_func .= ')';

	return $reference_func;
}

/**
 * Color variant string.
 *
 * @param string $color
 * @param array  $process_functions
 *
 * @return string
 */
function kalium_color_variant( $color, $process_functions ) {
	return sprintf( 'kalium-color-variant(%s, [%s])', $color, implode( ', ', $process_functions ) );
}

/**
 * General active color variant.
 *
 * @param string $path
 *
 * @return string
 */
function kalium_color_variant_active( $path ) {
	return kalium_color_variant( kalium_color_reference( $path ), [ 'shift(30)' ] );
}

/**
 * Check if given string is color reference.
 *
 * @param string $str
 *
 * @return bool
 */
function kalium_is_color_reference( $str ) {
	if ( is_scalar( $str ) ) {
		return (bool) preg_match( KALIUM_COLOR_REFERENCE_REGEX, $str );
	}

	return false;
}

/**
 * Checks if given string is color variant.
 *
 * @param string $str
 *
 * @return bool
 */
function kalium_is_color_variant( $str ) {
	if ( is_scalar( $str ) ) {
		return (bool) preg_match( KALIUM_COLOR_VARIANT_REGEX, $str );
	}

	return false;
}

/**
 * Color reference unprefixed var name.
 *
 * @parma string $str
 *
 * @return string
 */
function kalium_color_reference_var_name( $str ) {
	static $color_vars, $cached = [];

	// Return cached
	if ( isset( $cached[ $str ] ) ) {
		return $cached[ $str ];
	}

	if ( is_null( $color_vars ) ) {
		$color_vars = kalium_get_theme_color_vars();
	}

	if ( preg_match( KALIUM_COLOR_REFERENCE_REGEX, $str, $matches ) ) {
		$path = explode( '.', $matches['path'] );
		$id   = array_shift( $path );

		foreach ( $color_vars as $color_var ) {
			// Only theme mods
			if ( ! empty( $color_var['id'] ) && $id === $color_var['id'] ) {
				$css_var = $color_var['css_var'];

				while ( $sub_key = array_shift( $path ) ) {
					$css_var = $css_var[ $sub_key ] ?? null;
				}

				// Cache and return matched option
				if ( is_string( $css_var ) ) {
					$cached[ $str ] = $css_var;

					return $cached[ $str ];
				}
			}
		}
	}

	return null;
}

/**
 * Get color reference value.
 *
 * @param string $str
 *
 * @return string
 */
function kalium_color_reference_value( $str ) {
	static $color_vars, $cached = [];

	// Return cached
	if ( isset( $cached[ $str ] ) ) {
		return $cached[ $str ];
	}

	if ( is_null( $color_vars ) ) {
		$color_vars = kalium_get_theme_color_vars();
	}

	// Check if given string is color reference
	if ( preg_match( KALIUM_COLOR_REFERENCE_REGEX, $str, $matches ) ) {
		$value = kalium_get_theme_option_by_path( $matches['path'] );

		return kalium_color_reference_value( $value );
	}
	// Color value
	elseif ( $color = kalium()->style->color( $str ) ) {
		$hex = (string) $color->toHex();

		$cached[ $str ] = $hex;

		return $cached[ $str ];
	}

	return null;
}

/**
 * Parse color.
 *
 * Returns either CSS var or hex color.
 *
 * @param string $str
 *
 * @return string
 * @since 4.0
 */
function kalium_parse_color( $str ) {
	if ( ! $str ) {
		return null;
	}

	// Parse color variant (1)
	$color_variant_regex = trim( KALIUM_COLOR_VARIANT_REGEX, '/' );

	if ( preg_match( "/^{$color_variant_regex}$/", $str, $matches ) ) {
		$color = kalium_color_reference_value( $matches['color'] );

		// Parse process functions
		$process_functions = [];

		if ( preg_match_all( '/(?<functions>[a-z]+)\((?<args>.*?)\)/', $matches['process_functions'], $func_matches ) ) {
			foreach ( $func_matches['functions'] as $i => $function_name ) {
				$function_args = ! empty( $func_matches['args'][ $i ] ) ? explode( ',', $func_matches['args'][ $i ] ) : [];

				$process_functions[] = [
					'name' => $function_name,
					'args' => array_map( 'trim', $function_args ),
				];
			}
		}

		// Process color
		foreach ( $process_functions as $function ) {
			switch ( $function['name'] ) {
				// Color shift
				case 'shift':
					$color = kalium()->style->color_shift( $color, $function['args'][0] ?? 0 );
					break;

				// Alpha
				case 'alpha':
					$color_instance = kalium()->style->color( $color );

					if ( $color_instance ) {
						$color = (string) $color_instance->toRgb()->toRgba( floatval( $function['args'][0] ?? 1 ) );
					}
					break;
			}
		}

		return $color;
	}

	// Parse color reference (2)
	$color_reference_regex = trim( KALIUM_COLOR_REFERENCE_REGEX, '/' );

	if ( preg_match( "/^{$color_reference_regex}$/", $str, $matches ) ) {
		$var_name = kalium_color_reference_var_name( $str );
		$opacity  = $matches['opacity'] ?? 1;

		if ( is_numeric( $opacity ) && $opacity < 1 ) {
			return 'color-mix(in srgb, ' . kalium_css_var_reference( $var_name ) . ' calc(' . $opacity . ' * 100%), transparent)';
		}

		return kalium_css_var_reference( $var_name );
	}

	return null;
}

/**
 * Replace all color references in a string to CSS vars.
 *
 * @param string|array $str
 *
 * @return string|array
 */
function kalium_replace_color_references( $str ) {
	if ( is_array( $str ) ) {
		return array_map( 'kalium_replace_color_references', $str );
	}

	if ( is_string( $str ) ) {
		foreach ( [ KALIUM_COLOR_VARIANT_REGEX, KALIUM_COLOR_REFERENCE_REGEX ] as $regex ) {
			$str = preg_replace_callback(
				$regex,
				function ( $matches ) {
					return kalium_parse_color( $matches[0] );
				},
				$str
			);
		}
	}

	return $str;
}

/**
 * Theme base colors control args.
 *
 * @return array
 * @since 4.0
 */
function kalium_get_theme_base_colors_args() {
	return [
		'choices'  => [
			'color-1' => [
				'label'   => 'Color 1',
				'css_var' => true,
			],
			'color-2' => [
				'label'   => 'Color 2',
				'css_var' => true,
			],
			'color-3' => [
				'label'   => 'Color 3',
				'css_var' => true,
			],
			'color-4' => [
				'label'   => 'Color 4',
				'css_var' => true,
			],
			'color-5' => [
				'label'   => 'Color 5',
				'css_var' => true,
			],
			'color-6' => [
				'label'   => 'Color 6',
				'css_var' => true,
			],
			'color-7' => [
				'label'   => 'Color 7',
				'css_var' => true,
			],
			'color-8' => [
				'label'   => 'Color 8',
				'css_var' => true,
			],
		],
		'manage'   => [
			'color_id'      => 'color-%d',
			'color_label'   => 'Color %d',
			'color_css_var' => true,
			'palette_label' => 'Palette %d',
		],
		'palettes' => kalium_get_theme_palettes(),
		'default'  => kalium_get_default_theme_palette(),
	];
}

/**
 * Theme base colors.
 *
 * @return array
 * @since 4.0
 */
function kalium_get_theme_base_colors() {
	$theme_colors_var   = 'theme_colors';
	$theme_colors_args  = kalium_get_theme_base_colors_args();
	$theme_colors_value = kalium_get_theme_option( $theme_colors_var );
	$colors_choices     = $theme_colors_args['choices'];
	$colors_manage      = $theme_colors_args['manage'];

	$base_colors = [];

	// Base colors
	foreach ( $colors_choices as $color_id => $choice ) {
		$base_colors[] = [
			'id'        => $color_id,
			'label'     => $choice['label'],
			'reference' => kalium_color_reference( "{$theme_colors_var}.{$color_id}" ),
			'value'     => $theme_colors_value[ $color_id ] ?? null,
		];
	}

	// Custom colors
	if ( is_array( $theme_colors_value ) ) {
		$match_pattern = '/^' . str_replace( '%d', '([0-9]+)', $colors_manage['color_id'] ?? 'color-%d' ) . '$/';
		$custom_label  = $colors_manage['color_label'] ?? 'Color %d';

		foreach ( $theme_colors_value as $color_id => $color_value ) {
			if ( isset( $colors_choices[ $color_id ] ) ) {
				continue;
			}

			if ( preg_match( $match_pattern, $color_id, $matches ) ) {
				$id_num = $matches[1];

				$base_colors[] = [
					'id'        => $color_id,
					'label'     => str_replace( '%d', $id_num, $custom_label ),
					'reference' => kalium_color_reference( "{$theme_colors_var}.{$color_id}" ),
					'value'     => $color_value,
				];
			}
		}
	}

	return $base_colors;
}

/**
 * Get theme palettes.
 *
 * @return array
 * @since 4.0
 */
function kalium_get_theme_palettes() {
	return apply_filters(
		'kalium_theme_palettes',
		[
			[
				'title'  => 'Default',
				'colors' => [
					'color-1' => '#1A73E8',
					'color-2' => '#1964C6',
					'color-3' => '#333333',
					'color-4' => '#222222',
					'color-5' => '#F3F5F7',
					'color-6' => '#F3F5F6',
					'color-7' => '#FFFFFF',
					'color-8' => '#EEEEEE',
				],
			],
			[
				'title'  => 'Construction',
				'colors' => [
					'color-1' => '#F7C016',
					'color-2' => '#D3A107',
					'color-3' => '#444444',
					'color-4' => '#333333',
					'color-5' => '#F2F2F2',
					'color-6' => '#555555',
					'color-7' => '#FFFFFF',
					'color-8' => '#F9F9F9',
				],
			],
			[
				'title'  => 'Wedding',
				'colors' => [
					'color-1' => '#E0AFA0',
					'color-2' => '#B1867A',
					'color-3' => '#635B57',
					'color-4' => '#463F3A',
					'color-5' => '#F4F3EE',
					'color-6' => '#FFFFFF',
					'color-7' => '#FFFFFF',
					'color-8' => '#F9F2EB',
				],
			],
			[
				'title'  => 'Landing',
				'colors' => [
					'color-1' => '#1669D5',
					'color-2' => '#1964C6',
					'color-3' => '#333333',
					'color-4' => '#222222',
					'color-5' => '#F3F5F7',
					'color-6' => '#FFF9F0',
					'color-7' => '#FFFFFF',
					'color-8' => '#F5F5F7',
				],
			],
			[
				'title'  => 'Restaurant',
				'colors' => [
					'color-1' => '#CD9B33',
					'color-2' => '#9B741F',
					'color-3' => '#6B6B6B',
					'color-4' => '#242321',
					'color-5' => '#EEEEEE',
					'color-6' => '#ECECEC',
					'color-7' => '#FFFFFF',
					'color-8' => '#FCF5EE',
				],
			],
			[
				'title'  => 'Photography',
				'colors' => [
					'color-1' => '#968255',
					'color-2' => '#907C51',
					'color-3' => '#333333',
					'color-4' => '#222222',
					'color-5' => '#F3F5F7',
					'color-6' => '#F3F5F6',
					'color-7' => '#FFFFFF',
					'color-8' => '#EEEEEE',
				],
			],
			[
				'title'  => 'Bookstore',
				'colors' => [
					'color-1' => '#76323F',
					'color-2' => '#58252F',
					'color-3' => '#353432',
					'color-4' => '#2B2A29',
					'color-5' => '#E1DDD9',
					'color-6' => '#F3F0EF',
					'color-7' => '#FFFFFF',
					'color-8' => '#F3F0EF',
				],
			],
			[
				'title'  => 'Personal Blog',
				'colors' => [
					'color-1' => '#D1480F',
					'color-2' => '#C64610',
					'color-3' => '#333333',
					'color-4' => '#222222',
					'color-5' => '#E0DED7',
					'color-6' => '#FCFAF4',
					'color-7' => '#F2F0EB',
					'color-8' => '#FFFFFF',
				],
			],
			[
				'title'  => 'Automotive',
				'colors' => [
					'color-1' => '#D90707',
					'color-2' => '#B60303',
					'color-3' => '#444444',
					'color-4' => '#111111',
					'color-5' => '#DDDDDD',
					'color-6' => '#1F1F1F',
					'color-7' => '#FFFFFF',
					'color-8' => '#EEEEEE',
				],
			],
			[
				'title'  => 'Architecture',
				'colors' => [
					'color-1' => '#C15627',
					'color-2' => '#91401D',
					'color-3' => '#222222',
					'color-4' => '#000000',
					'color-5' => '#F3F5F7',
					'color-6' => '#F3F5F6',
					'color-7' => '#FFFFFF',
					'color-8' => '#E6E7E8',
				],
			],
			[
				'title'  => 'Fitness',
				'colors' => [
					'color-1' => '#D71E25',
					'color-2' => '#BA0F15',
					'color-3' => '#3E5777',
					'color-4' => '#1B3453',
					'color-5' => '#DFE4E6',
					'color-6' => '#9EA3AF',
					'color-7' => '#FFFFFF',
					'color-8' => '#F3F5F6',
				],
			],
		]
	);
}

/**
 * Get default theme palette colors.
 *
 * @return array
 */
function kalium_get_default_theme_palette() {
	$default_palette = current( kalium_get_theme_palettes() );

	return apply_filters( 'kalium_theme_palette_default', $default_palette['colors'] );
}

/**
 * Theme color vars with default values.
 *
 * @return array
 */
function kalium_get_theme_color_vars() {
	/**
	 * Hook: kalium_theme_color_vars.
	 */
	return apply_filters(
		'kalium_theme_color_vars',
		[

			// Theme base colors
			'theme'                          => [
				'id'      => 'theme_colors',
				'css_var' => wp_list_pluck( kalium_get_theme_base_colors(), 'id', 'id' ),
			],

			// Color 8 alt
			'color-8-alt'                    => [
				'css_var'  => 'color-8-alt',
				'generate' => [
					'source' => 'theme_colors.color-8',
					'shift'  => 10,
				],
			],

			// Links
			'link'                           => [
				'id'      => 'theme_colors_link',
				'css_var' => [
					'normal' => 'link-color',
					'hover'  => 'link-hover-color',
				],
			],

			// Text
			'text'                           => [
				'id'      => 'theme_colors_text',
				'css_var' => [
					'normal' => 'body-color',
					'muted'  => 'text-muted-color',
				],
			],

			// Text selection
			'text-selection'                 => [
				'id'      => 'theme_colors_text_selection',
				'css_var' => [
					'text'       => 'text-selection-color',
					'background' => 'text-selection-bg',
				],
			],

			// Headings
			'headings'                       => [
				'id'      => 'theme_colors_headings',
				'css_var' => 'heading-color',
			],

			// Heading 1
			'h1'                             => [
				'id'      => 'theme_colors_headings_h1',
				'css_var' => 'h1-color',
			],

			// Heading 2
			'h2'                             => [
				'id'      => 'theme_colors_headings_h2',
				'css_var' => 'h2-color',
			],

			// Heading 3
			'h3'                             => [
				'id'      => 'theme_colors_headings_h3',
				'css_var' => 'h3-color',
			],

			// Heading 4
			'h4'                             => [
				'id'      => 'theme_colors_headings_h4',
				'css_var' => 'h4-color',
			],

			// Heading 5
			'h5'                             => [
				'id'      => 'theme_colors_headings_h5',
				'css_var' => 'h5-color',
			],

			// Heading 6
			'h6'                             => [
				'id'      => 'theme_colors_headings_h6',
				'css_var' => 'h6-color',
			],

			// Borders
			'border'                         => [
				'id'      => 'theme_colors_border',
				'css_var' => 'border-color',
			],

			// Footer
			'footer'                         => [
				'id'      => 'theme_colors_footer',
				'css_var' => [
					'background' => 'footer-bg',
					'text'       => 'footer-color',
					'headings'   => 'footer-headings',
					'link'       => 'footer-link-color',
					'link-hover' => 'footer-link-hover-color',
				],
			],

			// Body
			'body'                           => [
				'id'      => 'theme_colors_body',
				'css_var' => 'body-bg',
			],

			// Overlay
			'overlay'                        => [
				'id'      => 'theme_colors_overlay',
				'css_var' => 'overlay-bg',
			],

			/**
			 * Buttons.
			 */

			// Button text
			'button'                         => [
				'id'      => 'button_color',
				'css_var' => [
					'normal' => 'button-color',
					'hover'  => 'button-hover-color',
					'active' => 'button-active-color',
				],
			],

			// Button background color
			'button-background'              => [
				'id'      => 'button_bg',
				'css_var' => [
					'normal' => 'button-bg',
					'hover'  => 'button-hover-bg',
					'active' => 'button-active-bg',
				],
			],

			/**
			 * Header.
			 */

			// Background
			'header-background'              => [
				'id'      => 'header_background',
				'css_var' => 'header-bg',
			],

			// Links
			'header-links'                   => [
				'id'      => 'header_style_links',
				'css_var' => [
					'normal' => 'header-link-color',
					'hover'  => 'header-link-hover-color',
					'active' => 'header-link-active-color',
				],
			],

			// Text
			'header-text'                    => [
				'id'      => 'header_style_text',
				'css_var' => 'header-color',
			],

			// Menu Trigger
			'header-menu-trigger'            => [
				'id'      => 'header_style_menu_trigger',
				'css_var' => [
					'normal' => 'header-menu-trigger-color',
					'hover'  => 'header-menu-trigger-hover-color',
					'active' => 'header-menu-trigger-active-color',
				],
			],

			// Submenu Container
			'header-submenu-bg'              => [
				'id'      => 'header_style_submenu_bg',
				'css_var' => 'header-submenu-bg',
			],

			// Submenu Links
			'header-submenu-links'           => [
				'id'      => 'header_style_submenu_links',
				'css_var' => [
					'normal' => 'header-submenu-link-color',
					'hover'  => 'header-submenu-link-hover-color',
					'active' => 'header-submenu-link-active-color',
				],
			],

			// Fullscreen menu background
			'header-fullscreen-bg'           => [
				'id'      => 'fullscreen_menu_bg_color',
				'css_var' => 'header-fullscreen-bg',
			],

			// Fullscreen menu links
			'header-fullscreen-links'        => [
				'id'      => 'header_style_fullscreen_links',
				'css_var' => [
					'normal' => 'header-fullscreen-link-color',
					'hover'  => 'header-fullscreen-link-hover-color',
					'active' => 'header-fullscreen-link-active-color',
				],
			],

			// Fullscreen menu text
			'header-fullscreen-text'         => [
				'id'      => 'header_style_fullscreen_text',
				'css_var' => 'header-fullscreen-text',
			],

			// Off-Canvas Side Overlay
			'header-ocs-overlay-bg'          => [
				'id'          => 'header_style_off_canvas_side_overlay',
				'css_var'     => 'header-ocs-overlay-bg',
				'transparent' => true,
			],

			// Off-Canvas Side Background
			'header-ocs-bg'                  => [
				'id'      => 'header_style_off_canvas_side_bg',
				'css_var' => 'header-ocs-bg',
			],

			// Off-Canvas Side Links
			'header-ocs-links'               => [
				'id'      => 'header_style_off_canvas_side_links',
				'css_var' => [
					'normal' => 'header-ocs-link',
					'hover'  => 'header-ocs-link-hover',
					'active' => 'header-ocs-link-active',
				],
			],

			// Off-Canvas Side Text
			'header-ocs-color'               => [
				'id'      => 'header_style_off_canvas_side_text',
				'css_var' => 'header-ocs-color',
			],

			// Off-Canvas Top Background
			'header-oct-bg'                  => [
				'id'      => 'header_style_off_canvas_top_bg',
				'css_var' => 'header-oct-bg',
			],

			// Off-Canvas Side Links
			'header-oct-links'               => [
				'id'      => 'header_style_off_canvas_top_links',
				'css_var' => [
					'normal' => 'header-oct-link-color',
					'hover'  => 'header-oct-link-hover-color',
					'active' => 'header-oct-link-active-color',
				],
			],

			// Off-Canvas Top Text
			'header-oct-text'                => [
				'id'      => 'header_style_off_canvas_top_text',
				'css_var' => 'header-oct-color',
			],

			// Top Header Bar
			'top-header-bar-container'       => [
				'id'      => 'top_header_bar_container_colors',
				'css_var' => [
					'background' => 'header-top-bar-bg',
					'border'     => 'header-top-bar-border',
					'separator'  => 'header-top-bar-separator',
				],
			],

			// Top Header Bar Links
			'top-header-bar-links'           => [
				'id'      => 'top_header_bar_link_colors',
				'css_var' => [
					'normal' => 'header-top-bar-link',
					'hover'  => 'header-top-bar-link-hover',
					'active' => 'header-top-bar-link-active',
				],
			],

			// Top Header Bar Text
			'top-header-bar-text'            => [
				'id'      => 'top_header_bar_text_color',
				'css_var' => 'header-top-bar-text',
			],

			/**
			 * Mobile menu.
			 */

			// Content Background
			'mobile-menu-bg'                 => [
				'id'      => 'mobile_menu_bg',
				'css_var' => 'mobile-menu-bg',
			],

			// Overlay background
			'mobile-menu-overlay-bg'         => [
				'id'          => 'mobile_menu_overlay_bg',
				'css_var'     => 'mobile-menu-overlay-bg',
				'transparent' => true,
			],

			// Mobile menu link colors
			'mobile-menu-link-colors'        => [
				'id'      => 'mobile_menu_link_colors',
				'css_var' => [
					'normal' => 'mobile-menu-link-color',
					'hover'  => 'mobile-menu-link-hover-color',
					'active' => 'mobile-menu-link-active-color',
				],
			],

			// Text
			'mobile-menu-color'              => [
				'id'      => 'mobile_menu_text',
				'css_var' => 'mobile-menu-color',
			],

			// Close Button
			'mobile-menu-close-button-color' => [
				'id'      => 'mobile_menu_close_button_color',
				'css_var' => [
					'normal' => 'mobile-menu-close-button-color',
					'hover'  => 'mobile-menu-close-button-hover-color',
				],
			],
		]
	);
}

/**
 * Global colors are meant to be used in Gutenberg editor colors and other builders which theme supports.
 *
 * @param array $args
 *
 * @return array
 * @since 4.0
 */
function kalium_get_theme_global_colors( $args = [] ) {
	static $color_vars;

	$args = wp_parse_args(
		$args,
		[
			'prefix'    => false,
			'separator' => '-',
		]
	);

	if ( is_null( $color_vars ) ) {
		$color_vars = kalium_get_theme_color_vars();
	}

	$global_colors = [];

	// Base colors
	foreach ( kalium_get_theme_base_colors() as $color_entry ) {
		$global_colors[ 'theme_colors.' . $color_entry['id'] ] = [
			'title' => 'Theme ' . $color_entry['label'],
		];
	}

	// Optional colors
	$global_colors = array_merge(
		$global_colors,
		[
			'theme_colors_headings'    => [
				'title'    => 'Theme Headings (H1 - H6)',
				'optional' => true,
			],
			'theme_colors_headings_h1' => [
				'title'    => 'Theme Heading 1 (H1)',
				'optional' => true,
			],
			'theme_colors_headings_h2' => [
				'title'    => 'Theme Heading 2 (H2)',
				'optional' => true,
			],
			'theme_colors_headings_h3' => [
				'title'    => 'Theme Heading 3 (H3)',
				'optional' => true,
			],
			'theme_colors_headings_h4' => [
				'title'    => 'Theme Heading 4 (H4)',
				'optional' => true,
			],
			'theme_colors_headings_h5' => [
				'title'    => 'Theme Heading 5 (H5)',
				'optional' => true,
			],
			'theme_colors_headings_h6' => [
				'title'    => 'Theme Heading 6 (H6)',
				'optional' => true,
			],
		]
	);

	/**
	 * Get color entry.
	 *
	 * @param string $color_path
	 * @param string $title
	 *
	 * @return array|null
	 */
	$get_color_entry = function ( $color_path, $title, $optional = false ) use ( &$color_vars, &$args ) {
		$color_path = explode( '.', $color_path );
		$theme_mod  = $color_path[0];
		$key        = $color_path[1] ?? null;

		$value   = kalium_get_theme_option( $theme_mod );
		$default = kalium_get_default_theme_option( $theme_mod );

		foreach ( $color_vars as $color_var ) {
			if ( ! empty( $color_var['id'] ) && $theme_mod === $color_var['id'] ) {
				$slug = $color_var['css_var'];

				if ( ! empty( $key ) ) {
					$slug = $slug[ $key ] ?? null;
				}
			}
		}

		// Empty color ID or value
		if ( empty( $value ) || empty( $slug ) ) {
			return null;
		}

		// Access key
		if ( ! empty( $key ) ) {
			$value   = $value[ $key ] ?? null;
			$default = $default[ $key ] ?? null;
		}

		// Skip defaults
		if ( $optional && $default === $value ) {
			return null;
		}

		// Separator
		if ( '-' !== $args['separator'] ) {
			$slug = str_replace( '-', $args['separator'], $slug );
		}

		// Prefix
		if ( $args['prefix'] ) {
			$slug = str_replace( '--', '', kalium_css_var_name() ) . $args['separator'] . $slug;
		}

		return [
			'slug'  => $slug,
			'name'  => $title,
			'color' => $value,
		];
	};

	return array_filter(
		array_map(
			function ( $color_path_options, $color_path ) use ( &$get_color_entry ) {
				return $get_color_entry( $color_path, $color_path_options['title'], $color_path_options['optional'] ?? false );
			},
			$global_colors,
			array_keys( $global_colors )
		)
	);
}

/**
 * Link style class.
 *
 * Applies to elements the link style that is defined in Styling > Links.
 *
 * @return string
 * @since 4.0
 */
function kalium_get_link_style_class() {
	return 'link-style';
}

/**
 * Link style class by context.
 *
 * @param string $context
 *
 * @return string|null
 */
function kalium_get_link_style_class_by_context( $context, $fallback = null ) {
	/**
	 * Filter link style by context hook.
	 *
	 * @param bool $should_apply
	 * @param string $link_style
	 */
	$should_apply = apply_filters( 'kalium_link_style_class_by_context', true, $context );

	if ( $should_apply ) {
		return kalium_get_link_style_class();
	}

	return $fallback;
}

/**
 * Link title class.
 *
 * @return string
 * @since 4.0
 */
function kalium_get_link_title_class() {
	return 'link-title';
}

/**
 * Link text class.
 *
 * @return string
 * @since 4.0
 */
function kalium_get_link_text_class() {
	return 'link-text';
}

/**
 * Link inline class.
 *
 * @return string
 * @since 4.0
 */
function kalium_get_link_inline_class() {
	return 'link-inline';
}

/**
 * Link block class.
 *
 * @return string
 * @since 4.0
 */
function kalium_get_link_block_class() {
	return 'link-block';
}

/**
 * Link plain class.
 *
 * @return string
 * @since 4.0
 */
function kalium_get_link_plain_class() {
	return 'link-plain';
}

/**
 * Get SVG icon from icon packs in assets.
 *
 * @param string $name
 *
 * @return string
 * @since 4.0
 */
function kalium_get_icon( $name ) {
	$cache_id = 'kalium-icon-' . $name;

	return kalium_get_cached_var(
		$cache_id,
		function () use ( $name ) {
			$icon_packs = [
				'kalium',
				'kalium-social',
			];

			foreach ( $icon_packs as $icon_pack ) {
				$icon_path = kalium()->assets_path( sprintf( 'images/svg-icons/%s/%s.svg', $icon_pack, $name ) );

				if ( $icon_path ) {
					$icon = trim( preg_replace( [ '/<!--.*?-->/', '/<title>.*?<\/title>\s*/' ], '', file_get_contents( $icon_path ) ) );
					return sprintf( '<span class="inline-svg-icon %s">%s</span>', "{$icon_pack}-svg-icon-{$name}", $icon );
				}
			}

			return null;
		}
	);
}

/**
 * Get root selector based on current screen.
 *
 * @return string
 */
function kalium_get_root_selector() {
	if ( kalium_is_block_editor() || kalium_is_customizer_preview_admin() ) {
		// return 'html .editor-styles-wrapper';
	}

	return ':root';
}

/**
 * Device visibility classes.
 *
 * @param array $devices
 *
 * @return array
 * @since 4.0
 */
function kalium_get_device_visibility_classes( $devices ) {
	$all_devices    = [ 'desktop', 'tablet', 'mobile' ];
	$hidden_classes = [];

	foreach ( $all_devices as $device ) {
		if ( ! in_array( $device, $devices, true ) ) {
			$hidden_classes[] = 'hidden-' . $device;
		}
	}

	return apply_filters( 'kalium_device_visibility_classes', $hidden_classes, $devices );
}

/**
 * Bootstrap responsive classname infix.
 *
 * @param array           $value
 * @param string|callable $prefix_callback
 *
 * @since 4.0
 */
function kalium_responsive_classname_infix( $value, $prefix_callback ) {
	static $viewports, $prefixes = [
		'desktop' => 'xl',
		'tablet'  => 'md',
		'mobile'  => '',
	];

	if ( is_null( $viewports ) ) {
		$viewports = array_reverse( array_keys( kalium_get_responsive_viewports() ) );
	}

	// Classes
	$classes = [];

	// String prefix
	if ( is_string( $prefix_callback ) ) {
		$prefix = $prefix_callback;

		$prefix_callback = static function ( $value, $bs_viewport ) use ( $prefix ) {
			if ( is_null( $value ) ) {
				return null;
			}

			return $prefix . kalium_if_else( $bs_viewport, "-{$bs_viewport}" ) . '-' . $value;
		};
	}

	if ( is_callable( $prefix_callback ) ) {
		$responsive_value = kalium_parse_responsive_value( $value, null, false );
		$classes_filtered = false;

		foreach ( $viewports as $viewport ) {
			$bs_viewport = kalium_if_else( count( $classes ), $prefixes[ $viewport ] );
			$class_name  = $prefix_callback( kalium_get_responsive_value( $responsive_value, $viewport ), $bs_viewport, $viewport );

			if ( false === $class_name ) {
				$classes_filtered = true;
				continue;
			}

			$classes[] = $class_name;
		}

		if ( $classes_filtered ) {
			$classes = array_values( array_filter( $classes ) );
		}
	}

	return $classes;
}

/**
 * Animate on viewport props.
 *
 * @param array $args
 * @param bool  $return
 *
 * @return array|void
 */
function kalium_animate_on_viewport( $args = [], $return = true ) {
	static $animations = [
		'fade-in',
		'fade-in-up',
		'zoom-in',
	];

	$args = wp_parse_args(
		$args,
		[
			'prefix'    => 'data-aov',
			'animation' => null,
			'duration'  => null, // [1,2,3,4]
			'delay'     => null, // [1,2,3,4]
			'stagger'   => null, // [1,2,3,4]
			'add_space' => true,
		]
	);

	$props = [];

	// Animation name/type
	if ( in_array( $args['animation'], $animations ) ) {
		$props['animation'] = $args['animation'];
	} else {
		$props['animation'] = kalium()->helpers->get_array_first( $animations );
	}

	// Set duration, delay and stagger
	foreach ( [ 'duration', 'delay', 'stagger' ] as $prop_name ) {
		if ( isset( $args[ $prop_name ] ) && is_numeric( $args[ $prop_name ] ) && 4 >= $args[ $prop_name ] && 1 <= $args[ $prop_name ] ) {
			$props[ $prop_name ] = $args[ $prop_name ];
		} elseif ( ! empty( $args[ $prop_name ] ) ) {
			$props[ $prop_name ] = true;
		}
	}

	// Add prefix
	if ( ! empty( $args['prefix'] ) ) {
		$props = array_combine(
			array_map(
				function ( $prop_name ) use ( &$args ) {
					if ( 'animation' === $prop_name ) {
						return $args['prefix'];
					}

					return $args['prefix'] . '-' . $prop_name;
				},
				array_keys( $props )
			),
			array_values( $props )
		);
	}

	// Print on screen
	if ( ! $return ) {

		// Add space before attributes list
		if ( $args['add_space'] ) {
			echo ' ';
		}

		// Print attributes
		kalium_render_attributes( $props );
	}
	// Return value
	else {
		return $props;
	}
}

/**
 * Parses the legacy WOW effect type to new implementation of AOV.
 *
 * @param string $type
 *
 * @return array
 * @since 4.0
 */
function kalium_parse_legacy_wow_args( $type ) {
	$args = [];

	// Effect
	switch ( $type ) {
		// Fade
		case 'fadeIn':
		case 'fadeIn-one':
			$args['animation'] = 'fade-in';
			break;

		// Fade and slide up
		case 'fadeInLab':
		case 'fadeInLab-one':
			$args['animation'] = 'fade-in-up';
			break;
	}

	if ( ! empty( $args['animation'] ) ) {
		$args['duration'] = 2;

		// Staggering
		$stagger = strpos( $type, '-one' );

		if ( $stagger ) {
			$args['stagger'] = 1;
		}
	}

	return $args;
}

/**
 * Get theme lightbox class.
 *
 * @return string
 * @since 4.2.2
 */
function kalium_get_lightbox_class() {
	return 'theme-lightbox';
}

/**
 * Get color object by attribute values.
 *
 * @param string $slug
 * @param string $custom
 *
 * @return string
 * @since 4.3
 */
function kalium_get_color_object_by_attribute_values( $slug = null, $custom = null ) {
	static $palette;

	if ( is_null( $palette ) ) {
		$palette = call_user_func_array( 'array_merge', array_values( wp_get_global_settings( [ 'color','palette' ] ) ) );
	}

	$colors = wp_list_pluck( $palette, 'color', 'slug' );

	return $colors[ $slug ] ?? $custom ?? $slug;
}
