<?php
/**
 * Kalium WordPress Theme
 *
 * Util functions.
 *
 * @author Laborator
 * @link   https://kaliumtheme.com
 */
if ( ! defined( 'ABSPATH' ) ) {
	exit; // Direct access not allowed.
}

/**
 * Filter terms array and remove duplicates.
 *
 * @param array $terms
 *
 * @return array
 */
function kalium_unique_terms( $terms ) {
	if ( ! is_array( $terms ) ) {
		return $terms;
	}

	$terms_unique = [];

	foreach ( $terms as $term ) {
		if ( $term instanceof WP_Term ) {
			if ( ! isset( $terms_unique[ $term->term_id ] ) ) {
				$terms_unique[ $term->term_id ] = $term;
			}
		}
	}

	return array_values( $terms_unique );
}

/**
 * Convert an english word to number.
 *
 * @param string|int $word
 *
 * @return int
 */
function kalium_get_number_from_word( $word ) {
	if ( is_numeric( $word ) ) {
		return $word;
	}

	$find    = [ 'ten', 'nine', 'eight', 'seven', 'six', 'five', 'four', 'three', 'two', 'one', 'zero' ];
	$replace = [ 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0 ];

	$word = str_replace( $find, $replace, $word );

	if ( is_numeric( $word ) ) {
		return (int) $word;
	}

	return 0;
}

/**
 * Clean post excerpt.
 *
 * @param string $content
 * @param bool   $strip_tags
 *
 * @return string
 */
function kalium_clean_excerpt( $content, $strip_tags = false ) {
	$content = strip_shortcodes( $content );
	$content = preg_replace( '#<style.*?>(.*?)</style>#i', '', $content ); // Remove style
	$content = preg_replace( '#<script.*?>(.*?)</script>#i', '', $content ); // Remove scripts
	$content = preg_replace( '#<img[^>]+>#i', '', $content ); // Remove images
	$content = trim( str_replace( '&nbsp;', ' ', $content ) ); // Trim content

	return $strip_tags ? wp_strip_all_tags( $content ) : $content;
}

/**
 * Extract aspect ratio from string.
 *
 * @param string $str
 *
 * @return array
 */
function kalium_extract_aspect_ratio( $str = '' ) {
	$ratio = [];

	if ( ! empty( $str ) && preg_match( '/^(?<w>[0-9]+)(:|x)(?<h>[0-9]+)$/', trim( $str ), $matches ) ) {
		$ratio = [
			'width'  => $matches['w'],
			'height' => $matches['h'],
		];
	}

	return $ratio;
}

/**
 * Wrap image with image placeholder element.
 *
 * @param string $image
 *
 * @return string
 */
function kalium_image_placeholder_wrap_element( $image ) {
	if ( false !== strpos( $image, '<img' ) ) {
		return kalium()->images->get_image( $image );
	}

	return $image;
}

/**
 * Default Value Set for Visual Composer Loop Parameter Type.
 *
 * @param string $query
 * @param string $field
 * @param string $value
 *
 * @return string
 */
function kalium_vc_loop_param_set_default_value( &$query, $field, $value = '' ) {
	$field = strval( $field );

	if ( ! preg_match( '/(\|?)' . preg_quote( $field ) . ':/', $query ) ) {
		$query .= "|{$field}:{$value}";
	}

	return ltrim( '|', $query );
}

/**
 * Compress text function.
 *
 * @param string $str
 *
 * @return string
 */
function kalium_compress_text( $str ) {

	/* remove comments */
	$str = preg_replace( '!/\*[^*]*\*+([^/][^*]*\*+)*/!', '', $str );

	/* remove tabs, spaces, newlines, etc. */
	$str = str_replace( [ "\r\n", "\r", "\n", "\t", '	', '	', '	' ], '', $str );

	return $str;
}
