<?php
/**
 * Kalium WordPress Theme
 *
 * WooCommerce functions.
 *
 * @author Laborator
 * @link   https://kaliumtheme.com
 */
if ( ! defined( 'ABSPATH' ) ) {
	exit; // Direct access not allowed.
}

/**
 * Check if shop is in catalog mode.
 *
 * @return bool
 */
function kalium_woocommerce_is_catalog_mode() {
	return kalium_get_theme_option( 'shop_catalog_mode' );
}

/**
 * Catalog mode hide prices.
 *
 * @return bool
 * @since 4.0
 */
function kalium_woocommerce_catalog_mode_hide_prices() {
	return kalium_woocommerce_is_catalog_mode() && kalium_get_theme_option( 'shop_catalog_mode_hide_prices' );
}

/**
 * Use default built-in WooCommerce product image gallery.
 *
 * @return bool
 * @since 4.0
 */
function kalium_woocommerce_use_default_image_gallery() {
	$legacy_product_image_gallery = apply_filters( 'kalium_woocommerce_use_custom_product_image_gallery_layout', true );

	return apply_filters( 'kalium_woocommerce_use_default_image_gallery', ! $legacy_product_image_gallery );
}

/**
 * Check if zoom is enabled.
 *
 * @return bool
 */
function kalium_woocommerce_is_product_gallery_zoom_enabled() {
	return (bool) get_theme_support( 'wc-product-gallery-zoom' );
}

/**
 * Check if gallery lightbox is enabled.
 *
 * @return boolean
 */
function kalium_woocommerce_is_product_gallery_lightbox_enabled() {
	return kalium_get_theme_option( 'shop_single_gallery_image_enlarge' );
}

/**
 * Get cart contents count.
 *
 * @return int
 */
function kalium_woocommerce_get_items_in_cart() {
	if ( function_exists( 'WC' ) && ! empty( WC()->cart ) ) {
		return WC()->cart->get_cart_contents_count();
	}

	return 0;
}

/**
 * Get cart items message.
 *
 * @return string
 * @since 4.2.3
 */
function kalium_woocommerce_get_cart_items_message() {
	$cart_items = kalium_woocommerce_get_items_in_cart();
	return sprintf( _n( 'You\'ve got 1 item in the cart', 'You\'ve got %d items in the cart', $cart_items, 'kalium' ), $cart_items );
}

/**
 * Check if login form can be shown in checkout page.
 *
 * @return bool
 * @since 4.0
 */
function kalium_woocommerce_has_checkout_login_form() {
	static $show_login_form;

	if ( ! isset( $show_login_form ) ) {
		$show_login_form = is_checkout() && ! is_user_logged_in() && 'no' !== get_option( 'woocommerce_enable_checkout_login_reminder' );
	}

	return $show_login_form;
}

/**
 * Get number of orders.
 *
 * @return int
 * @since 4.0
 */
function kalium_woocommerce_get_orders_count() {
	static $orders_count;

	// Get orders
	if ( ! isset( $orders_count ) ) {
		$customer_orders = wc_get_orders(
			apply_filters(
				'woocommerce_my_account_my_orders_query',
				[
					'customer' => get_current_user_id(),
					'limit'    => -1,
				]
			)
		);

		$orders_count = count( $customer_orders );
	}

	return $orders_count;
}

/**
 * Get product images.
 *
 * @param array $args
 *
 * @return int[]
 * @since 4.0
 */
function kalium_woocommerce_get_product_images( $args ) {
	$args = wp_parse_args(
		$args,
		[
			'product'    => null,
			'max_images' => -1,
		]
	);

	/**
	 * Get product.
	 *
	 * @var WC_Product $product
	 */
	if ( isset( $args['product'] ) ) {
		$product = wc_get_product( $args['product'] );
	} else {
		$product = $GLOBALS['product'];
	}

	$images = [];

	// If product is missing
	if ( ! ( $product instanceof WC_Product ) ) {
		return $images;
	}

	// Parent product ID
	$parent_id = $product->get_parent_id();

	// Product ID
	$product_id = $product->get_id();

	// Featured image
	if ( has_post_thumbnail( $product_id ) ) {
		$images[] = get_post_thumbnail_id( $product_id );
	} 
	
	// Parent product
	if ( $parent_id ) {
		$product = wc_get_product( $parent_id );
		
		// Add parent image as first image if no images are set
		$parent_image_id = empty( $images ) ? get_post_thumbnail_id( $parent_id ) : null;

		if ( $parent_image_id && ! in_array( $parent_image_id, $images ) ) {
			$images[] = $parent_image_id;
		}
	}

	// Append other images
	if ( $gallery_image_ids = $product->get_gallery_image_ids() ) {
		$images = array_merge( $images, $gallery_image_ids );
	}

	// Max images
	if ( $args['max_images'] > 0 ) {
		$images = array_slice( $images, 0, $args['max_images'] );
	}

	return apply_filters( 'kalium_woocommerce_product_images', $images, $product, $args );
}

/**
 * Get archive products view (shop layout).
 *
 * @return string
 * @since 4.0
 */
function kalium_woocommerce_get_products_view() {
	$products_view = kalium_get_theme_option( 'shop_products_view' );

	if ( ! empty( wc_get_loop_prop( 'name' ) ) ) {
		return 'grid';
	}

	return $products_view;
}

/**
 * Show category thumbnail.
 *
 * @param WP_Term $category
 *
 * @since 4.0
 */
function kalium_woocommerce_category_thumbnail( $category ) {
	static $cached_images = [];

	if ( ! ( $category instanceof WP_Term ) ) {
		return;
	}

	$small_thumbnail_size = apply_filters( 'subcategory_archive_thumbnail_size', 'woocommerce_thumbnail' );
	$thumbnail_id         = get_term_meta( $category->term_id, 'thumbnail_id', true );

	if ( $thumbnail_id ) {
		echo kalium_get_attachment_image( $thumbnail_id, $small_thumbnail_size );
	} else {
		$image_src = wc_placeholder_img_src( $small_thumbnail_size );

		if ( isset( $cached_images[ $image_src ] ) ) {
			echo $cached_images[ $image_src ];
		} else {
			$upload_dir = wp_upload_dir();

			if ( 0 === strpos( $image_src, $upload_dir['baseurl'] ) ) {
				$image_path = $upload_dir['basedir'] . substr( $image_src, strlen( $upload_dir['baseurl'] ) );

				if ( file_exists( $image_path ) ) {
					$meta = wp_getimagesize( $image_path );

					if ( $meta ) {
						list( $width, $height ) = $meta;

						$image = kalium()->images->aspect_ratio_wrap(
							kalium_render_element(
								[
									'tag_name'   => 'img',
									'attributes' => [
										'src'    => $image_src,
										'width'  => $width,
										'height' => $height,
										'alt'    => $category->name,
									],
								],
								false
							),
							[
								'width'  => $width,
								'height' => $height,
							]
						);

						$cached_images[ $image_src ] = $image;

						echo $image;
					}
				}
			}
		}
	}
}

/**
 * Get free shipping minimum amount.
 *
 * @return int|float|false
 */
function kalium_woocommerce_get_free_shipping_minimum_amount() {
	$min_amount = null;

	if ( did_action( 'init' ) ) {
		// Default shipping zone
		foreach ( WC_Shipping_Zones::get_zone( 0 )->get_shipping_methods() as $method ) {
			if ( 'free_shipping' === $method->id ) {
				if ( is_numeric( $method->get_option( 'min_amount' ) ) ) {
					$min_amount = $method->get_option( 'min_amount' );
				}
			}
		}

		// User defined shipping zones
		foreach ( WC_Shipping_Zones::get_zones() as $zone ) {
			foreach ( $zone['shipping_methods'] as $method ) {
				if ( 'free_shipping' === $method->id ) {
					if ( is_numeric( $method->get_option( 'min_amount' ) ) ) {
						$min_amount = $method->get_option( 'min_amount' );
					}
				}
			}
		}
	}

	return is_numeric( $min_amount ) && 0 != $min_amount ? $min_amount : false;
}

/**
 * Checks if AJAX add to cart is supported.
 *
 * @return bool
 * @since 4.0
 */
function kalium_woocommerce_is_ajax_add_to_cart_enabled() {
	return apply_filters( 'kalium_woocommerce_ajax_add_to_cart', true );
}

/**
 * Get product link.
 *
 * @param int|WP_Post $post
 *
 * @return string
 * @since 4.0
 */
function kalium_woocommerce_get_product_link( $post = null ) {
	global $product;

	return apply_filters( 'woocommerce_loop_product_link', get_the_permalink( $post ), $product );
}
