<?php
/**
 * Kalium WordPress Theme
 *
 * WooCommerce hook functions.
 *
 * @author Laborator
 * @link   https://kaliumtheme.com
 */
if ( ! defined( 'ABSPATH' ) ) {
	exit; // Direct access not allowed.
}

/**
 * WooCommerce ready.
 */
function kalium_woocommerce_ready() {

	// Enqueue cart fragments when mini-cart functionality is used
	if ( 'mini-cart' === kalium_get_theme_option( 'shop_add_to_cart_action' ) ) {
		wp_enqueue_script( 'wc-cart-fragments' );
	}

	// Rating color
	kalium_set_css_var( 'woocommerce-rating-color', kalium_get_theme_option( 'shop_single_reviews_rating_color' ) );
}

add_action( 'woocommerce_init', 'kalium_woocommerce_ready' );

/**
 * Category image size,
 *
 * @return string
 */
function kalium_woocommerce_subcategory_archive_thumbnail_size() {
	static $image_size;

	if ( is_null( $image_size ) ) {
		$category_image = kalium()->customize->get_image_size_settings( 'shop_category_image' );

		if ( 'custom' === $category_image['size'] ) {
			$width  = $category_image['size_custom']['width'] ?? null;
			$height = $category_image['size_custom']['height'] ?? null;

			$image_size = [ 500, 290 ];

			if ( $width || $height ) {
				$image_size = [ intval( $width ), intval( $height ) ];
			}
		} else {
			$image_size = $category_image['size'];
		}
	}

	return $image_size;
}

add_filter( 'subcategory_archive_thumbnail_size', 'kalium_woocommerce_subcategory_archive_thumbnail_size' );

/**
 * Display product loop card.
 *
 * @since 4.0
 */
function kalium_woocommerce_product_default_template() {
	laborator_builder_render(
		[
			'content_type'   => 'product_loop_card',
			'content_source' => 'product_loop_card',
		],
		[
			'parent_selector' => '.product',
		]
	);
}

add_action( 'kalium_woocommerce_shop_loop_item', 'kalium_woocommerce_product_default_template' );

/**
 * Cart Fragments for mini cart.
 *
 * @param array $fragments_arr
 *
 * @return array
 */
function kalium_woocommerce_woocommerce_add_to_cart_fragments( $fragments_arr ) {
	$kalium_args = [];

	// Totals
	$totals = WC()->cart->get_totals();

	$kalium_args['totals'] = array_map(
		'wc_price',
		[
			'cartTotal'      => $totals['total'],
			'cartSubtotal'   => $totals['subtotal'],
			'cartTotalExTax' => $totals['total'] - $totals['total_tax'],
		]
	);

	// Total items in cart
	$kalium_args['totals']['items'] = kalium_woocommerce_get_items_in_cart();

	// Mini cart drawer
	ob_start();
	kalium_woocommerce_mini_cart_drawer();
	$cart_drawer = ob_get_clean();

	$kalium_args['miniCartDrawer'] = $cart_drawer;

	// Header mini cart
	ob_start();
	kalium_woocommerce_mini_cart_popup();
	$cart_contents = ob_get_clean();

	$kalium_args['miniCartPopup'] = $cart_contents;

	// Cart items message
	$kalium_args['cartItemsMessage'] = kalium_woocommerce_get_cart_items_message();

	// Free shipping bar
	if ( kalium_get_theme_option( 'shop_free_shipping_bar' ) ) {
		ob_start();
		kalium_woocommerce_free_shipping_bar();
		$free_shipping_bar = ob_get_clean();

		$kalium_args['freeShippingBar'] = $free_shipping_bar;
	}

	// Notices
	if ( kalium_woocommerce_is_ajax_add_to_cart_enabled() ) {
		$error_notices = kalium_hook_return_value( [ 'error' ] );

		add_filter( 'woocommerce_notice_types', $error_notices, 100 );
		$notices = wc_print_notices( true );
		remove_filter( 'woocommerce_notice_types', $error_notices, 100 );

		$kalium_args['notices'] = $notices;
	}

	return array_merge(
		$fragments_arr,
		[
			'kalium' => $kalium_args,
		]
	);
}

add_filter( 'woocommerce_add_to_cart_fragments', 'kalium_woocommerce_woocommerce_add_to_cart_fragments' );

/**
 * Return to shop button text.
 *
 * @param string $text
 *
 * @return string
 * @since 4.0
 */
function kalium_woocommerce_return_to_shop_text( $text ) {
	return kalium_xtranslate( '__', 'Browse products', 'woocommerce' );
}

add_filter( 'woocommerce_return_to_shop_text', 'kalium_woocommerce_return_to_shop_text' );

/**
 * Product image on checkout page.
 *
 * @param string $product_name
 * @param array  $cart_item
 * @param array  $cart_item_key
 *
 * @return string
 * @since 4.0
 */
function kalium_woocommerce_checkout_product_image( $product_name, $cart_item, $cart_item_key ) {
	if ( is_checkout() ) {
		$_product = apply_filters( 'woocommerce_cart_item_product', $cart_item['data'], $cart_item, $cart_item_key );

		// Link to product
		if ( kalium_get_theme_option( 'shop_checkout_link_to_product' ) ) {
			$product_permalink = apply_filters( 'woocommerce_cart_item_permalink', $_product->is_visible() ? $_product->get_permalink( $cart_item ) : '', $cart_item, $cart_item_key );

			if ( false === strpos( $product_name, '<a' ) ) {
				$product_name = sprintf( '<a href="%s">%s</a>', esc_url( $product_permalink ), $product_name );
			}
		}

		// Thumbnail
		if ( kalium_get_theme_option( 'shop_checkout_product_image' ) ) {
			$thumbnail = apply_filters( 'woocommerce_cart_item_thumbnail', $_product->get_image(), $cart_item, $cart_item_key );

			if ( isset( $product_permalink ) ) {
				$thumbnail = sprintf( '<a href="%s">%s</a>', $product_permalink, $thumbnail );
			}

			$product_name = $thumbnail . '</td><td class="product-name product-name--text">' . $product_name;
		}
	}

	return $product_name;
}

add_filter( 'woocommerce_cart_item_name', 'kalium_woocommerce_checkout_product_image', 10, 3 );

/**
 * Load WPBakery shortcodes on Checkout page via AJAX.
 *
 * @since 3.4.1
 */
function kalium_woocommerce_load_wpbakery_shortcodes_on_checkout() {
	if ( kalium()->is->wpbakery_active() ) {
		WPBMap::addAllMappedShortcodes();
	}
}

add_action( 'woocommerce_checkout_update_order_review', 'kalium_woocommerce_load_wpbakery_shortcodes_on_checkout' );

/**
 * Products per page.
 *
 * @since 4.0
 */
function kalium_woocommerce_set_products_per_page() {
	return kalium_get_theme_option( 'shop_products_per_page' );
}

add_filter( 'loop_shop_per_page', 'kalium_woocommerce_set_products_per_page' );

/**
 * Product columns.
 *
 * @since 4.0
 */
function kalium_woocommerce_set_loop_product_columns() {
	$columns = kalium_parse_responsive_value( kalium_get_theme_option( 'shop_grid_columns' ), null, false );

	$columns_loop_props = [
		'desktop' => 'columns',
		'tablet'  => 'columns_tablet',
		'mobile'  => 'columns_mobile',
	];

	// Use inside shortcodes
	if ( 0 === strpos( current_action(), 'woocommerce_shortcode' ) ) {
		unset( $columns_loop_props['desktop'] );

		if ( 2 >= wc_get_loop_prop( 'columns' ) ) {
			$columns['tablet'] = $columns['mobile'] = wc_get_loop_prop( 'columns' );
		}
	}

	foreach ( $columns_loop_props as $viewport => $loop_prop ) {
		if ( isset( $columns[ $viewport ] ) && is_numeric( $columns[ $viewport ] ) ) {
			wc_set_loop_prop( $loop_prop, (int) $columns[ $viewport ] );
		}
	}
}

foreach ( [
	'woocommerce_before_shop_loop',
	'woocommerce_shortcode_before_products_loop',
	'woocommerce_shortcode_before_recent_products_loop',
	'woocommerce_shortcode_before_featured_products_loop',
	'woocommerce_shortcode_before_sale_products_loop',
	'woocommerce_shortcode_before_best_selling_products_loop',
	'woocommerce_shortcode_before_top_rated_products_loop',
	'woocommerce_shortcode_before_product_category_loop',
] as $hook ) {
	add_action( $hook, 'kalium_woocommerce_set_loop_product_columns', 40 );
}

/**
 * Print WooCommerce styles.
 *
 * @since 4.0
 */
function kalium_print_woocommerce_general_style() {
	$vars = [];

	// Products grid gap
	$vars['grid-gap'] = kalium_get_theme_option( 'shop_grid_gap' );

	// Print WooCommerce general style
	kalium_print_inline_style(
		[
			'id'       => 'woocommerce-general-style',
			'selector' => ':root',
			'vars'     => $vars,
		]
	);
}

add_action( 'wp_head', 'kalium_print_woocommerce_general_style' );

/**
 * Print badge styles.
 */
function kalium_print_woocommerce_badge_style() {
	$css_styles = $vars = [];

	// Badge style vars
	$style   = kalium_get_theme_option( 'shop_badges_style' );
	$size    = kalium_get_theme_option( 'shop_badges_size' );
	$outline = 'badge' === $style && kalium_get_theme_option( 'shop_badges_outline' );

	switch ( $style ) {

		// Circle
		case 'circle':
			// Size
			if ( false === kalium_is_empty_responsive( $size ) ) {
				$size_var_name = kalium_if_else( kalium_get_theme_option( 'shop_badges_fixed_size' ), 'product-badge-size-fixed', 'product-badge-size' );

				$vars[ $size_var_name ] = $size;
			}

			// Border radius
			if ( $radius = kalium_get_theme_option( 'shop_badges_radius' ) ) {
				$vars['product-badge-radius'] = $radius;
			}
			break;

		// Triangle
		case 'triangle':
			// Size
			if ( false === kalium_is_empty_responsive( $size ) ) {
				$vars['product-badge-size'] = $size;
			}
			break;
	}

	$css_styles[] = [
		'selector' => '.product-badge',
		'vars'     => $vars,
	];

	// Colors
	$bade_colors = [
		'shop_badge_color_sale'         => 'sale',
		'shop_badge_color_featured'     => 'featured',
		'shop_badge_color_out_of_stock' => 'out-of-stock',
		'shop_badge_color_backorder'    => 'backorder',
	];

	foreach ( $bade_colors as $color_option => $selector ) {
		if ( $color = kalium_get_theme_option( $color_option, true ) ) {
			$css_styles[] = [
				'selector' => '.product-badge.product-badge--type-' . $selector,
				'vars'     => [
					'badge-color'         => $outline ? $color['bg'] : $color['text'],
					'badge-bg'            => $outline ? $color['text'] : $color['bg'],
					'badge-outline-color' => $color['bg'],
				],
			];
		}
	}

	kalium_print_inline_style(
		[
			'styles' => $css_styles,
		],
	);
}

add_action( 'wp_head', 'kalium_print_woocommerce_badge_style' );

/**
 * Products with masonry container.
 */
function kalium_woocommerce_masonry_products_container() {
	if ( ! kalium_get_theme_option( 'shop_masonry_mode' ) ) {
		return;
	}

	kalium_create_masonry_container(
		[
			'options' => [
				'container' => '.products',
				'item'      => '.product',
			],
		]
	);
}

add_action( 'woocommerce_before_shop_loop', 'kalium_woocommerce_masonry_products_container', 100 );

/**
 * Replace WooCommerce templates.
 *
 * @since 4.0
 */
function kalium_woocommerce_replace_templates() {

	// Content product template (archive/loop)
	kalium_replace_woocommerce_template(
		'content-product.php',
		function () {
			return 'list' === kalium_woocommerce_get_products_view() ? 'woocommerce/content-product-list.php' : 'woocommerce/content-product.php';
		}
	);

	// Category template
	kalium_replace_woocommerce_template( 'content-product-cat.php', 'woocommerce/content-product-cat.php' );

	// Product tabs
	kalium_replace_woocommerce_template( 'single-product/tabs/tabs.php', 'kalium_woocommerce_product_tabs' );
}

add_action( 'woocommerce_init', 'kalium_woocommerce_replace_templates' );

/**
 * Catalog mode.
 *
 * @since 4.0
 */
function kalium_woocommerce_set_catalog_mode() {
	if ( ! kalium_woocommerce_is_catalog_mode() ) {
		return;
	}

	// Remove add-to-cart button template
	kalium_replace_woocommerce_template(
		[
			'loop/add-to-cart.php',
			'single-product/add-to-cart/*.php',
			'cart/cart.php',
			'checkout/form-checkout.php',
		]
	);

	// Disable checkout and cart page
	add_action(
		'template_redirect',
		function () {
			if ( is_cart() || is_checkout() ) {
				wp_redirect( get_permalink( wc_get_page_id( 'shop' ) ) );
				die();
			}
		}
	);

	// Hide prices
	if ( kalium_woocommerce_catalog_mode_hide_prices() ) {
		add_filter( 'woocommerce_get_price_html', '__return_empty_string', 1000 );
	}
}

add_action( 'wp', 'kalium_woocommerce_set_catalog_mode' );

/**
 * Move breadcrumb inside shop toolbar when shop results are not showing.
 *
 * @since 4.0
 */
function kalium_woocommerce_move_breadcrumb_to_shop_toolbar() {
	if ( kalium_get_theme_option( 'shop_breadcrumb' ) && ! kalium_get_theme_option( 'shop_results_count' ) ) {
		remove_action( 'kalium_woocommerce_shop_heading', 'kalium_woocommerce_breadcrumb_display' );
		add_action( 'kalium_woocommerce_shop_toolbar_left', 'kalium_woocommerce_breadcrumb_display' );
	}
}

add_action( 'wp', 'kalium_woocommerce_move_breadcrumb_to_shop_toolbar' );

/**
 * Switch columns views.
 *
 * @since 4.0
 */
function kalium_woocommerce_switch_columns_view() {
	if ( ! kalium_get_theme_option( 'shop_columns_switcher' ) ) {
		return;
	}

	// Current view
	if ( kalium()->request->has( 'switch_columns' ) ) {
		$switch_columns = kalium()->request->request( 'switch_columns' );
	}
	// Stored view in cookie
	elseif ( kalium()->request->has( 'shop_columns_switcher_value', 'COOKIE' ) && ! is_customize_preview() ) {
		$switch_columns = kalium()->request->cookie( 'shop_columns_switcher_value' );
	}

	if ( ! empty( $switch_columns ) ) {

		// Products view
		add_filter(
			'kalium_theme_option_shop_products_view',
			function () use ( $switch_columns ) {
				return 'list' === $switch_columns ? 'list' : 'grid';
			}
		);

		// Number of columns
		if ( is_numeric( $switch_columns ) && $switch_columns > 1 && $switch_columns <= 6 ) {
			add_filter(
				'kalium_theme_option_shop_grid_columns',
				function ( $value ) use ( $switch_columns ) {
					if ( ! isset( $value['desktop'] ) ) {
						$value = [];
					}

					$value['desktop'] = $switch_columns;

					return $value;
				}
			);
		}
	}
}

add_action( 'woocommerce_init', 'kalium_woocommerce_switch_columns_view' );

/**
 * Wrap categories with a container.
 *
 * @param string $loop_html
 *
 * @return string
 */
function kalium_woocommerce_wrapped_categories( $loop_html = '' ) {
	if ( $categories = woocommerce_maybe_show_product_subcategories() ) {
		$current_loop = $GLOBALS['woocommerce_loop'] ?? [];

		ob_start();

		// Container start
		wc_get_template(
			'loop/loop-start.php',
			[
				'extra_classes' => 'shop-categories',
			]
		);

		// Categories
		echo $categories;

		// Container end
		wc_get_template( 'loop/loop-end.php' );

		// Reset loop props
		foreach ( $current_loop as $prop => $value ) {
			wc_set_loop_prop( $prop, $value );
		}

		$categories_wrapped = ob_get_clean();

		$loop_html = $categories_wrapped . $loop_html;
	}

	return $loop_html;
}

/**
 * Separate categories from products container.
 *
 * @since 4.0
 */
function kalium_woocommerce_separate_categories_from_products_container() {
	if ( ! kalium_get_theme_option( 'shop_category_separate' ) || kalium_get_theme_option( 'shop_masonry_mode' ) ) {
		return;
	}

	remove_filter( 'woocommerce_product_loop_start', 'woocommerce_maybe_show_product_subcategories' );
	add_filter( 'woocommerce_product_loop_start', 'kalium_woocommerce_wrapped_categories' );
}

add_action( 'wp_loaded', 'kalium_woocommerce_separate_categories_from_products_container' );

/**
 * Add "shop-categories" class for products container ([product_categories]).
 *
 * @param string $output
 * @param string $tag
 *
 * @return string|string[]|null
 */
function kalium_woocommerce_product_categories_shortcode_wrap( $output, $tag ) {
	if ( 'product_categories' === $tag ) {
		$output = preg_replace( '/(<ul.*?class=")/', '${1}shop-categories ', $output );
	}

	return $output;
}

add_filter( 'do_shortcode_tag', 'kalium_woocommerce_product_categories_shortcode_wrap', 100, 2 );

/**
 * Remove "wpb-content-wrapper" wrapper in checkout page.
 *
 * @since 4.0
 */
function kalium_woocommerce_remove_wpb_content_wrapper() {
	if ( did_action( 'vc_after_init' ) && is_checkout() ) {
		add_filter(
			'the_content',
			function ( $content ) {
				$search = '<section class="wpb-content-wrapper">';

				if ( 0 === strpos( $content, $search ) ) {
					$content = substr( $content, strlen( $search ), -strlen( '</section>' ) );
				}

				return $content;
			},
			12
		);
	}
}

add_action( 'wp', 'kalium_woocommerce_remove_wpb_content_wrapper' );

/**
 * Related and up-sell products columns.
 *
 * @param int $columns
 *
 * @return int
 */
function kalium_woocommerce_related_products_columns( $columns ) {
	$related_columns = kalium_get_theme_option( 'shop_single_related_products_columns' );

	if ( is_numeric( $related_columns ) ) {
		return $related_columns;
	}

	return $columns;
}

add_filter( 'woocommerce_related_products_columns', 'kalium_woocommerce_related_products_columns' );
add_filter( 'woocommerce_upsells_columns', 'kalium_woocommerce_related_products_columns' );

/**
 * Related products to show.
 *
 * @param array $args
 *
 * @return array
 */
function kalium_woocommerce_related_products_args( $args ) {
	$posts_per_page = kalium_get_theme_option( 'shop_single_related_products_limit' );

	if ( is_numeric( $posts_per_page ) ) {
		$args['posts_per_page'] = $posts_per_page;
	}

	return $args;
}

add_filter( 'woocommerce_output_related_products_args', 'kalium_woocommerce_related_products_args' );

/**
 * Rating style.
 *
 * @param string $html
 *
 * @since 4.0
 */
function kalium_woocommerce_rating_style( $html ) {
	$rating_style = kalium_get_theme_option( 'shop_single_reviews_rating_style' );

	// Allowed rating styles
	if ( in_array( $rating_style, [ 'stars', 'circles' ] ) ) {
		$html = str_replace( 'class="star-rating', 'class="star-rating star-rating--style-' . esc_attr( $rating_style ), $html );
	}

	return $html;
}

add_filter( 'woocommerce_product_get_rating_html', 'kalium_woocommerce_rating_style' );

/**
 * Rating style in comment form.
 *
 * @since 4.0
 */
function kalium_woocommerce_rating_style_comment_form( $args ) {
	if ( ! empty( $args['comment_field'] ) ) {
		$rating_style = kalium_get_theme_option( 'shop_single_reviews_rating_style' );
		$find         = 'class="comment-form-rating';
		$replace      = $find . ' comment-form-rating--style-' . $rating_style;

		$args['comment_field'] = str_replace( $find, $replace, $args['comment_field'] );
	}

	return $args;
}

add_filter( 'woocommerce_product_review_comment_form_args', 'kalium_woocommerce_rating_style_comment_form' );

/**
 * Single product breadcrumb position (or placement).
 *
 * @since 4.0
 */
function kalium_woocommerce_single_product_breadcrumb_position() {
	if ( 'above-title' === kalium_get_theme_option( 'shop_single_breadcrumb_position' ) ) {
		add_action( 'woocommerce_single_product_summary', 'kalium_woocommerce_single_breadcrumb_display', 0 );
	} else {
		add_action( 'woocommerce_before_main_content', 'kalium_woocommerce_single_breadcrumb_display' );
	}
}

add_action( 'wp', 'kalium_woocommerce_single_product_breadcrumb_position' );

/**
 * WooCommerce product tabs mapper for theme native tabs.
 *
 * @param array  $product_tab
 * @param string $key
 *
 * @return array
 *
 * @since 4.0
 */
function kalium_woocommerce_product_tabs_mapper( $product_tab, $key ) {
	$tab_entry = [
		'id'      => $key,
		'title'   => wp_kses_post( apply_filters( 'woocommerce_product_' . $key . '_tab_title', $product_tab['title'], $key ) ),
		'classes' => [
			'woocommerce-Tabs-panel',
			'woocommerce-Tabs-panel--' . esc_attr( $key ),
			'panel',
			'entry-content',
			'wc-tab',
			kalium_get_link_style_class(),
		],
	];

	// Content
	ob_start();

	if ( isset( $product_tab['callback'] ) ) {
		call_user_func( $product_tab['callback'], $key, $product_tab );
	}

	$tab_entry['content'] = trim( ob_get_clean() );

	return $tab_entry;
}

/**
 * Related and up-sells products.
 *
 * @since 4.0
 */
function kalium_woocommerce_related_up_sells_products() {
	if ( ! kalium_get_theme_option( 'shop_single_upsells_products' ) ) {
		remove_action( 'woocommerce_after_single_product_summary', 'woocommerce_upsell_display', 15 );
	}

	if ( ! kalium_get_theme_option( 'shop_single_related_products' ) ) {
		remove_action( 'woocommerce_after_single_product_summary', 'woocommerce_output_related_products', 20 );
	}
}

add_action( 'wp', 'kalium_woocommerce_related_up_sells_products' );

/**
 * Related products container style.
 *
 * @param array $args
 *
 * @return array
 * @since 4.2
 */
function kalium_woocommerce_related_products_container_style( $args ) {
	$background    = kalium_get_theme_option( 'shop_single_related_products_bg' );
	$padding       = kalium_get_theme_option( 'shop_single_related_products_padding' );
	$border_radius = kalium_get_theme_option( 'shop_single_related_products_border_radius' );

	$vars = [];

	if ( $background ) {
		$vars['related-products-bg'] = $background;
	}

	if ( ! kalium_is_empty_responsive( $padding ) ) {
		$vars['related-products-padding'] = kalium_map_responsive_value( $padding, 'kalium_to_length' );
	}

	if ( ! kalium_is_empty_responsive( $border_radius ) ) {
		$vars['related-products-border-radius'] = kalium_map_responsive_value( $border_radius, 'kalium_to_length' );
	}

	kalium_print_inline_style(
		[
			'id'       => 'related-products',
			'selector' => '.related.products',
			'vars'     => $vars,
		]
	);

	return $args;
}

add_action( 'woocommerce_output_related_products_args', 'kalium_woocommerce_related_products_container_style' );

/**
 * Accordion in tabs in product single page.
 *
 * @since 4.0
 */
function kalium_woocommerce_single_product_accordion_in_summary() {
	if ( 'accordion' === kalium_get_theme_option( 'shop_single_tabs_type' ) && kalium_get_theme_option( 'shop_single_accordion_in_summary' ) ) {

		// Move tabs in summary
		remove_action( 'woocommerce_after_single_product_summary', 'woocommerce_output_product_data_tabs' );
		add_action( 'woocommerce_single_product_summary', 'woocommerce_output_product_data_tabs', 70 );

		// Move reviews below product summary
		add_action( 'woocommerce_after_single_product_summary', 'comments_template' );

		// Remove reviews from tabs
		add_filter(
			'woocommerce_product_tabs',
			function ( $tabs ) {
				unset( $tabs['reviews'] );

				return $tabs;
			}
		);
	}
}

add_action( 'wp_loaded', 'kalium_woocommerce_single_product_accordion_in_summary' );

/**
 * Use default WooCommerce product image gallery when used within Elementor Page template.
 *
 * @param bool $template
 *
 * @return string
 * @since 4.4.3
 */
function kalium_woocommerce_use_default_gallery_on_elementor( $template ) {
	if (
		false !== strpos( $template, 'elementor/modules/page-templates/templates/header-footer.php' ) &&
		(
			is_product() ||
			(
				kalium()->elementor->is_preview( get_queried_object_id() ) &&
				'product' === get_post_meta( get_queried_object_id(), '_elementor_template_type', true )
			)
		)
	) {
		add_filter( 'kalium_woocommerce_use_default_image_gallery', '__return_true', 1000 );
	}

	return $template;
}

add_filter( 'template_include', 'kalium_woocommerce_use_default_gallery_on_elementor', 20 );

/**
 * Product image gallery setup.
 *
 * @since 4.0
 */
function kalium_woocommerce_product_image_gallery_setup() {
	if ( kalium_woocommerce_use_default_image_gallery() ) {
		return;
	}

	// Replace gallery
	remove_action( 'woocommerce_before_single_product_summary', 'woocommerce_show_product_images', 20 );
	add_action( 'woocommerce_before_single_product_summary', 'kalium_woocommerce_single_product_gallery_display', 20 );

	// Dequeue Zoom plugin when not needed
	if ( ! kalium_get_theme_option( 'shop_single_gallery_image_click' ) || 'zoom' !== kalium_get_theme_option( 'shop_single_gallery_image_click_action' ) ) {
		$dequeue_zoom_js = function () {
			if ( wp_script_is( 'zoom' ) ) {
				wp_dequeue_script( 'zoom' );
			}
		};

		add_action( 'wp_enqueue_scripts', $dequeue_zoom_js );
	}

	// Reposition sale badge when gallery type is carousel with thumbnails
	$gallery_type = kalium_get_theme_option( 'shop_single_gallery_type' );

	if ( in_array( $gallery_type, [ 'type-1', 'type-2', 'type-3' ], true ) ) {
		remove_action( 'woocommerce_before_single_product_summary', 'kalium_woocommerce_single_product_badge_display', 5 );
		add_action( 'kalium_woocommerce_product_gallery_main_before', 'kalium_woocommerce_single_product_badge_display' );
	}
}

add_action( 'wp_loaded', 'kalium_woocommerce_product_image_gallery_setup' );

/**
 * My account navigation wrapper start.
 *
 * @since 4.0
 */
function kalium_woocommerce_myaccount_navigation_wrapper_start() {
	echo '<div class="woocommerce-MyAccount-navigation-wrapper">';
}

add_action( 'woocommerce_before_account_navigation', 'kalium_woocommerce_myaccount_navigation_wrapper_start', 5 );

/**
 * My account navigation wrapper end.
 *
 * @since 4.0
 */
function kalium_woocommerce_myaccount_navigation_wrapper_end() {
	echo '</div>';
}

add_action( 'woocommerce_after_account_navigation', 'kalium_woocommerce_myaccount_navigation_wrapper_end', 1000 );

/**
 * Add product image in product variation.
 *
 * @param array                $variation
 * @param WC_Product_Variable  $product_variable
 * @param WC_Product_Variation $product_variation
 *
 * @return array
 * @since 4.0
 */
function kalium_woocommerce_variation_add_image_attribute( $variation, $product_variable, $product_variation ) {
	if ( kalium_woocommerce_use_default_image_gallery() ) {
		return $variation;
	}

	if ( ! empty( $variation['image_id'] ) && $variation['image_id'] !== get_post_thumbnail_id( $product_variable->get_id() ) ) {
		$image_id = $variation['image_id'];

		// Main image
		$variation['kalium_image'] = kalium_woocommerce_product_image(
			$image_id,
			[
				'lightbox_button' => kalium_woocommerce_is_product_gallery_lightbox_enabled(),
				'loading_eager'   => true,
				'echo'            => false,
			]
		);

		// Thumbnail image
		$variation['kalium_image_thumbnail'] = kalium_woocommerce_product_image(
			$image_id,
			[
				'size'          => 'thumbnail',
				'loading_eager' => true,
				'echo'          => false,
			]
		);

		// Additional Variation Images
		if ( defined( 'WOO_VARIATION_GALLERY_PLUGIN_VERSION' ) ) {
			$additional_images = get_post_meta( $variation['variation_id'], 'woo_variation_gallery_images', true );

			if ( is_array( $additional_images ) ) {
				$variation['kalium_variation_images'] = array_map(
					function ( $image_id ) {
						return [
							'main'      => kalium_woocommerce_product_image(
								$image_id,
								[
									'lightbox_button' => kalium_woocommerce_is_product_gallery_lightbox_enabled(),
									'echo'            => false,
								]
							),
							'thumbnail' => kalium_woocommerce_product_image(
								$image_id,
								[
									'size' => 'thumbnail',
									'echo' => false,
								]
							),
						];
					},
					$additional_images
				);
			}
		}
	}

	return $variation;
}

add_action( 'woocommerce_available_variation', 'kalium_woocommerce_variation_add_image_attribute', 10, 3 );

/**
 * Replace YITH add/added to wishlist icon.
 *
 * @param string $icon
 * @param array  $atts
 *
 * @return string
 * @since 4.0
 */
function kalium_yith_wcwl_add_to_wishlist_icon_replace( $icon, $atts ) {
	if ( 'fa-heart' === $atts['icon'] ) {
		$icon_name = 'kalium-icon-added-to-wishlist';
	} elseif ( 'fa-heart-o' === $atts['icon'] ) {
		$icon_name = 'kalium-icon-add-to-wishlist';
	}

	if ( ! empty( $icon_name ) ) {
		return sprintf( '<i class="%s"></i>', esc_attr( $icon_name ) );
	}

	return $icon;
}

add_filter( 'yith_wcwl_add_to_wishlist_icon_html', 'kalium_yith_wcwl_add_to_wishlist_icon_replace', 10, 2 );

/**
 * WooCommerce JS vars.
 *
 * @since 4.0
 */
function kalium_woocommerce_print_js_vars() {
	kalium_js_define_vars(
		[
			// Add to Cart Action
			'addToCartAction'      => kalium_get_theme_option( 'shop_add_to_cart_action' ),

			// Mini cart type
			'miniCartType'         => kalium_get_theme_option( 'shop_mini_cart_type' ),

			// Support for AJAX Add to Aart
			'ajaxAddToCartEnabled' => kalium_woocommerce_is_ajax_add_to_cart_enabled(),
		]
	);
}

add_action( 'wp_head', 'kalium_woocommerce_print_js_vars' );

/**
 * Mini cart print styles.
 *
 * @since 4.0
 */
function kalium_woocommerce_print_mini_cart_styles() {
	$vars = [];

	// Max width
	$max_width = kalium_get_theme_option( 'shop_mini_cart_max_width' );

	if ( is_numeric( $max_width ) ) {
		$vars['oc-max-width'] = $max_width . 'px';
	}

	// Style
	$separators_color = kalium_get_theme_option( 'shop_mini_cart_separators_color' );
	$background       = kalium_get_theme_option( 'shop_mini_cart_bg' );
	$headings_color   = kalium_get_theme_option( 'shop_mini_cart_headings' );
	$text_color       = kalium_get_theme_option( 'shop_mini_cart_text' );
	$padding          = kalium_get_theme_option( 'shop_mini_cart_padding' );
	$border           = kalium_get_theme_option( 'shop_mini_cart_border' );
	$border_radius    = kalium_get_theme_option( 'shop_mini_cart_border_radius' );
	$box_shadow       = kalium_get_theme_option( 'shop_mini_cart_box_shadow' );
	$offset           = kalium_get_theme_option( 'shop_mini_cart_offset' );

	// Off canvas style
	$vars['oc-bg']            = $background;
	$vars['oc-text']          = $text_color;
	$vars['oc-border-radius'] = kalium_map_responsive_value( $border_radius, 'kalium_to_length' );
	$vars['oc-box-shadow']    = kalium_map_responsive_value( $box_shadow, 'kalium_to_box_shadow' );

	// Border
	kalium_map_responsive_value(
		$border,
		function ( $value, $viewport ) use ( &$vars ) {
			$border_props = kalium_to_border_box( $value );

			foreach ( $border_props as $prop => $value ) {
				$vars[ 'oc-border-' . $prop ][ $viewport ] = $value;
			}
		}
	);

	// Offset
	kalium_map_responsive_value(
		$offset,
		function ( $value, $viewport ) use ( &$vars ) {
			$vars['oc-offset-y'][ $viewport ] = kalium_get_multi_numeric_prop( $value, 'y' );
			$vars['oc-offset-x'][ $viewport ] = kalium_get_multi_numeric_prop( $value, 'x' );
		}
	);

	// Drawer style
	$vars['drawer-separators-color'] = $separators_color;
	$vars['heading-color']           = $headings_color;
	$vars['drawer-padding']          = kalium_map_responsive_value( $padding, 'kalium_to_length' );

	// Print styles
	kalium_print_inline_style(
		[
			'styles' => [
				[
					'selector' => [
						'.mini-cart-drawer',
						'.shop-sidebar.off-canvas',
					],
					'vars'     => $vars,
				],
			],
		]
	);
}

add_action( 'wp_head', 'kalium_woocommerce_print_mini_cart_styles' );

/**
 * Free shipping bar styles.
 *
 * @since 4.0
 */
function kalium_woocommerce_print_free_shipping_bar() {
	$vars = [];

	if ( $colors = kalium_get_theme_option( 'shop_free_shipping_bar_colors' ) ) {
		$vars = array_merge(
			$vars,
			[
				'progress-bg'            => $colors['bg'] ?? null,
				'progress-color'         => $colors['progress'] ?? null,
				'progress-stripes-color' => $colors['stripes'] ?? null,
			]
		);
	}

	// Print styles
	kalium_print_inline_style(
		[
			'selector' => '.free-shipping-bar',
			'vars'     => $vars,
		]
	);
}

add_action( 'wp_head', 'kalium_woocommerce_print_free_shipping_bar' );

/**
 * Handle AJAX add to cart requests.
 *
 * @param string     $url
 * @param WC_Product $adding_to_cart
 *
 * @return string
 * @since 4.0
 */
function kalium_woocommerce_single_product_add_to_cart_handler( $url, $adding_to_cart ) {
	if ( ! empty( $_POST['ajax-add-to-cart'] ) ) {
		ob_start();
		woocommerce_mini_cart();
		$mini_cart = ob_get_clean();

		$data = [
			'fragments' => apply_filters(
				'woocommerce_add_to_cart_fragments',
				[
					'div.widget_shopping_cart_content' => '<div class="widget_shopping_cart_content">' . $mini_cart . '</div>',
				]
			),
			'cart_hash' => WC()->cart->get_cart_hash(),
		];

		printf( '<script type="text/template" id="refreshed_cart_fragments">%s</script>', wp_json_encode( $data ) );
	}

	return $url;
}

add_filter( 'woocommerce_add_to_cart_redirect', 'kalium_woocommerce_single_product_add_to_cart_handler', 10, 2 );

/**
 * Off-canvas shop filters.
 *
 * @param array $args
 *
 * @return array
 * @since 4.0
 */
function kalium_woocommerce_off_canvas_shop_filters( $args ) {
	if ( kalium_get_theme_option( 'shop_sidebar_toggle' ) ) {
		$max_viewport_width = apply_filters( 'kalium_off_canvas_shop_filters_max_viewport_width', 991 );

		// Add class
		$args['classes'][] = 'off-canvas-shop-filters';

		// Off-canvas activated
		if ( 'off-canvas' === kalium_get_theme_option( 'shop_sidebar_toggle_action' ) ) {
			$max_viewport_width = null;
			$args['classes'][]  = 'off-canvas-shop-filters--enabled';
		}

		// Off-Canvas header
		add_action(
			'kalium_dynamic_sidebar_before',
			function ( $sidebar_id ) {
				if ( 'shop_sidebar' === $sidebar_id ) {
					kalium_woocommerce_off_canvas_shop_filters_header();
				}
			}
		);

		// Print style
		kalium_print_inline_style(
			[
				'id'       => 'off-canvas-shop-filters',
				'selector' => '.off-canvas-shop-filters .shop-sidebar',
				'media'    => kalium_if_else( is_numeric( $max_viewport_width ), "(max-width: {$max_viewport_width}px)" ),
				'props'    => [
					// Force hide
					'display' => 'none !important',
				],
				'vars'     => [
					'off-canvas-shop-filters' => 'enabled',
				],
			],
		);

		// Set collapsed state initially (prevents flicking effect)
		add_action(
			'wp_footer',
			function () {
				$state_var_name  = kalium_css_var_name( 'off-canvas-shop-filters' );
				$state_toggle_js = <<<EOD
( function() {
	var sidebarContainer = document.querySelector( '.off-canvas-shop-filters .sidebar' );
	
	if ( sidebarContainer && 'enabled' === getComputedStyle( sidebarContainer ).getPropertyValue( '{$state_var_name}' ) ) {
		document.querySelectorAll( '.filters-toggle' ).forEach( function( toggle ) {
			toggle.classList.add( 'filters-toggle--expand' );
		} );
	}
} )();
EOD;
				wp_print_inline_script_tag( $state_toggle_js );
			}
		);
	}

	return $args;
}

add_filter( 'kalium_shop_archive_content_wrapper_args', 'kalium_woocommerce_off_canvas_shop_filters' );

/**
 * Enabled badges.
 *
 * @param array $badges
 *
 * @return array
 * @since 4.0
 */
function kalium_woocommerce_enabled_badges( $badges ) {
	$enabled_badges = kalium_get_theme_option( 'shop_enabled_badges' );

	return array_filter(
		$badges,
		function ( $badge ) use ( $enabled_badges ) {
			return in_array( $badge['type'], $enabled_badges );
		}
	);
}

add_filter( 'kalium_woocommerce_product_badges', 'kalium_woocommerce_enabled_badges' );

/**
 * Setup product navigation on single product page.
 *
 * @since 4.1.2
 */
function kalium_woocommerce_product_navigation_setup() {
	if ( ! kalium_get_theme_option( 'shop_single_product_navigation' ) ) {
		return;
	}

	switch ( kalium_get_theme_option( 'product_navigation_layout' ) ) {
		case 'type-4':
			add_action( 'kalium_woocommerce_single_product_header', 'kalium_woocommerce_product_navigation', 20 );
			break;

		default:
			add_action( 'woocommerce_after_single_product_summary', 'kalium_woocommerce_product_navigation', 5 );
	}
}

add_action( 'woocommerce_before_single_product', 'kalium_woocommerce_product_navigation_setup' );
