<?php
/**
 * Kalium WordPress Theme
 *
 * Starter Site Type - Theme Options class.
 *
 * @author Laborator
 * @link   https://kaliumtheme.com
 */
namespace Kalium\Importer;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Direct access not allowed.
}

class Import_Theme_Options extends Import_Type {

	/**
	 * Get content pack name.
	 *
	 * @return string
	 */
	public function get_name() {
		return 'Site Options';
	}

	/**
	 * Backup current site options and theme mods.
	 */
	public function do_backup() {
		$import_manager       = $this->get_content_pack()->import_manager();
		$backup_manager       = $this->get_content_pack()->backup_manager();
		$theme_options_backup = $backup_manager->get_backup_option( 'theme_options', [] );

		// Loop through each source
		foreach ( $this->get_sources() as $source ) {
			$theme_options_file = $import_manager->get_content_pack_import_source_path( $source['name'] );

			// Check if theme options file exists
			if ( kalium()->filesystem->exists( $theme_options_file ) ) {
				$theme_options = kalium_decode_json( file_get_contents( $theme_options_file ), true );

				// Backup site options
				if ( ! empty( $site_options = kalium_get_array_key( $theme_options, 'site_options' ) ) ) {
					$current_site_options = kalium_get_array_key( $theme_options_backup, 'site_options', [] );

					foreach ( $site_options as $option ) {
						$option_name = kalium_get_array_key( $option, 'name' );

						if ( ! array_key_exists( $option_name, $current_site_options ) && ! is_null( get_option( $option_name, null ) ) ) {
							$current_site_options[ $option_name ] = get_option( $option_name );
						}
					}

					$theme_options_backup['site_options'] = $current_site_options;
				}

				// Backup theme mods
				if ( ! empty( $theme_mods = kalium_get_array_key( $theme_options, 'theme_mods' ) ) ) {
					$current_theme_mods = kalium_get_array_key( $theme_options_backup, 'theme_mods', [] );

					foreach ( $theme_mods as $mod_name => $mod_value ) {
						if ( ! array_key_exists( $mod_name, $current_theme_mods ) && ! is_null( get_theme_mod( $mod_name, null ) ) ) {
							$current_theme_mods[ $mod_name ] = get_theme_mod( $mod_name );
						}
					}

					$theme_options_backup['theme_mods'] = $current_theme_mods;
				}

				// Backup custom CSS
				if ( ! empty( kalium_get_array_key( $theme_options, 'custom_css' ) ) ) {
					$theme_options_backup['custom_css'] = wp_get_custom_css();
				}
			}
		}

		// Save backup option
		if ( ! empty( $theme_options_backup ) ) {
			$backup_manager->update_backup_option( 'theme_options', $theme_options_backup );
		}

		// Reset theme mods
		$this->reset_theme_options();
	}

	/**
	 * Import theme options.
	 */
	public function do_import() {
		parent::do_import();

		// Do not run if there are errors reported or option is unchecked
		if ( $this->errors->has_errors() || ! $this->is_checked() ) {
			return;
		}

		// Vars
		$content_pack    = $this->get_content_pack();
		$import_manager  = $content_pack->import_manager();
		$import_instance = $content_pack->get_import_instance();

		// Loop through each source
		foreach ( $this->get_sources() as $source ) {
			$theme_options_file = $import_manager->get_content_pack_import_source_path( $source['name'] );

			// Check if theme options file exists
			if ( kalium()->filesystem->exists( $theme_options_file ) ) {
				$theme_options = kalium_decode_json( file_get_contents( $theme_options_file ), true );

				// Register theme options
				if ( is_array( $theme_options ) ) {

					// Import site options
					if ( ! empty( $site_options = kalium_get_array_key( $theme_options, 'site_options' ) ) ) {
						foreach ( $site_options as $option ) {
							$option_name  = kalium_get_array_key( $option, 'name' );
							$option_value = kalium_get_array_key( $option, 'value' );

							// Sub prop replacement
							if ( ! empty( $option['sub_props'] ) ) {
								$option_value = get_option( $option_name, [] );

								foreach ( $option['sub_props'] as $sub_prop_name => $sub_prop_value ) {
									$option_value[ $sub_prop_name ] = $sub_prop_value;
								}
							}

							// Update option
							update_option( $option_name, $option_value );
						}
					}

					// Import theme mods
					if ( ! empty( $theme_mods = kalium_get_array_key( $theme_options, 'theme_mods' ) ) ) {
						foreach ( $theme_mods as $mod_name => $mod_value ) {
							set_theme_mod( $mod_name, $mod_value );
						}
					}

					// Set custom CSS
					if ( ! empty( $custom_css = kalium_get_array_key( $theme_options, 'custom_css' ) ) ) {
						wp_update_custom_css_post( $custom_css );
					}

					// Mark as successful import
					$import_instance->set_import_success();
				} else {
					$this->errors->add( 'kalium_importer_theme_options_not_valid', 'Theme options import file is not valid!' );
				}
			} else {

				// Theme options file doesn't exists
				$this->errors->add( 'kalium_importer_theme_options_not_exists', 'Theme options file doesn\'t exists!' );
			}
		}

		// Add errors to import instance
		if ( $this->errors->has_errors() ) {
			$import_instance->add_error( $this->errors );
		}
	}

	/**
	 * Flush rewrite rules and other related stuff.
	 */
	public function do_complete() {
		parent::do_complete();

		// Max Mega Menu clear cache
		if ( class_exists( 'Mega_Menu' ) ) {
			do_action( 'megamenu_delete_cache' );
		}

		// Flush rewrite rules
		flush_rewrite_rules();
	}

	/**
	 * Restore previous theme options.
	 */
	public function do_remove() {
		$backup_manager       = $this->get_content_pack()->backup_manager();
		$theme_options_backup = $backup_manager->get_backup_option( 'theme_options', [] );

		// Revert site options
		if ( ! empty( $site_options = kalium_get_array_key( $theme_options_backup, 'site_options' ) ) ) {
			foreach ( $site_options as $option_name => $option_value ) {
				update_option( $option_name, $option_value );
			}
		}

		// Revert theme mods
		$this->reset_theme_options();

		if ( ! empty( $theme_mods = kalium_get_array_key( $theme_options_backup, 'theme_mods' ) ) ) {
			foreach ( $theme_mods as $mod_name => $mod_value ) {
				set_theme_mod( $mod_name, $mod_value );
			}
		}

		// Revert custom CSS
		if ( ! empty( $custom_css = kalium_get_array_key( $theme_options_backup, 'custom_css' ) ) ) {
			wp_update_custom_css_post( $custom_css );
		}

		// Delete theme options backup value
		$backup_manager->delete_backup_option( 'theme_options' );

		// Flush rewrite rules
		flush_rewrite_rules();

		// Mark as removed
		parent::do_remove();
	}

	/**
	 * Reset theme options.
	 * 
	 * @since 4.4
	 */
	private function reset_theme_options() {
		foreach ( kalium()->theme_options->get_all() as $id => $args ) {
			remove_theme_mod( $id );
		}
	}
}
