<?php
/**
 * Kalium WordPress Theme
 *
 * Starter Sites Type - WordPress Widgets class.
 *
 * @author Laborator
 * @link   https://kaliumtheme.com
 */
namespace Kalium\Importer;

use WP_Term;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Direct access not allowed.
}

class Import_WordPress_Widgets extends Import_Type {

	/**
	 * Get content pack name.
	 *
	 * @return string
	 */
	public function get_name() {
		return 'Widgets';
	}

	/**
	 * Backup widgets.
	 *
	 * @return void
	 */
	public function do_backup() {
		$backup_manager = $this->get_content_pack()->backup_manager();

		// Backup current widgets
		$backup_manager->update_backup_option( 'widgets', wp_get_sidebars_widgets() );
	}

	/**
	 * Import WordPress widgets.
	 *
	 * @return void
	 */
	public function do_import() {
		parent::do_import();

		// Do not run if there are errors reported or option is unchecked
		if ( $this->errors->has_errors() || ! $this->is_checked() ) {
			return;
		}

		// Widget importer functions
		if ( ! function_exists( '_wie_available_widgets' ) ) {
			require_once __DIR__ . '/plugins/widget-importer-exporter.php';
		}

		// Vars
		$content_pack    = $this->get_content_pack();
		$import_manager  = $content_pack->import_manager();
		$import_instance = $content_pack->get_import_instance();

		// Delete widgets
		$sidebars_widgets = wp_get_sidebars_widgets();

		foreach ( $sidebars_widgets as $sidebar => $widgets ) {
			if ( in_array( $sidebar, [ 'wp_inactive_widgets', 'array_version' ] ) ) {
				continue;
			}

			unset( $sidebars_widgets[ $sidebar ] );
		}

		wp_set_sidebars_widgets( $sidebars_widgets );

		// Loop through each source
		foreach ( $this->get_sources() as $source ) {

			// WordPress Widgets
			$widgets_file = $import_manager->get_content_pack_import_source_path( $source['name'] );

			// Check if widgets file exists
			if ( kalium()->filesystem->exists( $widgets_file ) ) {
				$widgets = kalium_decode_json( file_get_contents( $widgets_file ), true );

				// Import WordPress Widgets
				_wie_import_data( (object) $widgets );

				// Mark as successful import
				$import_instance->set_import_success();
			} else {

				// Widgets file doesn't exists
				$this->errors->add( 'kalium_importer_wordpress_widgets_not_exists', 'Widgets file doesn\'t exists!' );
			}
		}

		// Add errors to import instance
		if ( $this->errors->has_errors() ) {
			$import_instance->add_error( $this->errors );
		}
	}

	/**
	 * Adjust widgets settings and map menu ids.
	 *
	 * @return void
	 */
	public function do_complete() {
		$processed_terms = $this->get_content_pack()->backup_manager()->get_backup_option( 'processed_terms' );

		// Fix legacy widget hashes in widget_block widgets
		$widget_blocks = get_option( 'widget_block', [] );

		if ( ! empty( $widget_blocks ) ) {
			$updated = false;

			// Closure to process legacy widget blocks recursively
			$process_legacy_widget_blocks = function ( &$blocks ) use ( &$process_legacy_widget_blocks, $processed_terms ) {
				$modified = false;

				foreach ( $blocks as &$block ) {
					if ( 'core/legacy-widget' === $block['blockName'] && isset( $block['attrs']['instance']['encoded'] ) ) {
						$encoded             = $block['attrs']['instance']['encoded'];
						$serialized_instance = base64_decode( $encoded );

						// Also set the correct nav_menu ID
						$instance = maybe_unserialize( $serialized_instance );

						if ( isset( $instance['nav_menu'] ) && isset( $processed_terms[ $instance['nav_menu'] ] ) ) {
							$instance['nav_menu'] = $processed_terms[ $instance['nav_menu'] ];

							$serialized_instance = serialize( $instance );
						}

						// Reset the hash
						$correct_hash = wp_hash( $serialized_instance );

						if ( isset( $block['attrs']['instance']['hash'] ) && $block['attrs']['instance']['hash'] !== $correct_hash ) {
							$block['attrs']['instance']['hash'] = $correct_hash;

							$modified = true;
						}
					}

					// Process inner blocks recursively
					if ( ! empty( $block['innerBlocks'] ) ) {
						if ( $process_legacy_widget_blocks( $block['innerBlocks'] ) ) {
							$modified = true;
						}
					}
				}

				return $modified;
			};

			foreach ( $widget_blocks as & $widget_data ) {
				if ( ! isset( $widget_data['content'] ) ) {
					continue;
				}

				// Parse blocks from content
				$blocks = parse_blocks( $widget_data['content'] );

				if ( $process_legacy_widget_blocks( $blocks ) ) {
					$widget_data['content'] = serialize_blocks( $blocks );

					$updated = true;
				}
			}

			if ( $updated ) {
				update_option( 'widget_block', $widget_blocks );
			}
		}

		// Fix legacy nav menu widget menu ids
		$widget_nav_menu = get_option( 'widget_nav_menu', [] );

		if ( ! empty( $widget_nav_menu ) ) {
			$updated = false;

			foreach ( $widget_nav_menu as & $widget_data ) {
				if ( isset( $widget_data['nav_menu'] ) && isset( $processed_terms[ $widget_data['nav_menu'] ] ) ) {
					$widget_data['nav_menu'] = $processed_terms[ $widget_data['nav_menu'] ];

					$updated = true;
				}
			}

			if ( $updated ) {
				update_option( 'widget_nav_menu', $widget_nav_menu );
			}
		}
		// Mark as successful task
		parent::do_complete();
	}

	/**
	 * Remove imported widgets.
	 */
	public function do_remove() {
		$backup_manager = $this->get_content_pack()->backup_manager();

		// Restore widgets
		if ( $widgets = $backup_manager->get_backup_option( 'widgets' ) ) {
			wp_set_sidebars_widgets( $widgets );

			$backup_manager->delete_backup_option( 'widgets' );
		}

		// Mark as removed
		parent::do_remove();
	}
}
