<?php
/**
 * Kalium WordPress Theme
 *
 * Kalium WP Import class.
 *
 * @author Laborator
 * @link   https://kaliumtheme.com
 */
namespace Kalium\Importer;

use WP_Error;
use WP_Import;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Direct access not allowed.
}

// Load importers
define( 'WP_LOAD_IMPORTERS', true );

// Load WordPress Importer
require_once __DIR__ . '/plugins/wordpress-importer/wordpress-importer.php';

class Kalium_WP_Import extends WP_Import {

	/**
	 * Import instance ID meta key.
	 *
	 * @const string
	 */
	const IMPORT_INSTANCE_ID_META_KEY = '_content_pack_import_id';

	/**
	 * Content pack URL.
	 *
	 * @var string
	 */
	public $content_pack_url;

	/**
	 * Constructor.
	 *
	 * @param string $import_instance_id
	 * @param bool   $fetch_attachments
	 * @param string $content_pack_url
	 */
	public function __construct( $import_instance_id, $fetch_attachments = true, $content_pack_url = null ) {
		parent::__construct();

		// Import instance ID
		$this->id = $import_instance_id;

		// Set fetch attachments
		$this->fetch_attachments = $fetch_attachments;

		// Content pack URL
		$this->content_pack_url = $content_pack_url;

		// Set '_content_pack_import_id' meta for posts, terms and comment terms with $import_instance_id
		add_filter( 'wp_import_post_meta', [ $this, 'set_import_instance_id_meta' ] );
		add_filter( 'wp_import_term_meta', [ $this, 'set_import_instance_id_meta' ] );
		add_filter( 'wp_import_post_comments', [ $this, 'set_import_instance_id_comment_meta' ] );

		// Attribute imported content to the current user
		add_action( 'import_start', [ $this, 'clear_authors' ] );

		// WooCommerce importer compatibility
		add_action( 'import_start', [ $this, '_woocommerce_post_importer_compatibility' ] );

		// Sort terms by ID in ascending order
		add_action( 'import_start', [ $this, 'sort_terms_by_id' ], 0 );
	}

	/**
	 * Sort terms by their ID in ascending order.
	 */
	public function sort_terms_by_id() {
		$sort_by_term_id = function ( $a, $b ) {
			return $a['term_id'] - $b['term_id'];
		};

		if ( ! empty( $this->terms ) ) {
			usort( $this->terms, $sort_by_term_id );
		}

		// Sort categories by ID
		if ( ! empty( $this->categories ) ) {
			usort( $this->categories, $sort_by_term_id );
		}

		// Sort tags by ID
		if ( ! empty( $this->tags ) ) {
			usort( $this->tags, $sort_by_term_id );
		}
	}

	/**
	 * Check if the current file is already downloaded.
	 *
	 * @param string $url
	 * @param array  $post
	 *
	 * @return array|WP_Error
	 */
	public function fetch_remote_file( $url, $post ) {
		// Check if current file already exists
		if ( ! empty( $this->content_pack_url ) ) {
			$relative_path = str_replace( $this->content_pack_url . '/', '', $url );
			$file_path     = ABSPATH . $relative_path;

			if ( file_exists( $file_path ) ) {
				$file_name = basename( $relative_path );
				$file_url  = site_url( $relative_path );

				$wp_filetype = wp_check_filetype_and_ext( $file_path, $file_name );

				$this->url_remap[ $url ]          = $file_url;
				$this->url_remap[ $post['guid'] ] = $file_url;

				return [
					'file'  => $file_path,
					'url'   => $file_url,
					'type'  => $wp_filetype['type'],
					'error' => false,
				];
			}
		}

		return parent::fetch_remote_file( $url, $post );
	}

	/**
	 * Custom handling for menu items.
	 *
	 * @param {array} $item
	 *
	 * @since 4.2.5
	 */
	public function process_menu_item( $item ) {
		parent::process_menu_item( $item );

		if ( ! isset( $this->processed_menu_items[ $item['post_id'] ] ) ) {
			return;
		}

		$menu_id   = $this->processed_menu_items[ $item['post_id'] ];
		$post_meta = wp_list_pluck( $item['postmeta'], 'value', 'key' );

		// Import Mega Menu settings
		if ( isset( $post_meta['_megamenu'] ) ) {
			update_post_meta( $menu_id, '_megamenu', maybe_unserialize( $post_meta['_megamenu'] ) );
		}
	}

	/**
	 * Assign import instance ID for post (or terms) meta.
	 *
	 * @param array $meta
	 *
	 * @return array
	 */
	public function set_import_instance_id_meta( $meta ) {
		$meta[] = [
			'key'   => self::IMPORT_INSTANCE_ID_META_KEY,
			'value' => $this->id,
		];

		return $meta;
	}

	/**
	 * Assign import instance ID for comments.
	 *
	 * @param array $comments
	 *
	 * @return array
	 */
	public function set_import_instance_id_comment_meta( $comments ) {
		foreach ( $comments as & $comment ) {
			$comment['commentmeta'] = $this->set_import_instance_id_meta( $comment['commentmeta'] );
		}

		return $comments;
	}

	/**
	 * Clear authors from import file.
	 *
	 * @return void
	 */
	public function clear_authors() {
		$this->authors = [];
	}

	/**
	 * When running the WP XML importer, ensure attributes exist.
	 *
	 * WordPress import should work - however, it fails to import custom product attribute taxonomies.
	 * This code grabs the file before it is imported and ensures the taxonomies are created.
	 *
	 * Borrowed from WooCommerce (4.0.1)
	 * ./woocommerce/includes/admin/class-wc-admin-importers.php:147
	 */
	public function _woocommerce_post_importer_compatibility() {

		// Only when WooCommerce plugin is active
		if ( ! kalium()->is->woocommerce_active() ) {
			return;
		}

		if ( ! empty( $this->posts ) ) {
			foreach ( $this->posts as $post ) {
				if ( 'product' === $post['post_type'] && ! empty( $post['terms'] ) ) {
					foreach ( $post['terms'] as $term ) {
						if ( strstr( $term['domain'], 'pa_' ) ) {
							if ( ! taxonomy_exists( $term['domain'] ) ) {
								$attribute_name = wc_attribute_taxonomy_slug( $term['domain'] );

								// Create the taxonomy.
								if ( ! in_array( $attribute_name, wc_get_attribute_taxonomies(), true ) ) {
									wc_create_attribute(
										[
											'name'         => $attribute_name,
											'slug'         => $attribute_name,
											'type'         => 'select',
											'order_by'     => 'menu_order',
											'has_archives' => false,
										]
									);
								}

								// Register the taxonomy now so that the import works!
								register_taxonomy(
									$term['domain'],
									apply_filters( 'woocommerce_taxonomy_objects_' . $term['domain'], [ 'product' ] ),
									apply_filters(
										'woocommerce_taxonomy_args_' . $term['domain'],
										[
											'hierarchical' => true,
											'show_ui'      => false,
											'query_var'    => true,
											'rewrite'      => false,
										]
									)
								);
							}
						}
					}
				}
			}
		}
	}
}
