<?php
/**
 * WooCommerce Product Description element.
 */

namespace Laborator_Builder\Elements;

use Laborator_Builder\Element;
use Laborator_Builder\Attribute;
use Laborator_Builder\Attribute_Sets\Advanced_Border_Set;
use Laborator_Builder\Attribute_Sets\Advanced_Custom_CSS_Set;
use Laborator_Builder\Attribute_Sets\Advanced_Dimensions_Set;
use Laborator_Builder\Attribute_Sets\Advanced_Layout_Set;
use Laborator_Builder\Attribute_Sets\Advanced_Text_Set;
use Laborator_Builder\Attribute_Sets\Advanced_Visibility_Hover_Set;
use Laborator_Builder\Attribute_Sets\Product_Common_Attributes_Set;

class WooCommerce_Product_Description extends Element {

	/**
	 * Name.
	 *
	 * @var string
	 */
	public static $name = 'woocommerce_product_description';

	/**
	 * Title.
	 *
	 * @var string
	 */
	public static $title = 'Description';

	/**
	 * Icon.
	 *
	 * @var string
	 */
	public static $icon = 'postContent';

	/**
	 * Maximum instances of element.
	 *
	 * @var int
	 * @static
	 */
	public static $max_instances = 1;

	/**
	 * Get attribute sets.
	 *
	 * @return array
	 */
	public function get_attribute_sets() {
		return array_merge(
			parent::get_attribute_sets(),
			[
				Advanced_Layout_Set::class,
				Advanced_Visibility_Hover_Set::class,
				Advanced_Border_Set::class,
				Advanced_Text_Set::class,
				Advanced_Dimensions_Set::class,
				Advanced_Custom_CSS_Set::class,
				Product_Common_Attributes_Set::class,
			]
		);
	}

	/**
	 * Attributes.
	 *
	 * @return Attribute[]
	 */
	public function create_attributes() {
		return array_merge(
			parent::create_attributes(),
			[
				// Content / Content
				Attribute::tab_section(
					Attribute::TAB_CONTENT,
					Attribute::SECTION_CONTENT,
					[
						Attribute::create(
							'max_words',
							[
								'type'         => 'quantity',
								'title'        => 'Max. words',
								'inline_label' => true,
								'min'          => 5,
							]
						),
						Attribute::create(
							'max_words_more',
							[
								'type'              => 'text',
								'title'             => 'Read more',
								'inline_label'      => true,
								'tooltip'           => 'What to append if text needs to be trimmed',
								'tooltip_placement' => 'top-end',
								'placeholder'       => '...',
								'dependency'        => [
									'max_words' => [
										'validate' => 'is_numeric',
									],
								],
							]
						),
						Attribute::create(
							'max_lines',
							[
								'type'         => 'quantity',
								'title'        => 'Max. lines',
								'inline_label' => true,
								'min'          => 1,
								'default'      => 3,
							]
						),
					]
				),

				// Style / Show More Button
				Attribute::tab_section(
					Attribute::TAB_STYLE,
					'Show More Button',
					[
						Attribute::create(
							'show_more_type',
							[
								'type'         => 'select',
								'title'        => 'Button type',
								'inline_label' => true,
								'choices'      => [
									'fade-gradient' => 'Fade gradient',
									'button'        => 'Button',
								],
							]
						),
						Attribute::create(
							'show_more_fade_gradient',
							[
								'type'         => 'color',
								'title'        => 'Gradient',
								'inline_label' => true,
								'separator'    => false,
								'choices'      => [
									'start' => 'Start',
									'end'   => 'End',
								],
								'dependency'   => [
									'show_more_type' => 'fade-gradient',
								],
							]
						),
						Attribute::create(
							'show_more_button',
							[
								'type'         => 'color',
								'title'        => 'Background',
								'inline_label' => true,
								'separator'    => false,
								'choices'      => [
									'background' => 'Button background',
									'shadow'     => 'Box shadow',
								],
								'dependency'   => [
									'show_more_type' => 'button',
								],
							]
						),
						Attribute::create(
							'show_more_dots_color',
							[
								'type'         => 'color',
								'title'        => 'Dots color',
								'inline_label' => true,
							]
						),
					]
				),
			]
		);
	}

	/**
	 * Generate styles.
	 */
	public function generate_styles() {
		parent::generate_styles();

		$this->add_style(
			[
				'&' => [
					kalium_css_var_name( 'expanding-text-lines' )      => $this->get_attribute( 'max_lines' ),
					kalium_css_var_name( 'expanding-text-dots-color' ) => $this->get_attribute( 'show_more_dots_color' ),
				],
			]
		);

		// Button type
		switch ( $this->get_attribute_value( 'show_more_type' ) ) {

			// Fade gradient
			case 'fade-gradient':
				$fade_gradient = $this->get_attribute_value( 'show_more_fade_gradient' );

				$this->add_style(
					[
						'&' => [
							kalium_css_var_name( 'expanding-text-more-bg-start' ) => $fade_gradient['start'] ?? null,
							kalium_css_var_name( 'expanding-text-more-bg-end' )   => $fade_gradient['end'] ?? null,
						],
					]
				);
				break;

			// Button
			case 'button':
				$button_style = $this->get_attribute_value( 'show_more_button' );

				$this->add_style(
					[
						'&' => [
							kalium_css_var_name( 'expanding-text-more-button-bg' )         => $button_style['background'] ?? null,
							kalium_css_var_name( 'expanding-text-more-button-box-shadow' ) => $button_style['shadow'] ?? null,
						],
					]
				);
				break;
		}
	}
}

// Register element
\Laborator_Builder::register_element_type( WooCommerce_Product_Description::class );
