<?php
/**
 * Social Icon base element class.
 */

namespace Laborator_Builder\Elements;

use Laborator_Builder\Element;
use Laborator_Builder\Attribute;

class Social_Icon extends Element {

	/**
	 * Element name.
	 *
	 * @var string
	 * @static
	 */
	public static $name = 'social_icon';

	/**
	 * Dynamic title.
	 *
	 * @var string
	 * @static
	 */
	public static $dynamic_title = '{{ data.attributes.label }}';

	/**
	 * Attributes.
	 *
	 * @return Attribute[]
	 */
	public function create_attributes() {
		return [
			Attribute::create(
				'label',
				[
					'type'         => 'text',
					'title'        => 'Label',
					'inline_label' => true,
					'placeholder'  => $this->get_title(),
				]
			),
			Attribute::create(
				'link',
				[
					'type'         => 'text',
					'title'        => 'Link',
					'inline_label' => true,
					'placeholder'  => 'https://',
				]
			),
			Attribute::create(
				'color',
				[
					'type'         => 'color',
					'title'        => 'Brand Color',
					'inline_label' => true,
					'reset'        => true,
					'default'      => kalium_color_reference( 'theme_colors.color-1' ),
				]
			),
		];
	}

	/**
	 * Init.
	 */
	public function init() {
		parent::init();

		// Display as A href link
		$this->tag_name = 'a';

		// Set href
		$this->dom_attributes['href'] = $this->get_link();

		// Aria label
		$this->dom_attributes['aria-label'] = $this->get_attribute_value( 'label' ) ?: $this->get_title();

		// Rel
		$rel = [];

		// New tab
		if ( $this->get_option( 'new_tab', true ) && ! preg_match( '/^(#|javascript:)/', $this->get_link() ) ) {
			$this->dom_attributes['target'] = '_blank';

			$rel[] = 'noopener';
		}

		// Nofollow
		if ( $this->get_option( 'no_follow' ) ) {
			$rel[] = 'nofollow';
		}

		// Me
		$rel[] = 'me';

		// Set rel
		if ( ! empty( $rel ) ) {
			$this->dom_attributes['rel'] = implode( ' ', $rel );
		}
	}

	/**
	 * Template name.
	 *
	 * @return string
	 */
	public function get_template_name() {
		return 'social_icon';
	}

	/**
	 * Element DOM class.
	 *
	 * @return array
	 */
	public function get_dom_class() {
		$classes = [
			'social-icon',
		];

		return array_merge( parent::get_dom_class(), $classes );
	}

	/**
	 * Generate styles.
	 */
	public function generate_styles() {
		$this->add_style(
			[
				'&' => [
					kalium_css_var_name( 'si-brand-color' ) => $this->get_attribute_value( 'color' ),
				],
			]
		);
	}

	/**
	 * Get social icon.
	 *
	 * @return string
	 */
	public function get_social_icon() {
		$icon_type = $this->get_attribute_value( 'icon_type' );

		// Custom Image
		if ( 'image' === $icon_type ) {
			$image_id = $this->get_attribute_value( 'icon_image' );
			$file     = get_attached_file( $image_id );

			if ( $file ) {

				// Print SVGs inline
				if ( kalium()->is->svg( $file ) ) {
					return file_get_contents( $file );
				}

				// Image
				if ( $image = wp_get_attachment_image( $image_id, 'full' ) ) {
					return $image;
				}
			}
		}
		// Inline HTML
		elseif ( 'inline-html' === $icon_type ) {
			$inline_html = $this->get_attribute_value( 'icon_inline_html' );

			// Backwards compatibility with FontAwesome custom icon
			// Will be removed in future versions of Kalium
			if ( preg_match( '/class="fa(r|s|b)? fa-[a-z]+"/i', $inline_html ) ) {
				kalium_enqueue( 'fontawesome' );
			}

			return $inline_html;
		}

		return kalium_get_icon( str_replace( 'kalium-social-icon-', '', $this->get_icon() ) );
	}

	/**
	 * Get link.
	 *
	 * @return string
	 */
	public function get_link() {
		$link = $this->get_attribute_value( 'link' );

		if ( ! empty( $link ) ) {

			// Email
			if ( 'kalium_social_icon_email' === $this->get_name() ) {
				$link = 'mailto:' . $link;
			} // Phone
			elseif ( 'kalium_social_icon_phone' === $this->get_name() ) {
				$link = 'tel:' . $link;
			}
		} else {
			$link = '#';
		}

		return $link;
	}
}

// Register element
\Laborator_Builder::register_element_type( Social_Icon::class );
