<?php
/**
 * Visibility attributes.
 */

namespace Laborator_Builder\Attribute_Sets;

use Laborator_Builder\Attribute;
use Laborator_Builder\Attribute_Set;

class Advanced_Visibility_Hover_Set extends Attribute_Set {

	/**
	 * Get attributes.
	 *
	 * @return array
	 */
	public function get_attributes() {
		return Attribute::tab_section(
			Attribute::TAB_ADVANCED,
			Attribute::SECTION_VISIBILITY,
			[
				Attribute::create(
					'visibility',
					[
						'type'       => 'radio',
						'title'      => 'Visibility',
						'responsive' => true,
						'reset'      => true,
						'choices'    => [
							'always' => 'Always',
							'hover'  => 'Hover',
							'hidden' => 'Hidden',
						],
						'buttons'    => true,
					]
				),
				Attribute::create(
					'hover_animation',
					[
						'type'         => 'select',
						'title'        => 'Hover animation',
						'inline_label' => true,
						'separator'    => false,
						'choices'      => [
							'none'        => 'None',

							'fade'        => 'Fade',
							'fade-up'     => 'Fade Up',
							'fade-down'   => 'Fade Down',
							'fade-left'   => 'Fade Left',
							'fade-right'  => 'Fade Right',

							'scale-up'    => 'Scale Up',
							'scale-down'  => 'Scale Down',

							'slide-up'    => 'Slide Up',
							'slide-down'  => 'Slide Down',
							'slide-left'  => 'Slide Left',
							'slide-right' => 'Slide Right',
						],
						'dependency'   => [
							'visibility' => 'hover',
						],
					]
				),
				Attribute::create(
					'hover_animation_duration',
					[
						'type'         => 'select',
						'title'        => 'Animation Duration',
						'inline_label' => true,
						'separator'    => false,
						'choices'      => [
							'slow'   => 'Slow',
							'normal' => 'Normal',
							'fast'   => 'Fast',
						],
						'default'      => 'normal',
						'dependency'   => [
							'visibility' => 'hover',
						],
					]
				),
				Attribute::create(
					'hover_animation_delay',
					[
						'type'         => 'text',
						'title'        => 'Animation Delay',
						'inline_label' => true,
						'separator'    => false,
						'numeric'      => true,
						'suffix'       => 'ms',
						'default'      => 'normal',
						'dependency'   => [
							'visibility' => 'hover',
						],
					]
				),
			]
		);
	}

	/**
	 * Get DOM classes.
	 *
	 * @return array
	 */
	public function get_dom_class() {
		$classes = [];
		$element = $this->get_element();

		// Visibility
		$visibility         = $element->get_attribute_value( 'visibility' );
		$visibility_classes = kalium_responsive_classname_infix( $visibility, 'visible' );

		// Visibility classes
		$classes = array_merge( $classes, $visibility_classes );

		// Visible on hover
		kalium_map_responsive_value(
			$visibility,
			function ( $value, $viewport ) use ( &$element, &$classes ) {
				if ( 'hover' !== $value ) {
					return;
				}

				$class_name = 'visible-hover' . ( kalium_get_viewport_media( $viewport ) ? "-{$viewport}" : '' );

				$hover_animation          = $element->get_attribute_value( 'hover_animation' );
				$hover_animation_duration = $element->get_attribute_value( 'hover_animation_duration' );

				// Has animation
				if ( 'none' !== $hover_animation ) {
					$classes[] = $class_name . '--animate';

					// Duration
					if ( in_array( $hover_animation_duration, [ 'slow', 'fast' ] ) ) {
						$classes[] = $class_name . '--animate-' . $hover_animation_duration;
					}

					// Animation type
					$classes[] = $class_name . '--' . $hover_animation;

					// Slide animation
					if ( 0 === strpos( $hover_animation, 'slide-' ) ) {
						$classes[] = $class_name . '--slide';
					}
				}
			}
		);

		return $classes;
	}

	/**
	 * Generate styles.
	 */
	public function generate_styles() {
		$element = $this->get_element();

		// Animation delay
		$visibility = $element->get_attribute_value( 'visibility' );

		if ( is_array( $visibility ) && in_array( 'hover', $visibility, true ) ) {
			$hover_animation_delay = $element->get_attribute_value( 'hover_animation_delay' );

			if ( is_numeric( $hover_animation_delay ) && $hover_animation_delay > 0 ) {
				$element->add_style(
					[
						'&' => [
							'transition-delay' => $hover_animation_delay . 'ms',
						],
					]
				);
			}
		}
	}
}
