<?php
/**
 * Background attributes.
 */

namespace Laborator_Builder\Attribute_Sets;

use Laborator_Builder\Attribute;
use Laborator_Builder\Attribute_Set;

class Style_Background extends Attribute_Set {

	/**
	 * Background attributes.
	 *
	 * @return array
	 */
	public function get_attributes() {
		return Attribute::tab_section(
			Attribute::TAB_STYLE,
			'Background',
			[
				// Color
				Attribute::create(
					'bg_color',
					[
						'type'         => 'color',
						'title'        => 'Color',
						'inline_label' => true,
						'reset'        => true,
					]
				),

				// Background Image
				Attribute::create(
					'bg_image',
					[
						'type'         => 'media',
						'title'        => 'Image',
						'inline_label' => true,
						'height'       => 50,
					]
				),

				// Background Image
				Attribute::create(
					'bg_image_resolution',
					[
						'type'         => 'select',
						'title'        => 'Image Resolution',
						'inline_label' => true,
						'choices'      => static function () {
							return kalium_get_image_size_names();
						},
						'default'      => 'full',
					]
				),

				// Position
				Attribute::create(
					'bg_position',
					[
						'type'         => 'select',
						'title'        => 'Position',
						'inline_label' => true,
						'responsive'   => true,
						'choices'      => [
							'default'       => 'Default',
							'center-center' => 'Center Center',
							'center-left'   => 'Center Left',
							'center-right'  => 'Center Right',
							'top-center'    => 'Top Center',
							'top-left'      => 'Top Left',
							'top-right'     => 'Top Right',
							'bottom-center' => 'Bottom Center',
							'bottom-left'   => 'Bottom Left',
							'bottom-right'  => 'Bottom Right',
							'custom'        => 'Custom',
						],
					]
				),
				Attribute::create(
					'bg_position_custom_x',
					[
						'type'         => 'text',
						'title'        => 'X Position',
						'inline_label' => true,
						'responsive'   => true,
						'separator'    => false,
						'units'        => kalium_get_default_units(),
						'dependency'   => [
							'bg_position' => 'custom',
						],
					]
				),
				Attribute::create(
					'bg_position_custom_y',
					[
						'type'         => 'text',
						'title'        => 'Y Position',
						'inline_label' => true,
						'responsive'   => true,
						'separator'    => false,
						'units'        => kalium_get_default_units(),
						'dependency'   => [
							'bg_position' => 'custom',
						],
					]
				),

				// Repeat
				Attribute::create(
					'bg_repeat',
					[
						'type'         => 'select',
						'title'        => 'Repeat',
						'inline_label' => true,
						'responsive'   => true,
						'choices'      => [
							'default'   => 'Default',
							'repeat'    => 'Repeat',
							'repeat-x'  => 'Repeat X',
							'repeat-y'  => 'Repeat Y',
							'no-repeat' => 'No Repeat',
						],
					]
				),

				// Size
				Attribute::create(
					'bg_size',
					[
						'type'         => 'select',
						'title'        => 'Size',
						'inline_label' => true,
						'responsive'   => true,
						'choices'      => [
							'default' => 'Default',
							'auto'    => 'Auto',
							'cover'   => 'Cover',
							'contain' => 'Contain',
							'custom'  => 'Custom',
						],
					]
				),
				Attribute::create(
					'bg_size_custom',
					[
						'type'         => 'text',
						'title'        => 'Custom Size',
						'inline_label' => true,
						'responsive'   => true,
						'separator'    => false,
						'dependency'   => [
							'bg_size' => 'custom',
						],
					]
				),
			]
		);
	}

	/**
	 * Generate styles.
	 */
	public function generate_styles() {
		$element = $this->get_element();

		$bg_style = [
			'background-color' => $element->get_attribute( 'bg_color' ),
		];

		if ( is_array( $bg_image = $element->get_attribute_value( 'bg_image' ) ) && is_array( $image = wp_get_attachment_image_src( $bg_image['id'] ?? null, $element->get_attribute_value( 'bg_image_resolution' ) ) ) ) {
			$bg_style['background-image'] = sprintf( 'url(%s)', esc_url( $image[0] ) );

			// Position
			$bg_position_x = $element->get_attribute_value( 'bg_position_custom_x' );
			$bg_position_y = $element->get_attribute_value( 'bg_position_custom_y' );

			$bg_style['background-position'] = $element->get_attribute( 'bg_position' )->get_value_object()->map(
				function ( $value, $viewport ) use ( $bg_position_x, $bg_position_y ) {
					if ( is_null( $value ) ) {
						return null;
					}

					if ( 'custom' === $value ) {
						return implode(
							' ',
							[
								kalium_get_responsive_value( $bg_position_x, $viewport ),
								kalium_get_responsive_value( $bg_position_y, $viewport ),
							]
						);
					}

					return str_replace( '-', ' ', $value );
				}
			);

			// Repeat
			$bg_style['background-repeat'] = $element->get_attribute( 'bg_repeat' )->get_value_object()->map(
				function ( $value ) {
					if ( 'default' === $value ) {
							return null;
					}

					return $value;
				}
			);

			// Size
			$bg_size_custom = $element->get_attribute_value( 'bg_size_custom' );

			$bg_style['background-size'] = $element->get_attribute( 'bg_size' )->get_value_object()->map(
				function ( $value, $viewport ) use ( $bg_size_custom ) {
					if ( 'custom' === $value ) {
						return kalium_get_responsive_value( $bg_size_custom, $viewport );
					}

					return $value;
				}
			);
		}

		$element->add_style(
			[
				'&' => $bg_style,
			]
		);
	}
}
