<?php
/**
 * Laborator Builder.
 *
 * Base attribute class for element.
 */

namespace Laborator_Builder;

class Attribute implements Attribute_Data {

	/**
	 * Uses exporter.
	 */
	use Exporter;

	/**
	 * Attribute type.
	 *
	 * @var string
	 */
	public $type;

	/**
	 * Name.
	 *
	 * @var string
	 */
	public $name;

	/**
	 * Title.
	 *
	 * @var string
	 */
	public $title;

	/**
	 * Description.
	 *
	 * @var string
	 */
	public $description;

	/**
	 * Responsive value attribute.
	 *
	 * @var bool
	 */
	public $responsive = false;

	/**
	 * Enable reset.
	 *
	 * @var bool
	 */
	public $reset = false;

	/**
	 * Default value.
	 *
	 * @var mixed
	 */
	public $default;

	/**
	 * Attribute value.
	 *
	 * @var Value|Responsive_Value
	 */
	public $value;

	/**
	 * Choices container.
	 *
	 * @var array
	 */
	public $choices = [];

	/**
	 * Tooltip content.
	 *
	 * @var string
	 */
	public $tooltip;

	/**
	 * Tooltip placement.
	 *
	 * @var string
	 */
	public $tooltip_placement = 'top';

	/**
	 * Validate rules.
	 *
	 * @var array
	 */
	public $validate = [];

	/**
	 * Dependency rules.
	 *
	 * @var array
	 */
	public $dependency = [];

	/**
	 * Tab.
	 *
	 * @var string
	 */
	public $tab;

	/**
	 * Section.
	 *
	 * @var string
	 */
	public $section;

	/**
	 * Order/attribute position.
	 *
	 * @var int
	 */
	public $order;

	/**
	 * Inline label.
	 *
	 * @var bool
	 */
	public $inline_label = false;

	/**
	 * Separator line.
	 *
	 * @var bool
	 */
	public $separator = true;

	/**
	 * Translation preference.
	 *
	 * @var bool
	 */
	public $translate = false;

	/**
	 * Contextual.
	 *
	 * If set to true, the value will be not exported.
	 *
	 * @var bool
	 */
	public $contextual = false;

	/**
	 * Custom attribute options.
	 *
	 * @var array
	 */
	public $options = [];

	/**
	 * Element that contains this attribute.
	 *
	 * @var Element
	 */
	public $element;

	/**
	 * Constructor.
	 *
	 * @param string|array $name
	 * @param array        $args
	 */
	public function __construct( $name, $args = [] ) {
		if ( is_array( $name ) ) {
			$args = array_merge( $name, $args );
			$name = $args['name'] ?? null;
		}

		// Set attribute name
		$this->name = $name;

		foreach ( $args as $prop => $value ) {
			if ( in_array( $prop, [ 'name', 'options' ] ) ) {
				continue;
			}

			$this->set_prop( $prop, $value );
		}

		// Initialize attribute value container
		$this->value = $this->is_responsive() ? new Responsive_Value() : new Value();
	}

	/**
	 * Create attribute instance.
	 *
	 * @param string|array $name
	 * @param array        $args
	 *
	 * @return self
	 */
	public static function create( $name, $args = [] ) {
		return new Attribute( $name, $args );
	}

	/**
	 * Create tabbed attributes.
	 *
	 * Syntax:
	 * $attribute_tabs = [
	 *  'Tab title 1' => Attributes[],
	 *  'Tab title 2' => Attributes[],
	 * ];
	 *
	 * @param array $attribute_tabs
	 * @param array $tabs_args
	 *
	 * @return array
	 */
	public static function create_tabbed( $attribute_tabs, $tabs_args = [] ) {
		$tabs_args = wp_parse_args(
			$tabs_args,
			[
				'toggles_size' => null,
				'dependency'   => [],
				'section'      => null,
				'tab'          => null,
			]
		);

		// Grouped attributes
		$attributes = [];

		// Tabs attribute
		$tabs_container = self::create(
			wp_generate_uuid4(),
			[
				'type'       => 'radio',
				'contextual' => true,
				'dependency' => $tabs_args['dependency'],
				'section'    => $tabs_args['section'],
				'tab'        => $tabs_args['tab'],
				'buttons'    => [
					'size' => $tabs_args['toggles_size'],
				],
			]
		);

		// Add tabs toggles
		$attributes[] = $tabs_container;

		// Register tabs and add attributes
		if ( is_array( $attribute_tabs ) ) {
			foreach ( $attribute_tabs as $tab_title => $tab_attributes ) {
				$tab_name = sanitize_title( $tab_title );

				// Tab entry
				$tabs_container->add_choice( $tab_name, $tab_title );

				// Add attributes to tab
				if ( is_array( $tab_attributes ) ) {
					foreach ( $tab_attributes as $attribute ) {
						if ( is_array( $attribute ) ) { // Convert to Attribute
							$attribute = self::create( $attribute );
						}

						// Inherit dependencies from tabs
						$attribute->dependency = array_merge( $attribute->get_dependency_rules(), $tabs_container->get_dependency_rules() );

						// Add dependency for the tab
						$attribute->dependency[ $tabs_container->get_name() ] = $tab_name;

						// Set section and tab
						$attribute->section = $tabs_container->get_section();
						$attribute->tab     = $tabs_container->get_tab();

						// Add to list
						$attributes[] = $attribute;
					}
				}
			}
		}

		return $attributes;
	}

	/**
	 * Tab section define.
	 *
	 * @param string $tab_name
	 * @param string $section_name
	 * @param array  $attributes
	 *
	 * @return array
	 */
	public static function tab_section( $tab_name, $section_name, $attributes ) {
		$grouped_attributes = [];

		if ( is_array( $attributes ) ) {
			foreach ( $attributes as $attribute ) {
				if ( $attribute instanceof Attribute ) {
					$attribute->tab     = $tab_name;
					$attribute->section = $section_name;

					$grouped_attributes[] = $attribute;
				} elseif ( is_array( $attribute ) ) {
					$grouped_attributes = array_merge( $grouped_attributes, self::tab_section( $tab_name, $section_name, $attribute ) );
				}
			}
		}

		return $grouped_attributes;
	}

	/**
	 * Flatten attributes into single array.
	 *
	 * @param array|Attribute[] $attributes
	 */
	public static function flatten_attributes( $attributes ) {
		$flattened = [];

		foreach ( $attributes as $attribute ) {
			if ( $attribute instanceof Attribute ) {
				$flattened[] = $attribute;
			} elseif ( is_array( $attribute ) ) {
				$flattened = array_merge( $flattened, self::flatten_attributes( $attribute ) );
			}
		}

		return $flattened;
	}

	/**
	 * Set attribute prop.
	 *
	 * @param string $prop
	 * @param mixed  $value
	 */
	public function set_prop( $prop, $value ) {

		// Set prop value
		if ( property_exists( $this, $prop ) ) {
			$this->{$prop} = $value;
		} else {
			$this->options[ $prop ] = $value;
		}
	}

	/**
	 * Get attribute type.
	 *
	 * @return string
	 */
	public function get_type() {
		return $this->type;
	}

	/**
	 * Get attribute name.
	 *
	 * @return string
	 */
	public function get_name() {
		return $this->name;
	}

	/**
	 * Get attribute title.
	 *
	 * @return string
	 */
	public function get_title() {
		return $this->title;
	}

	/**
	 * Get attribute description.
	 *
	 * @return string
	 */
	public function get_description() {
		return $this->description;
	}

	/**
	 * Get default value.
	 *
	 * @return mixed
	 */
	public function get_default_value() {
		return $this->default;
	}

	/**
	 * Get tooltip content.
	 *
	 * @return string
	 */
	public function get_tooltip() {
		return $this->tooltip;
	}

	/**
	 * Get tooltip placement.
	 *
	 * @return string
	 */
	public function get_tooltip_placement() {
		return $this->tooltip_placement;
	}

	/**
	 * Get value object (container).
	 *
	 * @return Value|Responsive_Value
	 */
	public function get_value_object() {
		return $this->value;
	}

	/**
	 * Reset value.
	 */
	public function reset_value() {
		$value = $this->get_default_value();

		// Choices
		$choices = $this->get_choices();

		if ( $choices instanceof \Closure ) {
			$choices = $choices();
		}

		if ( ! empty( $choices ) ) {
			$choice_values = array_map(
				function ( $key, $choice ) {
					if ( is_array( $choice ) && isset( $choice['value'] ) ) {
						return $choice['value'];
					}

					return $key;
				},
				array_keys( $choices ),
				$choices
			);

			if ( ! in_array( $value, $choice_values ) && ! in_array( $this->get_type(), [ 'checkbox' ] ) ) {
				$value = array_shift( $choice_values );
			}
		}

		// Set default value
		$this->get_value_object()->assign_value( $value );
	}

	/**
	 * Get value.
	 *
	 * @return mixed
	 */
	public function get_value() {
		$value        = $this->get_default_value();
		$value_object = $this->get_value_object();

		if ( $value_object instanceof Responsive_Value ) {
			$value = $value_object->get_all_values();
		} elseif ( ! is_null( $value_object->value() ) ) {
			$value = $value_object->value();
		}

		/**
		 * Filter attribute value.
		 *
		 * @param mixed $value
		 * @param self  $attribute
		 *
		 * @since 4.2.1
		 */
		return apply_filters( 'laborator_builder_attribute_value', $value, $this );
	}

	/**
	 * Get choices.
	 *
	 * @return array
	 */
	public function get_choices() {
		return $this->choices;
	}

	/**
	 * Add choice.
	 *
	 * @param string $value
	 * @param string $title
	 */
	public function add_choice( $value, $title ) {
		$this->choices[ $value ] = $title;
	}

	/**
	 * Check if attribute has responsive value.
	 *
	 * @return bool
	 */
	public function is_responsive() {
		return $this->responsive;
	}

	/**
	 * Check if attribute can be reset.
	 *
	 * @return bool
	 */
	public function is_resetable() {
		return $this->reset;
	}

	/**
	 * Get attribute tab.
	 *
	 * @return string
	 */
	public function get_tab() {
		return $this->tab;
	}

	/**
	 * Get attribute section.
	 *
	 * @return string
	 */
	public function get_section() {
		return $this->section;
	}

	/**
	 * Get attribute order number.
	 *
	 * @return int
	 */
	public function get_order() {
		return $this->order;
	}

	/**
	 * Check if attribute is using inline label.
	 *
	 * @return bool
	 */
	public function is_inline_label() {
		return $this->inline_label;
	}

	/**
	 * Has separator.
	 *
	 * @return bool
	 */
	public function has_separator() {
		return $this->separator;
	}

	/**
	 * Check if attribute is translatable.
	 *
	 * @return bool
	 */
	public function is_translatable() {
		return $this->translate;
	}

	/**
	 * Get the translation key used to differentiate the attribute string.
	 *
	 * @return string
	 * @since 4.2.1
	 */
	public function get_translation_key_name() {
		$element = $this->get_element();

		return sprintf( '%s_%s_%s', $element->get_name(), $this->get_name(), $element->get_instance_id() );
	}

	/**
	 * Is contextual attribute.
	 *
	 * @return bool
	 */
	public function is_contextual() {
		return $this->contextual;
	}

	/**
	 * Get options.
	 *
	 * @return array
	 */
	public function get_options() {
		return $this->options;
	}

	/**
	 * Get single option.
	 *
	 * @param string $name
	 * @param mixed  $default
	 *
	 * @return mixed
	 */
	public function get_option( $name, $default = null ) {
		return $this->options[ $name ] ?? $default;
	}

	/**
	 * Get validate rules.
	 *
	 * @return array
	 */
	public function get_validate_rules() {
		return $this->validate;
	}

	/**
	 * Get dependency rules.
	 *
	 * @return array
	 */
	public function get_dependency_rules() {
		return $this->dependency;
	}

	/**
	 * Get element that contains this attribute.
	 *
	 * @return Element
	 */
	public function get_element() {
		return $this->element;
	}
}
