<?php
/**
 * Laborator Builder.
 *
 * Builder functions.
 */

use Laborator_Builder\Content_Type;
use Laborator_Builder\Element;
use Laborator_Builder\Parser;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Register builder content type.
 *
 * @param string $id
 * @param array  $args
 */
function laborator_builder_register_content_type( $id, $args = [] ) {
	global $laborator_builder_content_types;

	// Initialize var
	if ( ! isset( $laborator_builder_content_types ) ) {
		$laborator_builder_content_types = [];
	}

	// Add content type
	$laborator_builder_content_types[ $id ] = new Content_Type( $id, $args );
}

/**
 * Get registered content type.
 *
 * @param string $id
 *
 * @return Content_Type|null
 */
function laborator_builder_get_content_type( $id ) {
	global $laborator_builder_content_types;

	laborator_builder_maybe_load_content_type( $id );

	if ( isset( $laborator_builder_content_types[ $id ] ) ) {
		return $laborator_builder_content_types[ $id ];
	}

	return null;
}

/**
 * Maybe load content type.
 *
 * @param string $id
 *
 * @since 1.1.0
 */
function laborator_builder_maybe_load_content_type( $id ) {
	global $laborator_builder_content_types;

	if ( ! isset( $laborator_builder_content_types[ $id ] ) ) {
		$content_type_file = __DIR__ . '/../content-types/' . $id . '/content-type.php';

		if ( file_exists( $content_type_file ) ) {
			$content_type_args = include $content_type_file;

			if ( is_array( $content_type_args ) ) {
				laborator_builder_register_content_type( $id, $content_type_args );
			}
		}
	}
}

/**
 * Render Laborator Builder content.
 *
 * @param array $content_type_args
 * @param array $options
 */
function laborator_builder_render( $content_type_args, $options = [] ) {
	Parser::render_content_type( $content_type_args, $options );
}

/**
 * Enqueue builder library.
 */
function laborator_builder_enqueue_library() {

	// Builder library
	wp_register_script(
		'laborator-builder',
		kalium()->file_url( 'includes/libraries/laborator/laborator-builder/assets/customize-control.min.js' ),
		[
			'backbone',
			'customize-controls',
			'wp-components',
			'wp-util',
			kalium_prefix_script_handle( 'theme-customizer' ),
		],
		Laborator_Builder::$version,
		true
	);

	// Builder style
	wp_register_style(
		'laborator-builder',
		kalium()->file_url( 'includes/libraries/laborator/laborator-builder/assets/laborator-builder.min.css' ),
		[
			'wp-components',
		],
		Laborator_Builder::$version
	);

	// Enqueue scripts and styles
	wp_enqueue_script( 'laborator-builder' );
	wp_enqueue_style( 'laborator-builder' );

	// Other enqueues
	kalium_enqueue( 'theme-icons-default' );
	kalium_enqueue( 'theme-icons-social' );
}

/**
 * Get attribute value from an unparsed element.
 *
 * @param array  $element
 * @param string $attribute_name
 * @param mixed  $default
 *
 * @return mixed
 */
function laborator_builder_get_attribute_value( $element, $attribute_name, $default = null ) {
	if ( ! empty( $element['attributes'] ) ) {
		foreach ( $element['attributes'] as $attribute ) {
			if ( $attribute_name === $attribute['name'] ) {
				return $attribute['value'];
			}
		}
	}

	return $default;
}

/**
 * Traverse element index.
 *
 * @param Element $element
 * @param array   $arr
 *
 * @return array
 */
function laborator_builder_traverse_index( $element, $arr = [] ) {
	array_unshift( $arr, $element->get_index() + 1 );

	if ( $element->get_parent() ) {
		$arr = laborator_builder_traverse_index( $element->get_parent(), $arr );
	}

	return $arr;
}
