<?php
/**
 * Laborator Builder.
 *
 * Parser and renderer.
 */
namespace Laborator_Builder;

class Parser {

	/**
	 * Parse Laborator Builder content.
	 *
	 * @param Content_Type|string $content_type
	 * @param array               $elements
	 * @param array               $options
	 * @param Element             $parent
	 *
	 * @return Elements
	 */
	public static function parse( $content_type, $elements, $options = [], $parent = null ) {
		$elements_list = [];

		// Content Type
		if ( is_string( $content_type ) ) {
			$content_type = laborator_builder_get_content_type( $content_type );
		}

		if ( $content_type instanceof Content_Type && is_array( $elements ) ) {
			foreach ( $elements as $index => $element ) {
				if ( ! empty( $element['name'] ) && ( $element_type = $content_type->get_element( $element['name'] ) ) ) {
					/** @var Element $element_instance */
					$element_instance = new $element_type['constructor'](
						array_merge(
							$element_type['extend'],
							[
								'visible'    => $element['visible'] ?? true,
								'attributes' => $element['attributes'] ?? [],
								'options'    => $options,
								'parent'     => $parent,
								'id'         => $element['id'] ?? null,
								'index'      => $index,
							]
						)
					);

					// Child elements
					if ( ! empty( $element['elements'] ) ) {
						$element_instance->children = self::parse( $content_type, $element['elements'], $options, $element_instance );
					}

					// Init
					$element_instance->init();

					// Add element to array
					$elements_list[] = $element_instance;
				}
			}
		}

		return Elements::create( $elements_list );
	}

	/**
	 * Render elements.
	 *
	 * @param Elements $elements
	 * @param array    $args
	 */
	public static function render_elements( $elements, $args = [] ) {
		if ( ! ( $elements instanceof Elements ) ) {
			return;
		}

		$args = wp_parse_args(
			$args,
			[
				'level'            => 0,
				'check_visibility' => true,
			]
		);

		// Render elements
		foreach ( $elements->get_elements() as $element ) {
			if ( $args['check_visibility'] && ! $element->is_visible() ) { // If element is not visible, skip
				continue;
			}

			// Vars
			$template   = $element->get_template_file();
			$attributes = $element->get_attributes();
			$children   = $element->get_children();

			// Buffer output
			ob_start();

			/**
			 * Before content hook.
			 *
			 * @param Element $element
			 */
			do_action( 'laborator_builder_element_before', $element );

			// Element template or content
			if ( $template ) {
				$template_vars = [
					'element' => $element,
				];

				foreach ( $attributes as $attribute ) {
					$template_vars[ $attribute->get_name() ] = $attribute->get_value();
				}

				self::require_template( $template, $template_vars );
			} else {
				$element->element_content();
			}

			// Render children recursively
			if ( $children instanceof Elements ) {
				self::render_elements(
					$children,
					array_merge(
						$args,
						[
							'level' => $args['level'] + 1,
						]
					)
				);
			}

			/**
			 * After content hook.
			 *
			 * @param Element $element
			 */
			do_action( 'laborator_builder_element_after', $element );

			// Show non-empty content
			if ( $content = ob_get_clean() ) {

				// Element wrapper start
				$element->element_start();

				// Element content
				echo $content;

				// Element wrapper end
				$element->element_end();
			}
		}
	}

	/**
	 * Render Laborator Builder content type.
	 *
	 * @param array $args
	 * @param array $options
	 */
	public static function render_content_type( $args, $options = [] ) {
		static $cached_content, $styles_print;

		$args = wp_parse_args(
			$args,
			[
				'content_type'   => null,
				'content_source' => null,
				'wrapper'        => null,
			]
		);

		// Content Type ID
		$content_type_id = $args['content_type'];

		// Get content type
		if ( $content_type = laborator_builder_get_content_type( $content_type_id ) ) {

			// Content source hash
			$content_source_hash = md5( serialize( $args['content_source'] ) );

			// Cache ID
			$cache_id = $content_type_id . '_' . $content_source_hash;

			// Set content source
			$content_type->set_content_source( $args['content_source'] );

			// Get builder content (or use cached)
			if ( empty( $cached_content[ $cache_id ] ) ) {
				$builder_content = $cached_content[ $cache_id ] = $content_type->get_content();
			} else {
				$builder_content = $cached_content[ $cache_id ];
			}

			// Parse elements
			if ( ! empty( $builder_content['elements'] ) ) {
				$elements = self::parse( $content_type, $builder_content['elements'], $options );
			}

			// Render elements and print styles
			if ( ! empty( $elements ) ) {
				ob_start();

				// Print styles (once)
				if ( empty( $styles_print[ $cache_id ] ) ) {
					self::print_styles( $elements );
					$styles_print[ $cache_id ] = 1;
				}

				// Render elements
				self::render_elements(
					$elements,
					[
						'check_visibility' => $content_type->supports_visibility(),
					]
				);

				// Buffered content
				$rendered = ob_get_clean();

				if ( ! empty( $args['wrapper'] ) ) {
					$rendered = sprintf( $args['wrapper'], $rendered );
				}

				echo $rendered;
			}
		}
	}

	/**
	 * Print styles.
	 *
	 * @param Elements $elements
	 */
	public static function print_styles( $elements ) {
		if ( ! ( $elements instanceof Elements ) ) {
			return;
		}

		/**
		 * Generate styles recursively.
		 *
		 * @param Elements $elements
		 *
		 * @return string[]
		 */
		$generate_styles_recursion = function ( $elements ) use ( &$generate_styles_recursion ) {
			$styles = [];

			foreach ( $elements->get_elements() as $element ) {
				if ( ! $element->is_enabled() || ! $element->is_visible() ) { // If element is not visible or not enabled, skip
					continue;
				}

				// Generate styles
				$element->generate_styles();

				// Add styles
				$styles[] = $element->get_css();

				// Recursion
				$children = $element->get_children();

				if ( $children && $children->has_elements() ) {
					$styles = array_merge( $styles, $generate_styles_recursion( $children ) );
				}
			}

			return array_filter( $styles );
		};

		// Styles
		if ( $styles = implode( '', $generate_styles_recursion( $elements ) ) ) {
			printf( '<style data-lb-style>%s</style>', $styles );
		}
	}

	/**
	 * Require template.
	 *
	 * @param string $template
	 * @param array  $args
	 */
	private static function require_template( $template, $args = [] ) {
		extract( $args, EXTR_SKIP );

		require $template;
	}
}
