<?php
/**
 * Laborator Builder.
 *
 * Styles generator.
 */

namespace Laborator_Builder;

trait Styles {

	/**
	 * Registered styles.
	 *
	 * @var array
	 */
	private $styles = [];

	/**
	 * Custom CSS.
	 *
	 * @var array
	 */
	private $custom_css = [];

	/**
	 * Add style.
	 *
	 * There are two ways how you can add style:
	 *
	 * 1.
	 * $this->add_style( [
	 *   {selector} => [
	 *     {css-prop-name} => {css-prop-value},
	 *     ...
	 *   ]
	 * ] )
	 *
	 * 2.
	 * $this->add_style( [
	 *   {selector} => [
	 *     {css-prop-name} => [
	 *       'value' => {css-prop-value},
	 *       ...
	 *     ],
	 *     ...
	 *   ],
	 * ] )
	 *
	 * {css-prop-value} can be:
	 * a. primitive value (string or number)
	 * b. Value object or Responsive object
	 * c. Attribute object
	 * d. an array of "prop => value" pairs with a, b or c type for example:
	 * [
	 *  'margin-top' => {css-prop-value},
	 *  'margin-bottom' => {css-prop-value},
	 * ]
	 *
	 * @param array $styles
	 */
	public function add_style( $styles ) {
		if ( is_array( $styles ) ) {
			foreach ( $styles as $selector => $props ) {
				$selector = $this->get_selector( $selector );

				if ( is_array( $props ) ) {
					foreach ( $props as $prop_name => $prop_value ) {
						if ( ! isset( $this->styles[ $selector ] ) ) {
							$this->styles[ $selector ] = [];
						}

						$css_value = new CSS_Value( $prop_name, $prop_value );

						$this->styles[ $selector ] = array_merge( $this->styles[ $selector ], $css_value->get_all_props() );
					}
				}
			}
		}
	}

	/**
	 * Get styles for selector.
	 *
	 * @param string $selector
	 *
	 * @return CSS_Value[]|null
	 */
	public function get_styles_for_selector( $selector ) {
		$selector = $this->get_selector( $selector );

		if ( isset( $this->styles[ $selector ] ) ) {
			return $this->styles[ $selector ];
		}

		return null;
	}

	/**
	 * Get single style for selector and prop.
	 *
	 * @param string $prop_name
	 * @param string $selector
	 *
	 * @return CSS_Value|null
	 */
	public function get_style( $prop_name, $selector = null ) {
		$styles = $this->get_styles_for_selector( $selector );

		if ( is_array( $styles ) ) {
			foreach ( $styles as $style ) {
				if ( $prop_name === $style->prop ) {
					return $style;
				}
			}
		}

		return null;
	}

	/**
	 * Remove style by selector and CSS props.
	 *
	 * @param string $selector
	 * @param array  $props
	 */
	public function remove_style( $selector, $props = [] ) {
		$style = $this->get_styles_for_selector( $selector );
		$props = is_string( $props ) ? array_map( 'trim', explode( ',', $props ) ) : $props;

		if ( $style ) {

			// Parsed selector
			$selector = $this->get_selector( $selector );

			// All props
			if ( empty( $props ) ) {
				/** @var CSS_Value $css_value */
				$props = array_map(
					function ( $css_value ) {
						return $css_value->prop;
					},
					$this->styles[ $selector ]
				);
			}

			// Filter styles
			$this->styles[ $selector ] = array_filter(
				$style,
				function ( $css_value ) use ( $props ) {
					/** @var CSS_Value $css_value */
					return ! in_array( $css_value->prop, $props );
				}
			);
		}
	}

	/**
	 * Get styles.
	 *
	 * @return array[]
	 */
	public function get_styles() {
		return $this->styles;
	}

	/**
	 * Add custom CSS.
	 *
	 * @param string                 $namespace
	 * @param string|Value|Attribute $custom_css
	 */
	public function add_custom_css( $namespace, $custom_css ) {
		if ( $custom_css instanceof Attribute || $custom_css instanceof Value ) {
			$custom_css = $custom_css->get_value();
		}

		if ( ! isset( $this->custom_css[ $namespace ] ) ) {
			$this->custom_css[ $namespace ] = $custom_css;
		}
	}

	/**
	 * Remove custom CSS.
	 *
	 * @param string $namespace
	 */
	public function remove_custom_css( $namespace ) {
		if ( isset( $this->custom_css[ $namespace ] ) ) {
			unset( $this->custom_css[ $namespace ] );
		}
	}

	/**
	 * Get Custom CSS styles.
	 *
	 * @return array
	 */
	public function get_custom_css() {
		return $this->custom_css;
	}

	/**
	 * Generate CSS.
	 *
	 * @return string
	 */
	public function get_css() {
		$styles_parsed = $css = [];

		foreach ( $this->get_styles() as $selector => $css_values ) {
			/** @var CSS_Value $css_value */
			foreach ( $css_values as $css_value ) {
				$value = $css_value->value;

				if ( $value instanceof Value ) {
					$value->map(
						function ( $value, $viewport ) use ( &$styles_parsed, $selector, $css_value ) {
							$media  = $this->get_media( $viewport );
							$prop   = $css_value->prop;
							$append = $css_value->get_option( 'append' );

							// Add non-empty value only
							if ( ! empty( $value ) || 0 === $value || '0' === $value ) {
								$styles_parsed[ $media ][ $selector ][ $prop ] = $value;

								// Append extra CSS
								if ( is_array( $append ) && ! empty( $append ) ) {
									$styles_parsed[ $media ][ $selector ] = array_merge( $styles_parsed[ $media ][ $selector ], $append );
								}
							}
						}
					);
				}
			}
		}

		// Generate CSS
		$bracket_open  = '{';
		$bracket_close = '}';

		foreach ( $styles_parsed as $media => $selectors ) {

			// @media
			if ( $media ) {
				$css[] = $media . $bracket_open;
			}

			foreach ( $selectors as $selector => $props ) {

				// .selector
				$css[] = $selector . $bracket_open;

				foreach ( $props as $prop_name => $prop_value ) {

					// Skip self referencing CSS vars
					if ( '--' === substr( $prop_name, 0, 2 ) && kalium_is_self_referencing_css_var( $prop_name, $prop_value ) ) {
						continue;
					}

					// prop: value;
					if ( is_scalar( $prop_value ) ) {
						$css[] = sprintf( '%s: %s;', $prop_name, $prop_value );
					}
				}

				$css[] = $bracket_close;
			}

			if ( $media ) {
				$css[] = $bracket_close;
			}
		}

		// Add custom CSS
		foreach ( $this->get_custom_css() as $custom_css ) {
			if ( ! empty( $custom_css ) ) {
				$css[] = str_replace( '#self', $this->get_selector(), $custom_css );
			}
		}

		return implode( '', $css );
	}

	/**
	 * Get/parse selector.
	 *
	 * @param string $selector
	 *
	 * @return string
	 */
	private function get_selector( $selector = null ) {
		$base_selector = '.' . $this->get_common_class();

		// Parent selector / class namespace
		if ( $parent_selector = $this->get_option( 'parent_selector' ) ) {
			$base_selector = $parent_selector . ' ' . $base_selector;
		}

		// Empty selector is self reference
		if ( empty( $selector ) ) {
			$selector = '&';
		}

		// Assign/replace self reference
		if ( false !== strpos( $selector, '&' ) ) {
			$selector = str_replace( '&', $base_selector, $selector );
		} else { // Otherwise simply append as sub selector
			$selector = $base_selector . ' ' . $selector;
		}

		return trim( $selector );
	}

	/**
	 * Get/parse media.
	 *
	 * @param string $viewport
	 *
	 * @return string
	 */
	private function get_media( $viewport ) {
		if ( $viewport = \Laborator_Builder::get_responsive_viewport( $viewport ) ) {
			$max_width = $viewport['max_width'] ?? null;

			if ( is_numeric( $max_width ) ) {
				return sprintf( '@media (max-width: %1$spx)', $max_width );
			}
		}

		return '';
	}
}
