<?php
/**
 * Laborator Builder - drag and drop layout builder for WP Customizer.
 *
 * @author  Laborator
 * @link    https://laborator.co
 * @version 1.0
 */

use Laborator_Builder\Customize_Control;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Laborator Builder main class.
 */
class Laborator_Builder {

	/**
	 * Version.
	 *
	 * @var string
	 */
	public static $version = '1.1.0';

	/**
	 * Registered elements.
	 *
	 * @var array
	 */
	public static $registered_elements = [];

	/**
	 * Builder instance.
	 *
	 * @var self
	 */
	private static $instance;

	/**
	 * Instantiate.
	 *
	 * @return self
	 */
	public static function instance() {
		if ( is_null( self::$instance ) ) {
			self::$instance = new self();
		}

		return self::$instance;
	}

	/**
	 * Register element type.
	 *
	 * @param string $element_class
	 */
	public static function register_element_type( $element_class ) {
		$element_name = $element_class::$name;
		$class_path   = explode( '\\', $element_class );
		$class_name   = end( $class_path );

		self::$registered_elements[ $element_name ] = [
			'class' => $element_class,
			'name'  => $class_name,
		];
	}

	/**
	 * Get registered element.
	 *
	 * @param string $element_name
	 *
	 * @return array|false
	 */
	public static function get_element( $element_name ) {
		foreach ( self::$registered_elements as $name => $element ) {
			if ( $name === $element_name ) {
				return $element;
			}
		}

		return false;
	}

	/**
	 * Get responsive viewports.
	 *
	 * @return array
	 */
	public static function get_responsive_viewports() {
		return kalium_get_responsive_viewports();
	}

	/**
	 * Get single responsive viewport.
	 *
	 * @param string $viewport_id
	 *
	 * @return array|false
	 */
	public static function get_responsive_viewport( $viewport_id ) {
		return kalium_get_responsive_viewport( $viewport_id );
	}

	/**
	 * Register Laborator Builder control type.
	 *
	 * @param WP_Customize_Manager $wp_customize
	 */
	public function register_control( $wp_customize ) {

		// Control type class
		require_once __DIR__ . '/includes/customize-control.php';

		// Register control type
		$wp_customize->register_control_type( Customize_Control::class );
	}

	/**
	 * Templates.
	 */
	public function templates() {

		?>
		<!-- Tabs -->
		<script type="text/template" id="tmpl-laborator-builder-tabs">
			<ul class="{{ data.tabsLinksClass }}">
				<# _.each( data.tabs, function( tab ) { #>
				<li class="{{ data.tabsLinksClass }}__tab" data-tab-id="{{ tab.id }}">
					<a href="#{{ tab.id }}">{{ tab.title }}</a>
				</li>
				<# } ) #>
			</ul>
			<div class="{{ data.tabsContentClass }}">
				<# _.each( data.tabs, function( tab ) { #>
				<div class="{{ data.tabsContentClass }}__tab" data-tab-id="{{ tab.id }}">
					<# if ( tab.description ) { #>
					<div class="{{ data.tabsContentClass }}__tab__description">{{{ tab.description }}}</div>
					<# } #>
					<div class="{{ data.tabsContentClass }}__tab__content"></div>
				</div>
				<# } ) #>
			</div>
		</script>

		<!-- Templates -->
		<script type="text/template" id="tmpl-laborator-builder-templates">
			<#
			_.each( data.templates, function( template ) {
			var templateClass = data.className + '__template';
			#>
			<div class="{{ templateClass }}" data-template="{{ template.id }}">
				<a href="#">
					<# if ( template.icon ) { #>
					<span class="{{ templateClass }}__icon">
						<img src="{{ template.icon }}" alt="{{ template.name }}"/>
					</span>
					<# } #>

					<span class="{{ templateClass }}__name">
						{{ template.name }}

						<# if ( data.modify ) { #>
						<span class="{{ templateClass }}__name__modify" title="Modify template">
							{{{ data.settingsIcon }}}
						</span>
						<# } #>
					</span>
				</a>
			</div>
			<# } ) #>
		</script>

		<!-- User Templates -->
		<script type="text/template" id="tmpl-laborator-builder-user-templates">
			<#
			_.each( data.templates, function( template ) {
			var templateClass = data.className + '__template';
			var templateActionClass = templateClass + '__action';
			#>
			<div class="{{ templateClass }}" data-template="{{ template.id }}">
				<a href="#" class="{{ templateClass }}__checked">
					{{{ data.checkIcon }}}
				</a>

				<a href="#" class="{{ templateClass }}__name">{{ template.name }}</a>

				<a href="#" class="{{ templateActionClass }} {{ templateActionClass }}--duplicate" title="Duplicate">
					{{{ data.duplicateIcon }}}
				</a>

				<a href="#" class="{{ templateActionClass }} {{ templateActionClass }}--edit" title="Edit">
					{{{ data.editIcon }}}
				</a>

				<a href="#" class="{{ templateActionClass }} {{ templateActionClass }}--delete" title="Delete">
					{{{ data.deleteIcon }}}
				</a>
			</div>
			<# } ) #>

			<div class="{{ data.className }}__add">
				<button type="button" class="button-primary">
					{{{ data.addIcon }}}
					<span class="{{ data.className }}__add__text">Add</span>
				</button>
			</div>
		</script>

		<!-- Edit content -->
		<script type="text/template" id="tmpl-laborator-builder-edit">
			<h3>Template Layout</h3>
			<div class="{{ data.editTemplateContentClass }}"></div>

			<div class="{{ data.editTemplateNameClass }}">
				<h3>Save Template</h3>
				<div class="{{ data.editTemplateNameClass }}__input">
					<input type="text" placeholder="Template name"/>
					<button type="button" class="button-primary">Save</button>
				</div>
			</div>
		</script>

		<!-- Debug Content -->
		<script type="text/template" id="tmpl-laborator-builder-debug-mode">
			<div class="{{ data.className }}__debug-mode">
				<button type="button" class="button copy-template-content">Export Content</button>
			</div>
		</script>

		<!-- Element -->
		<script type="text/template" id="tmpl-laborator-builder-element">
			<#
			var classes = [
				data.containerClass,
			];

			if ( data.isRegion ) {
				classes.push( data.containerClass + '--region' );
			}
			#>
			<li class="{{ classes.join( ' ' ) }}" data-id="{{ data.id }}" data-element="{{ data.name }}">
				<div class="{{ data.headingClass }}">
					<# if ( data.isEnabled ) { #>
					<# if ( data.supportsVisibility ) { #>
					<a href="#" class="{{ data.containerClass }}-visibility-toggle">
						<span class="when-visible">{{{ data.visibilityOnIcon }}}</span>
						<span class="when-invisible">{{{ data.visibilityOffIcon }}}</span>
					</a>
					<# } #>
					<# if ( data.displayElementIcon && data.elementIcon ) { #>
					<span class="{{ data.containerClass }}-icon">{{{ data.elementIcon }}}</span>
					<# } #>
					<# } else { #>
					<span class="{{ data.containerClass }}-disabled" data-tooltip-content="{{ data.disabledMessage }}" data-tooltip-placement="top-start">
						{{{ data.disabledIcon }}}
					</span>
					<# } #>

					<h3 class="{{ data.containerClass }}-title">{{ data.title }}</h3>
					<# if ( data.isChildContainer ) { #>
					<a href="#" class="{{ data.containerClass }}-collapse-status">
						<span class="when-collapsed">{{{ data.expandIcon }}}</span>
						<span class="when-expanded">{{{ data.collapseIcon }}}</span>
					</a>
					<# } #>
					<# if ( ! data.isEnabled ) { #>
					<# } #>
					<# if ( data.isEnabled && data.hasAttributes ) { #>
					<a href="#" class="{{ data.containerClass }}-attributes-toggle">
						{{{ data.settingsIcon }}}
					</a>
					<# } #>
					<# if ( data.isDeletable ) { #>
					<a href="#" class="{{ data.containerClass }}-remove">
						{{{ data.deleteIcon }}}
					</a>
					<# } #>
				</div>
			</li>
		</script>

		<!-- Root Element -->
		<script type="text/template" id="tmpl-laborator-builder-root-element">
			<div class="{{ data.containerClass }}"></div>
		</script>

		<!-- Elements container footer -->
		<script type="text/template" id="tmpl-laborator-builder-elements-footer">
			<a href="#" class="{{ data.addElementClass }} {{ data.addElementClass }}-button">
				{{{ data.addElementIcon }}}
				<span class="{{ data.addElementClass }}__text">{{ data.addElementLabel }}</span>
			</a>
		</script>

		<!-- Elements container footer for Root element -->
		<script type="text/template" id="tmpl-laborator-builder-elements-footer-root">
			<a href="#" class="button-primary {{ data.addElementClass }}">
				{{{ data.addElementIcon }}}
				<span class="{{ data.addElementClass }}__text">{{ data.addElementLabel }}</span>
			</a>
		</script>

		<!-- Add element list -->
		<script type="text/template" id="tmpl-laborator-builder-add-element-list">
			<ul>
				<#
				_.each( data.elements, function( element ) {
				var title = element.title || element.name;
				var classes = [ data.elementEntryClass ];

				if ( ! element.enabled ) {
				classes.push( data.elementEntryClass + '--disabled' );
				}
				#>
				<# if ( element.category ) { #>
				<li class="category">
					<h3>{{ element.category }}</h3>
				</li>
				<# } #>
				<li>
					<a href="#" class="{{ classes.join( ' ' ) }}"<# if ( element.name ) { #> data-element="{{ element.name }}"<# } #><# if ( element.variationId ) { #> data-element-variation="{{ element.variationId }}"<# } #>>
						<# if ( element.icon ) { #>
						<span class="{{ data.elementEntryClass }}__icon">
							{{{ element.icon }}}
						</span>
						<# } #>

						<# if ( title ) { #>
						<span class="{{ data.elementEntryClass }}__title">{{ title }}</span>
						<# } #>

						<# if ( ! element.enabled ) { #>
						<span class="{{ data.elementEntryClass }}__disabled" data-tooltip-content="{{ element.disabledMessage }}">{{{ data.disabledIcon }}}</span>
						<# } #>
					</a>
				</li>
				<# } ); #>
			</ul>
		</script>

		<!-- Attribute tabs -->
		<script type="text/template" id="tmpl-laborator-builder-attribute-tabs">
			<ul class="{{ data.containerClass }}">
				<# _.each( data.tabs, function( tabName, tabId ) { #>
				<li class="{{ data.containerClass }}__tab" data-tab="{{ tabId }}">
					<a href="#{{ tabId }}">{{ tabName }}</a>
				</li>
				<# } ); #>
			</ul>
			<div class="{{ data.containerClass }}-content">
				<# _.each( data.tabs, function( tabName, tabId ) { #>
				<div class="{{ data.containerClass }}-content__tab" data-tab="{{ tabId }}"></div>
				<# } ); #>
			</div>
		</script>

		<!-- Attribute section -->
		<script type="text/template" id="tmpl-laborator-builder-attribute-section">
			<div class="{{ data.containerClass }}" data-section="{{ data.id }}">
				<h4><a href="#{{ data.id }}">{{ data.title }}</a></h4>
				<div class="{{ data.containerClass }}-content"></div>
			</div>
		</script>

		<!-- Attribute wrapper -->
		<script type="text/template" id="tmpl-laborator-builder-attribute-wrapper">
			<# var inputStyle = data.inputStyle ? ( ' style="' + data.inputStyle + '"' ) : ''; #>

			<ul class="{{ data.notificationsContainerClass }}"></ul>
			<div class="{{ data.attributeClass }}-label-input-wrapper">
				<# if ( data.title ) { #>
				<div class="{{ data.attributeClass }}-label">
					<label class="{{ data.attributeClass }}-title" for="{{ data.inputId }}">{{ data.title }}</label>
					<div class="{{ data.responsiveControlsClass }}"></div>
					<# if ( data.reset ) { #>
					<a href="#" class="{{ data.resetValueClass }}" data-tooltip-content="Restore default">
						{{{ data.resetValueIcon }}}
					</a>
					<# } #>
				</div>
				<# } #>
				<div class="{{ data.inputWrapperClass }}"{{{ inputStyle }}}></div>
			</div>
			<# if ( data.description ) { #>
			<p class="{{ data.attributeClass }}-description">{{{ data.description }}}</p>
			<# } #>
		</script>

		<!-- Responsive control -->
		<script type="text/template" id="tmpl-laborator-builder-responsive-control">
			<#
			var tooltipContent = data.icon ? data.title : '';
			#>
			<a href="#{{ data.id }}" class="{{ data.className }}" data-viewport="{{ data.id }}" data-tooltip-content="{{ tooltipContent }}" data-tooltip-placement="right">
				<# if ( data.icon ) { #>
				{{{ data.icon }}}
				<# } else { #>
				{{ data.title }}
				<# } #>
			</a>
		</script>

		<!-- Notification template -->
		<script type="text/template" id="tmpl-laborator-builder-notification">
			<div class="notification-message">{{ data.message }}</div>
			<# if ( data.dismissible ) { #>
			<button type="button" class="notification-dismiss">
				{{{ data.dismissIcon }}}
			</button>
			<# } #>
		</script>
		<?php
	}

	/**
	 * Init builder.
	 */
	public function init() {
		$class_name = self::class;

		// Functions
		require_once __DIR__ . '/includes/functions.php';

		// Autoload core classes
		spl_autoload_register(
			kalium_class_autoloader(
				[
					'namespace' => $class_name,
					'base_dir'  => __DIR__ . '/includes',
				]
			)
		);

		/**
		 * Elements.
		 */

		// Core elements
		require_once __DIR__ . '/elements/core/row.php';
		require_once __DIR__ . '/elements/core/container-row.php';
		require_once __DIR__ . '/elements/core/column.php';
		require_once __DIR__ . '/elements/core/text.php';
		require_once __DIR__ . '/elements/core/sidebar.php';

		// Query Loop elements
		require_once __DIR__ . '/elements/query-loop/woocommerce/product-hover-block.php';
		require_once __DIR__ . '/elements/query-loop/woocommerce/product-row.php';
		require_once __DIR__ . '/elements/query-loop/woocommerce/product-column.php';
		require_once __DIR__ . '/elements/query-loop/woocommerce/product-title.php';
		require_once __DIR__ . '/elements/query-loop/woocommerce/product-description.php';
		require_once __DIR__ . '/elements/query-loop/woocommerce/product-category.php';
		require_once __DIR__ . '/elements/query-loop/woocommerce/product-price.php';
		require_once __DIR__ . '/elements/query-loop/woocommerce/product-add-to-cart.php';
		require_once __DIR__ . '/elements/query-loop/woocommerce/product-images.php';
		require_once __DIR__ . '/elements/query-loop/woocommerce/product-sale-badge.php';
		require_once __DIR__ . '/elements/query-loop/woocommerce/product-attribute.php';
		require_once __DIR__ . '/elements/query-loop/woocommerce/product-wishlist.php';
		require_once __DIR__ . '/elements/query-loop/woocommerce/product-swap-on-hover.php';
		require_once __DIR__ . '/elements/query-loop/woocommerce/product-quick-view.php';

		// Footer elements
		require_once __DIR__ . '/elements/footer/footer-row-container.php';
		require_once __DIR__ . '/elements/footer/footer-sidebar.php';

		// Legacy header elements
		require_once __DIR__ . '/elements/legacy-header/legacy-header-base-element.php';
		require_once __DIR__ . '/elements/legacy-header/legacy-header-row.php';
		require_once __DIR__ . '/elements/legacy-header/legacy-header-menu.php';
		require_once __DIR__ . '/elements/legacy-header/legacy-header-menu-mobile.php';
		require_once __DIR__ . '/elements/legacy-header/legacy-header-menu-trigger.php';
		require_once __DIR__ . '/elements/legacy-header/legacy-header-raw-text.php';
		require_once __DIR__ . '/elements/legacy-header/legacy-header-search-field.php';
		require_once __DIR__ . '/elements/legacy-header/legacy-header-search-input.php';
		require_once __DIR__ . '/elements/legacy-header/legacy-header-social-icons.php';
		require_once __DIR__ . '/elements/legacy-header/legacy-header-button.php';
		require_once __DIR__ . '/elements/legacy-header/legacy-header-date-time.php';
		require_once __DIR__ . '/elements/legacy-header/legacy-header-breadcrumb.php';
		require_once __DIR__ . '/elements/legacy-header/legacy-header-wpml-language-switcher.php';
		require_once __DIR__ . '/elements/legacy-header/legacy-header-woocommerce-mini-cart.php';
		require_once __DIR__ . '/elements/legacy-header/legacy-header-woocommerce-cart-totals.php';
		require_once __DIR__ . '/elements/legacy-header/legacy-header-woocommerce-my-account-link.php';

		// Social Sharing Link element
		require_once __DIR__ . '/elements/social-share-link.php';

		// Social Icon element
		require_once __DIR__ . '/elements/social-icon.php';

		/**
		 * Hook: Laborator Builder init.
		 */
		do_action( 'laborator_builder_init' );
	}

	/**
	 * Constructor.
	 */
	private function __construct() {
		add_action( 'after_setup_theme', [ $this, 'init' ] );
		add_action( 'customize_register', [ $this, 'register_control' ] );
		add_action( 'customize_controls_print_scripts', [ $this, 'templates' ] );
	}
}

// Create instance
Laborator_Builder::instance();
