<?php
/**
 * TypoLab - ultimate font management library.
 *
 * Data holder class.
 *
 * @author Laborator
 * @link   https://laborator.co
 */
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class TypoLab_Data {

	/**
	 * Get font sources.
	 *
	 * @return array
	 */
	public static function get_font_sources() {
		return apply_filters(
			'typolab_font_sources',
			[
				'google'    => [
					'name'        => 'Google Fonts',
					'description' => "Google's free font directory is one of the most exciting developments in web typography in a very long time.\n\nGoogle Fonts catalog are published under licenses that allow you to use them on any website, whether it’s commercial or personal.\n\nChoose between <strong>1600+</strong> available fonts to use with your site.",
				],
				'laborator' => [
					'name'        => 'Premium Fonts',
					'description' => "Premium fonts worth of <strong>$149</strong> (per site) are available for Laborator customers only.\n\nIt has the same installation procedures as Google Fonts, you need to download and install fonts that you want to use in this site.\n\nTheme registration is required in order to install and use fonts from this source.",
				],
				'adobe'     => [
					'name'        => 'Adobe Fonts',
					'description' => "Adobe Fonts is a subscription service for fonts which you can use on a website.\n\nInstead of licensing individual fonts, you can sign up for the plan that best suits your needs and get a library of fonts from which to choose.\n\nTo import Adobe Fonts fonts in your site, simply enter the <strong>Adobe Fonts API Token</strong> in settings page and you are all set.",
				],
				'hosted'    => [
					'name'        => 'Self-Hosted Fonts',
					'description' => "We have made it easier to upload web font formats such WOFF2, WOFF, TTF, EOT and SVG.\n\nFor better support you can upload all file formats, however WOFF2 is enough for modern browsers.\n\nThis method also complies with GDPR regulations by hosting the font on your website rather than fetching from external sources.",
				],
				'external'  => [
					'name'        => 'External Fonts',
					'description' => "If you can't find the right font from above sources then Custom Fonts got covered you.\n\nTo import a custom font, simply enter the stylesheet URL that includes @font-face's and specify font variant names.\n\nThis font type is suitable for services that provide stylesheet URL only and not the web fonts individually.",
				],
				'system'    => [
					'name'        => 'System Fonts',
					'description' => 'A System Font is a font which is compatible with the respective operating system. What that means is that when a website loads, the font doesn’t have to be downloaded by the browser.

They are used by Weather.com, GitHub, Bootstrap, Medium, Ghost, Booking.com and even this WordPress dashboard. This can help reduce the overall page weight on your website. While this is not huge, remember every little optimization you make adds up to a speedy website.',
				],
			]
		);
	}

	/**
	 * Get font source.
	 *
	 * @param string $font_source_id
	 *
	 * @return array|null
	 */
	public static function get_font_source( $font_source_id ) {
		return kalium_get_array_key( self::get_font_sources(), $font_source_id );
	}

	/**
	 * Get base selectors.
	 *
	 * @return array
	 */
	public static function get_base_selectors() {
		return apply_filters(
			'typolab_base_selectors',
			[

				// Sitewide
				'sitewide'   => [
					'name'        => 'Sitewide',
					'description' => 'Applied as default font for all elements in the page.',
					'selectors'   => TypoLab_CSS_Selectors::$SELECTORS_SITEWIDE,
					'css_vars'    => [
						'font-family' => 'body-font-family',
						'font-size'   => 'body-font-size',
					],
					'selected'    => true,
				],

				// Headings
				'headings'   => [
					'name'        => 'Headings',
					'description' => 'Applied on all title headings: h1, h2, h3, h4, h5, h6.',
					'selectors'   => TypoLab_CSS_Selectors::group(
						TypoLab_CSS_Selectors::$SELECTORS_H1,
						TypoLab_CSS_Selectors::$SELECTORS_H2,
						TypoLab_CSS_Selectors::$SELECTORS_H3,
						TypoLab_CSS_Selectors::$SELECTORS_H4,
						TypoLab_CSS_Selectors::$SELECTORS_H5,
						TypoLab_CSS_Selectors::$SELECTORS_H6
					),
					'css_vars'    => true,
				],

				// Paragraphs
				'paragraphs' => [
					'name'        => 'Paragraphs',
					'description' => 'Applied all over the site on paragraphs and plain text.',
					'selectors'   => TypoLab_CSS_Selectors::$SELECTORS_PARAGRAPHS,
				],
			]
		);
	}

	/**
	 * Get single base selector.
	 *
	 * @param string $id
	 *
	 * @return array|null
	 */
	public static function get_base_selector( $id ) {
		foreach ( self::get_base_selectors() as $base_selector_id => $base_selector ) {
			if ( $base_selector_id === $id ) {
				return $base_selector;
			}
		}

		return null;
	}

	/**
	 * Get predefined selectors for use in custom selectors.
	 *
	 * @return array
	 */
	public static function get_predefined_selectors() {
		return apply_filters(
			'typolab_predefined_selectors',
			[
				'root'        => [
					'name'      => ':root',
					'selectors' => TypoLab_CSS_Selectors::$SELECTORS_ROOT,
				],
				'paragraphs'  => [
					'name'      => 'Paragraphs',
					'selectors' => TypoLab_CSS_Selectors::$SELECTORS_PARAGRAPHS,
				],
				'headings'    => [
					'name'      => 'Headings',
					'selectors' => TypoLab_CSS_Selectors::group(
						TypoLab_CSS_Selectors::$SELECTORS_H1,
						TypoLab_CSS_Selectors::$SELECTORS_H2,
						TypoLab_CSS_Selectors::$SELECTORS_H3,
						TypoLab_CSS_Selectors::$SELECTORS_H4,
						TypoLab_CSS_Selectors::$SELECTORS_H5,
						TypoLab_CSS_Selectors::$SELECTORS_H6
					),
					'css_vars'  => true,
				],
				'bold'        => [
					'name'      => 'Bold',
					'selectors' => TypoLab_CSS_Selectors::$SELECTORS_BOLD,
				],
				'blockquotes' => [
					'name'      => 'Blockquotes',
					'selectors' => TypoLab_CSS_Selectors::$SELECTORS_BLOCKQUOTE,
				],
				'form_inputs' => [
					'name'      => 'Form Inputs',
					'selectors' => TypoLab_CSS_Selectors::$SELECTORS_FORM_INPUTS,
				],
			]
		);
	}

	/**
	 * Get single predefined selector.
	 *
	 * @param string $id
	 *
	 * @return array|null
	 */
	public static function get_predefined_selector( $id ) {
		foreach ( self::get_predefined_selectors() as $selector_id => $selector ) {
			if ( $selector_id === $id ) {
				return $selector;
			}
		}

		return null;
	}

	/**
	 * Get font appearance groups.
	 *
	 * @return array
	 */
	public static function get_font_appearance_groups() {
		return apply_filters(
			'typolab_font_appearance_groups',
			[
				// Base font group
				[
					'id'       => 'base-font',
					'name'     => 'Base Font',
					'elements' => [
						'body' => [
							'name'      => 'Base Font',
							'selectors' => TypoLab_CSS_Selectors::$SELECTORS_BASE,
							'css_vars'  => [
								'font-size'   => 'body-font-size',
								'line-height' => 'body-line-height',
							],
						],
					],
				],

				// Headings group
				[
					'id'       => 'headings',
					'name'     => 'Headings',
					'elements' => [
						'h1' => [
							'name'      => 'H1',
							'selectors' => TypoLab_CSS_Selectors::$SELECTORS_H1,
							'css_vars'  => true,
						],
						'h2' => [
							'name'      => 'H2',
							'selectors' => TypoLab_CSS_Selectors::$SELECTORS_H2,
							'css_vars'  => true,
						],
						'h3' => [
							'name'      => 'H3',
							'selectors' => TypoLab_CSS_Selectors::$SELECTORS_H3,
							'css_vars'  => true,
						],
						'h4' => [
							'name'      => 'H4',
							'selectors' => TypoLab_CSS_Selectors::$SELECTORS_H4,
							'css_vars'  => true,
						],
						'h5' => [
							'name'      => 'H5',
							'selectors' => TypoLab_CSS_Selectors::$SELECTORS_H5,
							'css_vars'  => true,
						],
						'h6' => [
							'name'      => 'H6',
							'selectors' => TypoLab_CSS_Selectors::$SELECTORS_H6,
							'css_vars'  => true,
						],
					],
				],

				// Header group
				[
					'id'       => 'header',
					'name'     => 'Header',
					'elements' => [
						'standard_menu'     => [
							'name'      => 'Menu',
							'selectors' => TypoLab_CSS_Selectors::$SELECTORS_MENU_STANDARD,
						],
						'standard_menu_sub' => [
							'name'      => 'Sub Menu',
							'selectors' => TypoLab_CSS_Selectors::$SELECTORS_MENU_STANDARD_SUB,
						],
						'header_logo'       => [
							'name'      => 'Logo',
							'selectors' => TypoLab_CSS_Selectors::$SELECTORS_HEADER_LOGO,
						],
						'header_top_bar'    => [
							'name'      => 'Top Bar',
							'selectors' => TypoLab_CSS_Selectors::$SELECTORS_HEADER_TOP_BAR,
						],
						'header_mini_cart'  => [
							'name'      => 'Mini Cart',
							'selectors' => TypoLab_CSS_Selectors::$SELECTORS_HEADER_MINI_CART,
						],
					],
				],

				// Fullscreen Menu group
				[
					'id'       => 'fullscreen-menu',
					'name'     => 'Header: Fullscreen',
					'elements' => [
						'fullscreen_menu'        => [
							'name'      => 'Menu',
							'selectors' => TypoLab_CSS_Selectors::$SELECTORS_MENU_FULLSCREEN,
						],
						'fullscreen_menu_sub'    => [
							'name'      => 'Sub Menu',
							'selectors' => TypoLab_CSS_Selectors::$SELECTORS_MENU_FULLSCREEN_SUB,
						],
						'fullscreen_menu_footer' => [
							'name'      => 'Fullscreen Footer',
							'selectors' => TypoLab_CSS_Selectors::$SELECTORS_MENU_FULLSCREEN_FOOTER,
						],
					],
				],

				// Menu: Off-Canvas Top group
				[
					'id'       => 'top-menu',
					'name'     => 'Header: Off-Canvas Top',
					'elements' => [
						'top_menu'                 => [
							'name'      => 'Menu',
							'selectors' => TypoLab_CSS_Selectors::$SELECTORS_MENU_TOP,
						],
						'top_menu_sub'             => [
							'name'      => 'Sub Menu',
							'selectors' => TypoLab_CSS_Selectors::$SELECTORS_MENU_TOP_SUB,
						],
						'top_menu_widgets_title'   => [
							'name'      => 'Widgets Title',
							'selectors' => TypoLab_CSS_Selectors::$SELECTORS_MENU_TOP_WIDGETS_TITLE,
						],
						'top_menu_widgets_content' => [
							'name'      => 'Widgets Content',
							'selectors' => TypoLab_CSS_Selectors::$SELECTORS_MENU_TOP_WIDGETS_CONTENT,
						],
					],
				],

				// Sidebar Menu group
				[
					'id'       => 'sidebar-menu',
					'name'     => 'Header: Off-Canvas Side',
					'elements' => [
						'sidebar_menu'                 => [
							'name'      => 'Menu',
							'selectors' => TypoLab_CSS_Selectors::$SELECTORS_MENU_SIDEBAR,
						],
						'sidebar_menu_sub'             => [
							'name'      => 'Sub Menu',
							'selectors' => TypoLab_CSS_Selectors::$SELECTORS_MENU_SIDEBAR_SUB,
						],
						'sidebar_menu_widgets_title'   => [
							'name'      => 'Widgets Title',
							'selectors' => TypoLab_CSS_Selectors::$SELECTORS_MENU_SIDEBAR_WIDGETS_TITLE,
						],
						'sidebar_menu_widgets_content' => [
							'name'      => 'Widgets Content',
							'selectors' => TypoLab_CSS_Selectors::$SELECTORS_MENU_SIDEBAR_WIDGETS_CONTENT,
						],
					],
				],

				// Mobile Menu group
				[
					'id'       => 'mobile-menu',
					'name'     => 'Header: Mobile Menu',
					'elements' => [
						'mobile_menu' => [
							'name'      => 'Menu',
							'selectors' => TypoLab_CSS_Selectors::$SELECTORS_MENU_MOBILE,
						],
					],
				],

				// Page titles group
				[
					'id'       => 'page-title',
					'name'     => 'Page Title',
					'elements' => [
						'page_title'       => [
							'name'      => 'Page Title',
							'selectors' => TypoLab_CSS_Selectors::$SELECTORS_PAGE_TITLE_TITLE,
						],
						'page_description' => [
							'name'      => 'Page Description',
							'selectors' => TypoLab_CSS_Selectors::$SELECTORS_PAGE_TITLE_DESCRIPTION,
						],
						'breadcrumbs'      => [
							'name'      => 'Breadcrumbs',
							'selectors' => TypoLab_CSS_Selectors::$SELECTORS_PAGE_TITLE_BREADCRUMBS,
						],
					],
				],

				// Sidebar group
				[
					'id'       => 'sidebar',
					'name'     => 'Sidebars',
					'elements' => [
						'sidebar_widgets_title'   => [
							'name'      => 'Widgets Title',
							'selectors' => TypoLab_CSS_Selectors::$SELECTORS_SIDEBAR_WIDGETS_TITLE,
						],
						'sidebar_widgets_content' => [
							'name'      => 'Widgets Content',
							'selectors' => TypoLab_CSS_Selectors::$SELECTORS_SIDEBAR_WIDGETS_CONTENT,
						],
					],
				],

				// Button group
				[
					'id'       => 'button',
					'name'     => 'Button',
					'elements' => [
						'button'    => [
							'name'      => 'Standard',
							'selectors' => TypoLab_CSS_Selectors::$SELECTORS_BUTTON,
							'css_vars'  => [
								'font-size' => 'button-font-size',
							],
						],

						'button-xs' => [
							'name'      => 'Extra Small',
							'selectors' => TypoLab_CSS_Selectors::$SELECTORS_BUTTON_XS,
							'css_vars'  => [
								'font-size' => 'button-font-size',
							],
						],

						'button-s'  => [
							'name'      => 'Small',
							'selectors' => TypoLab_CSS_Selectors::$SELECTORS_BUTTON_S,
							'css_vars'  => [
								'font-size' => 'button-font-size',
							],
						],

						'button-l'  => [
							'name'      => 'Large',
							'selectors' => TypoLab_CSS_Selectors::$SELECTORS_BUTTON_L,
							'css_vars'  => [
								'font-size' => 'button-font-size',
							],
						],

						'button-xl' => [
							'name'      => 'Extra Large',
							'selectors' => TypoLab_CSS_Selectors::$SELECTORS_BUTTON_XL,
							'css_vars'  => [
								'font-size' => 'button-font-size',
							],
						],
					],
				],

				// Footer group
				[
					'id'       => 'footer',
					'name'     => 'Footer',
					'elements' => [
						'footer_widgets_title'   => [
							'name'      => 'Widgets Title',
							'selectors' => TypoLab_CSS_Selectors::$SELECTORS_FOOTER_WIDGETS_TITLE,
						],
						'footer_widgets_content' => [
							'name'      => 'Widgets Content',
							'selectors' => TypoLab_CSS_Selectors::$SELECTORS_FOOTER_WIDGETS_CONTENT,
						],
					],
				],

				// Portfolio group
				[
					'id'       => 'portfolio',
					'name'     => 'Portfolio Page',
					'elements' => [
						'portfolio_item_title'      => [
							'name'      => 'Card title',
							'selectors' => TypoLab_CSS_Selectors::$SELECTORS_PORTFOLIO_ITEM_TITLE,
						],
						'portfolio_item_categories' => [
							'name'      => 'Card subtitle',
							'selectors' => TypoLab_CSS_Selectors::$SELECTORS_PORTFOLIO_ITEM_SUBTITLE_CATEGORY,
						],
						'portfolio_category_filter' => [
							'name'      => 'Category Filter',
							'selectors' => TypoLab_CSS_Selectors::$SELECTORS_PORTFOLIO_CATEGORY_FILTER,
						],
					],
				],

				// Project group
				[
					'id'       => 'portfolio-project',
					'name'     => 'Project Page',
					'elements' => [
						'portfolio_item_title_single' => [
							'name'      => 'Title',
							'selectors' => TypoLab_CSS_Selectors::$SELECTORS_PORTFOLIO_ITEM_TITLE_SINGLE,
						],
						'portfolio_item_sub_titles'   => [
							'name'      => 'Sub Title',
							'selectors' => TypoLab_CSS_Selectors::$SELECTORS_PORTFOLIO_ITEM_SUB_TITLES,
						],
						'portfolio_item_content'      => [
							'name'      => 'Description',
							'selectors' => TypoLab_CSS_Selectors::$SELECTORS_PORTFOLIO_ITEM_CONTENT,
						],
						'portfolio_checklist_title'   => [
							'name'      => 'Checklist title',
							'selectors' => TypoLab_CSS_Selectors::$SELECTORS_PORTFOLIO_ITEM_CHECKLIST_TITLES,
						],
						'portfolio_navigation'        => [
							'name'      => 'Navigation',
							'selectors' => TypoLab_CSS_Selectors::$SELECTORS_PORTFOLIO_ITEM_NAVIGATION,
						],
					],
				],

				// Shop group
				[
					'id'       => 'shop',
					'name'     => 'Shop Page',
					'elements' => [
						'shop_card_title'       => [
							'name'      => 'Card Title',
							'selectors' => TypoLab_CSS_Selectors::$SELECTORS_SHOP_CARD_TITLE,
						],
						'shop_card_price'       => [
							'name'      => 'Card Price',
							'selectors' => TypoLab_CSS_Selectors::$SELECTORS_SHOP_CARD_PRICE,
						],
						'shop_card_description' => [
							'name'      => 'Card Description',
							'selectors' => TypoLab_CSS_Selectors::$SELECTORS_SHOP_CARD_DESCRIPTION,
						],
						'shop_category_title'   => [
							'name'      => 'Category Title',
							'selectors' => TypoLab_CSS_Selectors::$SELECTORS_SHOP_CATEGORY_TITLE,
						],
					],
				],

				// Shop Product group
				[
					'id'       => 'shop-product',
					'name'     => 'Product Page',
					'elements' => [
						'product_title'         => [
							'name'      => 'Title',
							'selectors' => TypoLab_CSS_Selectors::$SELECTORS_SHOP_PRODUCT_PRODUCT_TITLE,
						],
						'product_price'         => [
							'name'      => 'Price',
							'selectors' => TypoLab_CSS_Selectors::$SELECTORS_SHOP_PRODUCT_PRODUCT_PRICE,
						],
						'product_related_title' => [
							'name'      => 'Related Title',
							'selectors' => TypoLab_CSS_Selectors::$SELECTORS_SHOP_PRODUCT_PRODUCT_RELATED_TITLE,
						],
						'product_tabs_content'  => [
							'name'      => 'Tabs',
							'selectors' => TypoLab_CSS_Selectors::$SELECTORS_SHOP_PRODUCT_PRODUCT_TABS,
						],
						'product_tabs_titles'   => [
							'name'      => 'Tabs Titles',
							'selectors' => TypoLab_CSS_Selectors::$SELECTORS_SHOP_PRODUCT_PRODUCT_TABS_TITLE,
						],
					],
				],

				// Blog group
				[
					'id'       => 'blog',
					'name'     => 'Blog Page',
					'elements' => [
						'blog_post_title'   => [
							'name'      => 'Card Title',
							'selectors' => TypoLab_CSS_Selectors::$SELECTORS_BLOG_POST_TITLE,
						],
						'blog_post_excerpt' => [
							'name'      => 'Card Excerpt',
							'selectors' => TypoLab_CSS_Selectors::$SELECTORS_BLOG_POST_EXCERPT,
						],
						'blog_post_meta'    => [
							'name'      => 'Card Meta',
							'selectors' => TypoLab_CSS_Selectors::$SELECTORS_BLOG_POST_META,
						],
					],
				],

				// Blog Post group
				[
					'id'       => 'blog-post',
					'name'     => 'Post Page',
					'elements' => [
						'blog_post_title_single'           => [
							'name'      => 'Title',
							'selectors' => TypoLab_CSS_Selectors::$SELECTORS_BLOG_POST_TITLE_SINGLE,
						],
						'blog_post_meta_single'            => [
							'name'      => 'Meta',
							'selectors' => TypoLab_CSS_Selectors::$SELECTORS_BLOG_POST_TITLE_META_SINGLE,
						],
						'blog_post_author_box_title'       => [
							'name'      => 'Author Box Title',
							'selectors' => TypoLab_CSS_Selectors::$SELECTORS_BLOG_POST_AUTHOR_BOX_TITLE,
						],
						'blog_post_author_box_description' => [
							'name'      => 'Author Box Description',
							'selectors' => TypoLab_CSS_Selectors::$SELECTORS_BLOG_POST_AUTHOR_BOX_DESCRIPTION,
						],
						'blog_post_related_title'          => [
							'name'      => 'Related Posts Title',
							'selectors' => TypoLab_CSS_Selectors::$SELECTORS_BLOG_POST_RELATED_TITLE,
						],
						'blog_post_post_navigation'        => [
							'name'      => 'Post Navigation',
							'selectors' => TypoLab_CSS_Selectors::$SELECTORS_BLOG_POST_NAVIGATION,
						],
						'blog_post_comments_title'         => [
							'name'      => 'Comments Title',
							'selectors' => TypoLab_CSS_Selectors::$SELECTORS_BLOG_POST_COMMENTS_TITLE,
						],
						'blog_post_commenter_name'         => [
							'name'      => 'Commenter Name',
							'selectors' => TypoLab_CSS_Selectors::$SELECTORS_BLOG_POST_COMMENTER_NAME,
						],
					],
				],

				// Search Results group
				[
					'id'       => 'search-results',
					'name'     => 'Search Results',
					'elements' => [
						'search_results_title'       => [
							'name'      => 'Title',
							'selectors' => TypoLab_CSS_Selectors::$SELECTORS_SEARCH_RESULTS_TITLE,
						],
						'search_results_description' => [
							'name'      => 'Description',
							'selectors' => TypoLab_CSS_Selectors::$SELECTORS_SEARCH_RESULTS_DESCRIPTION,
						],
					],
				],
			]
		);
	}

	/**
	 * Get font appearance element.
	 *
	 * @param string $group_id
	 * @param string $id
	 *
	 * @return array|null
	 */
	public static function get_font_appearance_element( $group_id, $id ) {
		foreach ( self::get_font_appearance_groups() as $group ) {
			if ( $group['id'] === $group_id ) {
				foreach ( $group['elements'] as $element_id => $element ) {
					if ( $element_id === $id ) {
						return $element;
					}
				}
			}
		}

		return null;
	}

	/**
	 * Get responsive breakpoints.
	 *
	 * @return array
	 */
	public static function get_responsive_breakpoints() {
		return apply_filters(
			'typolab_responsive_breakpoints',
			[

				// General
				'general' => [
					'name'     => 'General',
					'icon'     => 'kalium-admin-icon-device-desktop',
					'min_size' => null,
					'max_size' => null,
					'default'  => true,
				],

				// Tablet
				'tablet'  => [
					'name'     => 'Tablet',
					'icon'     => 'kalium-admin-icon-device-tablet',
					'min_size' => null,
					'max_size' => 992,
					'inherit'  => 'general',
				],

				// Mobile
				'mobile'  => [
					'name'     => 'Mobile',
					'icon'     => 'kalium-admin-icon-device-phone',
					'min_size' => null,
					'max_size' => 768,
					'inherit'  => 'tablet',
				],
			]
		);
	}

	/**
	 * Get single responsive breakpoint.
	 *
	 * @param string $id
	 *
	 * @return array|null
	 */
	public static function get_responsive_breakpoint( $id ) {
		foreach ( self::get_responsive_breakpoints() as $breakpoint_id => $responsive_breakpoint ) {
			if ( $id === $breakpoint_id ) {
				return $responsive_breakpoint;
			}
		}

		return null;
	}

	/**
	 * Get font units.
	 *
	 * @return array
	 */
	public static function get_units() {
		return apply_filters(
			'typolab_units',
			[
				'px'  => 'PX',
				'em'  => 'EM',
				'rem' => 'REM',
				'pt'  => 'PT',
				'vw'  => 'VW',
				'%'   => '%',
			]
		);
	}
}
