<?php
/**
 * Kalium WordPress Theme
 *
 * Portfolio widget.
 *
 * @author Laborator
 * @link   https://kaliumtheme.com
 */
if ( ! defined( 'ABSPATH' ) ) {
	exit; // Direct access not allowed.
}

class WPBakeryShortCode_Lab_Portfolio_Items extends WPBakeryShortCode {

	/**
	 * Register widget.
	 */
	public static function register_widget() {
		if ( ! kalium()->portfolio->is_enabled() ) {
			return;
		}

		if ( kalium()->is->ajax_action( 'vc_edit_form' ) && 'lab_portfolio_items' === kalium()->request->request( 'tag' ) ) {
			$portfolio_categories = wp_list_pluck(
				get_terms(
					[
						'taxonomy' => 'portfolio_category',
					]
				),
				'slug',
				'name'
			);
		}

		vc_map(
			[
				'base'        => 'lab_portfolio_items',
				'name'        => 'Portfolio',
				'description' => 'Show portfolio',
				'category'    => [ 'Laborator', 'Portfolio' ],
				'icon'        => kalium()->file_url( 'includes/wpbakery/assets/images/portfolio.svg' ),
				'params'      => [
					[
						'type'        => 'loop',
						'heading'     => 'Portfolio',
						'param_name'  => 'portfolio_query',
						'settings'    => [
							'size'      => [
								'hidden' => false,
								'value'  => 4 * 3,
							],
							'order_by'  => [ 'value' => 'date' ],
							'post_type' => [
								'value'  => 'portfolio',
								'hidden' => false,
							],
						],
						'description' => 'Create WordPress loop, to populate content from your site.',
					],
					[
						'type'        => 'dropdown',
						'heading'     => 'Title tag',
						'param_name'  => 'title_tag',
						'std'         => 'h2',
						'value'       => [
							'H1' => 'h1',
							'H2' => 'h2',
							'H3' => 'h3',
							'H4' => 'h4',
						],
						'description' => 'Select title tag for widget title.',
					],
					[
						'type'        => 'textfield',
						'heading'     => 'Title',
						'param_name'  => 'title',
						'admin_label' => true,
						'value'       => '',
						'description' => 'Main title of this widget. (Optional)',
					],
					[
						'type'        => 'textarea',
						'heading'     => 'Description',
						'param_name'  => 'description',
						'value'       => '',
						'description' => 'Description under main portfolio title. (Optional)',
					],
					[
						'type'        => 'dropdown',
						'heading'     => 'Category Filter',
						'param_name'  => 'category_filter',
						'value'       => [
							'Yes' => 'yes',
							'No'  => 'no',
						],
						'description' => 'Show category filter above the projects.',
					],
					[
						'type'        => 'dropdown',
						'heading'     => 'Default filter category',
						'param_name'  => 'default_filter_category',
						'value'       => array_merge(
							[
								'Default (All)' => 'default',
							],
							$portfolio_categories ?? []
						),
						'description' => 'Set default category to filter projects at first page load.',
						'dependency'  => [
							'element' => 'category_filter',
							'value'   => [ 'yes' ],
						],
					],
					[
						'type'       => 'checkbox',
						'heading'    => 'Hide "All" filter link from portfolio',
						'param_name' => 'filter_category_hide_all',
						'value'      => [
							'Yes' => 'yes',
						],
						'dependency' => [
							'element'            => 'default_filter_category',
							'value_not_equal_to' => [ 'default' ],
						],
					],
					[
						'group'       => 'Layout',
						'type'        => 'dropdown',
						'heading'     => 'Portfolio Type',
						'admin_label' => true,
						'param_name'  => 'portfolio_type',
						'std'         => 'type-1',
						'value'       => [
							'Thumbnails with Visible Titles' => 'type-1',
							'Thumbnails with Titles Inside'  => 'type-2',
						],
						'description' => 'Select portfolio type to show items.',
					],
					[
						'group'       => 'Layout',
						'type'        => 'dropdown',
						'heading'     => 'Columns',
						'admin_label' => true,
						'param_name'  => 'columns',
						'std'         => 'inherit',
						'value'       => [
							'Inherit from Theme Options' => 'inherit',
							'1 Item per Row'             => 1,
							'2 Items per Row'            => 2,
							'3 Items per Row'            => 3,
							'4 Items per Row'            => 4,
							'5 Items per Row'            => 5,
							'6 Items per Row'            => 6,
						],
						'description' => 'Number of columns to show projects.',
					],
					[
						'group'       => 'Layout',
						'type'        => 'dropdown',
						'heading'     => 'Reveal Effect',
						'param_name'  => 'reveal_effect',
						'std'         => 'inherit',
						'value'       => [
							'Inherit from Theme Options'  => 'inherit',
							'None'                        => 'none',
							'Fade'                        => 'fade',
							'Slide and Fade'              => 'slidenfade',
							'Zoom In'                     => 'zoom',
							'Fade (one by one)'           => 'fade-one',
							'Slide and Fade (one by one)' => 'slidenfade-one',
							'Zoom In (one by one)'        => 'zoom-one',
						],
						'description' => 'Reveal effect for projects.',
					],
					[
						'group'       => 'Layout',
						'type'        => 'dropdown',
						'heading'     => 'Item Spacing',
						'param_name'  => 'portfolio_spacing',
						'description' => 'Spacing between projects.',
						'std'         => 'inherit',
						'value'       => [
							'Inherit from Theme Options' => 'inherit',
							'Yes'                        => 'yes',
							'No'                         => 'no',
						],
						'dependency'  => [
							'element' => 'portfolio_type',
							'value'   => [ 'type-2' ],
						],
					],
					[
						'group'       => 'Layout',
						'type'        => 'dropdown',
						'heading'     => 'Dynamic Image Height',
						'param_name'  => 'dynamic_image_height',
						'description' => 'Use proportional image height for each item.',
						'std'         => 'no',
						'value'       => [
							'Yes' => 'yes',
							'No'  => 'no',
						],
					],
					[
						'group'       => 'Layout',
						'type'        => 'textfield',
						'heading'     => 'Image Size',
						'param_name'  => 'image_size',
						'description' => 'Enter custom image size (Example: "thumbnail", "medium", "large", "full" or other sizes defined by theme). Alternatively enter size in pixels (Example: 200x100 (Width x Height)).',
					],
					[
						'group'       => 'Layout',
						'type'        => 'dropdown',
						'heading'     => 'Title and Filter Container',
						'param_name'  => 'portfolio_full_width_title_container',
						'description' => 'Include title and filter within container.',
						'std'         => 'yes',
						'value'       => [
							'Inherit from Theme Options' => 'inherit',
							'Yes'                        => 'yes',
							'No'                         => 'no',
						],
						'dependency'  => [
							'element' => 'portfolio_full_width',
							'value'   => [ 'yes', 'no' ],
						],
					],
					[
						'group'       => 'Layout',
						'type'        => 'dropdown',
						'heading'     => 'Full-width Container',
						'param_name'  => 'portfolio_full_width',
						'description' => 'Extend portfolio container to the browser edge. <br><small>Note: If you  use full-width container, you need to set this VC row container to Full width as well.</small>',
						'std'         => 'inherit',
						'value'       => [
							'Inherit from Theme Options' => 'inherit',
							'Yes'                        => 'yes',
							'No'                         => 'no',
						],
					],
					[
						'group'       => 'Pagination',
						'type'        => 'dropdown',
						'heading'     => 'Pagination Type',
						'param_name'  => 'pagination_type',
						'description' => 'Select pagination type to use with this widget.',
						'std'         => 'static',
						'value'       => [
							'No "Show More" button'     => 'hide',
							'Static "Show More" button' => 'static',
							'Endless Pagination'        => 'endless',
						],
					],
					[
						'group'       => 'Pagination',
						'type'        => 'vc_link',
						'heading'     => 'More Link',
						'param_name'  => 'more_link',
						'value'       => '',
						'description' => 'This will show "More" button in the end of projects.',
						'dependency'  => [
							'element' => 'pagination_type',
							'value'   => [ 'static' ],
						],
					],
					[
						'group'      => 'Pagination',
						'type'       => 'checkbox',
						'heading'    => 'Auto Reveal',
						'param_name' => 'endless_auto_reveal',
						'value'      => [
							'Yes' => 'yes',
						],
						'dependency' => [
							'element' => 'pagination_type',
							'value'   => [ 'endless' ],
						],
					],
					[
						'group'      => 'Pagination',
						'type'       => 'textfield',
						'heading'    => 'Show more button text',
						'param_name' => 'endless_show_more_button_text',
						'value'      => 'Show More',
						'dependency' => [
							'element' => 'pagination_type',
							'value'   => [ 'endless' ],
						],
					],
					[
						'group'      => 'Pagination',
						'type'       => 'textfield',
						'heading'    => 'No more items to show text',
						'param_name' => 'endless_no_more_items_button_text',
						'value'      => 'No more projects to show',
						'dependency' => [
							'element' => 'pagination_type',
							'value'   => [ 'endless' ],
						],
					],
					[
						'type'        => 'textfield',
						'heading'     => 'Extra class name',
						'param_name'  => 'el_class',
						'description' => 'If you wish to style particular content element differently, then use this field to add a class name and then refer to it in your css file.',
					],
					[
						'type'       => 'css_editor',
						'heading'    => 'Css',
						'param_name' => 'css',
						'group'      => 'Design options',
					],
				],
			]
		);
	}

	/**
	 * Shortcode content.
	 *
	 * @param array  $atts
	 * @param string $content
	 *
	 * @return string
	 */
	public function content( $atts, $content = '' ) {
		static $instance_id = 0;

		// Portfolio is not enabled
		if ( ! kalium()->portfolio->is_enabled() ) {
			return '';
		}

		// Attributes
		$atts = vc_map_get_attributes( $this->getShortcode(), $atts );

		// Set loop options
		$custom_loop_options = [
			'layout'     => $atts['portfolio_type'],
			'url'        => get_permalink( get_queried_object() ),
			'query_vars' => array_merge(
				[
					'post_type' => kalium_get_portfolio_loop_option( 'post_type' ),
				],
				kalium_wpb_build_query_args( $atts['portfolio_query'] )
			),
		];

		// Execute query
		$custom_loop_options['query'] = new WP_Query(
			kalium_get_portfolio_loop_query_args(
				$custom_loop_options['query_vars'],
				false
			)
		);

		// Heading options
		$custom_loop_options['heading']['title']['tag_name'] = $atts['title_tag'];
		$custom_loop_options['heading']['title']['text']     = $atts['title'];
		$custom_loop_options['heading']['title']['content']  = $atts['description'];

		if ( empty( $atts['title'] ) && empty( $atts['description'] ) ) {
			$custom_loop_options['heading']['title']['visible'] = false;
		}

		// Category filter
		$custom_loop_options['filtering']['enabled'] = kalium_validate_boolean( $atts['category_filter'] );

		// Default category
		$default_category = $atts['default_filter_category'];
		$hide_all         = $atts['filter_category_hide_all'];

		if ( 'default' !== $default_category ) {
			$custom_loop_options['filtering']['current']['portfolio_category'] = $default_category;

			if ( $hide_all ) {
				$custom_loop_options['filtering']['reset'] = false;
			}
		}

		// Columns
		$columns = $atts['columns'];

		if ( is_numeric( $columns ) ) {
			$custom_loop_options['grid']['columns'] = array_merge(
				kalium_get_portfolio_loop_option( 'grid/columns' ),
				[
					'desktop' => intval( $columns ),
				]
			);
		}

		// Reveal effect
		if ( 'inherit' !== $atts['reveal_effect'] ) {
			$custom_loop_options['item']['aov']['enabled']          = 'none' !== $atts['reveal_effect'];
			$custom_loop_options['item']['aov']['legacy_animation'] = $atts['reveal_effect'];
			$custom_loop_options['item']['aov']['animation']        = null;
		}

		// Grid spacing
		$grid_spacing = $atts['portfolio_spacing'];

		if ( 'yes' === $grid_spacing ) {
			$custom_loop_options['grid']['gap'] = 30;
		} elseif ( 'no' === $grid_spacing ) {
			$custom_loop_options['grid']['gap'] = 0;
		}

		// Dynamic image height
		$dynamic_image_height = kalium_validate_boolean( $atts['dynamic_image_height'] );

		if ( $atts['image_size'] ) {
			$thumbnail_size = preg_match( '/^\d+x\d+$/', $atts['image_size'] ) ? explode( 'x', $atts['image_size'] ) : $atts['image_size'];
		} else {
			$thumbnail_size = $dynamic_image_height ? 'medium_large' : [ 655, 545 ];
		}

		$custom_loop_options['item']['thumbnail']['size'] = $thumbnail_size;

		if ( $dynamic_image_height ) {
			$custom_loop_options['grid']['masonry'] = [
				'enabled' => true,
				'mode'    => 'packery',
			];
		}

		// Contained title
		$heading_contained = 'yes' === $atts['portfolio_full_width'] && 'yes' === $atts['portfolio_full_width_title_container'];

		// Pagination
		switch ( $atts['pagination_type'] ) {

			// Load more
			case 'endless':
				$custom_loop_options['pagination']['enabled'] = true;
				$custom_loop_options['pagination']['type']    = 'load-more';

				$custom_loop_options['pagination']['load_more']['infinite_scroll'] = kalium_validate_boolean( $atts['endless_auto_reveal'] );
				$custom_loop_options['pagination']['load_more']['view_more_text']  = $atts['endless_show_more_button_text'];
				$custom_loop_options['pagination']['load_more']['last_page_text']  = $atts['endless_no_more_items_button_text'];
				break;

			// Static more link
			case 'static':
				$more_link = vc_build_link( $atts['more_link'] );

				$custom_loop_options['pagination']['type'] = 'more-link';

				$custom_loop_options['pagination']['enabled']             = true;
				$custom_loop_options['pagination']['more_link']['url']    = $more_link['url'];
				$custom_loop_options['pagination']['more_link']['target'] = $more_link['target'];
				$custom_loop_options['pagination']['more_link']['text']   = $more_link['title'];
				break;

			// Hide
			case 'hide':
				$custom_loop_options['pagination']['enabled'] = false;
				break;
		}

		// Element class
		$css       = $atts['css'];
		$el_class  = $atts['el_class'];
		$class     = $this->getExtraClass( $el_class );
		$css_class = apply_filters( VC_SHORTCODE_CUSTOM_CSS_FILTER_TAG, $class, $this->settings['base'], $atts ) . vc_shortcode_custom_css_class( $css, ' ' );

		$classes = [
			'lab-portfolio-items',
			'portfolio',
			$css_class,
		];

		if ( $heading_contained ) {
			$classes[] = 'portfolio--heading-contained';
		}

		// Output template
		ob_start();

		// Init portfolio options
		kalium_init_portfolio_loop_options( 'wpb-' . ++$instance_id, $custom_loop_options );

		// Heading
		kalium_portfolio_loop_heading();

		// Loop before
		kalium_portfolio_loop_before();

		// Portfolio loop
		kalium_portfolio_items_loop();

		// Pagination
		kalium_portfolio_items_pagination();

		// Loop after
		kalium_portfolio_loop_after();

		// Reset portfolio loop options
		kalium_reset_portfolio_loop_options();

		return kalium_render_element(
			[
				'tag_name'   => 'div',
				'attributes' => [
					'class' => $classes,
				],
				'content'    => ob_get_clean(),
			],
			false
		);
	}
}

// Register widget
WPBakeryShortCode_Lab_Portfolio_Items::register_widget();
